/* -*-  Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2; -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "ChromeUtils.h"

#include "mozilla/Base64.h"
#include "mozilla/BasePrincipal.h"
#include "mozJSComponentLoader.h"

namespace mozilla {
namespace dom {

/* static */ void
ChromeUtils::NondeterministicGetWeakMapKeys(GlobalObject& aGlobal,
                                            JS::Handle<JS::Value> aMap,
                                            JS::MutableHandle<JS::Value> aRetval,
                                            ErrorResult& aRv)
{
  if (!aMap.isObject()) {
    aRetval.setUndefined();
  } else {
    JSContext* cx = aGlobal.Context();
    JS::Rooted<JSObject*> objRet(cx);
    JS::Rooted<JSObject*> mapObj(cx, &aMap.toObject());
    if (!JS_NondeterministicGetWeakMapKeys(cx, mapObj, &objRet)) {
      aRv.Throw(NS_ERROR_OUT_OF_MEMORY);
    } else {
      aRetval.set(objRet ? JS::ObjectValue(*objRet) : JS::UndefinedValue());
    }
  }
}

/* static */ void
ChromeUtils::NondeterministicGetWeakSetKeys(GlobalObject& aGlobal,
                                            JS::Handle<JS::Value> aSet,
                                            JS::MutableHandle<JS::Value> aRetval,
                                            ErrorResult& aRv)
{
  if (!aSet.isObject()) {
    aRetval.setUndefined();
  } else {
    JSContext* cx = aGlobal.Context();
    JS::Rooted<JSObject*> objRet(cx);
    JS::Rooted<JSObject*> setObj(cx, &aSet.toObject());
    if (!JS_NondeterministicGetWeakSetKeys(cx, setObj, &objRet)) {
      aRv.Throw(NS_ERROR_OUT_OF_MEMORY);
    } else {
      aRetval.set(objRet ? JS::ObjectValue(*objRet) : JS::UndefinedValue());
    }
  }
}

/* static */ void
ChromeUtils::Base64URLEncode(GlobalObject& aGlobal,
                             const ArrayBufferViewOrArrayBuffer& aSource,
                             const Base64URLEncodeOptions& aOptions,
                             nsACString& aResult,
                             ErrorResult& aRv)
{
  size_t length = 0;
  uint8_t* data = nullptr;
  if (aSource.IsArrayBuffer()) {
    const ArrayBuffer& buffer = aSource.GetAsArrayBuffer();
    buffer.ComputeLengthAndData();
    length = buffer.Length();
    data = buffer.Data();
  } else if (aSource.IsArrayBufferView()) {
    const ArrayBufferView& view = aSource.GetAsArrayBufferView();
    view.ComputeLengthAndData();
    length = view.Length();
    data = view.Data();
  } else {
    MOZ_CRASH("Uninitialized union: expected buffer or view");
  }

  auto paddingPolicy = aOptions.mPad ? Base64URLEncodePaddingPolicy::Include :
                                       Base64URLEncodePaddingPolicy::Omit;
  nsresult rv = mozilla::Base64URLEncode(length, data, paddingPolicy, aResult);
  if (NS_WARN_IF(NS_FAILED(rv))) {
    aResult.Truncate();
    aRv.Throw(rv);
  }
}

/* static */ void
ChromeUtils::Base64URLDecode(GlobalObject& aGlobal,
                             const nsACString& aString,
                             const Base64URLDecodeOptions& aOptions,
                             JS::MutableHandle<JSObject*> aRetval,
                             ErrorResult& aRv)
{
  Base64URLDecodePaddingPolicy paddingPolicy;
  switch (aOptions.mPadding) {
    case Base64URLDecodePadding::Require:
      paddingPolicy = Base64URLDecodePaddingPolicy::Require;
      break;

    case Base64URLDecodePadding::Ignore:
      paddingPolicy = Base64URLDecodePaddingPolicy::Ignore;
      break;

    case Base64URLDecodePadding::Reject:
      paddingPolicy = Base64URLDecodePaddingPolicy::Reject;
      break;

    default:
      aRv.Throw(NS_ERROR_INVALID_ARG);
      return;
  }
  FallibleTArray<uint8_t> data;
  nsresult rv = mozilla::Base64URLDecode(aString, paddingPolicy, data);
  if (NS_WARN_IF(NS_FAILED(rv))) {
    aRv.Throw(rv);
    return;
  }

  JS::Rooted<JSObject*> buffer(aGlobal.Context(),
                               ArrayBuffer::Create(aGlobal.Context(),
                                                   data.Length(),
                                                   data.Elements()));
  if (NS_WARN_IF(!buffer)) {
    aRv.Throw(NS_ERROR_OUT_OF_MEMORY);
    return;
  }
  aRetval.set(buffer);
}

/* static */ void
ChromeUtils::Import(const GlobalObject& aGlobal,
                    const nsAString& aResourceURI,
                    const Optional<JS::Handle<JSObject*>>& aTargetObj,
                    JS::MutableHandle<JSObject*> aRetval,
                    ErrorResult& aRv)
{

  RefPtr<mozJSComponentLoader> moduleloader = mozJSComponentLoader::Get();
  MOZ_ASSERT(moduleloader);

  NS_ConvertUTF16toUTF8 registryLocation(aResourceURI);

  JSContext* cx = aGlobal.Context();
  JS::Rooted<JS::Value> targetObj(cx);
  uint8_t optionalArgc;
  if (aTargetObj.WasPassed()) {
    targetObj.setObjectOrNull(aTargetObj.Value());
    optionalArgc = 1;
  } else {
    targetObj.setUndefined();
    optionalArgc = 0;
  }

  JS::Rooted<JS::Value> retval(cx);
  nsresult rv = moduleloader->ImportInto(registryLocation, targetObj, cx,
                                         optionalArgc, &retval);
  if (NS_FAILED(rv)) {
    aRv.Throw(rv);
    return;
  }

  // Import() on the component loader can return NS_OK while leaving an
  // exception on the JSContext.  Check for that case.
  if (JS_IsExceptionPending(cx)) {
    aRv.NoteJSContextException(cx);
    return;
  }

  // Now we better have an object.
  MOZ_ASSERT(retval.isObject());
  aRetval.set(&retval.toObject());
}

namespace module_getter {
  static const size_t SLOT_ID = 0;
  static const size_t SLOT_URI = 1;

  static bool
  ExtractArgs(JSContext* aCx, JS::CallArgs& aArgs,
              JS::MutableHandle<JSObject*> aCallee,
              JS::MutableHandle<JSObject*> aThisObj,
              JS::MutableHandle<jsid> aId)
  {
    aCallee.set(&aArgs.callee());

    JS::Handle<JS::Value> thisv = aArgs.thisv();
    if (!thisv.isObject()) {
      JS_ReportErrorASCII(aCx, "Invalid target object");
      return false;
    }

    aThisObj.set(&thisv.toObject());

    JS::Rooted<JS::Value> id(aCx, js::GetFunctionNativeReserved(aCallee, SLOT_ID));
    MOZ_ALWAYS_TRUE(JS_ValueToId(aCx, id, aId));
    return true;
  }

  static bool
  ModuleGetter(JSContext* aCx, unsigned aArgc, JS::Value* aVp)
  {
    JS::CallArgs args = JS::CallArgsFromVp(aArgc, aVp);

    JS::Rooted<JSObject*> callee(aCx);
    JS::Rooted<JSObject*> thisObj(aCx);
    JS::Rooted<jsid> id(aCx);
    if (!ExtractArgs(aCx, args, &callee, &thisObj, &id)) {
      return false;
    }

    JS::Rooted<JSString*> moduleURI(
      aCx, js::GetFunctionNativeReserved(callee, SLOT_URI).toString());
    JSAutoByteString bytes;
    if (!bytes.encodeUtf8(aCx, moduleURI)) {
      return false;
    }
    nsDependentCString uri(bytes.ptr());

    RefPtr<mozJSComponentLoader> moduleloader = mozJSComponentLoader::Get();
    MOZ_ASSERT(moduleloader);

    JS::Rooted<JSObject*> moduleGlobal(aCx);
    JS::Rooted<JSObject*> moduleExports(aCx);
    nsresult rv = moduleloader->Import(aCx, uri, &moduleGlobal, &moduleExports);
    if (NS_FAILED(rv)) {
      Throw(aCx, rv);
      return false;
    }

    JS::RootedValue value(aCx);
    {
      JSAutoCompartment ac(aCx, moduleExports);

      if (!JS_GetPropertyById(aCx, moduleExports, id, &value)) {
        return false;
      }
    }

    if (!JS_WrapValue(aCx, &value) ||
        !JS_DefinePropertyById(aCx, thisObj, id, value,
                               JSPROP_ENUMERATE)) {
      return false;
    }

    args.rval().set(value);
    return true;
  }

  static bool
  ModuleSetter(JSContext* aCx, unsigned aArgc, JS::Value* aVp)
  {
    JS::CallArgs args = JS::CallArgsFromVp(aArgc, aVp);

    JS::Rooted<JSObject*> callee(aCx);
    JS::Rooted<JSObject*> thisObj(aCx);
    JS::Rooted<jsid> id(aCx);
    if (!ExtractArgs(aCx, args, &callee, &thisObj, &id)) {
      return false;
    }

    return JS_DefinePropertyById(aCx, thisObj, id, args.get(0),
                                 JSPROP_ENUMERATE);
  }

  static bool
  DefineGetter(JSContext* aCx,
               JS::Handle<JSObject*> aTarget,
               const nsAString& aId,
               const nsAString& aResourceURI)
  {
    JS::RootedValue uri(aCx);
    JS::RootedValue idValue(aCx);
    JS::Rooted<jsid> id(aCx);
    if (!xpc::NonVoidStringToJsval(aCx, aResourceURI, &uri) ||
        !xpc::NonVoidStringToJsval(aCx, aId, &idValue) ||
        !JS_ValueToId(aCx, idValue, &id)) {
      return false;
    }
    idValue = js::IdToValue(id);


    JS::Rooted<JSObject*> getter(aCx, JS_GetFunctionObject(
      js::NewFunctionByIdWithReserved(aCx, ModuleGetter, 0, 0, id)));

    JS::Rooted<JSObject*> setter(aCx, JS_GetFunctionObject(
      js::NewFunctionByIdWithReserved(aCx, ModuleSetter, 0, 0, id)));

    if (!getter || !setter) {
      JS_ReportOutOfMemory(aCx);
      return false;
    }

    js::SetFunctionNativeReserved(getter, SLOT_ID, idValue);
    js::SetFunctionNativeReserved(setter, SLOT_ID, idValue);

    js::SetFunctionNativeReserved(getter, SLOT_URI, uri);

    return JS_DefinePropertyById(aCx, aTarget, id, JS::UndefinedHandleValue,
                                 JSPROP_GETTER | JSPROP_SETTER | JSPROP_ENUMERATE,
                                 JS_DATA_TO_FUNC_PTR(JSNative, getter.get()),
                                 JS_DATA_TO_FUNC_PTR(JSNative, setter.get()));
  }
} // namespace module_getter

/* static */ void
ChromeUtils::DefineModuleGetter(const GlobalObject& global,
                                JS::Handle<JSObject*> target,
                                const nsAString& id,
                                const nsAString& resourceURI,
                                ErrorResult& aRv)
{
  if (!module_getter::DefineGetter(global.Context(), target, id, resourceURI)) {
    aRv.NoteJSContextException(global.Context());
  }
}

/* static */ void
ChromeUtils::OriginAttributesToSuffix(dom::GlobalObject& aGlobal,
                                      const dom::OriginAttributesDictionary& aAttrs,
                                      nsCString& aSuffix)

{
  OriginAttributes attrs(aAttrs);
  attrs.CreateSuffix(aSuffix);
}

/* static */ bool
ChromeUtils::OriginAttributesMatchPattern(dom::GlobalObject& aGlobal,
                                          const dom::OriginAttributesDictionary& aAttrs,
                                          const dom::OriginAttributesPatternDictionary& aPattern)
{
  OriginAttributes attrs(aAttrs);
  OriginAttributesPattern pattern(aPattern);
  return pattern.Matches(attrs);
}

/* static */ void
ChromeUtils::CreateOriginAttributesFromOrigin(dom::GlobalObject& aGlobal,
                                       const nsAString& aOrigin,
                                       dom::OriginAttributesDictionary& aAttrs,
                                       ErrorResult& aRv)
{
  OriginAttributes attrs;
  nsAutoCString suffix;
  if (!attrs.PopulateFromOrigin(NS_ConvertUTF16toUTF8(aOrigin), suffix)) {
    aRv.Throw(NS_ERROR_FAILURE);
    return;
  }
  aAttrs = attrs;
}

/* static */ void
ChromeUtils::FillNonDefaultOriginAttributes(dom::GlobalObject& aGlobal,
                                 const dom::OriginAttributesDictionary& aAttrs,
                                 dom::OriginAttributesDictionary& aNewAttrs)
{
  aNewAttrs = aAttrs;
}


/* static */ bool
ChromeUtils::IsOriginAttributesEqual(dom::GlobalObject& aGlobal,
                                     const dom::OriginAttributesDictionary& aA,
                                     const dom::OriginAttributesDictionary& aB)
{
  return IsOriginAttributesEqual(aA, aB);
}

/* static */ bool
ChromeUtils::IsOriginAttributesEqual(const dom::OriginAttributesDictionary& aA,
                                     const dom::OriginAttributesDictionary& aB)
{
  return aA.mAppId == aB.mAppId &&
         aA.mInIsolatedMozBrowser == aB.mInIsolatedMozBrowser &&
         aA.mUserContextId == aB.mUserContextId &&
         aA.mPrivateBrowsingId == aB.mPrivateBrowsingId;
}

} // namespace dom
} // namespace mozilla
