/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "ksimframe.h"
#include "ksimframe.moc"
#include <themeloader.h>
#include <ksim.h>
#include "themetypes.h"

#include <klocale.h>
#include <kiconloader.h>
#include <kdebug.h>

#include <qpainter.h>

KSim::Frame::Frame(KSim::MainWindow *topLevel, 
   int type, QWidget *parent, const char *name)
   : QWidget (parent, name)
{
  setType(type);
  installEventFilter(this);
  m_topLevel = topLevel;
  m_resizing = false;

  setBackgroundMode(NoBackground);
  configureObject();
}

KSim::Frame::~Frame()
{
}

void KSim::Frame::configureObject(bool repaintWidget)
{
  m_image.load(themeLoader().current().framePixmap(type()));

  switch (type()) {
    case KSim::Types::TopFrame:
      setFrameHeight(themeLoader().current().frameTopHeight());
      break;
    case KSim::Types::BottomFrame:
      setFrameHeight(themeLoader().current().frameBottomHeight());
      break;
    case KSim::Types::LeftFrame:
      setFrameWidth(themeLoader().current().frameLeftWidth());
      break;
    case KSim::Types::RightFrame:
      setFrameWidth(themeLoader().current().frameRightWidth());
      break;
  }

  themeLoader().reColourImage(m_image);
  m_background.convertFromImage(m_image.smoothScale(size()));

  if (repaintWidget)
    update();
}

void KSim::Frame::paintEvent(QPaintEvent *)
{
  bitBlt(this, 0, 0, &m_background);
}

void KSim::Frame::resizeEvent(QResizeEvent *ev)
{
  m_background.convertFromImage(m_image.smoothScale(ev->size()));
}

void KSim::Frame::mousePressEvent(QMouseEvent *e)
{
  if (e->button() != LeftButton || !m_topLevel)
    return;

  m_resizing = true;
  m_origWidth = m_topLevel->width();
  m_origHeight = m_topLevel->height();

  int origLeft = (m_topLevel->rect()).left();
  int origTop = (m_topLevel->rect()).top();
  int origRight = (m_topLevel->rect()).right();
  int origBottom = (m_topLevel->rect()).bottom();

  m_globalTopLeft = m_topLevel->mapToGlobal(QPoint(origLeft, origTop));
  m_globalBottomRight = m_topLevel->mapToGlobal(QPoint(origRight, origBottom));

  m_origPos = e->globalPos();
}

void KSim::Frame::mouseReleaseEvent(QMouseEvent *)
{
  m_resizing = false;
}

void KSim::Frame::mouseMoveEvent(QMouseEvent *e)
{
  // The method used here is this:
  // Make sure all resizing and positioning is done from the original
  // settings to avoid weird problems when you have to move and resize.
  // This is done for resizing from the top and from the left, the other
  // 2 sides dont have this problem, a resize is enough there.
  if (!m_resizing || !m_topLevel)
    return;

  QPoint pos = e->globalPos();
  QPoint vector = m_origPos - pos;

  switch(type()) {
    case KSim::Types::LeftFrame:
      // globalbottomright.x() has the right coord in global coords
      // globaltopleft.y() has the top coord in global coords
      // vector.x() has the change compared to when the mousebutton was pressed
      if ((m_origWidth + vector.x()) >= m_topLevel->minimumWidth())
        m_topLevel->setGeometry(m_globalBottomRight.x() - m_origWidth -
           vector.x(), m_globalTopLeft.y(), m_origWidth + vector.x(), m_origHeight);
      break;
    case KSim::Types::RightFrame:
      // from the right only a resize is needed so resize is easier than setGeometry
      m_topLevel->resize(m_origWidth - vector.x(), m_origHeight);
      break;
  }
}

bool KSim::Frame::eventFilter(QObject *o, QEvent *e)
{
  if ((o == this) && e->type() == QEvent::MouseButtonPress) {
    mousePressEvent(static_cast<QMouseEvent *>(e));
    return true;
  }

  return QWidget::eventFilter(o, e);
}

void KSim::Frame::enterEvent(QEvent *)
{
  if (!m_topLevel)
    return;

  switch(type()) {
    case KSim::Types::LeftFrame:
    case KSim::Types::RightFrame:
      setCursor(sizeHorCursor);
      break;
  }
}

void KSim::Frame::leaveEvent(QEvent *)
{
  if (m_resizing || !m_topLevel)
    return;

  unsetCursor();
}

void KSim::Frame::setFrameHeight(int height)
{
  if (!height) {
    setFixedHeight(m_image.height());
    return;
  }

  setFixedHeight(height);
}

void KSim::Frame::setFrameWidth(int width)
{
  if (!width) {
    setFixedWidth(m_image.width());
    return;
  }

  setFixedWidth(width);
}
