/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>

#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include <libgnome/libgnome.h>

#include <gmime/gmime.h>

#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-mime.h>
#include <pan/base/util-file.h>

/***
**** YENC
***/

#define YENC_MARKER_BEGIN      "=ybegin"
#define YENC_MARKER_BEGIN_LEN  7
#define YENC_MARKER_PART       "=ypart"
#define YENC_MARKER_PART_LEN   6
#define YENC_MARKER_END        "=yend"
#define YENC_MARKER_END_LEN    5
#define YENC_TAG_PART          " part="
#define YENC_TAG_LINE          " line="
#define YENC_TAG_SIZE          " size="
#define YENC_TAG_NAME          " name="
#define YENC_TAG_BEGIN         " begin="
#define YENC_TAG_END           " end="
#define YENC_TAG_PCRC32                " pcrc32="
#define YENC_TAG_CRC32         " crc32="
#define YENC_FULL_LINE_LEN     256
#define YENC_HALF_LINE_LEN     128
#define YENC_ESC_NULL          "=@"
#define YENC_ESC_TAB           "=I"
#define YENC_ESC_LF            "=J"
#define YENC_ESC_CR            "=M"
#define YENC_ESC_ESC           "={"
#define YENC_SHIFT             42
#define YENC_QUOTE_SHIFT       64

static const char*
__yenc_extract_tag_val_char (const char * line, const char *tag)
{
	const char * retval = NULL;
	const char * tmp;
       
	tmp = strstr (line, tag);
	if (tmp != NULL) {
		tmp += strlen (tag);
		if (*tmp != '\0')
			retval = tmp;
	}

	return retval;
}

static guint
__yenc_extract_tag_val_int_base (const char * line,
                                 const char * tag,
                                 int          base)
{
	guint retval = 0;
	const char * tmp;

	tmp = __yenc_extract_tag_val_char (line, tag);

	if (tmp != NULL) {
		char * tail = NULL;
		retval = strtoul (tmp, &tail, base);
		if (tmp == tail)
			retval = 0;
	}

	return retval;
}

static int
__yenc_extract_tag_val_hex_int (const char * line,
                                const char * tag)
{
	return __yenc_extract_tag_val_int_base (line, tag, 16);
}

static int
__yenc_extract_tag_val_int (const char *line,
                            const char *tag)
{
	return __yenc_extract_tag_val_int_base (line, tag, 0);
}


static int
yenc_parse_end_line (const char * b,
                     size_t     * size,
                     int        * part,
                     guint      * pcrc,
                     guint      * crc)
{
	size_t _size = 0;
	int _part = 0;
	guint _pcrc = 0;
	guint _crc = 0;
	const char * pch;

	/* find size */
	pch = __yenc_extract_tag_val_char (b, YENC_TAG_SIZE);
	if (pch!=NULL)
		_size = strtoul(pch, NULL, 10);
	g_return_val_if_fail (_size!=0, -1);

	/* part is optional */
	_part = __yenc_extract_tag_val_int (b, YENC_TAG_PART);
	_pcrc = __yenc_extract_tag_val_hex_int (b, YENC_TAG_PCRC32);
	if (part != 0)
		g_return_val_if_fail( _pcrc != 0, -1 );

	_crc = __yenc_extract_tag_val_hex_int( b, YENC_TAG_CRC32);

	if (size)
		*size = _size;
	if (part)
		*part = _part;
	if (pcrc)
		*pcrc = _pcrc;
	if (crc)
		*crc = _crc;

	return 0;
}

/**
 * yenc_parse_being_line
 * @param line the line to check for "begin [part] line size filename"
 * @param filename if parse is successful, is set with the
 *        starting character of the filename.
 * @param line_len if parse is successful, is set with the line length
 * @param part is parse is successful & set to the cuurent attachements 
 *       part number
 * @return 0 on success, -1 on failure
 */
static int
yenc_parse_begin_line (const char  * b,
                       char       ** file,
                       int         * line_len,
                       int         * attach_size,
                       int         * part)
{
	int ll;
	int part_num;
	int a_sz;
	const char *fname;

	ll = __yenc_extract_tag_val_int (b, YENC_TAG_LINE);
	g_return_val_if_fail (ll != 0, -1);

	// part is optional
	part_num = __yenc_extract_tag_val_int (b, YENC_TAG_PART);
	
	a_sz = __yenc_extract_tag_val_int( b, YENC_TAG_SIZE );
	g_return_val_if_fail( a_sz != 0, -1 );
	
	fname = __yenc_extract_tag_val_char (b, YENC_TAG_NAME);
	g_return_val_if_fail( fname, -1 );

	if (line_len)
		*line_len = ll;
	if (file) {
		const char * pch = strchr (fname, '\n');
		*file = g_strstrip (g_strndup (fname, pch-fname));
	}
	if (part)
		*part = part_num;
	if (attach_size)
		*attach_size = a_sz;

	return 0;
}

/*
 * a =ypart line requires both begin & end offsets. These are the location
 * of the part inside the end file
 */
static int
yenc_parse_part_line (const char * b, int *begin_offset, int *end_offset)
{
	int bg;
	int end;

	bg = __yenc_extract_tag_val_int( b, YENC_TAG_BEGIN );
	if (bg == 0)
		return -1;

	end = __yenc_extract_tag_val_int( b, YENC_TAG_END );
	if (end == 0)
		return -1;

	if (begin_offset)
		*begin_offset = bg;
	if (end_offset)
		*end_offset = end;

	return 0;
}


#if 0
static gboolean
is_yenc_line (const char * line, int len)
{
	if (len != YENC_FULL_LINE_LEN || len != YENC_HALF_LINE_LEN) {
		if (len != YENC_FULL_LINE_LEN + 1 || YENC_HALF_LINE_LEN + 1)
			return FALSE;
	}
	
	// TODO if +1 than line length check for escape chars at the end
	return TRUE;
}
#endif

/**
 * yenc_is_beginning
 * @param line line to test & see if it's the beginning of a yenc block
 * @return true if it is, false otherwise
 */
static gboolean
yenc_is_beginning_line (const gchar * line)
{
	return !strncmp (line, YENC_MARKER_BEGIN, YENC_MARKER_BEGIN_LEN) &&
	       !yenc_parse_begin_line (line, NULL, NULL, NULL, NULL);
}

static gboolean
yenc_is_part_line (const gchar * line)
{
	return !strncmp (line, YENC_MARKER_PART, YENC_MARKER_PART_LEN) &&
	       !yenc_parse_part_line (line, NULL, NULL);
}

/**
 * yenc_is_ending_line
 * @param line line to test & see if it's the end of a yenc block
 * @return true if it is, false otherwise
 */
static gboolean
yenc_is_ending_line (const gchar * line)
{
	return !strncmp (line, YENC_MARKER_END, YENC_MARKER_END_LEN) &&
	       !yenc_parse_end_line (line, NULL, NULL, NULL, NULL);
}


/***
**** UU
***/

/**
 * parse_uu_begin_line
 * @param line the line to check for "begin filename mode"
 * @param filename if parse is successful, is set with the
 *        starting character of the filename.
 * @param mode if parse is successful, mode is set with the
 *        mode requested by the line.
 * @return 0 on success, -1 on failure
 */
int
uu_parse_begin_line (const gchar  * b,
                     const gchar ** file,
                            gint  * mode)
{
	gint m = 0;

	/* skip to the permissions */
	while (*b==' ') ++b;
	if (!isdigit((guchar)*b)) return -1;

	/* parse the permissions */
	while (isdigit((guchar)*b)) {
		m = m*8 + (*b-'0');
		++b;
	};

	/* skip to the filename */
	if (*b!=' ') return -1;
	while (*b==' ') ++b;
	if (*b < ' ') return -1;

	/* return the results */
	if (mode!=NULL) *mode = m;
	if (file!=NULL) *file = b;
	return 0;
}

/**
 * uu_is_beginning
 * @param line line to test & see if it's the beginning of a uu-encoded block
 * @return true if it is, false otherwise
 */
gboolean
uu_is_beginning_line (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='b' || line[0]=='B')
	 	&& (line[1]=='e' || line[1]=='E')
	 	&& (line[2]=='g' || line[2]=='G')
	 	&& (line[3]=='i' || line[3]=='I')
	 	&& (line[4]=='n' || line[4]=='N')
		&& line[5]==' '
		&& !uu_parse_begin_line (line+6, NULL, NULL);
}


/**
 * uu_is_ending
 * @param line line to test & see if it's the end of a uu-encoded block
 * @return true if it is, false otherwise
 */
gboolean
uu_is_ending_line (const gchar * line)
{
	return line!=NULL
	 	&& (line[0]=='e' || line[0]=='E')
	 	&& (line[1]=='n' || line[1]=='N')
	 	&& (line[2]=='d' || line[2]=='D')
	 	&& (line[3]=='\0' || line[3]=='\n' || line[3]==' ');
}

void
uu_get_file_info (const gchar       * begin,
                  gchar            ** setme_filename,
                  gulong            * setme_mode)
{
	gchar * filename;
	const gchar * end;

	g_return_if_fail (uu_is_beginning_line(begin));
	g_return_if_fail (setme_filename != NULL);
	g_return_if_fail (setme_mode != NULL);

	*setme_mode = strtol (begin+6, NULL, 8);
	
	begin += 10;
	end = strchr (begin, '\n');
	filename = g_strndup (begin, end-begin);
	g_strstrip (filename);
	*setme_filename = filename;
}

int
uu_get_char_len (gint octet_len)
{
	gint char_len = (octet_len / 3) * 4;
	switch (octet_len % 3) {
		case 0: break;
		case 1: char_len += 2; break;
		case 2: char_len += 3; break;
	}
	return char_len;
}

gboolean
is_uu_line (const gchar * line, gint len)
{
	gint i;
	gint octet_len;
	gint char_len;

	g_return_val_if_fail (is_nonempty_string(line), FALSE);

	if (*line=='\0' || len<1)
		return FALSE;

	if (len==1 && *line=='`')
		return TRUE;

	/* get octet length */
	octet_len = *line - 0x20;
	if (octet_len > 45)
		return FALSE;

	/* get character length */
	char_len = uu_get_char_len (octet_len);
	if (char_len+1 > len)
		return FALSE;

	/* if there's a lot of noise at the end, be suspicious.
	   This is to keep out lines of nntp-server-generated
	   taglines that have a space as the first character */
	if (char_len+10 < len)
		return FALSE;

	/* make sure each character is in the uuencoded range */
	for (i=0; i<char_len; ++i)
		if (line[i+1]<0x20 || line[i+1]>0x60)
			return FALSE;

	/* looks okay */
	return TRUE;
}

typedef enum
{
	ENC_PLAIN,
	ENC_YENC,
	ENC_UU
}
EncType;
typedef struct
{
	off_t start;
	off_t end;
	char * decoded_filename;

	int y_line_len;
	int y_attach_size;
	int y_part;
	guint y_offset_begin;
	guint y_offset_end;
	guint y_crc;
	guint y_pcrc;
	size_t y_size;

	EncType type;
}
EncTempFilePart;

static void
separate_encoded_parts (GMimeStream * istream,
                        GPtrArray   * appendme_file_parts)
{
	EncTempFilePart * cur = NULL;
	EncType type = ENC_PLAIN;
	GByteArray * line;
	gboolean yenc_looking_for_part_line = FALSE;
	off_t linestart_pos = 0;

	/* sanity clause */
	g_return_if_fail (istream!=NULL);
	g_return_if_fail (appendme_file_parts!=NULL);

	line = g_byte_array_new ();

	for (;;)
	{
		const char * line_str;

		linestart_pos = g_mime_stream_tell (istream);
		g_byte_array_set_size (line, 0);
		g_mime_stream_buffer_readln (istream, line);
		if (!line->len)
			break;

		g_byte_array_append (line, "", 1);
		line_str = (const char*) line->data;

		/* beginning of a UU part */
		if (type==ENC_PLAIN && uu_is_beginning_line(line_str))
		{
			gulong mode = 0;
			char * decoded_filename = NULL;

			if (cur != NULL) {
				cur->type = ENC_PLAIN; /* either this was text, or an encoded block aborted unexpectedly */
				cur->end = linestart_pos;
				g_ptr_array_add (appendme_file_parts, cur);
				cur = NULL;
			}

			/* new entry */
			uu_get_file_info (line_str, &decoded_filename, &mode);
			cur = g_new0 (EncTempFilePart, 1);
			cur->start = linestart_pos;
			cur->type = type = ENC_UU;
			cur->decoded_filename = decoded_filename;
		}

		/* ending of a UU part */
		else if (type==ENC_UU && uu_is_ending_line(line_str))
		{
			if (cur != NULL) {
				cur->end = g_mime_stream_tell (istream);
				g_ptr_array_add (appendme_file_parts, cur);
				cur = NULL;
			}
			type = ENC_PLAIN;
		}

		/* beginning of a yenc part */
		else if (type==ENC_PLAIN && yenc_is_beginning_line (line_str))
		{
			if (cur != NULL) {
				cur->type = ENC_PLAIN; /* either this was text, or an encoded block aborted unexpectedly */
				cur->end = linestart_pos;
				g_ptr_array_add (appendme_file_parts, cur);
				cur = NULL;
			}

			/* new entry */
			cur = g_new0 (EncTempFilePart, 1);
			cur->start = linestart_pos;
			cur->type = type = ENC_YENC;
			yenc_parse_begin_line (line_str,
			                       &cur->decoded_filename, 
			                       &cur->y_line_len,
			                       &cur->y_attach_size,
			                       &cur->y_part);
			if (cur->y_part != 0)
				yenc_looking_for_part_line = TRUE;

		}

		/* yenc part line */
		else if (type==ENC_YENC
			&& yenc_looking_for_part_line
			&& yenc_is_part_line(line_str))
		{
			yenc_parse_part_line (line_str, &cur->y_offset_begin, &cur->y_offset_end);
			yenc_looking_for_part_line = FALSE;
		}

		/* continuing a yenc part */
		else if (type==ENC_YENC && yenc_is_ending_line (line_str))
		{
			if (cur != NULL) {
				yenc_parse_end_line (line_str,
						     &cur->y_size,
						     NULL,
						     &cur->y_pcrc,
						     &cur->y_crc);
				cur->end = g_mime_stream_tell (istream);
				g_ptr_array_add (appendme_file_parts, cur);
				cur = NULL;
			}
			type = ENC_PLAIN;
		}

		/* new non-encoded part */
		else if (cur == NULL)
		{
			cur = g_new0 (EncTempFilePart, 1);
			cur->start = linestart_pos;
			cur->type = type = ENC_PLAIN;
			cur->decoded_filename = NULL;
		}
	}

	/* close last entry */
	if (cur != NULL) {
		cur->type = ENC_PLAIN; /* either this was text, or an encoded block aborted unexpectedly */
		cur->end = g_mime_stream_tell (istream);
		g_ptr_array_add (appendme_file_parts, cur);
		cur = NULL;
	}

	g_byte_array_free (line, TRUE);
}

static void
guess_part_type_from_filename (const gchar   * filename,
                               gchar        ** setme_type,
                               gchar        ** setme_subtype)
{
	const gchar * type = gnome_mime_type (filename);
	const gchar * delimiter = strchr (type, '/');
	*setme_type = g_strndup (type, delimiter-type);
	*setme_subtype = g_strdup (delimiter+1);
}

static void
look_for_uuencoded_data (GMimePart * parent, gpointer data)
{
	gint i;
	guint content_len;
	GPtrArray * parts;
	GHashTable * skip = (GHashTable*) data;

	if (g_hash_table_lookup (skip, parent) != NULL)
		return;
	if (!g_mime_content_type_is_type (parent->mime_type, "text", "plain"))
		return;

	content_len = 0;
	parts = g_ptr_array_new ();

	/* convert into N temp files -- uu and text parts */
	if (parent->content)
	{
		const GMimeDataWrapper * content;
		GMimeStream * in_stream;
		GMimeStream * in_buf;
		
		content = g_mime_part_get_content_object (parent);
		in_stream = g_mime_data_wrapper_get_stream ((GMimeDataWrapper*)content);
		in_buf = g_mime_stream_buffer_new (in_stream, GMIME_STREAM_BUFFER_BLOCK_READ);
		g_mime_stream_unref (in_stream);
		in_stream = NULL;
		
		separate_encoded_parts (in_buf, parts);
		g_mime_stream_reset (in_buf);

		/* split? */
		if (parts->len>1 || (parts->len==1 && ((EncTempFilePart*)g_ptr_array_index(parts,0))->type!=ENC_PLAIN))
		{
			/* recast this part as a multipart/mixed instead of a text/plain */
			if (1) {
				GMimeContentType * mime_type;
				g_mime_part_set_content_object (parent, NULL);
				mime_type = g_mime_content_type_new ("multipart", "mixed");
				g_mime_part_set_content_type (parent, mime_type);
			}

			/* add the children */
			for (i=0; i<parts->len; ++i)
			{
				const EncTempFilePart * part = g_ptr_array_index(parts,i);
				GMimePart * child = NULL;

				if (part->type == ENC_UU)
				{
					gchar * pch = NULL;
					gchar * type = NULL;
					gchar * subtype = NULL;
					GMimeDataWrapper * child_wrapper;
					GMimeStream * child_stream;
					GMimeStream * child_filter_stream;
					GMimeFilter * child_filter;

					/* create the part */
					guess_part_type_from_filename (part->decoded_filename, &type, &subtype);
					child = g_mime_part_new_with_type (type, subtype);

					/* set part's attributes */
					g_mime_part_set_filename (child, part->decoded_filename);
					pch = g_strdup_printf ("<%s>", part->decoded_filename);
					g_mime_part_set_content_id (child, pch);
					g_free (pch);
					g_mime_part_set_content_disposition (child, "inline");

					/* set the part's content */
					child_stream = g_mime_stream_substream (in_buf, part->start, part->end);
					child_filter_stream = g_mime_stream_filter_new_with_stream (child_stream);
					g_mime_stream_unref (child_stream);
					
					child_filter = g_mime_filter_basic_new_type (GMIME_FILTER_BASIC_UU_DEC);
					g_mime_stream_filter_add (GMIME_STREAM_FILTER(child_filter_stream), child_filter);
					child_wrapper = g_mime_data_wrapper_new_with_stream (child_filter_stream, GMIME_PART_ENCODING_8BIT);
					g_mime_stream_unref (child_filter_stream);
					
					g_mime_part_set_content_object (child, child_wrapper);
					
					g_free (type);
					g_free (subtype);
				}
				else if (part->type == ENC_YENC)
				{
					char * pch = NULL;
					char * type = NULL;
					char * subtype = NULL;
					GMimeDataWrapper * child_wrapper;
					GMimeStream * child_stream;
					YencInfo * yenc_info;

					/* create the part */
					guess_part_type_from_filename (part->decoded_filename, &type, &subtype);
					child = g_mime_part_new_with_type (type, subtype);

					/* set part's attributes */
					g_mime_part_set_filename (child, part->decoded_filename);
					pch = g_strdup_printf ("<%s>", part->decoded_filename);
					g_mime_part_set_content_id (child, pch);
					g_free (pch);
					g_mime_part_set_content_disposition (child, "inline");

					/* set the part's content */
					child_stream = g_mime_stream_substream (in_buf, part->start, part->end);
					child_wrapper = g_mime_data_wrapper_new_with_stream (child_stream, GMIME_PART_ENCODING_8BIT);
					g_mime_part_set_content_object (child, child_wrapper);
					g_mime_stream_unref (child_stream);

					yenc_info = g_new0 (YencInfo, 1);
					yenc_info->filename = g_strdup (part->decoded_filename);
					yenc_info->line_len = part->y_line_len;
					yenc_info->attach_size = part->y_attach_size;
					yenc_info->part = part->y_part;
					yenc_info->offset_begin = part->y_offset_begin;
					yenc_info->offset_end = part->y_offset_end;
					yenc_info->crc = part->y_crc;
					yenc_info->pcrc = part->y_pcrc;
					yenc_info->size = part->y_size;
					GMIME_OBJECT(child)->user_data = yenc_info;

					g_free (type);
					g_free (subtype);
				}
				else /* PLAIN */
				{
					GMimeStream * child_stream;
					GMimeDataWrapper * child_wrapper;

					child = g_mime_part_new_with_type ("text", "plain");
					child_stream = g_mime_stream_substream (in_buf, part->start, part->end);
					child_wrapper = g_mime_data_wrapper_new_with_stream (child_stream, GMIME_PART_ENCODING_8BIT);
					g_mime_part_set_content_object (child, child_wrapper);
					g_mime_stream_unref (child_stream);
				}

				g_hash_table_insert (skip, child, child);
				g_mime_part_add_subpart (parent, child);

				g_mime_object_unref (GMIME_OBJECT (child));
			}
		}

		g_mime_stream_unref (in_buf);
	}

	/* remove parts */
	for (i=0; i<parts->len; ++i) {
		EncTempFilePart * part = g_ptr_array_index(parts,i);
		g_free (part->decoded_filename);
		g_free (part);
	}
	g_ptr_array_free (parts, TRUE);
}

/***
****
***/

static GMimeMessage*
construct_message_impl (GMimeStream ** istreams, int qty)
{
	int i;
	GMimeMessage * retval;
	GMimeMessage ** messages;

	/* sanity clause */
	g_return_val_if_fail (istreams!=NULL, NULL);
	g_return_val_if_fail (qty>=1, NULL);

	/* build the GMimeMessages */
	messages = g_new0 (GMimeMessage*, qty);
	for (i=0; i<qty; ++i)
		messages[i] = g_mime_parser_construct_message (istreams[i], TRUE);
	retval = *messages;

	/* merge all the aritcles into a single article */
	if (qty > 1)
	{
		/* recast parent as a subpart if necessary */
		if (!g_mime_content_type_is_type (retval->mime_part->mime_type, "multipart", "*"))
		{
			GMimePart * new_part = g_mime_part_new_with_type ("multipart", "mixed");
			GMimePart * old_part = retval->mime_part;

			g_mime_object_ref (GMIME_OBJECT(old_part));
			g_mime_message_set_mime_part (retval, new_part);
			g_mime_part_add_subpart (new_part, old_part);
			g_mime_object_unref (GMIME_OBJECT(old_part));
		}

		/* move all the other parts into the first message */
		for (i=1; i<qty; ++i)
		{
			GList * l;
			GMimePart * part;

			part = messages[i]->mime_part;
			if (!g_mime_content_type_is_type (retval->mime_part->mime_type, "multipart", "*"))
			{
				g_mime_object_ref (GMIME_OBJECT(part));
				g_mime_message_set_mime_part (messages[i], NULL);
				g_mime_part_add_subpart (retval->mime_part, part);
				g_mime_object_unref (GMIME_OBJECT(part));
			}
			else for (l=part->children; l!=NULL; l=l->next)
			{
				GMimePart * child = GMIME_PART(l->data);

				g_mime_object_ref (GMIME_OBJECT(child));
				g_mime_part_remove_subpart (part, child);
				g_mime_part_add_subpart (retval->mime_part, child);
				g_mime_object_unref (GMIME_OBJECT(child));
			}
		}
	}

	/* pick out yenc and uuenc data in text/plain messages */
	if (1) {
		GHashTable * skip = g_hash_table_new (g_direct_hash, g_direct_equal);
		g_mime_message_foreach_part (retval, look_for_uuencoded_data, skip);
		g_hash_table_destroy (skip);
	}

	/* cleanup */
	for (i=1; i<qty; ++i)
		g_mime_object_unref (GMIME_OBJECT(messages[i]));
	g_free (messages);

	return retval;
}

GMimeMessage*
pan_g_mime_parser_construct_message (const char ** raw_texts, int qty)
{
	int i;
	int stream_qty;
	GMimeMessage * retval;
	GMimeStream ** streams;

	/* sanity clause */
	g_return_val_if_fail (raw_texts!=NULL, NULL);
	g_return_val_if_fail (qty>=1, NULL);

	/* build the GMimeMessages */
	streams = g_new0 (GMimeStream*, qty);
	for (stream_qty=i=0; i<qty; ++i) {
		const char * txt = raw_texts[i];
		if (is_nonempty_string(txt))
			streams[stream_qty++] = g_mime_stream_mem_new_with_buffer (txt, strlen(txt));
	}
	retval = construct_message_impl (streams, stream_qty);
	for (i=0; i<stream_qty; ++i)
		g_mime_stream_unref (streams[i]);

	return retval;
}

GMimeMessage*
pan_g_mime_parser_construct_message_from_file (FILE ** fps, int qty)
{
	int i;
	GMimeStream ** istreams;
	GMimeStream ** buf_istreams;
	GMimeMessage * retval;

	g_return_val_if_fail (fps!=NULL, NULL);
	g_return_val_if_fail (qty>=1, NULL);

	istreams = g_new0 (GMimeStream*, qty);
	buf_istreams = g_new0 (GMimeStream*, qty);
	for (i=0; i<qty; ++i) {
		istreams[i] = g_mime_stream_file_new (fps[i]);
		buf_istreams[i] = g_mime_stream_buffer_new (istreams[i], GMIME_STREAM_BUFFER_BLOCK_READ);
	}
       	retval = construct_message_impl (buf_istreams, qty);
	for (i=0; i<qty; ++i) {
		g_mime_stream_unref (buf_istreams[i]);
		g_mime_stream_unref (istreams[i]);
	}

	return retval;
}

gboolean
pan_header_is_8bit_encoded (const gchar * pch)
{
	return pch!=NULL && strstr(pch,"=?")!=NULL;
}
