/******************************************************************************
  File:     $Id: hpdjdata.c,v 1.8 1999-09-07 16:45:50+02 Martin Rel $
  Contents: Ghostscript device driver "hpdj":
	    Constant data structures, in particular descriptions of printer
	    capabilities
  Author:   Martin Lottermoser, Metzgerfeldweg 9, 85737 Ismaning, Germany

*******************************************************************************
*									      *
*	Copyright (C) 1996, 1997, 1998 by Martin Lottermoser		      *
*	All rights reserved						      *
*									      *
******************************************************************************/

/* Configuration management identification */
#ifndef lint
static const char
  cm_id[] = "@(#)$Id: hpdjdata.c,v 1.8 1999-09-07 16:45:50+02 Martin Rel $";
#endif

/*****************************************************************************/

#ifndef _XOPEN_SOURCE
#define _XOPEN_SOURCE	500
#endif

/* hpdj headers */
#include "hpdj.h"

/*****************************************************************************/

/*  Table mapping PCL Page Size codes to media dimensions.

    The first dimension given is the width, the second the height, both in PCL
    space. See the comments for type 'med_dim' in hpdj.h for a detailed
    discussion.

    I have here made the assumption that the user always chooses an input
    orientation such that the media orientation is portrait (width <= height)
    in PCL space.

    I am not aware of any statement from Hewlett-Packard relating input
    orientation to orientation in PCL space. My DeskJet 850C handbook
    ("HP DeskJet 850C Handbuch", C2145-90248, 3/95), however, at least states
    that envelopes and index cards should always be inserted short edge first,
    and that not only seems sensible for purely mechanical reasons but seems to
    lead to the desired orientation in all cases as well.

    This is not a problem as long as the input orientation leading to portrait
    orientation in PCL space is the same for all supported Page Size codes. If
    there should happen to exist entries in the following list having the same
    media dimensions within the 5 bp (1.8 mm) tolerance limit accepted by
    PostScript and where different input orientations are needed, the user
    would need to know how hpdj chooses an entry (first match in this list) in
    order to insert the paper properly. Although there are at present no
    entries of this kind in the list, these remarks extend to custom page sizes
    where I've also assumed portrait orientation in PCL space. But to repeat:
    This is no problem for a user if the input orientation for portrait
    orientation in PCL space is the *same* for all Page Size codes supported by
    his/her printer, and not only does this seem to be the case, but the
    required input orientation seems to be uniformly "short edge first" on all
    DeskJets.

    Note also that there are no entries here where one cannot distinguish
    between landscape and portrait orientation within the 5 bp limit.
 */
const med_dim hpdj_mdim[] = {
  /* Sorted by width and height */
  {pcl_ps_monarch,	3.875*BP_PER_IN, 7.5*BP_PER_IN}, /* 98.4 x 190.5 mm */
  {pcl_ps_hagaki,	100*BP_PER_MM,	148*BP_PER_MM},
  {pcl_ps_4x6,		4.0*BP_PER_IN,	6.0*BP_PER_IN},	 /* 101.6 x 152.4 mm */
  {pcl_ps_com10,	4.125*BP_PER_IN, 9.5*BP_PER_IN}, /* 104.8 x 241.3 mm */
  {pcl_ps_A6,		105*BP_PER_MM,	148*BP_PER_MM},
  {pcl_ps_DL,		110*BP_PER_MM,	220*BP_PER_MM},
  {pcl_ps_US_A2,	4.375*BP_PER_IN, 5.75*BP_PER_IN}, /* 111.1 x 146.1 mm */
  {pcl_ps_C6,		114*BP_PER_MM,	162*BP_PER_MM},
  {pcl_ps_5x8,		5.0*BP_PER_IN,	8.0*BP_PER_IN},	 /* 127 x 203.2 mm */
  {pcl_ps_statement,	5.5*BP_PER_IN,	8.5*BP_PER_IN},	 /* 139.7 x 215.9 mm */
  {pcl_ps_A5,		148*BP_PER_MM,	210*BP_PER_MM},
  {pcl_ps_C5,		162*BP_PER_MM,	229*BP_PER_MM},
  {pcl_ps_JISB5,	182*BP_PER_MM,	257*BP_PER_MM},
  {pcl_ps_executive,	7.25*BP_PER_IN,	10.5*BP_PER_IN}, /* 184.2 x 266.7 mm */
  {pcl_ps_A4,		210*BP_PER_MM,	297*BP_PER_MM},
  {pcl_ps_letter,	8.5*BP_PER_IN,	11.0*BP_PER_IN}, /* 215.9 x 279.4 mm */
  {pcl_ps_legal,	8.5*BP_PER_IN,	14.0*BP_PER_IN}, /* 215.9 x 355.6 mm */
  {pcl_ps_JISB4,	257*BP_PER_MM,	364*BP_PER_MM},
  {pcl_ps_tabloid,	11*BP_PER_IN,	17*BP_PER_IN},	 /* 279.4 x 431.8 mm */
  {pcl_ps_A3,		297*BP_PER_MM,	420*BP_PER_MM},
  {pcl_ps_superB,	305*BP_PER_MM,	487*BP_PER_MM},
  {pcl_ps_none,		0.0,		0.0}		/* sentinel */
};

/*****************************************************************************/

/*
  Lists of media dimensions supported by each model and the (graphics mode)
  margins in force for them.

  Unless otherwise noted, the data are from the TRG (pp. 1-18 to 1-21).
  I had to guess some of the margins. This is mostly the case where the
  TRG states on page 3-2 that the size is supported but there are no margin
  specifications on pp. 1-18 to 1-21.

  Be careful when inserting margin specifications! In particular for envelopes,
  older (pre-1997) HP documentation frequently gives them in landscape
  orientation instead of in portrait orientation as needed here. A useful rule
  of thumb: If the bottom margin is not the largest of the four margins,
  it's very likely landscape. The value should be around 0.5 in (ca. 12 mm).
  Oh, yes: as far as I was able to check (DeskJet 850C---another point on
  which I could not find HP documentation), the relation between landscape
  and portrait is the same in PCL as in PostScript: rotate the page contents
  by +90 degrees (counterclockwise) from portrait and you get landscape.
  This means that if the bottom margin is not the largest, the left margin
  should be. If it isn't, THINK CAREFULLY!, because I can't help you any
  further.

  This discussion also assumes that the unprintable region does not change
  when the orientation changes. If this is not true, you can't convert
  margins given for landscape orientation to margins for portrait
  orientation. (No HP statement available for this point either...)
*/

static const margin_desc
  hpdj500_margins[] = {
    /* Order of margins: left, bottom, right, top. */
    {pcl_ps_A4,	3.1*BP_PER_MM, 0.57*BP_PER_IN, 3.6*BP_PER_MM, 0.1*BP_PER_IN},
    {pcl_ps_letter,
      0.25*BP_PER_IN, 0.57*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_legal,
      0.25*BP_PER_IN, 0.57*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_com10,	/* Supported, margins guessed */
      0.25*BP_PER_IN, 0.57*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_none}
  },
  hpdj500c_black_margins[] = {
    /* Order of margins: left, bottom, right, top. */
    {pcl_ps_A4,	3.1*BP_PER_MM, 0.4*BP_PER_IN, 3.6*BP_PER_MM, 0.1*BP_PER_IN},
    {pcl_ps_letter,
      0.25*BP_PER_IN, 0.4*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_legal,
      0.25*BP_PER_IN, 0.4*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_com10,	/* Supported, margins guessed */
      0.25*BP_PER_IN, 0.4*BP_PER_IN, 0.25*BP_PER_IN, 0.1*BP_PER_IN},
    {pcl_ps_none}
  },
  common_margins[] = { /* DJs 510, 520, 550C und 560C for printing in black */
    /* Order of margins: left, bottom, right, top. */
    {pcl_ps_A4,	3.1*BP_PER_MM, 0.46*BP_PER_IN, 3.6*BP_PER_MM, 0.04*BP_PER_IN},
    {pcl_ps_letter,
      0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.04*BP_PER_IN},
    {pcl_ps_legal,
      0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.04*BP_PER_IN},
    {pcl_ps_executive,
      0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.20*BP_PER_IN, 0.04*BP_PER_IN},
    {pcl_ps_DL,		/* Supported, margins guessed */
      0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.04*BP_PER_IN},
    {pcl_ps_com10,	/* Supported, margins guessed */
      0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.04*BP_PER_IN},
    {pcl_ps_none}
  },
  hpdj540_black_margins[] = {
    /* Order of margins: left, bottom, right, top. */
    {pcl_ps_A4,	3.2*BP_PER_MM, 11.7*BP_PER_MM, 3.2*BP_PER_MM, 0.0},
    {pcl_ps_A5,	3.2*BP_PER_MM, 18.0*BP_PER_MM, 3.2*BP_PER_MM, 0.0},
    {pcl_ps_A6,	3.2*BP_PER_MM, 18.0*BP_PER_MM, 3.2*BP_PER_MM, 0.0},
    {pcl_ps_C6, 0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.13*BP_PER_IN, 0.0},
      /* given in landscape by HP */
    {pcl_ps_DL, 0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.13*BP_PER_IN, 0.0},
      /* given in landscape by HP */
    {pcl_ps_JISB5, 4.2*BP_PER_MM, 11.7*BP_PER_MM, 4.2*BP_PER_MM, 0.0},
    {pcl_ps_executive, 0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.0},
    {pcl_ps_letter,    0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.0},
    {pcl_ps_legal,     0.25*BP_PER_IN, 0.46*BP_PER_IN, 0.25*BP_PER_IN, 0.0},
    {pcl_ps_4x6,       0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.13*BP_PER_IN, 0.0},
    {pcl_ps_5x8,       0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.13*BP_PER_IN, 0.0},
    {pcl_ps_com10,     0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.13*BP_PER_IN, 0.0},
      /* given in landscape by HP */
    {pcl_ps_hagaki, 4.2*BP_PER_MM, 18.0*BP_PER_MM, 4.2*BP_PER_MM, 0.0},
    {pcl_ps_none}
  },
  hpdj850c_margins[] = {	/* valid for black and colour printing */
   /* Apart from JIS B5, these specifications have been taken from the German
      HP user manual for the DeskJet 850C, "HP DeskJet 850C Handbuch",
      C2145-90248, 3/95.
      It agrees with "HP DeskJet 850C and 855C Printer -- Printable Regions"
      (BPD02523), HP FIRST #: 2789, 1995-09-05, and the newer "HP DeskJet 855C,
      850C, 870C and 890C Series Printers -- Printable Areas" (BPD02523),
      effective date 1996-06-06.
      First some shorthand notation for margin types. The order is:
	left, bottom, right, top.
    */
#define type1	6.4*BP_PER_MM, 11.7*BP_PER_MM, 6.4*BP_PER_MM, 1.0*BP_PER_MM
#define type2	3.4*BP_PER_MM, 11.7*BP_PER_MM, 3.4*BP_PER_MM, 1.0*BP_PER_MM
#define type3	3.4*BP_PER_MM, 11.7*BP_PER_MM, 3.4*BP_PER_MM, 11.7*BP_PER_MM
#define type4	3.2*BP_PER_MM, 11.7*BP_PER_MM, 3.2*BP_PER_MM, 1.0*BP_PER_MM
#define type5	3.2*BP_PER_MM, 22.0*BP_PER_MM, 3.2*BP_PER_MM, 1.0*BP_PER_MM
    {pcl_ps_A4,		type2},
    {pcl_ps_A5,		type2},
    {pcl_ps_A6,		type4},
    {pcl_ps_C6,		type5},
    {pcl_ps_DL,		type5},
    {pcl_ps_letter,	type1},
    {pcl_ps_legal,	type1},
    {pcl_ps_executive,	type1},
    {pcl_ps_JISB5,	type2},
      /*  Actually, the German handbook as well as BPD02523 state that this
	  should be type3. This is however demonstrably wrong for the DJ 850C.
	  Besides I can't see why the top margin should be so much larger for
	  JIS B5 than for the other sizes. Note that the DJ 540 has a zero top
	  margin for JIS B5. 'type2' agrees fairly well with what I've
	  measured on a DJ 850C. In particular the right and bottom clipping
	  margins are definitely close to 3 and 11.5 mm, respectively.
      */
    {pcl_ps_4x6,	type4},
    {pcl_ps_5x8,	type4},
    {pcl_ps_hagaki,	type4},
    {pcl_ps_com10,	type5},
    /*	BPD02926 claims that the series-800 DeskJets support also US A2 envelope
	size (pcl_ps_US_A2). I've experimented on a DeskJet 850C and I don't
	believe it: when sent this page size code, the printer establishes
	clipping regions which agree with those for US Letter size. This
	indicates that the PCL interpreter has not recognized the page size code
	and has therefore switched to the default size. */
    {pcl_ps_none}
  };

#undef type1
/* type2 will be needed in a moment */
#undef type3
#undef type4
#undef type5

/*---------------------------------------------------------------------------*/

/*  Custom page size descriptions.
    First the limits: width min and max, height min and max.
    Then the margins in the usual order: left, bottom, right, top.

    Again, I have assumed something I could find no documentation for, namely
    that these limits refer to PCL space for custom page size (see the type
    'med_dim' for a discussion). It is, however, clear from the upper limits
    for the DJ 850C that "height" does not refer to the direction orthogonal
    to the feeding direction. My assumption is therefore equivalent to
    assuming that custom pages will *always* have a portrait orientation in
    PCL space and should therefore *always* have the same input orientation,
    presumably short edge first.
 */
static const custom_page_desc
  hpdj540_custom[] = {
    /* These values are taken from the document "HP DeskJet 540 Printer -
       Print Regions", BPD02194, 1996, except for the right margin which is
       stated to be application-dependent. The TRG recommends that one should
       enforce at least 1/8 in.
       These are the correct values for printing in black. Note that the
       DeskJet 540 in colour mode needs a bottom margin which is larger by
       0.13 in.
    */
    { /* Page width 7.12 to 8.5 in */
      7.12*BP_PER_IN, 8.5*BP_PER_IN, 5.83*BP_PER_IN, 14*BP_PER_IN,
      0.13*BP_PER_IN, 0.46*BP_PER_IN, 0.125*BP_PER_IN, 0
    },
    { /* Page width 5 to 7.12 in */
      5*BP_PER_IN, 7.12*BP_PER_IN, 5.83*BP_PER_IN, 14*BP_PER_IN,
      0.13*BP_PER_IN, 0.71*BP_PER_IN, 0.125*BP_PER_IN, 0
    },
    {0, 0, 0, 0,  0, 0, 0, 0}
  },
  hpdj850c_custom[] = {
    { /*
	The following values have been taken from the German DeskJet 850C
	manual, "HP DeskJet 850C Handbuch", C2145-90248, 3/95.
	They agree with the values given in the newer version of BPD02523 for
	the DeskJets 850C, 855C, 870C and 890C except that the left and right
	margins are stated to be 3.44 mm instead of 3.4 mm. This is almost
	certainly a misprint because the accompanying value in inches is
	identical (0.13 in, i.e. 3.302 mm), and specifying the margins to a
	hundredth of a millimetre is ridiculous.
      */
      100*BP_PER_MM, 216*BP_PER_MM, 148*BP_PER_MM, 356*BP_PER_MM,
      type2
    },
    {0, 0, 0, 0,  0, 0, 0, 0}
  },
  unspec_custom[] = {
    /* Any value is permitted in this case. This is approximated by permitting
       1-10^37 bp for both, width and height (10^37 bp = 3.5*10^33 m).
       Note that 10^37 is the smallest upper bound for 'float' values permitted
       by the C standard. */
    {
      1, 1.0E37,  1, 1.0E37,
      type2	/* use DJ 850C margins */
    },
    {0, 0, 0, 0,  0, 0, 0, 0}
  };

#undef type2

/*****************************************************************************/

/*  Resolution lists
    These are given according to the TRG and to BPD02926. I know that already
    some series-500 DeskJets claim to support 600x300 ppi, and I also know that
    at least the DJ 540 recognizes the new Configure Raster Data command which
    would make such a feature accessible if it should be really supported for
    raster data and not only for the builtin fonts. If you want to know why
    I have not included these resolutions here, read hpdj's manual page and
    George Cameron's comments for the cdj/pj drivers in Devices.htm.
 */

static const supported_resolution
  old_resolutions[] = {
      { 75, 75, true }, { 100, 100, true}, { 150, 150, true },
      { 300, 300, true }
    },
  hp540_resolutions[] = {
      { 75, 75, true }, { 150, 150, true }, { 300, 300, true }
    },
  hp850_resolutions[] = {
      { 75, 75, true }, { 150, 150, true }, { 300, 300, true },
      { 600, 600, false }
    };

/*****************************************************************************/

/* Description of known models. The order of these entries must agree with
   the order of values defined in the type 'Model'. */

const model_description hpdj_model[] = {
  {hpdj_none,	"none",	cmyk,	hpdj500_margins,	0.0,	unspec_custom,
    NULL},
    /*  It's important to permit everything except printing for "none",
	otherwise a check might issue a message mentioning that something is
	not supported by the "DeskJet none". That would be confusing. */
  {hpdj500,	"500",	mono,	hpdj500_margins,	0.0,	NULL,
    old_resolutions},
  {hpdj500c,	"500C", cmy,	hpdj500c_black_margins,	0.17*BP_PER_IN,	NULL,
    old_resolutions},
  {hpdj510,	"510",	mono,	common_margins,		0.0,	NULL,
    old_resolutions},
  {hpdj520,	"520",	mono,	common_margins,		0.0,	NULL,
    old_resolutions},
  {hpdj540,	"540",	cmy,	hpdj540_black_margins,	0.13*BP_PER_IN,
    hpdj540_custom, hp540_resolutions},
  {hpdj550c,	"550C", cmy_plus_k, common_margins,	0.13*BP_PER_IN,	NULL,
    old_resolutions},
  {hpdj560c,	"560C", cmy_plus_k, common_margins,	0.13*BP_PER_IN,	NULL,
    old_resolutions},
  {hpdj850c,	"850C",	cmyk,	hpdj850c_margins,	0.0,  hpdj850c_custom,
    hp850_resolutions},
  {hpdj855c,	"855C",	cmyk,	hpdj850c_margins,	0.0,  hpdj850c_custom,
    hp850_resolutions},
  {hpdj_unspec,	"unspec", cmyk,	hpdj850c_margins,	0.0,  unspec_custom,
    NULL},
  {hpdj_none,	NULL,	mono,	NULL,	0.0,	NULL, 	NULL}	/* sentinel */
};
