/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2000 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "prefwidgets.h"
#include "resources.h"
#include "cmdedit.h"
#include "kbabeldictbox.h"

#include <kcombobox.h>
#include <klocale.h>
#include <kdialog.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <klineedit.h> 
#include <kurlcompletion.h>
#include <kfontdialog.h>
#include <kcolorbtn.h>
#include <ksconfig.h>
#include <kurldrag.h>
#include <kurlrequester.h>

#include <qlayout.h>
#include <qobjectlist.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qhbuttongroup.h>
#include <qpushbutton.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qtextcodec.h>
#include <qwhatsthis.h>


static QSize sizeHintForWidget(const QWidget* widget)
{
  //
  // The size is computed by adding the sizeHint().height() of all
  // widget children and taking the width of the widest child and adding
  // layout()->margin() and layout()->spacing()
  //

  QSize size;

  int numChild = 0;
  QObjectList *l = (QObjectList*)(widget->children());

  for( uint i=0; i < l->count(); i++ )
  {
    QObject *o = l->at(i);
    if( o->isWidgetType() )
    {
      numChild += 1;
      QWidget *w=((QWidget*)o);

      QSize s = w->sizeHint();
      if( s.isEmpty() == true )
      {
          s = QSize( 50, 100 ); // Default size
      }
      size.setHeight( size.height() + s.height() );
      if( s.width() > size.width() ) { size.setWidth( s.width() ); }
    }
  }

  if( numChild > 0 )
  {
    size.setHeight( size.height() + widget->layout()->spacing()*(numChild-1) );
    size += QSize( widget->layout()->margin()*2, widget->layout()->margin()*2 + 1 );
  }
  else
  {
    size = QSize( 1, 1 );
  }

  return( size );
}




SavePreferences::SavePreferences(QWidget *parent)
    : KTabCtl(parent)
{
    QWidget* page = new QWidget(this);
    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);

    box->setMargin(KDialog::marginHint());
    _updateButton = new QCheckBox(i18n("&Update header when saving"),box);
    _autoCheckButton = new QCheckBox(i18n("Chec&k syntax of file when saving"),box);

    QGroupBox* gridBox = new QGroupBox(2,Qt::Horizontal,i18n("Fields to update:"),page);
    layout->addWidget(gridBox);
    gridBox->setMargin(KDialog::marginHint());

    _revisionButton = new QCheckBox(i18n("Re&vision-Date"),gridBox);
    _lastButton = new QCheckBox(i18n("Last-&Translator"),gridBox);
    _languageButton = new QCheckBox(i18n("&Language"),gridBox);
    _charsetButton = new QCheckBox(i18n("Char&set"),gridBox);
    _encodingButton = new QCheckBox(i18n("&Encoding"),gridBox);

    QGroupBox* dateBox = new QButtonGroup(2,Qt::Horizontal,i18n("Format of Revision-Date:"),page);
    layout->addWidget(dateBox);
    box->setMargin(KDialog::marginHint());

    _defaultDateButton = new QRadioButton( i18n("De&fault date format"),dateBox );
    _localDateButton = new QRadioButton( i18n("Local date fo&rmat"),dateBox );
    _customDateButton = new QRadioButton( i18n("Custo&m date format:"),dateBox );
    _dateFormatEdit = new QLineEdit(dateBox);
    _dateFormatEdit->setEnabled(false);

    connect( _customDateButton, SIGNAL(toggled(bool)), this, SLOT( customDateActivated(bool) ) );


    QGroupBox* encodingBox = new QGroupBox(1,Qt::Horizontal,i18n("Encoding:")
                                  ,page);
    encodingBox->setMargin(KDialog::marginHint());
    layout->addWidget(encodingBox);
    QHBox *b = new QHBox(encodingBox);

    QLabel* tempLabel=new QLabel(i18n("Default:"),b);
    _encodingBox = new QComboBox(b);
    b->setStretchFactor(_encodingBox,2);
    b->setSpacing(KDialog::spacingHint());

    QString defaultName=charsetString(Locale);
    defaultName+=" "+i18n("(default)");
    QString utf8Name=charsetString(UTF8);
    QString utf16Name=charsetString(UTF16);

    _encodingBox->insertItem(defaultName,(int)Locale);
    _encodingBox->insertItem(utf8Name,(int)UTF8);

    // KBabel seems to crash somehow, when saving in utf16, so
    // it's better to disable this, since it is useless anyway
    // at the moment
    //_encodingBox->insertItem(utf16Name,(int)UTF16);

    tempLabel->setBuddy(_encodingBox);

    _oldEncodingButton = new QCheckBox(i18n("Kee&p the encoding of the file")
                     ,encodingBox);


    QWhatsThis::add(_updateButton,
        i18n("<qt><p><b>Update Header</b></p>\n"
             "<p>Check this button, to update the header "
             "information of the file "
             "every time when it is saved.</p>\n"
             "<p>The header normally keeps information about "
             "the date and time the file was last\n"
             "updated, the last translator etc.</p>\n"
             "<p>You can choose which information you want to update from the checkboxes below.\n"
             "Fields, that do not exist, are added to the header.\n"
             "If you want to add additional fields to the header you can edit the header manually by choosing\n"
             "<b>Edit->Edit Header</b> in the editor window.</p></qt>"));

    QWhatsThis::add(gridBox,i18n("<qt><p><b>Fields to update</b></p>\n"
 "<p>Choose which fields in the header you want to have updated when saving.\n"
 "If a field does not exist, it is appended to the header.</p>\n"
 "<p>If you want to add other information to the header, you have to edit the header manually\n"
 "by choosing <b>Edit->Edit Header</b> in the editor window.</p>\n"
 "<p>Deactivate <b>Update Header</b> above if you don't want to have the header\n"
 "updated when saving.</p></qt>"));

    QWhatsThis::add(encodingBox,i18n("<qt><p><b>Encoding</b></p>"
"<p>Choose how to encode characters when saving to a file. If you are unsure "
"what encoding to use, please ask your translation coordinator.</p>"
"<ul><li><b>%1</b>: this is the encoding, that fits to the character "
"set of your system language.</li>"
"<li><b>%2</b>: uses unicode(utf8) encoding.</li>"
"<li><b>%3</b>: uses unicode(utf16) encoding."
"This is not recommended at the moment, because gettext might have some trouble reading this.</li></ul></qt>").arg(defaultName).arg(utf8Name).arg(utf16Name) );


    QWhatsThis::add(_oldEncodingButton
        ,i18n("<qt><p><b>Keep the encoding of the file</b></p>"
        "<p>If this option is activated, files are always saved in the "
        "same encoding as they were read in. Files without charset "
        "information in the header (e.g. POT files) are saved in the "
        "encoding set above.</p></qt>"));

    QWhatsThis::add(_autoCheckButton,i18n("<qt><p><b>Check syntax of file when saving</b></p>\n"
"<p>Check this to automatically check syntax of file with \"msgfmt --statistics\"\n"
"when saving a file. You will only get a message, if an error occured.</p></qt>"));


    QWhatsThis::add(dateBox, i18n("<qt><p><b>Format of Revision-Date</b></p>"
"<p>Choose in which format the date and time of the header field\n"
"<i>PO-Revision-Date</i> is saved: <ul>\n"
"<li><b>Default</b> is the format normally used in PO-files.</li>\n"
"<li><b>Local</b> is the format specific to your country.\n"
"It can be configured in KDE's Control Center.</li>\n"
"<li><b>Custom</b> let you define your own format.</li></ul></p> "
"<p>For more information see section <b>The Preferences Dialog</b> "
"in the online help.</p></qt>") );


    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));
    addTab(page,i18n("title of page in preferences dialog","Save"));


    setMinimumSize(sizeHint());


    defaults();
}


void SavePreferences::setSettings(const SaveSettings& settings)
{
   _updateButton->setChecked(settings.autoUpdate);

   _lastButton->setChecked(settings.updateLastTranslator);
   _revisionButton->setChecked(settings.updateRevisionDate);
   _languageButton->setChecked(settings.updateLanguageTeam);
   _charsetButton->setChecked(settings.updateCharset);
   _encodingButton->setChecked(settings.updateEncoding);

   _encodingBox->setCurrentItem((int)settings.encoding);
   _oldEncodingButton->setChecked(settings.useOldEncoding);

   _autoCheckButton->setChecked(settings.autoSyntaxCheck);

   _dateFormatEdit->setText(settings.customDateFormat);

   switch(settings.dateFormat)
   {
      case Default:
         _defaultDateButton->setChecked(true);
         break;
      case Local:
         _localDateButton->setChecked(true);
         break;
      case Custom:
         _customDateButton->setChecked(true);
         break;
   }

}

SaveSettings SavePreferences::settings() const
{
   SaveSettings settings;

   settings.updateLastTranslator=_lastButton->isChecked();
   settings.updateRevisionDate=_revisionButton->isChecked();
   settings.updateLanguageTeam=_languageButton->isChecked();
   settings.updateCharset=_charsetButton->isChecked();
   settings.updateEncoding=_encodingButton->isChecked();
   settings.autoUpdate=_updateButton->isChecked();

   settings.encoding=(FileEncoding)_encodingBox->currentItem();
   settings.useOldEncoding=_oldEncodingButton->isChecked();

   settings.autoSyntaxCheck = _autoCheckButton->isChecked();


   settings.customDateFormat=_dateFormatEdit->text();

   if(_defaultDateButton->isChecked())
   {
      settings.dateFormat=Default;
   }
   else if(_localDateButton->isChecked())
   {
      settings.dateFormat=Local;
   }
   else
   {
      settings.dateFormat=Custom;
   }


   return settings;
}

void SavePreferences::defaults()
{
   _updateButton->setChecked(Defaults::Save::autoUpdate);

   _lastButton->setChecked(Defaults::Save::updateLastTranslator);
   _revisionButton->setChecked(Defaults::Save::updateRevisionDate);
   _languageButton->setChecked(Defaults::Save::updateLanguageTeam);
   _charsetButton->setChecked(Defaults::Save::updateCharset);
   _encodingButton->setChecked(Defaults::Save::updateEncoding);

   _encodingBox->setCurrentItem((int)Defaults::Save::encoding);
   _oldEncodingButton->setChecked(Defaults::Save::useOldEncoding);

   _autoCheckButton->setChecked(Defaults::Save::autoSyntaxCheck);

   _dateFormatEdit->setText(Defaults::Save::customDateFormat);

   switch(Defaults::Save::dateFormat)
   {
      case Default:
         _defaultDateButton->setChecked(true);
         break;
      case Local:
         _localDateButton->setChecked(true);
         break;
      case Custom:
         _customDateButton->setChecked(true);
         break;
   }

}


void SavePreferences::customDateActivated(bool on)
{
   _dateFormatEdit->setEnabled(on);
   _dateFormatEdit->setFocus();
}





IdentityPreferences::IdentityPreferences(QWidget* parent)
         : KTabCtl(parent)
{
    QWidget* page = new QWidget(this);
    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* group = new QGroupBox(2,Qt::Horizontal,page);
    layout->addWidget(group);
    group->setMargin(KDialog::marginHint());

    QLabel* tempLabel=new QLabel(i18n("&Name:"),group);
    _nameEdit = new QLineEdit(group);
    tempLabel->setBuddy(_nameEdit);


    tempLabel=new QLabel(i18n("E&mail:"),group);
    _mailEdit = new QLineEdit(group);
    tempLabel->setBuddy(_mailEdit);


    tempLabel=new QLabel(i18n("&Full language name:"),group);

    QHBox *hbox = new QHBox(group);
    hbox->setSpacing(KDialog::spacingHint());
    _langEdit = new QLineEdit(hbox);
    tempLabel->setBuddy(_langEdit);
    tempLabel=new QLabel(i18n("Lan&guage code:"),hbox);
    _langCodeEdit = new QLineEdit(hbox);
    tempLabel->setBuddy(_langCodeEdit);


    tempLabel=new QLabel(i18n("&Language mailing list:"),group);
    _listEdit = new QLineEdit(group);
    _listEdit->setMinimumSize(100,_listEdit->sizeHint().height());
    tempLabel->setBuddy(_listEdit);



    QString whatsThisMsg=i18n("<qt><p><b>Identity</b></p>\n"
"<p>Fill in information about you and your translation team.\n"
"This information is used when updating the header of a file.</p>\n"
"<p>You can find the options if and what fields in the header should be updated\n"
"on page <b>Save</b> in this dialog.</p></qt>");

    QWhatsThis::add(group,whatsThisMsg);

    group = new QGroupBox(2,Qt::Horizontal,page);
    layout->addWidget(group);
    group->setMargin(KDialog::marginHint());
    tempLabel=new QLabel(i18n("&Timezone:"),group);
    _timezoneEdit = new QLineEdit(group);
    _timezoneEdit->setMinimumSize(100,_timezoneEdit->sizeHint().height());
    tempLabel->setBuddy(_timezoneEdit);


    QWhatsThis::add(group,whatsThisMsg);


    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("Identity"));

    defaults();

    setMinimumSize(sizeHint());

    _mailEdit->installEventFilter(this);
    _listEdit->installEventFilter(this);
}

void IdentityPreferences::setSettings(const IdentitySettings& settings)
{
   _nameEdit->setText(settings.authorName);
   _mailEdit->setText(settings.authorEmail);
   _langEdit->setText(settings.languageName);
   _langCodeEdit->setText(settings.languageCode);
   _listEdit->setText(settings.mailingList);

   _timezoneEdit->setText(settings.timeZone);
}

IdentitySettings IdentityPreferences::settings() const
{
    IdentitySettings settings;

    settings.authorName=_nameEdit->text();
    settings.authorEmail=_mailEdit->text();
    settings.languageName=_langEdit->text();
    settings.languageCode=_langCodeEdit->text();
    settings.mailingList=_listEdit->text();

    settings.timeZone=_timezoneEdit->text();

    return settings;
}


void IdentityPreferences::defaults()
{
   _nameEdit->setText(Defaults::Identity::authorName());
   _mailEdit->setText(Defaults::Identity::authorEmail());
   _langEdit->setText(Defaults::Identity::languageName());
   _langCodeEdit->setText(Defaults::Identity::languageCode());
   _listEdit->setText(Defaults::Identity::mailingList());
   _timezoneEdit->setText(Defaults::Identity::timezone());
}

bool IdentityPreferences::eventFilter(QObject *o, QEvent *e)
{
    if(e->type() == QEvent::Drop)
    {
        QDropEvent *de = static_cast<QDropEvent*>(e);
        KURL::List urlList;
        if(de && KURLDrag::decode(de,urlList))
        {
            KURL url(urlList.first());
            if(url.protocol()== "mailto")
            {
                QString mail=url.path();

                bool handled=false;
                if(o == _mailEdit)
                {
                    handled=true;
                    _mailEdit->setText(mail);
                }
                else if(o == _listEdit)
                {
                    handled=true;
                    _listEdit->setText(mail);
                }

                if(handled)
                    return true;
            }
        }
    }

    return false;
}





EditorPreferences::EditorPreferences(QWidget* parent)
         : KTabCtl(parent)
{
    QWidget* page= new QWidget(this);
    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());
    _fuzzyButton = new QCheckBox(i18n("A&utomatically unset fuzzy status"),box);


    QWhatsThis::add(box,i18n("<qt><p><b>Automatically unset fuzzy status</b></p>\n"
"<p>If this is activated and you are editing a fuzzy entry, the fuzzy status is automatically\n"
"unset (this means the string <i>, fuzzy</i>\n"
"is removed from the entry's comment).</p></qt>"));


    box = new QGroupBox(4,Qt::Vertical,i18n("Automatic checks"), page);
        layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    _checkArgsButton = new QCheckBox(i18n("Chec&k Arguments"),box);
    _checkAccelButton = new QCheckBox(i18n("Check Accele&rator"),box);
    _checkEquationButton = new QCheckBox(i18n("Check &Equation"),box);
    _checkContextButton = new QCheckBox(i18n("Look for Translated Co&ntext Info"), box);

    _beepButton = new QCheckBox(i18n("&Beep on error"),box);
    _colorErrorButton = new QCheckBox(i18n("Change te&xt color on error"),box);

    QString msg = i18n("<qt><p><b>Automatic checks</b></p>"
    "<p>Activate here the things, you want to have checked "
    "while editing.<ul>"
    "<li><b>Check Arguments</b>: checks if the c-format arguments "
    "(%1, %s ...) in the original text and in the translated text "
    "match.</li>"
    "<li><b>Check Accelerator</b>: checks if the number of \"&amp;\" "
    "match</li>"
    "<li><b>Check Equation</b>: checks if the left side of an equation "
    "is the same. Useful for example when you edit KDE's desktop.po file."
    "<li><b>Check for Context</b>: checks if the context information "
    "has been translated. This means the first translated line starts "
    "with <b>_:</b>. </li></ul></p></qt>");

    QWhatsThis::add(_checkArgsButton,msg);
    QWhatsThis::add(_checkAccelButton,msg);
    QWhatsThis::add(_checkEquationButton,msg);
    QWhatsThis::add(_checkContextButton,msg);

    msg = i18n("<qt><p><b>Error recognition</b></p>"
        "<p>Here you can set how to show, that an error occured. "
        "<b>Beep on error</b> beeps and <b>Change text color on error"
        "</b> changes the color of the translated text. If none is "
        "activated, you will still see a message in the statusbar."
        "</p></qt>");

    QWhatsThis::add(_beepButton,msg);
    QWhatsThis::add(_colorErrorButton,msg);

    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("&General"));

    page= new QWidget(this);
    layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    box=new QGroupBox(2,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    _hlSyntaxButton = new QCheckBox(i18n("H&ighlight syntax"),box);
    _spacePointsButton = new QCheckBox(i18n("Mark &whitespaces with points"),box);
    _hlBgButton = new QCheckBox(i18n("Highlight backgrou&nd"),box);
    _quotesButton = new QCheckBox(i18n("&Show surrounding quotes"),box);

    QWhatsThis::add(box,i18n("<qt><p><b>Appearance</b></p>"
"<p>These options let you configure the appearance for the message editor.</p>"
"<p>Check <b>Highlight syntax</b> to let KBabel display special characters, "
"like \\\" and \\n, in different colors.</p>"
"<p>The three other options will help you to \"see\" whitespace characters. "
"<b>Mark whitespace with points</b> draws a small point instead of a whitespace, "
"<b>Highlight background</b> colors the background of the characters and "
"<b>Show surrounding quotes</b> displays quotes at the beginning and end "
"of each line.</p></qt>"));


    box= new QGroupBox(2,Qt::Horizontal,i18n("Colors"),page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    QLabel* tempLabel = new QLabel(i18n("&Background color:"),box);
    _bgColorButton = new KColorButton(box);
    tempLabel->setBuddy(_bgColorButton);

    tempLabel = new QLabel(i18n("Color for &quoted characters:"),box);
    _quotedColorButton = new  KColorButton(box);
    tempLabel->setBuddy(_quotedColorButton);

    tempLabel = new QLabel(i18n("Color for s&yntax errors:"),box);
    _errorColorButton = new  KColorButton(box);
    tempLabel->setBuddy(_errorColorButton);

    tempLabel = new QLabel(i18n("Color for c-for&mat characters:"),box);
    _cformatColorButton = new  KColorButton(box);
    tempLabel->setBuddy(_cformatColorButton);

    tempLabel = new QLabel(i18n("Color for &keyboard accelerators:"),box);
    _accelColorButton = new  KColorButton(box);
    tempLabel->setBuddy(_accelColorButton);

    QWhatsThis::add(box,i18n("<qt><p><b>Colors</b></p>"
"<p>Define here which colors you want to be used in the Editor. "
"<b>Background color</b> is used for highlighting the background of characters. "
"The other colors are used for syntax highlighting.</p></qt>"));



    box=new QGroupBox(2,Qt::Horizontal,i18n("Status LEDs"),page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    QButtonGroup* bgBox=new QButtonGroup(2,Qt::Horizontal,box);
    bgBox->setFrameStyle(QFrame::NoFrame);
    _ledInStatusbarBtn = new QRadioButton(i18n("Display in stat&usbar"),bgBox);
    _ledInEditorBtn = new QRadioButton(i18n("Display in edi&tor"),bgBox);

    QHBox* hBox=new QHBox(box);
    tempLabel = new QLabel(i18n("Colo&r:"),hBox);
    _ledColorButton = new  KColorButton(hBox);
    tempLabel->setBuddy(_ledColorButton);
    tempLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

    QWhatsThis::add(box,i18n("<qt><p><b>Status LEDs</b></p>\n"
"<p>Choose here where the status LEDs are displayed and what color they have.</p></qt>"));


    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("A&ppearance"));


    page= new QWidget(this);
    layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());
    spellConfig = new KSpellConfig(page,"spellConfigWidget",0,false);
    layout->addWidget(spellConfig);
    remIgnoredBtn = new QCheckBox(i18n("&Remember ignored words"),page);
    layout->addWidget(remIgnoredBtn);

    tempLabel = new QLabel(i18n("F&ile to store ignored words:"),page);
    layout->addWidget(tempLabel);
    ignoreURLEdit = new KURLRequester(page);
    layout->addWidget(ignoreURLEdit);
    tempLabel->setBuddy(ignoreURLEdit);

    connect(remIgnoredBtn,SIGNAL(toggled(bool)),ignoreURLEdit
                        ,SLOT(setEnabled(bool)));


    msg = i18n("<qt><p><b>Remember ignored words</b></p>"
        "<p>Activate this, to let KBabel ignore the words, where you have "
        "chosen <i>Ignore All</i> in the spell check dialog, "
        "in every spell check.</p></qt>");

    QWhatsThis::add(remIgnoredBtn,msg);
    QWhatsThis::add(tempLabel,msg);
    QWhatsThis::add(ignoreURLEdit,msg);

    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("Spe&ll check"));



    page= new QWidget(this);
    layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    box=new QGroupBox(1,Qt::Horizontal,i18n("Font for messages"),page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());

    _onlyFixedFontsButton = new QCheckBox(i18n("&Show only fixed fonts"),box);
    connect(_onlyFixedFontsButton, SIGNAL(toggled(bool))
            , this, SLOT(showOnlyFixedFonts(bool)) );

    _fontBox = new KFontChooser(box,0,false,QStringList(),false);

    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("&Fonts"));


    defaults();

    setMinimumSize(sizeHint());


    connect(_ledInStatusbarBtn,SIGNAL(toggled(bool)),this,SLOT(ledWarning(bool)));
    connect(_ledInEditorBtn,SIGNAL(toggled(bool)),this,SLOT(ledWarning(bool)));
}



void EditorPreferences::setSettings(const EditorSettings& settings)
{
    _fuzzyButton->setChecked(settings.autoUnsetFuzzy);

    _checkArgsButton->setChecked(settings.autoCheckArgs);
    _checkAccelButton->setChecked(settings.autoCheckAccel);
    _checkEquationButton->setChecked(settings.autoCheckEquation);
    _checkContextButton->setChecked(settings.autoCheckContext);
    _beepButton->setChecked(settings.beepOnError);
    _colorErrorButton->setChecked(settings.autoCheckColorError);

    _fontBox->setFont(settings.msgFont);

    _hlSyntaxButton->setChecked(settings.highlightSyntax);
    _hlBgButton->setChecked(settings.highlightBg);
    _quotesButton->setChecked(settings.quotes);
    _spacePointsButton->setChecked(settings.whitespacePoints);

    _bgColorButton->setColor(settings.bgColor);
    _quotedColorButton->setColor(settings.quotedColor);
    _errorColorButton->setColor(settings.errorColor);
    _cformatColorButton->setColor(settings.cformatColor);
    _accelColorButton->setColor(settings.accelColor);


    _ledInStatusbarBtn->blockSignals(true);
    _ledInEditorBtn->blockSignals(true);
    _ledInStatusbarBtn->setChecked(settings.ledInStatusbar);
    _ledInEditorBtn->setChecked(!settings.ledInStatusbar);
    _ledInStatusbarBtn->blockSignals(false);
    _ledInEditorBtn->blockSignals(false);

    _ledColorButton->setColor(settings.ledColor);

    spellConfig->setNoRootAffix(settings.noRootAffix);
    spellConfig->setRunTogether(settings.runTogether);
    spellConfig->setClient(settings.spellClient);
    spellConfig->setEncoding(settings.spellEncoding);
    spellConfig->setDictionary(settings.spellDict);

    remIgnoredBtn->setChecked(settings.rememberIgnored);
    ignoreURLEdit->setURL(settings.ignoreURL);
}


EditorSettings EditorPreferences::settings() const
{
    EditorSettings settings;

    settings.autoUnsetFuzzy=_fuzzyButton->isChecked();
    settings.autoCheckArgs=_checkArgsButton->isChecked();
    settings.autoCheckAccel=_checkAccelButton->isChecked();
    settings.autoCheckEquation=_checkEquationButton->isChecked();
    settings.autoCheckContext=_checkContextButton->isChecked();
    settings.beepOnError=_beepButton->isChecked();
    settings.autoCheckColorError=_colorErrorButton->isChecked();


    settings.msgFont=_fontBox->font();

    settings.highlightSyntax=_hlSyntaxButton->isChecked();
    settings.highlightBg=_hlBgButton->isChecked();
    settings.quotes=_quotesButton->isChecked();
    settings.whitespacePoints=_spacePointsButton->isChecked();

    settings.quotedColor=_quotedColorButton->color();
    settings.errorColor=_errorColorButton->color();
    settings.cformatColor=_cformatColorButton->color();
    settings.accelColor=_accelColorButton->color();

    settings.bgColor=_bgColorButton->color();


    settings.ledInStatusbar=_ledInStatusbarBtn->isChecked();
    settings.ledColor=_ledColorButton->color();

    settings.noRootAffix=spellConfig->noRootAffix();
    settings.runTogether=spellConfig->runTogether();
    settings.spellClient=spellConfig->client();
    settings.spellEncoding=spellConfig->encoding();
    settings.spellDict=spellConfig->dictionary();

    settings.rememberIgnored=remIgnoredBtn->isChecked();
    settings.ignoreURL=ignoreURLEdit->url();

    return settings;
}

void EditorPreferences::defaults()
{
    _fuzzyButton->setChecked(Defaults::Editor::autoUnsetFuzzy);

    _checkArgsButton->setChecked(Defaults::Editor::autoCheckArgs);
    _checkAccelButton->setChecked(Defaults::Editor::autoCheckAccel);
    _checkEquationButton->setChecked(Defaults::Editor::autoCheckEquation);
    _checkContextButton->setChecked(Defaults::Editor::autoCheckContext);
    _beepButton->setChecked(Defaults::Editor::beepOnError);
    _colorErrorButton->setChecked(Defaults::Editor::autoCheckColorError);


    _fontBox->setFont(Defaults::Editor::msgFont());

    _hlSyntaxButton->setChecked(Defaults::Editor::highlightSyntax);
    _hlBgButton->setChecked(Defaults::Editor::highlightBg);
    _quotesButton->setChecked(Defaults::Editor::quotes);
    _spacePointsButton->setChecked(Defaults::Editor::whitespacePoints);

    _bgColorButton->setColor(Defaults::Editor::bgColor);
    _quotedColorButton->setColor(Defaults::Editor::quotedColor);
    _errorColorButton->setColor(Defaults::Editor::errorColor);
    _cformatColorButton->setColor(Defaults::Editor::cformatColor);
    _accelColorButton->setColor(Defaults::Editor::accelColor);


    _ledInStatusbarBtn->setChecked(Defaults::Editor::ledInStatusbar);
    _ledInEditorBtn->setChecked(!Defaults::Editor::ledInStatusbar);
    _ledColorButton->setColor(Defaults::Editor::ledColor);


    remIgnoredBtn->setChecked(Defaults::Editor::rememberIgnored);
    ignoreURLEdit->setURL(Defaults::Editor::ignoreURL());

    KSpellConfig spCfg;
    *spellConfig = spCfg;
}


void EditorPreferences::showOnlyFixedFonts(bool on)
{
   _fontBox->setFont(_fontBox->font(),on);
}


void EditorPreferences::ledWarning(bool show)
{
   if(show)
      KMessageBox::information( this,i18n("This option takes no effect until KBabel is restarted.") );
}



SearchPreferences::SearchPreferences(QList<ModuleInfo> ml,QWidget* parent)
         : KTabCtl(parent)
{
    moduleList = ml;
    moduleList.setAutoDelete(true);

    // first page
    QWidget* page= new QWidget(this);
    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    layout->addWidget(box);
    box->setMargin(KDialog::marginHint());
    _searchButton = new QCheckBox(i18n("Au&tomatically start search"),box);

    QWhatsThis::add(_searchButton,i18n(
        "<qt><p><b>Automatically start search</b></p>"
        "<p>If this is activated, the search is automatically started "
        "whenever you switch to another entry in the editor. You can "
        "choose where to search with the combo box <b>Default Dictionary</b>."
        "</p><p>You can also start searching manually by choosing an entry in "
        "the popup menu that appears either when clicking at "
        "<b>Dictionaries->Find ...</b> or keeping the dictionary button "
        "in the toolbar pressed for a while.</p></qt>"));

    QHBox *hb = new QHBox(box);
    QLabel *label = new QLabel(i18n("D&efault Dictionary:"),hb);
    _defaultModuleBox = new KComboBox(hb,"defaultModuleBox");
    _defaultModuleBox->setAutoCompletion(true);
    label->setBuddy(_defaultModuleBox);

    ModuleInfo *info;
    for(info = ml.first(); info != 0; info = ml.next())
    {
        _defaultModuleBox->insertItem(info->name);
    }

    QString msg=i18n("<qt><p><b>Default Dictionary</b></p>\n"
        "<p>Choose here where to search as default. "
        "This setting is used when searching is started automatically "
        "or when pressing the dictionary button in the toolbar.</p>"
        "<p>You can configure the different dictionaries by selecting "
        "the wanted dictionary from <b>Settings->Configure Dictionary</b>."
        "</p></qt>");

    QWhatsThis::add(_defaultModuleBox,msg);
    QWhatsThis::add(label,msg);


    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    addTab(page,i18n("&General"));


    defaults();


    setMinimumSize(sizeHint());
}


SearchPreferences::~SearchPreferences()
{
}


void SearchPreferences::setSettings(const SearchSettings& settings)
{
    _searchButton->setChecked(settings.autoSearch);

    int i=0;
    ModuleInfo *info;
    for(info = moduleList.first(); info != 0; info = moduleList.next())
    {
        if(settings.defaultModule == info->id)
            break;

        i++;
    }
    
    _defaultModuleBox->setCurrentItem(i);
}


SearchSettings SearchPreferences::settings()
{
    SearchSettings settings;
    
    settings.autoSearch=_searchButton->isChecked();
    
    int i=_defaultModuleBox->currentItem();
    ModuleInfo *info = moduleList.at(i);

    if(info)
        settings.defaultModule=info->id;

    return settings;
}



void SearchPreferences::defaults()
{
    _searchButton->setChecked(Defaults::Search::autoSearch);
    
    int i=0;
    ModuleInfo *info;
    for(info = moduleList.first(); info != 0; info = moduleList.next())
    {
        if(Defaults::Search::defaultModule == info->id)
            break;
        
        i++;
    }
    
    _defaultModuleBox->setCurrentItem(i);
}





CatManPreferences::CatManPreferences(QWidget* parent)
         : KTabCtl(parent)
{
    QWidget* page = new QWidget(this);

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    QLabel* label=new QLabel(i18n("&Base directory of PO-Files:"),box);
    QHBox* hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _poDirEdit = new KLineEdit(hbox);
    _poDirEdit->setCompletionObject( new KURLCompletion() );
    _poDirEdit->setAutoDeleteCompletionObject(true);
    _poDirEdit->setMinimumSize(250,_poDirEdit->sizeHint().height());
    label->setBuddy(_poDirEdit);
    QPushButton* browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browsePO()));


    label=new QLabel(i18n("Ba&se directory of POT-Files:"),box);
    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    _potDirEdit = new KLineEdit(hbox);
    _potDirEdit->setCompletionObject( new KURLCompletion() );
    _potDirEdit->setAutoDeleteCompletionObject(true);
    _potDirEdit->setMinimumSize(250,_potDirEdit->sizeHint().height());
    label->setBuddy(_potDirEdit);
    browse = new QPushButton(i18n("Browse..."),hbox);
    connect(browse,SIGNAL(clicked()),this,SLOT(browsePOT()));



    QWhatsThis::add(box,i18n("<qt><p><b>Base directories</b></p>\n"
     "<p>Type in the directories which contains all your PO- and POT-files.\n"
     "The files and the directories in these directories will then be merged to one\n"
     "tree.</p></qt>"));


    box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    _openWindowButton = new QCheckBox(i18n("O&pen files in new window"),box);


    QWhatsThis::add(_openWindowButton,i18n("<qt><p><b>Open files in new window</b></p>\n"
"<p>If this is activated all files that are opened from the Catalog Manager are opened\n"
"in a new window.</p></qt>"));

    _killButton = new QCheckBox( i18n("&Kill processes on exit") , box );

    QWhatsThis::add( _killButton , i18n("<qt><p><b>Kill processes on exit</b></p>\n"
"<p>If you check this, KBabel tries to kill the processes, that are not exited already when KBabel exits,\n"
"by sending a kill signal to them.</p>\n"
"<p>NOTE: It's not guaranteed, that the processes are killed.</p></qt>") );


    layout->addStretch(1);

    page->setMinimumSize(sizeHintForWidget(page));

    addTab( page , i18n("&General") );



    page = new QWidget( this );
    layout = new QVBoxLayout( page );
    layout->setSpacing( KDialog::spacingHint() );
    layout->setMargin( KDialog::marginHint() );

    box = new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for directories") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _dirCmdEdit = new CmdEdit( box );
    label = new QLabel( i18n("Replaceables:\n@PACKAGE@, @PODIR@, @POTDIR@"), box);

    QWhatsThis::add( box , i18n("<qt><p><b>Commands for directories</b></p>"
"<p>Insert here the commands you want to execute in directories from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the directory without path</li>"
"<li>@PODIR@: The name of the PO-directory with path</li>"
"<li>@POTDIR@: The name of the template directory with path</li></ul></p>"
"</qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
    addTab( page , i18n("D&irectory Commands") );


    page = new QWidget( this );
    layout = new QVBoxLayout( page );
    layout->setSpacing( KDialog::spacingHint() );
    layout->setMargin( KDialog::marginHint() );

    box=new QGroupBox( 1 , Qt::Horizontal , i18n("Commands for files") , page );
    box->setMargin( KDialog::marginHint() );
    layout->addWidget( box );

    _fileCmdEdit = new CmdEdit( box );
    label = new QLabel( i18n("Replaceables:\n"
"@PACKAGE@, @POFILE@,@POTFILE@,\n@PODIR@, @POTDIR@"), box);



    QWhatsThis::add( box , i18n("<qt><p><b>Commands for files</b></p>"
"<p>Insert here the commands you want to execute on files from "
"the Catalog Manager. The commands are then shown in the submenu "
"<b>Commands</b> in the Catalog Manager's context menu.</p>"
"<p>The following strings will be replaced in a command:<ul>"
"<li>@PACKAGE@: The name of the file without path and extension</li>"
"<li>@POFILE@: The name of the PO-file with path and extension</li>"
"<li>@POTFILE@: The name of the corresponding template file with path "
"and extension</li>"
"<li>@PODIR@: The name of the directory, the PO-file is in, with path</li>"
"<li>@POTDIR@: The name of the directory, the template file is in, with "
"path</li></ul></p></qt>") );



    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
    addTab( page , i18n("&File Commands") );



    defaults();

    setMinimumSize(sizeHint());
}


CatManPreferences::~CatManPreferences()
{
}


void CatManPreferences::setSettings(const CatManSettings& settings)
{
   _poDirEdit->setText(settings.poBaseDir);
   _potDirEdit->setText(settings.potBaseDir);

   _openWindowButton->setChecked(settings.openWindow);

   _dirCmdEdit->setCommands( settings.dirCommands , settings.dirCommandNames );
   _fileCmdEdit->setCommands( settings.fileCommands , settings.fileCommandNames );

   _killButton->setChecked( settings.killCmdOnExit );
}


CatManSettings CatManPreferences::settings() const
{
    CatManSettings settings;

    settings.poBaseDir=_poDirEdit->text();
    settings.potBaseDir=_potDirEdit->text();
    settings.openWindow=_openWindowButton->isChecked();

    _dirCmdEdit->commands( settings.dirCommands , settings.dirCommandNames );
    _fileCmdEdit->commands( settings.fileCommands , settings.fileCommandNames );

    settings.killCmdOnExit = _killButton->isChecked() ;

    return settings;
}

void CatManPreferences::defaults()
{
   _poDirEdit->setText(Defaults::CatalogManager::poBaseDir());
   _potDirEdit->setText(Defaults::CatalogManager::potBaseDir());

   _openWindowButton->setChecked(Defaults::CatalogManager::openWindow);

   _dirCmdEdit->setCommands( Defaults::CatalogManager::dirCommands()
                            , Defaults::CatalogManager::dirCommandNames() );
   _fileCmdEdit->setCommands( Defaults::CatalogManager::fileCommands()
                            , Defaults::CatalogManager::fileCommandNames() );

   _killButton->setChecked( Defaults::CatalogManager::killCmdOnExit );
}


void CatManPreferences::browsePO()
{
    QString name=KFileDialog::getExistingDirectory(_poDirEdit->text(),this);
    if(!name)
    {
       return;
    }

    _poDirEdit->setText(name);
}


void CatManPreferences::browsePOT()
{
    QString name=KFileDialog::getExistingDirectory(_potDirEdit->text(),this);
    if(!name)
    {
       return;
    }

    _potDirEdit->setText(name);
}


MiscPreferences::MiscPreferences(QWidget *parent)
                : QWidget(parent)
{
    QWidget* page = this;

    QVBoxLayout* layout=new QVBoxLayout(page);
    layout->setSpacing(KDialog::spacingHint());
    layout->setMargin(KDialog::marginHint());

    QGroupBox* box=new QGroupBox(1,Qt::Horizontal,page);
    box->setMargin(KDialog::marginHint());
    layout->addWidget(box);

    QHBox *hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    QLabel *label = new QLabel(i18n("&Marker for keyboard accelerator:"),hbox);
    accelMarkerEdit = new KLineEdit(hbox);
    accelMarkerEdit->setMaxLength(1);
    label->setBuddy(accelMarkerEdit);
    hbox->setStretchFactor(accelMarkerEdit,1);
    QString msg=i18n("<qt><p><b>Marker for keyboard accelerator</b></p>"
        "<p>Define here, what character marks the following "
        "character as keyboard accelerator. For example in Qt it is "
        "'&amp;' and in Gtk it is '_'.</p></qt>");
    QWhatsThis::add(label,msg);
    QWhatsThis::add(accelMarkerEdit,msg);



    hbox = new QHBox(box);
    hbox->setSpacing(KDialog::spacingHint());

    label = new QLabel(i18n("&Regular expression for context information:")
            ,hbox);
    contextInfoEdit = new KLineEdit(hbox);
    label->setBuddy(contextInfoEdit);
    hbox->setStretchFactor(contextInfoEdit,1);
    msg=i18n("<qt><p><b>Regular expression for context information</b></p>"
        "<p>Enter here a regular expression, which defines, what is "
        "a context information in the message and must not get "
        "translated.</p></qt>");
    QWhatsThis::add(label,msg);
    QWhatsThis::add(contextInfoEdit,msg);

    layout->addStretch(1);
    page->setMinimumSize(sizeHintForWidget(page));
}

MiscPreferences::~MiscPreferences()
{
}

void MiscPreferences::defaults()
{
   accelMarkerEdit->setText(Defaults::Misc::accelMarker);
   setContextInfo(Defaults::Misc::contextInfo.pattern());
}

void MiscPreferences::setSettings(const MiscSettings& settings)
{
   accelMarkerEdit->setText(settings.accelMarker);
   setContextInfo(settings.contextInfo.pattern());
}

MiscSettings MiscPreferences::settings() const
{
    MiscSettings settings;

    QString temp=accelMarkerEdit->text();
    if(temp.isEmpty())
    {
        temp=Defaults::Misc::accelMarker;
        accelMarkerEdit->setText(temp);
    }
    settings.accelMarker=temp[0];

    settings.contextInfo.setPattern(contextInfo());

    return settings;
}


QString MiscPreferences::contextInfo() const
{
    QString temp=contextInfoEdit->text();

    bool quoted=false;
    QString newStr;

    for(uint i=0; i<temp.length(); i++)
    {
        if(temp[i]=='n')
        {
            quoted=!quoted;
            newStr+=temp[i];
        }
        else if(temp[i]=='n' && quoted)
        {
            newStr[newStr.length()-1]='\n';
            quoted=false;
        }
        else
        {
            quoted=false;
            newStr+=temp[i];
        }
    }

    return newStr;
}

void MiscPreferences::setContextInfo(QString reg)
{
    reg.replace(QRegExp("\n"),"\\n");
    contextInfoEdit->setText(reg);
}

#include "prefwidgets.moc"
