/*
 *  This file is part of the KDE System Control Tool,
 *  Copyright (C)1999 Thorsten Westheider <twesthei@physik.uni-bielefeld.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ****************************************************************************/

#include <qfile.h>
#include <qtextstream.h>

#include "isapnpdb.h"
#include "isapnpdevice.h"
#include "isapnpvendors.h"

extern ISAPnPDataBase          isapnpdb;
extern ISAPnPVendorDictionary  isapnpvendors;


ISAPnPDevice::ISAPnPDevice(int cardno, QString cardidstr, 
			   int logno, QString logidstr, 
			   QString vendorid, uint devid, QString devname,
			   uint logid, QString logname) : _cardidstr(cardidstr),
                           				  _logidstr(logidstr),
                                                          _cardno(cardno),
                                                          _logno(logno),
                           				  _vendorid(vendorid),
			   				  _devicename(devname),
					                  _logname(logname),
					                  _deviceid(devid),
							  _logid(logid),
							  _active(false)
{
  lookupData();
}


/*
 * Public methods
 *****************/

ResourceBaseConfig  *ISAPnPDevice::settingsBasedOn(int *index)
{
  ResourceBaseConfig  *baseconf;
  
  for (baseconf = _conflist.first(); baseconf; baseconf = _conflist.next())
  {
    if (baseconf->contains(_reslist)) 
    {
      *index = _conflist.at();
      return baseconf;
    }
  }
    
  return _conflist.first(); // Well, it must be based on something... FIXME!
}


QString  ISAPnPDevice::deviceClassName()
{
  switch (_class)
  {
    case ISAPnPClassNetworkOther    : return i18n("Network controller");
    case ISAPnPClassMultimediaAudio : return i18n("Multimedia audio controller");
    case ISAPnPClassInputOther	    : return i18n("Input device controller");
    default	                    : return i18n("Unknown device");
  }
}


Device::devstate  ISAPnPDevice::deviceState()
{
  Resource  *res;

  /*
   * If the device requires I/O addresses, I guess there must be
   * a range of 2 bytes at least; so if range > 1 that indicates
   * those I/O addresse were found in /proc/ioports and thus the
   * driver for this device must have registered them, that is,
   * the driver is present - otherwise we don't know.
   ***************************************************************/
  
  for (res = firstResource(); res; res = nextResource())
    if ((res->resourceType() == Resource::ResIOAddress) && (res->resourceRange() > 1)) 
      return DeviceStateReady;
    
  return DeviceStateUnknown;
}


/*
 * Change resource setting on physical device
 *********************************************/
void  ISAPnPDevice::changePhysicalResources(QList<Resource>& newsettings)
{
  Resource  *res;
  QString   resstr;
  QFile     isapnp("/proc/isapnp");

  _reslist.clear();
  
  for (res = newsettings.first(); res; res = newsettings.next())
    _reslist.append(res->copy());
  
  if (isapnp.open(IO_WriteOnly))
  {
    QTextStream  s(&isapnp);
    
    s << "card " << _cardno << " " << _cardidstr << endl;
    s << "dev "  << _logno  << " " << _logidstr  << endl;
    
    for (res = _reslist.first(); res; res = _reslist.next())
    {
      resstr.sprintf("0x%x", res->resourceValue());
    
      s << res->resourceShortName()        << " ";
      s << res->resourceIndex()            << " ";
      s << resstr                          << endl;
    }
    
    s << "activate" << endl;
  
    isapnp.close();
  }
}


/*
 * Private methods
 ******************/

void  ISAPnPDevice::lookupData()
{
  ISAPnPDeviceInfo  *info;
  
  _vendorname = isapnpvendors[_vendorid];

  if ((info = isapnpdb.find(_vendorid, _deviceid, _logid)))
  {
    _baseclass = info->baseClass();
    _class     = info->isapnpClass();
  }
}
