/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "prefs.h"
#include "mozilla_i18n.h"
#include "mozilla_prefs.h"
#include "window.h"
#include "toolbar.h"
#include "embed.h"
#include "misc.h"
#include "mime.h"
#include "state.h"

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtklist.h>
#include <gtk/gtkcheckmenuitem.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-help.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-font-picker.h>
#include <libgnomeui/gnome-color-picker.h>

/* default MIME types */
struct { gchar *name; MimeAction action; } default_mimetypes[] =
{
	{ "audio/mp3",				MIME_SAVE_TO_DISK },
	{ "video/mpeg",				MIME_SAVE_TO_DISK },
	{ "vidoe/x-msvideo",			MIME_SAVE_TO_DISK },
	{ "application/x-tar",			MIME_SAVE_TO_DISK },
	{ "application/x-gzip",			MIME_SAVE_TO_DISK },
	{ "application/x-bzip",			MIME_SAVE_TO_DISK },
	{ "application/x-bzip2",		MIME_SAVE_TO_DISK },
	{ "application/x-compressed-tar",	MIME_SAVE_TO_DISK },
	{ "application/x-bzip-compressed-tar",	MIME_SAVE_TO_DISK },
	{ "application/x-tar",			MIME_SAVE_TO_DISK },	
	{ "application/zip",			MIME_SAVE_TO_DISK },	
	{ "application/x-rpm",			MIME_SAVE_TO_DISK },
	{ "application/octet-stream",		MIME_SAVE_TO_DISK },
	{ NULL, MIME_UNKNOWN } /* terminator -- must be last */
};

/* local types */
typedef struct
{ 
	gint value;
	gchar *name;
	GtkCTreeNode **node_ref;
	gboolean expand;
} PreferencesTreeItem;

/* local function prototypes */
static void preferences_load_mime (void);
static void preferences_save_mime (void);
static void preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml);
static PreferencesDialog *preferences_build_dialog (void);
static void preferences_build_trees (PreferencesDialog *pd);
static void build_tree (PreferencesDialog *pd, GtkCTreeNode *parent, 
			PreferencesTreeItem *item);
static void preferences_set_dialog_to_config (PreferencesDialog *pd);
static void preferences_set_config_to_dialog (PreferencesDialog *pd);
static void set_settings_menu_global (void);
static void set_config_from_optionmenu (GtkWidget *optionmenu,
					gchar *config_name);
static void set_config_from_fontpicker (GtkWidget *fontpicker,
					gchar *config_name);
static void set_config_from_spin_button (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_config_from_togglebutton (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_config_from_int_entry (GtkWidget *entry, gchar *config_name);
static void set_config_from_color (GtkWidget *colorpicker, gchar *config_name);
static void set_optionmenu_from_config (GtkWidget *optionmenu,
					gchar *config_name);
static void set_fontpicker_from_config (GtkWidget *fontpicker,
					gchar *config_name);
static void set_spin_button_from_config (GtkWidget *spinbutton,
					 gchar *config_name);
static void set_int_entry_from_config (GtkWidget *entry, gchar *config_name);
static void set_togglebutton_from_config (GtkWidget *togglebutton,
					  gchar *config_name);
static void set_list_from_config (GtkWidget *list, gchar *config_name);
static void set_config_from_list (GtkWidget *list, gchar *config_name);
static void set_color_from_config (GtkWidget *colorpicker, gchar *config_name);
static void preferences_mime_set_config_to_dialog (PreferencesDialog *pd);
static void preferences_mime_set_dialog_to_config (PreferencesDialog *pd);
void prefs_mime_edit_cb (GtkWidget *w, GaleonWindow *window);
static void prefs_dialog_save_geometry (PreferencesDialog *pd);
static void cb_reset_font_picker (GtkWidget *menu_item, PreferencesDialog *pd);
static void find_active_radio_button (gpointer data, gpointer chosen_int);

/**
 * preferences_open: open the prefs dialog
 */
void
preferences_open (void)
{
	static PreferencesDialog *pd = NULL;

	/* dialog available? */
	if (pd == NULL)
	{
		/* nope, build it */
		pd = preferences_build_dialog ();
	}
	else if (GTK_WIDGET_VISIBLE (pd->dialog))
	{
		/* already visible, raise to top */
		gdk_window_raise (GTK_WIDGET (pd->dialog)->window);
		return;
	}
	
	/* set values in dialog to stored values in prefs */
	preferences_set_dialog_to_config (pd);

	/* display it */
	gtk_widget_show (GTK_WIDGET (pd->dialog));
	window_set_layer (GTK_WIDGET (pd->dialog));
}

/**
 * preferences_close: close the preferences dialog window (hides it)
 */
void
preferences_close (PreferencesDialog *pd)
{
	prefs_dialog_save_geometry (pd);
	gtk_widget_hide (GTK_WIDGET (pd->dialog));
}

/**
 * preferences_load: load Galeon preferences needed at startup
 */
void
preferences_load (void)
{
	preferences_load_mime ();
}

/**
 * preferences_save: save all Galeon preferences
 */
void
preferences_save (void)
{
	preferences_save_mime ();
	gnome_config_sync ();
}

/**
 * preferences_apply: make any global changes required for system state
 * to match config state
 */
void
preferences_apply (PreferencesDialog *pd)
{
	gint tab_pos, n_embeds, always_show, popup = 0;
	GList *w, *e;

	preferences_set_config_to_dialog (pd);
	autobookmarks_generate ();
	mozilla_prefs_set ();
	set_settings_menu_global ();

	tab_pos = gnome_config_get_int (CONF_APPEARANCE_TABBED_POSITION);
	always_show =
		gnome_config_get_bool (CONF_APPEARANCE_TABBED_ALWAYS_SHOW);

	for (w = all_windows; w != NULL; w = g_list_next (w))
	{
		GaleonWindow *window = (GaleonWindow *)(w->data);

		/* recreate toolbars */
		// FIXME, dont do every time
		window_toolbars_recreate (window);
		
		/* set the tab position */
		gtk_notebook_set_tab_pos (GTK_NOTEBOOK (window->notebook),
					  tab_pos);

		/* check if the tab(s) should be visible */
		n_embeds = g_list_length (window->embed_list);
		if (n_embeds == 1)
		{
			GaleonEmbed *embed;

			e = window->embed_list;
 			embed = (GaleonEmbed *) (e->data);
			popup = embed->is_popup;
		}
		gtk_notebook_set_show_tabs (GTK_NOTEBOOK (window->notebook), 
					    (n_embeds > 1) | 
					    (always_show && !popup));

		/* update notebook tabs */
		for (e = window->embed_list; e != NULL; e = g_list_next (e))
		{
			GaleonEmbed *embed = (GaleonEmbed *) (e->data);

			embed_set_notebook_label (embed);
			embed_update_notebook_closebutton (embed);
		}
	}
}

/**
 * preferences_help: show the prefs Galeon manual page   
 */
void
preferences_help (void)
{
	GnomeHelpMenuEntry pref_help = { "galeon", "preferences.html" };
	gchar *file = NULL;

	/* Try to find the preferences help page.  If we fail, use ghelp
	   to load the index, instead */
	file = gnome_help_file_find_file("galeon", "preferences.html");

	if (file)
	{
	        gnome_help_display (NULL, &pref_help);
	}
	else
	{
		gnome_help_goto (NULL, "ghelp:galeon");
	}

	g_free(file);
}

/**
 * preferences_load_mime: load MIME types and details of their handlers
 */
static void
preferences_load_mime(void)
{
	gint i, count;
	gchar *mime_path;
	gchar **mime_type;
	MimeAction action;

	/* free up any existing list of MIME types */
	if (mime_types != NULL)
	{
		g_list_foreach(mime_types, (GFunc) mime_item_free, NULL);
		g_list_free(mime_types);
		mime_types = NULL;
	}

	/* working in the galeon/mime configuration hierarchy */
	gnome_config_push_prefix(CONF_MIME);
     
	/* find out the number of configured MIME types */
	count = gnome_config_get_int("count=-1");

	/* if this isn't yet set, copy in the defaults */
	if (count == -1)
	{
		for (i = 0; default_mimetypes[i].name != NULL; i++)
		{
			mime_set_action (default_mimetypes[i].name, 
					 default_mimetypes[i].action);
		}
	}

	/* otherwise (count >= 0) read from configuration structure */
	for (i = 0; i < count; i++)
	{
		gint c;

		/* generate the key */
		mime_path = g_strdup_printf("mime%d", i);

		/* lookup the data */
		gnome_config_get_vector(mime_path, &c, &mime_type);
		if (c != 2) 
		{ 
                        /* Probably a galeon <= 0.8.0 config file */
			int j;
			g_warning ("Strange MIME info in prefs, ignoring");
			g_message ("If you have recently updated your version "
				   "of Galeon,\n you can safely ignore the "
				   "previous message.\n Galeon will install "
				   "the default MIME actions now.");
			/* Remove all MIME prefs */
			gnome_config_set_int ("count", -1);
			gnome_config_sync ();
			/* cleanup */
			g_free (mime_path);
			for (j = 0; j < 2; j++) 
				g_free (mime_type[j]);
			g_free (mime_type);
			gnome_config_pop_prefix ();
			/* call ourselves again */
			preferences_load_mime ();
			return;

		}
		if (!strcmp(mime_type[1], "SAVE_TO_DISK"))
			action = MIME_SAVE_TO_DISK;
		else if (!strcmp(mime_type[1], "RUN_PROGRAM"))
			action = MIME_RUN_PROGRAM;
		else if (!strcmp(mime_type[1], "ASK_ACTION"))
			action = MIME_ASK_ACTION;
		else if (!strcmp(mime_type[1], "SAVE_WITH_GTM"))
			action = MIME_SAVE_WITH_GTM;
		else 
		{
			g_warning("preferences_load_mime: "
				  "unknown mime action");
			action = MIME_UNKNOWN;
		}

		/* build the mime type */
		mime_set_action (mime_type[0], action);

		/* free allocated strings */
		g_free(mime_path);
		g_free (mime_type[0]);
		g_free (mime_type[1]);
		g_free (mime_type);
	}

	/* ! CONF_MIME */
	gnome_config_pop_prefix();
}

/**
 * preferences_save_mime: save MIME types and handlers 
 */
void
preferences_save_mime(void)
{
	GList *l;
	gchar *mime_type[2];
	gchar *mime_path;
	gint i, count;
  
	/* working in the galeon/mime configuration hierarchy */
	gnome_config_push_prefix(CONF_MIME);

	/* write out the number of MIME types */
	count = g_list_length(mime_types);
	gnome_config_set_int("count",  count);

	/* write out each MIME type entry as string vectors */
	for (l = mime_types, i = 0; l != NULL; l = g_list_next(l), i++)
	{
		/* get the MIME type */
		MimeItem *m = (MimeItem *)(l->data);
		if (m->action == MIME_UNKNOWN) continue; /* do not save this */

		/* fill the entries into the vector */
		mime_type[0] = m->name;
		switch (m->action)
		{
		case MIME_SAVE_TO_DISK :
			mime_type[1] = "SAVE_TO_DISK"; break;
		case MIME_RUN_PROGRAM :
			mime_type[1] = "RUN_PROGRAM";  break;
		case MIME_ASK_ACTION :
			mime_type[1] = "ASK_ACTION";   break;
		case MIME_SAVE_WITH_GTM : 
			mime_type[1] = "SAVE_WITH_GTM"; break;
		case MIME_OVERRIDE_MOZILLA :
			mime_type[1] = "OVERRIDE_MOZILLA"; break;
		case MIME_IGNORE :
			mime_type[1] = "IGNORE"; break;
		case MIME_UNKNOWN :
			mime_type[1] = "UNKNOWN"; break;
			g_assert_not_reached (); break;
		}

		/* write the vector out to the config */
		mime_path = g_strdup_printf("mime%d", i);
		gnome_config_set_vector(mime_path, 2,
					(const gchar **)mime_type);

		/* free allocated strings */
		g_free(mime_path);
	}

	/* ! CONF_MIME */
	gnome_config_pop_prefix();
}

static PreferencesDialog *
preferences_build_dialog (void)
{
	PreferencesDialog *pd;
	GladeXML *gxml;

	/* not been generated yet... */
	pd = g_new0 (PreferencesDialog, 1);
	
	/* build the widgets */
	gxml = glade_widget_new ("preferences.glade", 
				 "preferences_dialog", &(pd->dialog), pd);
		
	/* lookup all the widgets */
	preferences_lookup_widgets (pd, gxml);

	/* no longer needed */
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* build the tree */
	preferences_build_trees (pd);
	
	/* get the saved geometry */
	state_load_window_state (GTK_WIDGET (pd->dialog), "prefs_dialog",
				600, 500);
	/* setup paned view */
	state_load_pane_pos (pd->hpaned, "prefs_dialog_hpane=150");
	
	/* initialize the default charset combo */
	gtk_combo_set_popdown_strings (GTK_COMBO (pd->default_charset_combo), 
				       sorted_charset_titles);

	/* return completed dialog */
	return pd;
}

static void
preferences_build_trees (PreferencesDialog *pd)
{
	GtkCTreeNode *browsing_node;
	GtkCTreeNode *user_interface_node;
	GtkCTreeNode *handlers_node;
	GtkCTreeNode *rendering_node;
	GtkCTreeNode *advanced_node;

	PreferencesTreeItem main_tree[] =
	{
		{  0, _("Browsing"),        &browsing_node,       TRUE },
		{  0, _("User Interface"),  &user_interface_node, TRUE },
		{  0, _("Handlers"),        &handlers_node,       TRUE },
		{  0, _("Rendering"),       &rendering_node,      TRUE },
		{  0, _("Advanced"),        &advanced_node,       TRUE },
		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	PreferencesTreeItem browsing_tree[] =
	{
		{  1, _("Homepage"),        NULL, FALSE },
		{  2, _("Bookmarks"),       NULL, FALSE },
		{  3, _("History"),         NULL, FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
        };

	PreferencesTreeItem user_interface_tree[] =
	{
		{  4, _("Layout"),          NULL, FALSE },
		{  5, _("Tabbed mode"),     NULL, FALSE },
		{  6, _("Themes"),          NULL, FALSE },
		{  7, _("Mouse"),           NULL, FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
        };

	PreferencesTreeItem handlers_tree[] =
	{
		{  8, _("Downloading"),     NULL, FALSE },
		{  9, _("Programs"),        NULL, FALSE },
		{ 10, _("MIME Types"),      NULL, FALSE }, 
		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	PreferencesTreeItem rendering_tree[] =
	{
		{ 11, _("Fonts/Colors"),    NULL, FALSE },
		{ 12, _("Languages"),       NULL, FALSE },
		{ -1, NULL, NULL, FALSE } /* terminator */
	};

	PreferencesTreeItem advanced_tree[] =
	{
		{ 13, _("Network"),         NULL, FALSE },
		{ 14, _("Filtering"),       NULL, FALSE },
		{ 15, _("Crash Recovery"),  NULL, FALSE },
		{ 16, _("Java"),            NULL, FALSE },
		{ -1, NULL, NULL } /* terminator */
	};

	/* build the preferences tree */
	build_tree (pd, NULL, main_tree);
	build_tree (pd, browsing_node, browsing_tree);
	build_tree (pd, user_interface_node, user_interface_tree);
	build_tree (pd, handlers_node, handlers_tree);
	build_tree (pd, rendering_node, rendering_tree);
	build_tree (pd, advanced_node, advanced_tree);
}

/**
 * preferences_set_config_to_dialog: read all the preferences values 
 * out of the dialog box widget and store them in the configuration space
 */
static void
preferences_set_config_to_dialog (PreferencesDialog *pd)
{
	gchar buf[128];
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* set each of the general configuration options */
	gnome_config_push_prefix ("/galeon/General/");
	set_config_from_editable (pd->startpage_entry, "startpage");
	set_config_from_optionmenu (pd->startpage_optionmenu,
				    "startpage_type");
	set_config_from_optionmenu (pd->new_optionmenu, "newpage_type");
	set_config_from_optionmenu (pd->bookmarks_optionmenu,
				    "bookmark_title");
	set_config_from_optionmenu (pd->bm_extra_items_optionmenu,
				   "bm_extra_items_location");
	set_config_from_togglebutton (pd->favicons_enabled_check,
				     "favicons_enabled");
	set_config_from_togglebutton (pd->show_zoom_check, "show_zoom");
	set_config_from_togglebutton (pd->show_spinner_check, "show_spinner");
	set_config_from_togglebutton (pd->allow_popups_checkbutton, 
				      "allow_popups");
	gnome_config_pop_prefix ();	

	/* autobookmarks preferences */
	gnome_config_push_prefix ("/galeon/AutoBookmarks/");
	set_config_from_togglebutton (pd->autobookmarks_enabled_check,
				      "enabled");
	set_config_from_spin_button (pd->autobookmarks_count_spin, "count");
	set_config_from_spin_button (pd->autobookmarks_shorten_spin,
				     "shorten");
	gnome_config_pop_prefix ();	
	
	/* process downloading preferences */
	gnome_config_push_prefix ("/galeon/Downloading/");
	set_config_from_togglebutton (pd->autodownload_checkbutton,
				     "auto_download");
	set_config_from_togglebutton (pd->ask_download_dir_checkbutton,
				     "ask_for_download_dir");
	set_config_from_optionmenu (pd->ftp_program_optionmenu, 
				   "ftp_program");
	set_config_from_editable (pd->ftp_command_entry, "ftp_command");
	set_config_from_editable (pd->download_dir_entry, "download_dir");
	gnome_config_pop_prefix ();
	
	/* process appearance preferences */
	gnome_config_push_prefix ("/galeon/Appearance/");
	set_config_from_togglebutton (pd->tabbed_enable_check, "tabbed");
	set_config_from_togglebutton (pd->tabbed_popups_enable_check, 
				     "tabbed_popups");
	set_config_from_togglebutton (pd->tabbed_autojump_check, 
				     "tabbed_autojump");
	set_config_from_togglebutton (pd->tabbed_insert_new_tabs_check, 
				     "tabbed_insert_new_tabs");
	set_config_from_optionmenu (pd->tabbed_position_optionmenu, 
				   "tabbed_position");
        set_config_from_radiobuttongroup (pd->tabbed_style_radiobutton1, 
                                   "tabbed_shorten_style");
	set_config_from_togglebutton (pd->tabbed_always_show_check, 
				     "tabbed_always_show");
	set_config_from_togglebutton (pd->tabbed_closebutton_check, 
				     "tabbed_closebutton");
	set_config_from_spin_button (pd->tabbed_shorten_spin, 
				     "tabbed_shorten");
        set_config_from_spin_button (pd->tabbed_shorten_spin2, 
                                    "tabbed_shorten_points");
	set_config_from_togglebutton (pd->tabbed_separator_check,
				     "tabbed_separator");
	set_config_from_togglebutton (pd->tabbed_vowel_check,
				      "tabbed_vowel");
	set_config_from_togglebutton (pd->tabbed_prefix_check,
				      "tabbed_prefix");
	set_config_from_optionmenu (pd->location_optionmenu, "url_location");
	set_config_from_optionmenu (pd->toolbar_optionmenu, "toolbar_type");
	set_config_from_optionmenu (pd->lang_encode, "lang_encode");

	menu = GTK_OPTION_MENU (pd->lang_encode)->menu;
	list = GTK_MENU_SHELL (menu)->children;
	item = gtk_menu_get_active (GTK_MENU (menu));
	index = g_list_index (list, item);

	g_snprintf (buf, 128, "serif_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, "sansserif_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, "monospace_font_%s", lang_encode_item[index]);
	set_config_from_fontpicker (pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "size_font_%s", lang_encode_item[index]);
	set_config_from_spin_button (pd->size_font_spin, buf);

	set_config_from_togglebutton (pd->use_own_fonts_checkbutton,
				      "use_own_fonts");
 	set_config_from_optionmenu (pd->default_font_optionmenu,
				    "default_font");
	set_config_from_togglebutton (pd->show_go_button_check,
				     "show_go_button");
	set_config_from_togglebutton (pd->show_up_button_check,
				     "show_up_button");
	set_config_from_togglebutton (pd->show_new_button_check,
				     "show_new_button");
	set_config_from_togglebutton (pd->show_home_button_check,
				      "show_home_button");
	set_config_from_togglebutton (pd->show_drag_event_box_check,
				      "show_drag_event_box");
	set_config_from_togglebutton (pd->nautilus_icons_checkbutton,
				     "nautilus_icons");
	set_config_from_togglebutton (pd->show_toolbars_in_fullscreen_check,
				     "show_toolbars_in_fullscreen");
	set_config_from_togglebutton (pd->show_menubar_in_fullscreen_check,
				     "show_menubar_in_fullscreen");
	set_config_from_togglebutton (pd->show_statusbar_in_fullscreen_check,
				     "show_statusbar_in_fullscreen");
	set_config_from_editable (pd->windows_title_entry, "windows_title");
	set_config_from_togglebutton (pd->fullscreen_stacking_checkbutton,
				     "fullscreen_stacking");
	set_config_from_editable (pd->spinner_dir_entry, "spinner_dir");
	set_config_from_color (pd->background_color, "background_color");
	set_config_from_color (pd->text_color, "text_color");
	set_config_from_color (pd->unvisited_link_color,
			       "unvisited_link_color");
	set_config_from_color (pd->visited_link_color, "visited_link_color");
	set_config_from_togglebutton (pd->use_own_colors_checkbutton,
				      "use_own_colors");
	set_config_from_togglebutton (pd->underline_links_checkbutton, 
				      "underline_links");
	set_config_from_spin_button (pd->smart_bm_entry_width_spin,
				     "smart_bm_entry_width");
	gnome_config_pop_prefix ();

	/* process mouse options */
	gnome_config_push_prefix ("/galeon/Mouse/");
	set_config_from_togglebutton (pd->history_popup_checkbutton,
				      "history_popup");
	set_config_from_optionmenu (pd->middle_button_optionmenu,
				    "middle_button_action");
	set_config_from_optionmenu (pd->right_button_optionmenu,
				    "right_button_action");
	set_config_from_togglebutton (pd->links_drag_checkbutton,
				      "links_drag");
	gnome_config_pop_prefix ();

	/* process completion preferences */
	gnome_config_push_prefix ("/galeon/completion/");
	set_config_from_togglebutton (pd->autocompletion_enabled_checkbutton,
				     "autocompletion");
	set_config_from_optionmenu (pd->completion_sort_optionmenu,
				   "sorting_type");
	gnome_config_pop_prefix ();

        /* process handlers preferences */
	gnome_config_push_prefix ("/galeon/Handlers/");
	set_config_from_optionmenu (pd->ftp_optionmenu, "ftp");
	set_config_from_togglebutton (pd->use_external_source_viewer_check,
				      "use_external_source_viewer");
	set_config_from_editable (pd->external_source_viewer_entry, 
				  "external_source_viewer");
	gnome_config_pop_prefix ();
 
        /* process advanced preferences */
	gnome_config_push_prefix ("/galeon/Advanced/");
	set_config_from_togglebutton (pd->proxy_use_checkbutton, 
				      "proxy_enabled");
	set_config_from_editable (pd->http_proxy_entry, "http_proxy");
	set_config_from_editable (pd->ssl_proxy_entry, "ssl_proxy");
	set_config_from_editable (pd->ftp_proxy_entry, "ftp_proxy");
	set_config_from_editable (pd->no_proxies_for_entry, "no_proxies_for");
	set_config_from_spin_button (pd->http_proxy_port_spin,
				     "http_proxy_port");
	set_config_from_spin_button (pd->ssl_proxy_port_spin, 
				     "ssl_proxy_port");
	set_config_from_spin_button (pd->ftp_proxy_port_spin, 
				     "ftp_proxy_port");
	set_config_from_optionmenu (pd->cache_compare_optionmenu, 
				    "cache_compare");
	set_config_from_int_entry (pd->disk_cache_entry, "disk_cache_size");
	set_config_from_int_entry (pd->mem_cache_entry, "mem_cache_size");
	set_config_from_togglebutton (pd->java_checkbutton, "java_enabled");
	set_config_from_togglebutton (pd->javascript_checkbutton,
				      "javascript_enabled");
	set_config_from_optionmenu (pd->image_optionmenu, 
				    "image_loading_type");
	set_config_from_togglebutton (pd->statusbar_rewrite_check,
				      "statusbar_rewrite");
	set_config_from_optionmenu (pd->crash_optionmenu, "crash_recovery");
	set_config_from_optionmenu (pd->autodetect_charset_optionmenu,
				    "autodetect_charset");
	set_config_from_editable (pd->default_charset_entry, 
				  "default_charset");
	set_config_from_list (pd->lang_list, "lang");
	set_config_from_optionmenu (pd->http_version_optionmenu, 
				    "http_version"); 
	set_config_from_togglebutton (pd->proxy_keepalive_checkbutton, 
				      "proxy_keepalive");
	set_config_from_togglebutton (pd->warn_cookie_checkbutton, 
				      "cookie_warn");
	set_config_from_optionmenu (pd->cookie_behavior_optionmenu, 
				    "cookie_behavior");
	set_config_from_togglebutton (pd->passwords_save_checkbutton,
				      "passwords_save");
	set_config_from_optionmenu (pd->wheel_nokey_action_optionmenu,
				    "wheel_nokey_action"); 
	set_config_from_optionmenu (pd->wheel_ctrl_action_optionmenu,
				    "wheel_ctrl_action"); 
	set_config_from_optionmenu (pd->wheel_shift_action_optionmenu,
				    "wheel_shift_action");
	set_config_from_optionmenu (pd->wheel_alt_action_optionmenu,
				    "wheel_alt_action");  
	set_config_from_spin_button (pd->wheel_nokey_step_spinbutton,
				     "wheel_nokey_step");
	set_config_from_spin_button (pd->wheel_ctrl_step_spinbutton,
				     "wheel_ctrl_step");
	set_config_from_spin_button (pd->wheel_shift_step_spinbutton,
				     "wheel_shift_step");
	set_config_from_spin_button (pd->wheel_alt_step_spinbutton,
				     "wheel_alt_step");
	set_config_from_togglebutton (pd->wheel_nokey_default_step_checkbutton,
				      "wheel_nokey_default_step");
	set_config_from_togglebutton (pd->wheel_ctrl_default_step_checkbutton,
				      "wheel_ctrl_default_step");
	set_config_from_togglebutton (pd->wheel_shift_default_step_checkbutton,
				      "wheel_shift_default_step");
	set_config_from_togglebutton (pd->wheel_alt_default_step_checkbutton,
				      "wheel_alt_default_step");
	set_config_from_optionmenu (pd->crash_behaviour_optionmenu,
				    "crash_behaviour");
	gnome_config_pop_prefix ();

	/* process history preferences */
	gnome_config_push_prefix ("/galeon/History/");
       	set_config_from_spin_button (pd->history_expire_spin, 
				     "expire");
	gnome_config_pop_prefix ();
	
	preferences_mime_set_config_to_dialog (pd);

	/* save all these preferences to disk */
	preferences_save ();
}

static void
preferences_set_dialog_to_config (PreferencesDialog *pd)
{
	GtkWidget *menu, *menu_item;
	gint smart_bm_entry_width;
	gchar buf[128];
	int i;

	/* fill general */
	gnome_config_push_prefix ("/galeon/General/");
	set_editable_from_config (pd->startpage_entry,
				 "startpage=www.gnome.org");
	set_optionmenu_from_config (pd->startpage_optionmenu,
				   "startpage_type=0");
	set_optionmenu_from_config (pd->new_optionmenu, "newpage_type=0");
	set_optionmenu_from_config (pd->bookmarks_optionmenu, 
				   "bookmark_title=0");
	set_optionmenu_from_config (pd->bm_extra_items_optionmenu,
				   "bm_extra_items_location=1");
	set_togglebutton_from_config (pd->favicons_enabled_check,
				     "favicons_enabled=TRUE");
	set_togglebutton_from_config (pd->show_zoom_check, "show_zoom=1");
	set_togglebutton_from_config (pd->show_spinner_check, 
				      "show_spinner=1");
	set_togglebutton_from_config (pd->allow_popups_checkbutton, 
				      "allow_popups=1");
	gnome_config_pop_prefix ();

	/* autobookmarks preferences */
	gnome_config_push_prefix ("/galeon/AutoBookmarks/");
	set_togglebutton_from_config (pd->autobookmarks_enabled_check,
				     "enabled=TRUE");
	set_spin_button_from_config (pd->autobookmarks_count_spin, "count=15");
	set_spin_button_from_config (pd->autobookmarks_shorten_spin, 
				    "shorten=25");
	gnome_config_pop_prefix ();
	
	/* process downloading preferences */
	gnome_config_push_prefix ("/galeon/Downloading/");
	set_togglebutton_from_config (pd->autodownload_checkbutton,
				     "auto_download=0");
	set_togglebutton_from_config (pd->ask_download_dir_checkbutton,
				     "ask_for_download_dir=0");
	set_optionmenu_from_config (pd->ftp_program_optionmenu, 
				    "ftp_program=0");
	set_editable_from_config (pd->ftp_command_entry, "ftp_command=");
	set_editable_from_config (pd->download_dir_entry, "download_dir=");
	gnome_config_pop_prefix ();
	
	/* process appearance preferences */
	gnome_config_push_prefix ("/galeon/Appearance/");
	set_togglebutton_from_config (pd->tabbed_enable_check, "tabbed=FALSE");
	set_togglebutton_from_config (pd->tabbed_popups_enable_check, 
				     "tabbed_popups=FALSE");
	set_togglebutton_from_config (pd->tabbed_autojump_check, 
				     "tabbed_autojump=FALSE");
	set_togglebutton_from_config (pd->tabbed_insert_new_tabs_check, 
				     "tabbed_insert_new_tabs=FALSE");
	set_optionmenu_from_config (pd->tabbed_position_optionmenu, 
				   "tabbed_position=2");
        set_radiobuttongroup_from_config (pd->tabbed_style_radiobutton1, 
                                   "tabbed_shorten_style=0");
	set_togglebutton_from_config (pd->tabbed_always_show_check, 
				   "tabbed_always_show=FALSE");
	set_togglebutton_from_config (pd->tabbed_closebutton_check, 
				   "tabbed_closebutton=TRUE");
	set_spin_button_from_config (pd->tabbed_shorten_spin,
				    "tabbed_shorten=30");
        set_spin_button_from_config (pd->tabbed_shorten_spin2,
                                    "tabbed_shorten_points=30");
	set_togglebutton_from_config (pd->tabbed_separator_check,
				    "tabbed_separator=TRUE");
	set_togglebutton_from_config (pd->tabbed_vowel_check,
				    "tabbed_vowel=FALSE");
	set_togglebutton_from_config (pd->tabbed_prefix_check,
				    "tabbed_prefix=TRUE");
	set_optionmenu_from_config (pd->location_optionmenu, "url_location=1");
	set_optionmenu_from_config (pd->toolbar_optionmenu, "toolbar_type=0");

	/* FIXME... */
	menu = gtk_menu_new ();
	for (i = 0; i < LANG_ENC_NUM; i++)
	{
	   menu_item = gtk_menu_item_new_with_label 
		   (gettext (lang_encode_name[i]));
	   gtk_menu_append (GTK_MENU (menu), menu_item);
	   gtk_object_set_data (GTK_OBJECT (menu_item), "index",
				GINT_TO_POINTER (i));
	   gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			       GTK_SIGNAL_FUNC (cb_reset_font_picker), pd);
	   gtk_widget_show (menu_item);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (pd->lang_encode), menu );
	set_optionmenu_from_config(pd->lang_encode, "lang_encode=0");

	i = gnome_config_get_int("lang_encode=0");
	g_snprintf (buf, 128, ("serif_font_%s=-*-times-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, ("sansserif_font_%s=-*-helvetica-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, ("monospace_font_%s=-*-courier-medium-r-"
			       "normal-*-14-*-*-*-*-*-*-*"),
		    lang_encode_item[i]);
	set_fontpicker_from_config(pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "size_font_%s=14", lang_encode_item[i]);
	set_spin_button_from_config (pd->size_font_spin, buf);

	set_togglebutton_from_config (pd->use_own_fonts_checkbutton, 
				     "use_own_fonts=0");
 	set_optionmenu_from_config (pd->default_font_optionmenu,
				   "default_font=1");
	set_togglebutton_from_config (pd->show_go_button_check,
				     "show_go_button=FALSE");
	set_togglebutton_from_config (pd->show_up_button_check,
				     "show_up_button=FALSE");
	set_togglebutton_from_config (pd->show_new_button_check,
				     "show_new_button=FALSE");
	set_togglebutton_from_config (pd->show_home_button_check,
				      "show_home_button=TRUE");
	set_togglebutton_from_config (pd->show_drag_event_box_check,
				      "show_drag_event_box=TRUE");
	set_togglebutton_from_config (pd->nautilus_icons_checkbutton,
				     "nautilus_icons=FALSE");
	set_togglebutton_from_config (pd->show_toolbars_in_fullscreen_check,
				     "show_toolbars_in_fullscreen=FALSE");
	set_togglebutton_from_config (pd->show_menubar_in_fullscreen_check,
				     "show_menubar_in_fullscreen=FALSE");
	set_togglebutton_from_config (pd->show_statusbar_in_fullscreen_check,
				     "show_statusbar_in_fullscreen=FALSE");
	set_editable_from_config (pd->windows_title_entry,
				 "windows_title=%s - Galeon");
	set_togglebutton_from_config (pd->fullscreen_stacking_checkbutton,
				     "fullscreen_stacking=TRUE");
	set_editable_from_config (pd->spinner_dir_entry,
				 "spinner_dir=" SHARE_DIR "/spinners/pipeon/");
	set_color_from_config (pd->background_color, 
			       "background_color=#FFFFFF");
	set_color_from_config (pd->text_color, "text_color=#000000");
	set_color_from_config (pd->unvisited_link_color, 
			       "unvisited_link_color=#0000FF");
	set_color_from_config (pd->visited_link_color, 
			       "visited_link_color=#FF0000");
	set_togglebutton_from_config (pd->use_own_colors_checkbutton,
				      "use_own_colors=0");
	set_togglebutton_from_config (pd->underline_links_checkbutton, 
				      "underline_links=TRUE");
	set_spin_button_from_config (pd->smart_bm_entry_width_spin,
				     "smart_bm_entry_width=100");
	smart_bm_entry_width = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON (pd->smart_bm_entry_width_spin));
	gtk_widget_set_usize (pd->smart_bm_entry_width_spin,
			      smart_bm_entry_width, -2);
	gnome_config_pop_prefix ();

	/* process mouse options */
	gnome_config_push_prefix ("/galeon/Mouse/");
	set_togglebutton_from_config (pd->history_popup_checkbutton,
				      "history_popup=FALSE");
	set_optionmenu_from_config (pd->middle_button_optionmenu,
				    "middle_button_action=1");
	set_optionmenu_from_config (pd->right_button_optionmenu,
				    "right_button_action=0");
	set_togglebutton_from_config (pd->links_drag_checkbutton,
				      "links_drag=FALSE");
	gnome_config_pop_prefix ();

	/* process completion preferences */
	gnome_config_push_prefix ("/galeon/completion/");
	set_togglebutton_from_config (pd->autocompletion_enabled_checkbutton,
				     "autocompletion=TRUE");
	set_optionmenu_from_config (pd->completion_sort_optionmenu,
				   "sorting_type=0");
	gnome_config_pop_prefix ();
	
        /* process handlers preferences */
	gnome_config_push_prefix ("/galeon/Handlers/");
	set_optionmenu_from_config (pd->ftp_optionmenu, "ftp=0");
	set_togglebutton_from_config (pd->use_external_source_viewer_check, 
				      "use_external_source_viewer=FALSE");
	set_editable_from_config (pd->external_source_viewer_entry, 
				  "external_source_viewer=gnome-edit");
	gnome_config_pop_prefix ();
 
        /* process advanced preferences */
	gnome_config_push_prefix ("/galeon/Advanced/");
	set_togglebutton_from_config (pd->proxy_use_checkbutton,
				     "proxy_enabled=FALSE");
	set_editable_from_config (pd->http_proxy_entry, "http_proxy=");
	set_editable_from_config (pd->ssl_proxy_entry, "ssl_proxy=");
	set_editable_from_config (pd->ftp_proxy_entry, "ftp_proxy=");
	set_editable_from_config (pd->no_proxies_for_entry, "no_proxies_for=");
	set_spin_button_from_config (pd->http_proxy_port_spin,
				    "http_proxy_port=8080");
	set_spin_button_from_config (pd->ssl_proxy_port_spin,
				    "ssl_proxy_port=8080");
	set_spin_button_from_config (pd->ftp_proxy_port_spin,
				    "ftp_proxy_port=8080");
	set_optionmenu_from_config (pd->http_version_optionmenu,
				   "http_version=1.1");
	set_togglebutton_from_config (pd->proxy_keepalive_checkbutton,
				     "proxy_keepalive=TRUE");
	set_optionmenu_from_config (pd->cache_compare_optionmenu,
				    "cache_compare=0");
	set_int_entry_from_config (pd->disk_cache_entry,
				   "disk_cache_size=7680");
	set_int_entry_from_config (pd->mem_cache_entry, "mem_cache_size=1024");
	set_togglebutton_from_config (pd->java_checkbutton, "java_enabled=1");
	set_togglebutton_from_config (pd->javascript_checkbutton,
				     "javascript_enabled=1");
	set_optionmenu_from_config (pd->image_optionmenu,
				   "image_loading_type=0");
	set_togglebutton_from_config (pd->statusbar_rewrite_check,
				  "statusbar_rewrite=TRUE");
	set_optionmenu_from_config (pd->crash_optionmenu, "crash_recovery=1");
	set_optionmenu_from_config (pd->autodetect_charset_optionmenu,
				   "autodetect_charset=0");
 	set_editable_from_config (pd->default_charset_entry,
				  "default_charset=Western (ISO-8859-1)"); 
 	set_list_from_config (pd->lang_list,"lang=English\\ (en)"); 
	set_togglebutton_from_config (pd->warn_cookie_checkbutton,
				     "cookie_warn=FALSE");
	set_optionmenu_from_config (pd->cookie_behavior_optionmenu,
				   "cookie_behavior=0");
	set_togglebutton_from_config (pd->passwords_save_checkbutton,
				     "passwords_save=FALSE");
	set_optionmenu_from_config (pd->wheel_nokey_action_optionmenu,
				   "wheel_nokey_action=0"); 
	set_optionmenu_from_config (pd->wheel_ctrl_action_optionmenu,
				   "wheel_ctrl_action=2"); 
	set_optionmenu_from_config (pd->wheel_shift_action_optionmenu,
				   "wheel_shift_action=1");
	set_optionmenu_from_config (pd->wheel_alt_action_optionmenu,
				   "wheel_alt_action=3");  
	set_spin_button_from_config (pd->wheel_nokey_step_spinbutton,
				    "wheel_nokey_step=1");
	set_spin_button_from_config (pd->wheel_ctrl_step_spinbutton,
				    "wheel_ctrl_step=1");
	set_spin_button_from_config (pd->wheel_shift_step_spinbutton,
				    "wheel_shift_step=1");
	set_spin_button_from_config (pd->wheel_alt_step_spinbutton,
				    "wheel_alt_step=1");
	set_togglebutton_from_config (pd->wheel_nokey_default_step_checkbutton,
				     "wheel_nokey_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_ctrl_default_step_checkbutton,
				     "wheel_ctrl_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_shift_default_step_checkbutton,
				     "wheel_shift_default_step=TRUE");
	set_togglebutton_from_config (pd->wheel_alt_default_step_checkbutton,
				     "wheel_alt_default_step=TRUE");
	set_optionmenu_from_config (pd->crash_behaviour_optionmenu,
				   "crash_behaviour=0");
	gnome_config_pop_prefix ();

	/* process history preferences */
	gnome_config_push_prefix ("/galeon/History/");
       	set_spin_button_from_config (pd->history_expire_spin, 
				     "expire=100");
	gnome_config_pop_prefix ();

        /* process mime types */
	preferences_mime_set_dialog_to_config (pd);
}

static void
preferences_mime_set_config_to_dialog (PreferencesDialog *pd)
{
	int i;
	MimeItem *item;

	/* free up any existing list of MIME types */
	if (mime_types != NULL)
	{
		g_list_foreach (mime_types, (GFunc)mime_item_free, NULL);
		g_list_free (mime_types);
		mime_types = NULL;
	}

	for (i = 0; i < GTK_CLIST (pd->mime_list)->rows; i++)
	{
		item = gtk_clist_get_row_data (GTK_CLIST (pd->mime_list), i);
		mime_types = g_list_append (mime_types, 
					    mime_item_clone (item));
	}
}

static void
preferences_mime_set_dialog_to_config (PreferencesDialog *pd)
{
	GList *l;
	gchar *text[3] = { NULL, NULL, NULL};
	GtkWidget *clist = pd->mime_list;
	GtkWidget *action_optionmenu = pd->mime_action_optionmenu;
	gint row;
        GtkWidget *menuitem;
	GtkWidget *action_menu;
        
        /* Connect the action menu signals, 
	 * I can't connect them in a better way :(*/
        action_menu = GTK_OPTION_MENU (action_optionmenu)->menu;
        for (l = GTK_MENU_SHELL (action_menu)->children; 
	     l != NULL; l = l->next)
	{
		menuitem = GTK_WIDGET (l->data);
		gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
				    prefs_mime_edit_cb, pd);
	}

	/* freeze and empty the list */
	gtk_clist_freeze (GTK_CLIST(clist));
	gtk_clist_clear (GTK_CLIST(clist));

	/* re-insert every element */
	for (l = mime_types; l != NULL; l = g_list_next(l))
	{
		MimeItem *m = (MimeItem *)(l->data);
		MimeItem *clone;

		/* insert the row */
		text[0] = m->name;
		switch (m->action) {
		case MIME_SAVE_WITH_GTM:
			text[1] = N_("Save to disk with External Download Handler");
			break;
		case MIME_SAVE_TO_DISK:
			text[1] = N_("Save to disk with Built-in Download Handler");
			break;
		case MIME_RUN_PROGRAM:  
			text[1] = N_("Run with Helper App");
			break;
		case MIME_OVERRIDE_MOZILLA:
		case MIME_UNKNOWN:
			text[1] = N_("Unknown");
			break;
		case MIME_IGNORE:
			text[1] = N_("Ignored");
			break;
		case MIME_ASK_ACTION:
			text[1] = N_("Ask the user");
			break;
		}
		row = gtk_clist_append (GTK_CLIST(clist), text);
		clone = mime_item_clone (m);
		gtk_clist_set_row_data (GTK_CLIST(clist), row, clone);
	}

	/* thaw the list -- make changes visible */
	gtk_clist_thaw (GTK_CLIST(clist));	
}

static void 
set_settings_menu_global (void)
{
	GList *l;

	for (l = all_windows; l != NULL; l = l->next)
	{
		set_settings_menu_window((GaleonWindow *) l->data);
	}
}

void
set_settings_menu_window (GaleonWindow *window)
{
	gint load_images;
	gint animate;
	gboolean use_own_fonts;
	gboolean use_own_colors;
	gboolean enable_java;
	gboolean enable_javascript;
	gboolean proxy_enabled;

	load_images = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	animate = gnome_config_get_int (CONF_ADVANCED_ANIMATE_TYPE);

	use_own_fonts = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_FONTS);
	use_own_colors = 
		gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_COLORS);
	enable_java = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
	enable_javascript = 
		gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	proxy_enabled = gnome_config_get_bool (CONF_ADVANCED_PROXY_ENABLED);

	if (!window->menuBarOn) return;

	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->use_own_fonts), 
		 use_own_fonts);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->use_own_colors), 
		 use_own_colors);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_java), 
		 enable_java);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_javascript),
		 enable_javascript);
	gtk_check_menu_item_set_active 
		(GTK_CHECK_MENU_ITEM (window->enable_proxy),
		 proxy_enabled);
	
	switch (load_images) {
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->load_images_from_current_server_only), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->load_images_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_ADVANCED_IMAGE_LOADING_TYPE);
	}

	switch (animate)
	{
	case 0:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_always), TRUE);
		break;
	case 1:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM 
			 (window->animate_once_through), TRUE);
		break;
	case 2:
		gtk_check_menu_item_set_active
			(GTK_CHECK_MENU_ITEM (window->animate_never), TRUE);
		break;
	default:
		g_warning ("strange option for " CONF_ADVANCED_ANIMATE_TYPE);
	}
}

static void
preferences_lookup_widgets (PreferencesDialog *pd, GladeXML *gxml)
{
	WidgetLookup lookup_table[] =
	{
{ "hpaned2", /* FIXME */            &(pd->hpaned)                            },
{ "notebook1", /* FIXME */          &(pd->notebook)                          },
{ "menu_ctree",                     &(pd->menu_ctree)                        },
{ "preferences_mime_list",          &(pd->mime_list)                         },
{ "prefs_mime_action_optionmenu",   &(pd->mime_action_optionmenu)            },
{ "prefs_mime_name_entry",          &(pd->mime_name_entry)                   },
{ "scrolledwindow9",                &(pd->scrolledwindow9)                   },
{ "themes_page",                    &(pd->themes_page)                       },
{ "spinner_iconlist",               &(pd->iconlist)                          },
{ "smart_bm_example_entry",         &(pd->smart_bm_example_entry)            },
{ "lang_entry",                     &(pd->lang_entry)                        },
{ "autodownload_checkbutton",       &(pd->autodownload_checkbutton)          },
{ "ask_download_dir_checkbutton",   &(pd->ask_download_dir_checkbutton)      },
{ "location_optionmenu",            &(pd->location_optionmenu)               },
{ "toolbar_optionmenu",             &(pd->toolbar_optionmenu)                },
{ "startpage_entry",                &(pd->startpage_entry)                   },
{ "startpage_optionmenu",           &(pd->startpage_optionmenu)              },
{ "new_optionmenu",                 &(pd->new_optionmenu)                    },
{ "bookmarks_optionmenu",           &(pd->bookmarks_optionmenu)              },
{ "bm_extra_items_optionmenu",      &(pd->bm_extra_items_optionmenu)         },
{ "favicons_enabled_check",         &(pd->favicons_enabled_check)            },
{ "autobookmarks_enabled_check",    &(pd->autobookmarks_enabled_check)       },
{ "autobookmarks_count_spin",       &(pd->autobookmarks_count_spin)          },
{ "autobookmarks_shorten_spin",     &(pd->autobookmarks_shorten_spin)        },
{ "smart_bm_entry_width_spin",      &(pd->smart_bm_entry_width_spin)         },
{ "java_checkbutton",               &(pd->java_checkbutton)                  },
{ "javascript_checkbutton",         &(pd->javascript_checkbutton)            },
{ "download_dir_entry",             &(pd->download_dir_entry)                },
{ "tabbed_enable_check",            &(pd->tabbed_enable_check)               },
{ "tabbed_popups_enable_check",     &(pd->tabbed_popups_enable_check)        },
{ "tabbed_autojump_check",          &(pd->tabbed_autojump_check)             },
{ "tabbed_insert_new_tabs_check",   &(pd->tabbed_insert_new_tabs_check)      },
{ "tabbed_position_optionmenu",     &(pd->tabbed_position_optionmenu)        },
{ "radiobutton1", /* FIXME */       &(pd->tabbed_style_radiobutton1)         },
{ "tabbed_always_show_check",       &(pd->tabbed_always_show_check)          },
{ "tabbed_closebutton_check",       &(pd->tabbed_closebutton_check)          },
{ "tabbed_shorten_spin",            &(pd->tabbed_shorten_spin)               },
{ "tabbed_shorten_spin2",/* FIXME */&(pd->tabbed_shorten_spin2)              },
{ "tabbed_separator_check",         &(pd->tabbed_separator_check)            },
{ "tabbed_vowel_check",             &(pd->tabbed_vowel_check)                },
{ "tabbed_prefix_check",            &(pd->tabbed_prefix_check)               },
{ "autocompletion_enabled_checkbutton", 
                                   &(pd->autocompletion_enabled_checkbutton) },
{ "completion_sort_optionmenu",     &(pd->completion_sort_optionmenu)        },
{ "show_zoom_checkbutton",          &(pd->show_zoom_check)                   },
{ "show_spinner_checkbutton",       &(pd->show_spinner_check)                },
{ "http_proxy_entry",               &(pd->http_proxy_entry)                  },
{ "ssl_proxy_entry",                &(pd->ssl_proxy_entry)                   },
{ "ftp_proxy_entry",                &(pd->ftp_proxy_entry)                   },
{ "no_proxies_for_entry",           &(pd->no_proxies_for_entry)              },
{ "http_proxy_port_spin",           &(pd->http_proxy_port_spin)              },
{ "ssl_proxy_port_spin",            &(pd->ssl_proxy_port_spin)               },
{ "ftp_proxy_port_spin",            &(pd->ftp_proxy_port_spin)               },
{ "disk_cache_entry",               &(pd->disk_cache_entry)                  },
{ "mem_cache_entry",                &(pd->mem_cache_entry)                   },
{ "lang_encode",                    &(pd->lang_encode)                       },
{ "serif_font_fpicker",             &(pd->serif_font_fpicker)                },
{ "sansserif_font_fpicker",         &(pd->sansserif_font_fpicker)            },
{ "monospace_font_fpicker",         &(pd->monospace_font_fpicker)            },
{ "size_font_spin",                 &(pd->size_font_spin)                    },
{ "use_own_fonts_checkbutton",      &(pd->use_own_fonts_checkbutton)         },
{ "default_font_optionmenu",        &(pd->default_font_optionmenu)           },
{ "history_expire_spin",            &(pd->history_expire_spin)               },
{ "image_optionmenu",               &(pd->image_optionmenu)                  },
{ "ftp_optionmenu",                 &(pd->ftp_optionmenu)                    },
{ "ftp_program_optionmenu",         &(pd->ftp_program_optionmenu)            },
{ "ftp_command_entry",              &(pd->ftp_command_entry)                 },
{ "show_go_button_check",           &(pd->show_go_button_check)              },
{ "show_up_button_check",           &(pd->show_up_button_check)              },
{ "show_new_button_check",          &(pd->show_new_button_check)             },
{ "show_home_button_check",         &(pd->show_home_button_check)            },
{ "show_drag_event_box_check",      &(pd->show_drag_event_box_check)         },
{ "show_toolbars_in_fullscreen_check",
                                  &(pd->show_toolbars_in_fullscreen_check)   },
{ "show_menubar_in_fullscreen_check", 
                                  &(pd->show_menubar_in_fullscreen_check)    },
{ "show_statusbar_in_fullscreen_check",
                                  &(pd->show_statusbar_in_fullscreen_check)  },
{ "use_external_source_viewer_check",
                                  &(pd->use_external_source_viewer_check)    },
{ "external_source_viewer_entry",   &(pd->external_source_viewer_entry)      },
{ "crash_optionmenu",               &(pd->crash_optionmenu)                  },
{ "allow_popups_checkbutton",       &(pd->allow_popups_checkbutton)          },
{ "windows_title_entry",            &(pd->windows_title_entry)               },
{ "fullscreen_stacking_checkbutton",&(pd->fullscreen_stacking_checkbutton)   },
{ "spinner_dir_entry",              &(pd->spinner_dir_entry)                 },
{ "autodetect_charset_optionmenu",  &(pd->autodetect_charset_optionmenu)     },
{ "default_charset_combo",          &(pd->default_charset_combo)             },
{ "default_charset_entry",          &(pd->default_charset_entry)             },
{ "lang_list",                      &(pd->lang_list)                         },
{ "background_color",               &(pd->background_color)                  },
{ "text_color",                     &(pd->text_color)                        },
{ "unvisited_link_color",           &(pd->unvisited_link_color)              },
{ "visited_link_color",             &(pd->visited_link_color)                },
{ "use_own_colors_checkbutton",     &(pd->use_own_colors_checkbutton)        },
{ "underline_links_checkbutton",    &(pd->underline_links_checkbutton)       },
{ "proxy_keepalive_checkbutton",    &(pd->proxy_keepalive_checkbutton)       },
{ "proxy_use_checkbutton",          &(pd->proxy_use_checkbutton)             },
{ "http_version_optionmenu",        &(pd->http_version_optionmenu)           },
{ "history_popup_checkbutton",      &(pd->history_popup_checkbutton)         },
{ "middle_button_optionmenu",       &(pd->middle_button_optionmenu)          },
{ "right_button_optionmenu",        &(pd->right_button_optionmenu)           },
{ "cookie_behavior_optionmenu",     &(pd->cookie_behavior_optionmenu)        },
{ "cookie_manager_button",          &(pd->cookie_manager_button)             },
{ "warn_cookie_checkbutton",        &(pd->warn_cookie_checkbutton)           },
{ "passwords_save_checkbutton",     &(pd->passwords_save_checkbutton)        },
{ "wheel_nokey_action_optionmenu",  &(pd->wheel_nokey_action_optionmenu)     },
{ "wheel_nokey_step_spinbutton",    &(pd->wheel_nokey_step_spinbutton)       },
{ "wheel_nokey_default_step_checkbutton",
                                 &(pd->wheel_nokey_default_step_checkbutton) },
{ "wheel_alt_action_optionmenu",    &(pd->wheel_alt_action_optionmenu)       },
{ "wheel_alt_step_spinbutton",      &(pd->wheel_alt_step_spinbutton)         },
{ "wheel_alt_default_step_checkbutton",
                                 &(pd->wheel_alt_default_step_checkbutton)   },
{ "wheel_ctrl_action_optionmenu",   &(pd->wheel_ctrl_action_optionmenu)      },
{ "wheel_ctrl_step_spinbutton",     &(pd->wheel_ctrl_step_spinbutton)        },
{ "wheel_ctrl_default_step_checkbutton",
                                 &(pd->wheel_ctrl_default_step_checkbutton)  },
{ "wheel_shift_action_optionmenu",  &(pd->wheel_shift_action_optionmenu)     },
{ "wheel_shift_step_spinbutton",    &(pd->wheel_shift_step_spinbutton)       },
{ "wheel_shift_default_step_checkbutton", 
                                 &(pd->wheel_shift_default_step_checkbutton) },
{ "crash_behaviour_optionmenu",     &(pd->crash_behaviour_optionmenu)        },
{ "links_drag_checkbutton",         &(pd->links_drag_checkbutton)            },
{ "nautilus_icons_checkbutton",     &(pd->nautilus_icons_checkbutton)        },
{ "cache_compare_optionmenu",       &(pd->cache_compare_optionmenu)          },
{ "statusbar_rewrite_check",        &(pd->statusbar_rewrite_check)           },

{ NULL, NULL } /* terminator, must be last */
        };

        /* look them all up */
        lookup_widgets (gxml, lookup_table);
}

static void
build_tree (PreferencesDialog *pd, GtkCTreeNode *parent, 
	    PreferencesTreeItem *item)
{
	GtkCTreeNode *node;
	gchar *text[] = { NULL, NULL };

	for (; item->name != NULL; item++)
	{
		text[0] = item->name;
		node = gtk_ctree_insert_node 
			(GTK_CTREE (pd->menu_ctree), parent,
			 NULL, text, 0, NULL,
			 NULL, NULL, NULL,
			 FALSE, item->expand);

		gtk_ctree_node_set_row_data (GTK_CTREE (pd->menu_ctree), node,
					     GINT_TO_POINTER (item->value));

		if (item->value == 0)
		{
			gtk_ctree_node_set_selectable 
				(GTK_CTREE (pd->menu_ctree),
				 GTK_CTREE_NODE (node),
				 FALSE);
		}

		if (item->node_ref != NULL)
		{
			*(item->node_ref) = node;
		}
	}
}

void
set_config_from_editable (GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get all the text into a new string */
	value = gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1);

	/* set it in the configuration space */
	gnome_config_set_string (config_name, value);

	/* free the allocated string */
	g_free (value);
}

static void
set_config_from_optionmenu (GtkWidget *optionmenu, gchar *config_name)
{
	GtkWidget *menu;
	GList *list;
	gpointer item;
	gint index;

	/* extract the selection */
	menu = GTK_OPTION_MENU(optionmenu)->menu;
	list = GTK_MENU_SHELL(menu)->children;
	item = gtk_menu_get_active (GTK_MENU(menu));
	index = g_list_index (list, item);
	
	/* set the configuration value */
	gnome_config_set_int (config_name, index);
}

static void
find_active_radio_button (gpointer data, gpointer user_data)
{
        GtkWidget *radiobutton = GTK_WIDGET(data);
        gboolean active;

        active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(radiobutton));

        if (active){
                GSList *list;
                gint index;
                gint *chosen = user_data;

                list = gtk_radio_button_group (GTK_RADIO_BUTTON(radiobutton));
                index = g_slist_index (list, radiobutton);
                *chosen = index;
        }
}

void
set_config_from_radiobuttongroup (GtkWidget *radiobutton, gchar *config_name)
{
        GSList *list;
        gint index, chosen, length;

        list = gtk_radio_button_group (GTK_RADIO_BUTTON(radiobutton));
        index = g_slist_index (list, radiobutton);
        length = g_slist_length(list);

        /* find active button and get its index to &chosen */
        g_slist_foreach (list, find_active_radio_button, (gpointer) &chosen);

        /* new buttons are *preppended* to the list, so button added as first 
         * has last position in the list */
        chosen = (length - 1) - chosen;

        /* set the configuration value */
        gnome_config_set_int (config_name, chosen);
}

static void
set_config_from_fontpicker (GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the name of the font */
	font = gnome_font_picker_get_font_name (GNOME_FONT_PICKER(fontpicker));

	/* set the configuration value */
	gnome_config_set_string (config_name, font);
}

static void
set_config_from_spin_button (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* read the value as an integer */
	value = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(spinbutton));

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_togglebutton (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* read the value */
	value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));

	/* set the configuration value */
	gnome_config_set_bool (config_name, value);
}

static void
set_config_from_int_entry (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the string */
	value_string = gtk_entry_get_text (GTK_ENTRY(entry));

	/* convert it into an integer */
	/* FIXME: atoi is a rubbish function to use, it
	 * doesn't detect any errors. We should fix this -
	 * perhaps by using spin buttons for these things? -- MattA */
	value = atoi(value_string);

	/* set the configuration value */
	gnome_config_set_int (config_name, value);
}

static void
set_config_from_list (GtkWidget *list, gchar *config_name)
{
	GList *children = g_list_copy (GTK_LIST(list)->children);
	guint size = g_list_length (children);
	char **strings = g_malloc (size * sizeof (char *));
	int x = 0;

	while (children) 
	{
		GtkObject       *list_item;
		gchar           *item_data_string;
        
		list_item = GTK_OBJECT(children->data);
		item_data_string = gtk_object_get_data(list_item, "string");
		strings[x] = item_data_string;
		children=children->next;
		x++;
	}
	
	/* set the configuration value */
	gnome_config_set_vector (config_name, size, (const char**)strings);
	g_free (children);
	g_free (strings);
}

static void
set_config_from_color (GtkWidget *colorpicker, gchar *config_name)
{
	guint8 r, g, b, a;
	gchar color_string[9];

	/* get color values from color picker */
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   &r, &g, &b, &a);

	/* write into string (bounded size) */
	snprintf (color_string, 9, "#%02X%02X%02X", r, g, b);

	/* set the configuration value */
	gnome_config_set_string (config_name, color_string);
}

void
set_editable_from_config (GtkWidget *editable, gchar *config_name)
{
	gchar *value;

	/* get the current value from the configuration space */
	value = gnome_config_get_string (config_name);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(editable), value);

	/* free the allocated string */
	g_free (value);
}

static void
set_optionmenu_from_config (GtkWidget *optionmenu, gchar *config_name)
{
	gint index;

	/* get the current value from the configuration space */
	index = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_option_menu_set_history(GTK_OPTION_MENU(optionmenu), index);
}

void
set_radiobuttongroup_from_config (GtkWidget *radiobutton, gchar *config_name)
{
        GSList *list;
        GtkWidget *button;
        gint index, length;

        /* get the current value from the configuration space */
        index = gnome_config_get_int (config_name);
        list = gtk_radio_button_group (GTK_RADIO_BUTTON(radiobutton));
        length = g_slist_length(list);

        /* new buttons are *preppended* to the list, so button added as first 
         * has last position in the list */
        index = (length - 1) - index;

        button = g_slist_nth_data(list, index);

        gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(button), TRUE);

}

static void
set_fontpicker_from_config (GtkWidget *fontpicker, gchar *config_name)
{
	gchar *font;

	/* get the current value from the configuration space */
	font = gnome_config_get_string (config_name);

	/* get the name of the font */
	gnome_font_picker_set_font_name (GNOME_FONT_PICKER(fontpicker), font);

	/* free the allocated string */
	g_free (font);
}

static void
set_spin_button_from_config (GtkWidget *spinbutton, gchar *config_name)
{
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* set this option value in the widget */
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinbutton), value);
}

static void
set_togglebutton_from_config (GtkWidget *togglebutton, gchar *config_name)
{
	gboolean value;

	/* get the current value from the configuration space */
	value = gnome_config_get_bool (config_name);

	/* set this option value in the widget */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (togglebutton), value);
}

static void
set_int_entry_from_config (GtkWidget *entry, gchar *config_name)
{
	gchar *value_string;
	gint value;

	/* get the current value from the configuration space */
	value = gnome_config_get_int (config_name);

	/* convert to a string */
	value_string = g_strdup_printf("%d", value);

	/* set this string value in the widget */
	gtk_entry_set_text(GTK_ENTRY(entry), value_string);

	/* free the allocated string */
	g_free (value_string);
}

static void
set_list_from_config (GtkWidget *list, gchar *config_name)
{
	int size;
	gchar **strings;
	GList *items = NULL;
	int k;

	/* empty the list */
	gtk_list_clear_items (GTK_LIST (list), 0, -1);

	/* get the array of strings */
	gnome_config_get_vector (config_name, &size, &strings);
	if (strings == NULL) 
	{
		return;
	}

	for (k = 0; k < size; k++)
	{	
		GtkWidget *list_item;
		char *label;
		
		label = strings[k];	
		list_item = gtk_list_item_new_with_label (label);
		gtk_object_set_data (GTK_OBJECT (list_item), "string", label);
		items = g_list_append (items, list_item);
		gtk_widget_show (list_item);
	}

	if (items != NULL)
	{
		gtk_list_append_items (GTK_LIST (list), items);
	}

	/* free the vector and list */
	g_free (strings);
}

static void
set_color_from_config (GtkWidget *colorpicker, gchar *config_name)
{
	gchar *color_string;
	guint r, g, b;

	/* get the string from config */
	color_string = gnome_config_get_string (config_name);

	/* parse it and setup the color picker */
	sscanf (color_string, "#%2X%2X%2X", &r, &g, &b);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (colorpicker), 
				   r, g, b, 0);

	/* free the string */
	g_free (color_string);
}

void
preferences_set_use_own_colors (gboolean new_value)
{
	gboolean old_value = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_COLORS);
	if (new_value != old_value) {
		gnome_config_set_bool (CONF_APPEARANCE_USE_OWN_COLORS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); /* we need a less general way to do this... */
		set_settings_menu_global (); /* update the menu in other windows */
	}
}

void 
preferences_set_enable_javascript (gboolean new_value)
{
	gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED);
	if (new_value != old_value)
	{
		gnome_config_set_bool (CONF_ADVANCED_JAVASCRIPT_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set ();
		set_settings_menu_global (); 
	}
}

void 
preferences_set_enable_java (gboolean new_value)
{
 	gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_JAVA_ENABLED);
  	if (new_value != old_value)
	{
		gnome_config_set_int (CONF_ADVANCED_JAVA_ENABLED, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
preferences_set_enable_proxy (gboolean value)
{
        gboolean old_value = gnome_config_get_bool (CONF_ADVANCED_PROXY_ENABLED);
        if (value != old_value) {
                gnome_config_set_bool (CONF_ADVANCED_PROXY_ENABLED, value);
                gnome_config_sync ();
                mozilla_prefs_set ();
                set_settings_menu_global ();
        }
}


void
preferences_set_use_own_fonts (gboolean new_value)
{	
	gboolean old_value = gnome_config_get_bool (CONF_APPEARANCE_USE_OWN_FONTS);
	if (new_value != old_value) {
		gnome_config_set_bool (CONF_APPEARANCE_USE_OWN_FONTS, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void 
preferences_set_image_loading (ImageLoadingType value)
{
	int new_value = value;
	int old_value = gnome_config_get_int (CONF_ADVANCED_IMAGE_LOADING_TYPE);
	if (new_value != old_value) {
		gnome_config_set_int (CONF_ADVANCED_IMAGE_LOADING_TYPE, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void 
preferences_set_animate (AnimateType value)
{
	int new_value = value;
	int old_value = gnome_config_get_int (CONF_ADVANCED_ANIMATE_TYPE);

	if (new_value != old_value)
	{
		gnome_config_set_int (CONF_ADVANCED_ANIMATE_TYPE, new_value);
		gnome_config_sync ();
		mozilla_prefs_set (); 
		set_settings_menu_global (); 
	}
}

void
mouse_wheel_controls_set_sensitive (PreferencesDialog *pd, WheelModKey key,
				    gboolean sensitive)
{
	GtkWidget *spinbutton = NULL, *checkbutton = NULL;
	
	switch (key)
	{
	case WHEEL_MOD_NOKEY:
		spinbutton = pd->wheel_nokey_step_spinbutton;
		checkbutton = pd->wheel_nokey_default_step_checkbutton;
		break;
	case WHEEL_MOD_ALT:
		spinbutton = pd->wheel_alt_step_spinbutton;
		checkbutton = pd->wheel_alt_default_step_checkbutton;
		break;
	case WHEEL_MOD_CTRL:
		spinbutton = pd->wheel_ctrl_step_spinbutton;
		checkbutton = pd->wheel_ctrl_default_step_checkbutton;
		break;
	case WHEEL_MOD_SHIFT:
		spinbutton = pd->wheel_shift_step_spinbutton;
		checkbutton = pd->wheel_shift_default_step_checkbutton;
		break;
	default:
		break;
	}

	if (spinbutton && checkbutton)
	{
		gtk_widget_set_sensitive (spinbutton, sensitive);
		gtk_widget_set_sensitive (checkbutton, sensitive);
	}
}

static void
prefs_dialog_save_geometry (PreferencesDialog *pd)
{
	state_save_window_state (GTK_WIDGET (pd->dialog)->window, 
				"prefs_dialog");
	state_save_pane_pos (pd->hpaned, "prefs_dialog_hpane");
}

/* FIXME */
static void
cb_reset_font_picker (GtkWidget *menu_item, PreferencesDialog *pd)
{
	int i;
	gchar buf[128];
	const gchar *default_font [3] =
	{
		"-*-times-medium-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*",
		"-*-courier-medium-r-normal-*-14-*-*-*-*-*-*-*"
	};

	i = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menu_item),
						  "index"));

	//gnome_config_set_int ("lang_encode", i);
	g_snprintf (buf, 128, "/galeon/Appearance/serif_font_%s=%s",
		    lang_encode_item[i], default_font[0]);
	set_fontpicker_from_config(pd->serif_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/sansserif_font_%s=%s",
		    lang_encode_item[i], default_font[1]);
	set_fontpicker_from_config(pd->sansserif_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/monospace_font_%s=%s",
		    lang_encode_item[i], default_font[2]);
	set_fontpicker_from_config(pd->monospace_font_fpicker, buf);

	g_snprintf (buf, 128, "/galeon/Appearance/size_font_%s=14", lang_encode_item[i]);
	set_spin_button_from_config(pd->size_font_spin, buf);
}
