/*  XMPS - X MPEG Player System
 *  Copyright (C) 1999 Damien Chavarria
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *  config.c
 *
 *  Code file for the options.
 *
 *  most of this code has been taken form
 *  the xmms project.
 *
 */

/*********************************************************************
 *                             INCLUDES                              *
 *********************************************************************/

/*
 * includes : local
 *
 */

#include "libxmps_config.h"

/*********************************************************************
 *                            FUNCTIONS                              *
 *********************************************************************/

/*
 * pre-defined
 *
 */

static ConfigSection *xmps_config_create_section(xmps_config_file_t * config, 
					      gchar * name);

static ConfigLine *xmps_config_create_string(ConfigSection * section, 
					  gchar * key, 
					  gchar * value);

static ConfigSection *xmps_config_find_section(xmps_config_file_t * config, 
					    gchar * name);

static ConfigLine *xmps_config_find_string(ConfigSection * section, 
					gchar * key);

/*
 * xmps_config_new
 *
 * - creates a new config file
 *
 */

xmps_config_file_t *xmps_config_file_new(void)
{
  xmps_config_file_t *config;

  config = g_malloc0(sizeof (xmps_config_file_t));
  
  return config;
}

/*
 * xmps_config_open_file
 *
 * - opens an existing file for reading
 *
 */

xmps_config_file_t *xmps_config_open_file(gchar * filename)
{
  xmps_config_file_t *config;

  FILE *file;
  gchar *buffer, **lines, *tmp;
  gint i;
  struct stat stats;
  ConfigSection *section = NULL;
  

  if (lstat(filename, &stats) == -1)
    return NULL;
    
  if (!(file = fopen(filename, "r")))
    return NULL;

  buffer = g_malloc(stats.st_size + 1);
  
  if (fread(buffer, 1, stats.st_size, file) != stats.st_size) {
    
    g_free(buffer);
    fclose(file);
    return NULL;
  }
  
  fclose(file);
  buffer[stats.st_size] = '\0';

  config = g_malloc0(sizeof (xmps_config_file_t));
  lines = g_strsplit(buffer, "\n", 0);
  g_free(buffer);
  
  i = 0;
  
  while (lines[i])
    {
      if (lines[i][0] == '[')
	{
	  if ((tmp = strchr(lines[i], ']')))
	    {
	      *tmp = '\0';
	      section = xmps_config_create_section(config, &lines[i][1]);
	    }
	}
      else if (lines[i][0] != '#' && section)
	{
	  if ((tmp = strchr(lines[i], '=')))
	    {
	      *tmp = '\0';
	      tmp++;
	      xmps_config_create_string(section, lines[i], tmp);
	    }
	}
      i++;
    }
  
  g_strfreev(lines);
  
  return config;
}

/*
 * xmps_config_get_default_filename
 *
 * - returns the default config filename
 *
 */

gchar *xmps_config_get_default_filename(void)
{
  static gchar *filename = NULL;
  
  if (!filename)
    filename = g_strconcat(g_get_home_dir(), "/.xmps/config", NULL);
  
  return filename;
}

/*
 * xmps_config_open_default_file
 *
 *
 *
 */

xmps_config_file_t * xmps_config_open_default_file(void)
{
  xmps_config_file_t *ret;

  ret = xmps_config_open_file(xmps_config_get_default_filename());
  
  if (!ret)
    ret = xmps_config_file_new();
    
  return ret;
}

/*
 * xmps_config_write_file
 *
 *
 *
 */

gboolean xmps_config_write_file(xmps_config_file_t * config, gchar * filename)
{
  FILE *file;
  GList *section_list, *line_list;
  ConfigSection *section;
  ConfigLine *line;

  if (!(file = fopen(filename, "w")))
    return FALSE;

  section_list = config->sections;
  while (section_list)
    {
      section = (ConfigSection *) section_list->data;
      if (section->lines)
	{
	  fprintf(file, "[%s]\n", section->name);
	  line_list = section->lines;
	  while (line_list)
	    {
	      line = (ConfigLine *) line_list->data;
	      fprintf(file, "%s=%s\n", line->key, line->value);
	      line_list = g_list_next(line_list);
	    }
	  fprintf(file, "\n");
	}
      section_list = g_list_next(section_list);
    }
  
  fclose(file);
 
  return TRUE;
}

gboolean xmps_config_write_default_file(xmps_config_file_t * config)
{
  return xmps_config_write_file(config, 
			     xmps_config_get_default_filename());
}

gboolean xmps_config_read_string(xmps_config_file_t * config, gchar * section, gchar * key, gchar ** value)
{
  ConfigSection *sect;
  ConfigLine *line;
  
  if (!(sect = xmps_config_find_section(config, section)))
    return FALSE;
    
  if (!(line = xmps_config_find_string(sect, key)))
    return FALSE;
    
  *value = g_strdup(line->value);
  
  return TRUE;
}

gboolean xmps_config_read_int(xmps_config_file_t * config, gchar * section, gchar * key, gint * value)
{
  gchar *str;
  
  if (!xmps_config_read_string(config, section, key, &str))
    return FALSE;
  *value = atoi(str);
  g_free(str);
  
  return TRUE;
}

gboolean xmps_config_read_boolean(xmps_config_file_t * config, gchar * section, gchar * key, gboolean * value)
{
  gchar *str;

  if (!xmps_config_read_string(config, section, key, &str))
    return FALSE;
  if (!strcasecmp(str, "TRUE"))
    *value = TRUE;
  else
    *value = FALSE;
  g_free(str);
  return TRUE;
}

gboolean xmps_config_read_float(xmps_config_file_t * config, gchar * section, gchar * key, gfloat * value)
{
  gchar *str;

  if (!xmps_config_read_string(config, section, key, &str))
    return FALSE;

  *value = (gfloat) g_strtod(str, NULL);
  g_free(str);

  return TRUE;
}

gboolean xmps_config_read_double(xmps_config_file_t * config, gchar * section, gchar * key, gdouble * value)
{
  gchar *str;

  if (!xmps_config_read_string(config, section, key, &str))
    return FALSE;

  *value = g_strtod(str, NULL);
  g_free(str);

  return TRUE;
}
void xmps_config_write_string(xmps_config_file_t * config, gchar * section, gchar * key, gchar * value)
{
  ConfigSection *sect;
  ConfigLine *line;

  sect = xmps_config_find_section(config, section);
  if (!sect)
    sect = xmps_config_create_section(config, section);
  if ((line = xmps_config_find_string(sect, key)))
    {
      g_free(line->value);
      line->value = g_strchug(g_strchomp(g_strdup(value)));
    }
  else
    xmps_config_create_string(sect, key, value);
}

void xmps_config_write_int(xmps_config_file_t * config, gchar * section, gchar * key, gint value)
{
  gchar *strvalue;

  strvalue = g_strdup_printf("%d", value);
  xmps_config_write_string(config, section, key, strvalue);
  g_free(strvalue);
}

void xmps_config_write_boolean(xmps_config_file_t * config, gchar * section, gchar * key, gboolean value)
{
  if (value)
    xmps_config_write_string(config, section, key, "TRUE");
  else
    xmps_config_write_string(config, section, key, "FALSE");
}

void xmps_config_write_float(xmps_config_file_t * config, gchar * section, gchar * key, gfloat value)
{
  gchar *strvalue;

  strvalue = g_strdup_printf("%g", value);
  xmps_config_write_string(config, section, key, strvalue);
  g_free(strvalue);
}

void xmps_config_write_double(xmps_config_file_t * config, gchar * section, gchar * key, gdouble value)
{
  gchar *strvalue;

  strvalue = g_strdup_printf("%g", value);
  xmps_config_write_string(config, section, key, strvalue);
  g_free(strvalue);
}

void xmps_config_remove_key(xmps_config_file_t * config, gchar * section, gchar * key)
{
  ConfigSection *sect;
  ConfigLine *line;

  if ((sect = xmps_config_find_section(config, section)) != NULL)
    {
      if ((line = xmps_config_find_string(sect, key)) != NULL)
	{
	  g_free(line->key);
	  g_free(line->value);
	  g_free(line);
	  sect->lines = g_list_remove(sect->lines, line);
	}
    }
}

void xmps_config_free(xmps_config_file_t * config)
{
  ConfigSection *section;
  ConfigLine *line;
  GList *section_list, *line_list;

  section_list = config->sections;
  while (section_list)
    {
      section = (ConfigSection *) section_list->data;
      g_free(section->name);

      line_list = section->lines;
      while (line_list)
	{
	  line = (ConfigLine *) line_list->data;
	  g_free(line->key);
	  g_free(line->value);
	  g_free(line);
	  line_list = g_list_next(line_list);
	}
      g_list_free(section->lines);
      g_free(section);

      section_list = g_list_next(section_list);
    }
  g_list_free(config->sections);
}

static ConfigSection *xmps_config_create_section(xmps_config_file_t * config, gchar * name)
{
  ConfigSection *section;

  section = g_malloc0(sizeof (ConfigSection));
  section->name = g_strdup(name);
  config->sections = g_list_append(config->sections, section);

  return section;
}

static ConfigLine *xmps_config_create_string(ConfigSection * section, gchar * key, gchar * value)
{
  ConfigLine *line;

  line = g_malloc0(sizeof (ConfigLine));
  line->key = g_strchug(g_strchomp(g_strdup(key)));
  line->value = g_strchug(g_strchomp(g_strdup(value)));
  section->lines = g_list_append(section->lines, line);

  return line;
}

static ConfigSection *xmps_config_find_section(xmps_config_file_t * config, gchar * name)
{
  ConfigSection *section;
  GList *list;

  list = config->sections;
  while (list)
    {
      section = (ConfigSection *) list->data;
      if (!strcasecmp(section->name, name))
	return section;
      list = g_list_next(list);
    }
  return NULL;
}

static ConfigLine *xmps_config_find_string(ConfigSection * section, gchar * key)
{
  ConfigLine *line;
  GList *list;

  list = section->lines;
  while (list)
    {
      line = (ConfigLine *) list->data;
      if (!strcasecmp(line->key, key))
	return line;
      list = g_list_next(list);
    }
  return NULL;
}








