/* vi:set ts=8 sts=0 sw=8:
 * $Id: main.c,v 1.39 2000/03/29 04:22:47 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#ifdef USE_GNOME
#include <gnome.h>
#endif
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include "config.h"
#include "prefs.h"
#include "win.h"
#include "doc.h"
#include "msgbar.h"
#include "recent.h"
#include "misc.h"
#include "file.h"
#include "gnpintl.h"

/*** external declarations ***/
extern char *build_user;	/* these are all in appinfo.c */
extern char *build_host;
extern char *build_date;
extern char *gtkver;
#ifdef USE_GNOME
extern char *gnomever;
#endif
extern char *compiler;
extern char *gtk_cflags;
extern char *cflags;
extern char *gtk_ldflags;
extern char *ldflags;
extern char *unamenfo;

extern void     win_set_title(doc_t *d);

#if defined(ENABLE_NLS) && defined(GTK_HAVE_FEATURES_1_1_0)
static char G_GNUC_UNUSED *dummyMsg[] = { N_("Untitled"),
					  N_("Unknown"),
					  N_("Welcome to "),
					  N_("Files left to open: ") };
#endif

/*** local declarations ***/
typedef struct {
#ifdef WANT_SPLASH
	GtkWidget *toplev;	/* top level init status window */
	GtkWidget *fnum;	/* label for num files remaining */
	GtkWidget *fname;	/* label for filename */
	bool_t do_splash;
#endif
	int timeout_id;
	win_t *w;
	char **files;
	int num;
} init_data_t;


/*** local function prototypes ***/
static void app_init(int num, char **files, bool_t);
static int  app_init_open_files(gpointer cbdata);
static void app_init_status_win_create(win_t *w, char **files, int num, bool_t);
#ifdef WANT_SPLASH
static void app_init_status_win_destroy(init_data_t *idp);
static void app_init_status_win_update(init_data_t *idp);
#endif
#ifdef USE_GNOME
static void main_init_gnome(int argc, char **argv);
#else
static void app_parse_args(int *argc, char ***argv, bool_t *, bool_t *);
static void app_show_help(char **argv);
static void main_init_gtk(int argc, char **argv);
#endif
static void app_print_appinfo(void);
static void app_show_info(void);
static void app_show_version(void);


/*** global function definitions ***/

int
main(int argc, char **argv)
{
#ifdef ENABLE_NLS
	bindtextdomain("gnotepad+", LOCALEDIR);
	textdomain("gnotepad+");
#endif
#ifdef USE_GNOME
	main_init_gnome(argc, argv);
#else
	main_init_gtk(argc, argv);
#endif
	gtk_main();

	return 0;
} /* main */


/*** local function definitions ***/
/*
 * PRIVATE: app_init
 *
 * performs all the initial creates of the first window.  creates "Untitled"
 * document if none were specified on the command line.  otherwise, creates a
 * popup showing the number of files remaining to be opened along with the
 * filename to be opened.
 */
static void
app_init(int num, char **files, bool_t do_splash)
{
	win_t *w;

	GNPDBG_GENERIC(("app_init: files to open = %d\n", num));

	/* initialize the window list */
	w = win_new();

	/* add "Untitled" doc if no files specified */
	if (!files || *files == NULL) {
		doc_new(w, NULL, DocText, (DO_LOAD | UPDATE_TITLE));
		gtk_widget_show(w->toplev);

		/*
		 * setting the switch page signal here is kind of hokey, but we
		 * need to do it here because the switchpage handler is where
		 * we determine when to actually load the file from disk.
		 * since we may be opening multiple files, we only want to
		 * enable the switchpage handler **AFTER** we've opened all the
		 * files at the start of the program.  technically, if we're in
		 * this "if" part, we're only opening one file.  however,
		 * there's no clean way of not doing this without passing an
		 * additional parameter to win_new(), which then gets passed to
		 * notebook_init().  we could do it that way, but it's too
		 * messy.
		 *
		 * note that this is done again, at the end of
		 * app_init_open_files().
		 */
		(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
					       GTK_SIGNAL_FUNC(doc_switch_page),
					       w);
		gtk_events_flush();

		msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME,
			      APP_VERSION, _("."));
		win_autosave_init();
	} else {
		g_assert(files);
		app_init_status_win_create(w, files, num, do_splash);
	}

} /* app_init */


/*
 * PRIVATE: app_init_open_files
 *
 * callback routine for app_init_status_win_create() which is attached to a
 * timeout during the creation of the status popup window.
 */
static int
app_init_open_files(gpointer cbdata)
{
	init_data_t *idp = (init_data_t *)cbdata;
	win_t *w;

	g_assert(idp != NULL);
	gtk_timeout_remove(idp->timeout_id);
	w = idp->w;
	/*
	 * for each file specified, create a document and add it to the current
	 * window.  since we are inserting into the doc notebook at the head of
	 * the list, we'll do the filenames backwards to preserve the order.
	 */
	while (idp->num > 0) {
#ifdef WANT_SPLASH
		if (idp->do_splash && IS_SHOW_SPLASH())
			app_init_status_win_update(idp);
#endif

#ifdef WANT_DOCUNLOAD
		doc_new(idp->w, idp->files[idp->num - 1], DocText,
			(idp->num - 1 == 0) ? DO_LOAD : 0);
#else
		doc_new(idp->w, idp->files[idp->num - 1], DocText, DO_LOAD);
#endif
#ifdef USE_RECENT
		if (idp->num <= prefs.maxrecent)
			recent_list_add(idp->w, idp->files[idp->num - 1]);
#endif	/* USE_RECENT */
		g_free(idp->files[idp->num - 1]);
		idp->files[idp->num - 1] = NULL;
		idp->num--;
	}

	/*
	 * establish the notebook's switch_page handler.  see app_init() for
	 * more details why it's done here
	 */
	(void)gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				       GTK_SIGNAL_FUNC(doc_switch_page), w);

#ifdef WANT_SPLASH
	if (idp->do_splash && IS_SHOW_SPLASH())
		app_init_status_win_destroy(idp);
#else
	g_assert(idp->files[0] == NULL);
	g_free(idp->files);
	g_free(idp);
#endif

	win_set_title(DOC_CURRENT(w));
	doc_info_label_update(w);
	gtk_widget_show(w->toplev);
	gtk_events_flush();

	msgbar_printf(w, "%s%s %s", WELCOME_MSG, APP_NAME, APP_VERSION, _("."));
	win_autosave_init();

	return FALSE;
} /* app_init_open_files */


/*
 * PRIVATE: app_init_status_win_create
 *
 * creates all the widgets necessary for the popup window to show the
 * initialization status.  the files are actually opened through a callback
 * routine which is attached to a gtk_timeout near the end of this function.
 */
#define FILES_LEFT	gettext("Files left to open: ")
static void
app_init_status_win_create(win_t *w, char **files, int num, bool_t do_splash)
{
#ifdef WANT_SPLASH
	GtkWidget *vbox;
#endif
	init_data_t *idp;
	int i;

	idp = g_new(init_data_t, 1);
	idp->w = w;
	idp->num = num;
	idp->files = g_new(char *, num);
	for (i = 0; i < num; i++)
		idp->files[i] = g_strdup(files[i]);

#ifdef WANT_SPLASH
	idp->do_splash = do_splash;
	if (do_splash && IS_SHOW_SPLASH()) {
		idp->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title(GTK_WINDOW(idp->toplev),
				     _("gnotepad+ Init..."));
		(void)gtk_signal_connect(GTK_OBJECT(idp->toplev),
					 "delete_event",
					 GTK_SIGNAL_FUNC(gtk_true), NULL);
		gtk_container_border_width(GTK_CONTAINER(idp->toplev), 10);

		vbox = gtk_vbox_new(FALSE, 4);
		gtk_container_add(GTK_CONTAINER(idp->toplev), vbox);

		/* for some reason, the labels below are not centered correctly
		 * when this is compiled with gtk-1.1 */
		idp->fnum = gtk_label_new(FILES_LEFT);
		gtk_box_pack_start(GTK_BOX(vbox), idp->fnum, FALSE, TRUE, 10);

		idp->fname = gtk_label_new(" ");
		gtk_box_pack_start(GTK_BOX(vbox), idp->fname, FALSE, TRUE, 10);
		gtk_window_position(GTK_WINDOW(idp->toplev),
				    GTK_WIN_POS_CENTER);
		gtk_widget_show_all(idp->toplev);
	}
#endif	/* WANT_SPLASH */

	/* this will kick-off opening all the files */
	idp->timeout_id = gtk_timeout_add(1,
					(GtkFunction)app_init_open_files, idp);
} /* app_init_status_win_create */


#ifdef WANT_SPLASH
/*
 * PRIVATE: app_init_status_win_destroy
 *
 * cleanup
 */
static void
app_init_status_win_destroy(init_data_t *idp)
{
	g_assert(idp->files[0] == NULL);
	g_free(idp->files);
	if (idp->toplev)
		gtk_widget_destroy(idp->toplev);
	g_free(idp);
} /* app_init_status_win_destroy */
#endif	/* WANT_SPLASH */


#ifdef WANT_SPLASH
/*
 * PRIVATE: app_init_status_win_update
 *
 * update the two labels in the status window.
 */
static void
app_init_status_win_update(init_data_t *idp)
{
	char *buf, *bfname;
	int len;

	bfname = (char *)my_basename(idp->files[idp->num - 1]);
	len = MAX(strlen(bfname) + 1, strlen(FILES_LEFT) + 10);
	buf = g_new(char, len);
	g_snprintf(buf, len, " %s%d ", FILES_LEFT, idp->num);
	gtk_label_set(GTK_LABEL(idp->fnum), buf);
	g_snprintf(buf, len, " %s ", bfname);
	gtk_label_set(GTK_LABEL(idp->fname), buf);
	g_free(buf);
	gtk_events_flush();
	gdk_flush();
} /* app_init_status_win_update */
#endif	/* WANT_SPLASH */


#ifdef USE_GNOME
/*
 * init for GNOME, using popt table
 */
static void
main_init_gnome(int argc, char **argv)
{
	GnomeClient *gncli;
	const char **args;
	poptContext ctx;
# ifdef APP_DEBUG
	char *dbgstr = NULL;
# endif
	bool_t nosplash = FALSE;
	bool_t print_ver = FALSE;
	bool_t print_info = FALSE;
# ifdef WANT_SESSION
	bool_t load_session = FALSE;
# endif
	const struct poptOption gnopts[] = {
		{
			"version",
			'v',
			POPT_ARG_NONE,
			&print_ver,
			0,
			N_("Print version information"),
			NULL
		},
		{
			"info",
			'i',
			POPT_ARG_NONE,
			&print_info,
			0,
			N_("Print compile/link options used"),
			NULL
		},
# ifdef WANT_SPLASH
		{
			"nosplash",
			's',
			POPT_ARG_NONE,
			&nosplash,
			0,
			N_("Don't show splash/init screen on startup"),
			NULL
		},
# endif
# ifdef WANT_SESSION
		{
			"ldsession",
			'l',
			POPT_ARG_NONE,
			&load_session,
			0,
			N_("Load open files from previous session"),
			NULL
		},
# endif
# ifdef APP_DEBUG
		{
			"debug",
			'd',
			POPT_ARG_STRING,
			&dbgstr,
			0,
			N_("Set debug flag to FLAGS"),
			N_("FLAGS")
		},
# endif
		{ NULL, }
	};

	gnome_init_with_popt_table(APP_NAME, APP_VERSION, argc, argv, gnopts,
				   0, &ctx);
	if (print_ver)
		app_show_version();	/* does not return */
	if (print_info)
		app_show_info();	/* does not return */
# ifdef APP_DEBUG
	if (dbgstr) {
		if (dbgstr[0] == '0')
			sscanf(dbgstr, "%lx", &dbg_flags);
		else    
			sscanf(dbgstr, "%ld", &dbg_flags);
	}
# endif

	glade_gnome_init();

	args = poptGetArgs(ctx);
	argc = 0;
	while (args && args[argc])
		argc++;
	prefs_init();

# ifdef WANT_SESSION
	if (load_session || IS_ENABLE_SESSION())
		load_session = win_session_load();
# endif

	gncli = gnome_master_client();
	gtk_signal_connect(GTK_OBJECT(gncli), "save_yourself",
			   GTK_SIGNAL_FUNC(win_session_save), argv[0]);
	gtk_signal_connect(GTK_OBJECT(gncli), "die",
			   GTK_SIGNAL_FUNC(win_session_die), NULL);

# ifdef WANT_SESSION
	if (!load_session)
# endif
		app_init(argc, (char **)args, !nosplash);
	poptFreeContext(ctx);

} /* main_init_gnome */

#else	/* no GNOME */

/*
 * init for non-GNOME, using getopt() or getopt_long(), if available.  if
 * neither is available, fallback to manual parsing.
 */
static void
main_init_gtk(int argc, char **argv)
{
	bool_t do_splash = TRUE;
	bool_t load_session = FALSE;

	gtk_set_locale();
	gtk_init(&argc, &argv);
	prefs_init();
	app_parse_args(&argc, &argv, &do_splash, &load_session);
# ifdef WANT_SESSION
	if (load_session || IS_ENABLE_SESSION())
		load_session = win_session_load();
	if (!load_session)
# endif
		app_init(argc - 1, argv + 1, do_splash);
} /* main_init_gtk */


#if defined(HAVE_GETOPT) || defined(HAVE_GETOPT_LONG)
/*
 * parse command line options using getopt() or getopt_long()
 */
static void
parse_getopt(int *argcp, char ***argvp, bool_t *do_splash, bool_t *load_session)
{
	int ch;
# ifdef HAVE_GETOPT_LONG
	int idx;

	struct option lopts[] = {
		{ "help", 0, 0, 'h' },
		{ "version", 0, 0, 'v' },
		{ "info", 0, 0, 'i' },
#  ifdef WANT_SPLASH
		{ "nosplash", 0, 0, 's' },
#  endif
#  ifdef WANT_SESSION
		{ "ldsession", 0, 0, 'l' },
#  endif
#  ifdef APP_DEBUG
		{ "debug", 1, 0, 'd' },
#  endif
		{ 0, 0, 0, 0 }
	};

	while ((ch = getopt_long(*argcp, *argvp, "hvifsld:", lopts, &idx)) !=
		EOF)
# else
	while ((ch = getopt(*argcp, *argvp, "hvifsld:")) != EOF)
# endif
	{
		switch (ch) {
		case 'h':
			app_show_help(*argvp);	/* does not return */
		case 'v':
			app_show_version();	/* does not return */
		case 'i':
			app_show_info();	/* does not return */
# ifdef WANT_SPLASH
		case 's':
			*do_splash = FALSE;
			break;
# endif
# ifdef WANT_SESSION
		case 'l':
			*load_session = TRUE;
			break;
# endif
# ifdef APP_DEBUG
		case 'd':
			if (optarg) {
				if (optarg[0] == '0')
					sscanf(optarg, "%lx", &dbg_flags);
				else    
					sscanf(optarg, "%ld", &dbg_flags);
			}
			GNPDBG_ALWAYS(("dbg_flags = 0x%lx (%ld)\n",
				       dbg_flags, dbg_flags));
			break;
# endif
		default:
			app_show_help(*argvp);	/* does not return */
		}
	}

	*argvp += (optind - 1);
	*argcp -= (optind - 1);
} /* parse_getopt */

#else	/* no getopt nor getopt_long */

/*
 * Parse command line options manually.
 */
# define SHIFT_ARGS(zz_argc, zz_argp)					\
	do {								\
		char **aptmp;						\
		(zz_argc)--;						\
		aptmp = zz_argp;					\
		while (*aptmp) {					\
			if (*(aptmp + 1) == NULL) {			\
				*aptmp = NULL;				\
			} else {					\
				*aptmp = *(aptmp + 1);			\
				aptmp++;				\
			}						\
		}							\
	} while (0)
static void
parse_manually(int *argcp, char ***argvp, bool_t *do_splash,
	       bool_t *load_session)
{
	char **ap = *argvp;

	while (*ap) {
		GNPDBG_GENERIC(("parse_manually: *ap = '%s'\n", *ap));
		if (strcmp(*ap, "--help") == 0)
			app_show_help(*argvp);	/* does not return */
		else if (strcmp(*ap, "--version") == 0)
			app_show_version();	/* does not return */
		else if (strcmp(*ap, "--info") == 0)
			app_show_info();	/* does not return */
# ifdef WANT_SPLASH
		else if (strcmp(*ap, "--nosplash") == 0) {
			*do_splash = FALSE;
			SHIFT_ARGS(*argcp, ap);
		}
# endif
# ifdef WANT_SESSION
		else if (strcmp(*ap, "--ldsession") == 0) {
			*load_session = TRUE;
			SHIFT_ARGS(*argcp, ap);
		}
# endif
# ifdef APP_DEBUG
		else if (strcmp(*ap, "--debug") == 0) {
			SHIFT_ARGS(*argcp, ap);
			if (*ap) {
				if (*ap[0] == '0')
					sscanf(*ap, "%lx", &dbg_flags);
				else
					sscanf(*ap, "%ld", &dbg_flags);
				SHIFT_ARGS(*argcp, ap);
			}
			GNPDBG_ALWAYS(("dbg_flags = 0x%lx (%ld)\n",
					dbg_flags, dbg_flags));
		}
# endif
		else
			ap++;
	}
} /* parse_manually */

#endif /* defined(HAVE_GETOPT) || defined(HAVE_GETOPT_LONG) */


/*
 * PRIVATE: app_parse_args
 *
 * parse for application specific arguments.  return argc and argv with any app
 * specific arguments omitted/removed.
 */
static void
app_parse_args(int *argcp, char ***argvp, bool_t *do_splash,
	       bool_t *load_session)
{
#if defined(HAVE_GETOPT) || defined(HAVE_GETOPT_LONG)
	parse_getopt(argcp, argvp, do_splash, load_session);
#else
	parse_manually(argcp, argvp, do_splash, load_session);
#endif
} /* app_parse_args */


/*
 * PRIVATE: app_show_help
 *
 * duh
 */
static void
app_show_help(char **argv)
{
	int i;
#ifdef HAVE_GETOPT_LONG
	char *usage_msg[] = {
		N_("Options:"),
		N_("  -h, --help         shows this screen"),
		N_("  -v, --version      prints app name and version"),
		N_("  -i, --info         prints compile/link options used"),
		" ",
# ifdef WANT_SPLASH
		N_("  -s, --nosplash     don't show splash screen on startup"),
# endif
		N_("  -l, --ldsession    load previous session"),
		NULL
	};
#elif HAVE_GETOPT
	char *usage_msg[] = {
		N_("Options:"),
		N_("  -h    shows this screen"),
		N_("  -v    prints app name and version"),
		N_("  -i    prints compile/link options used"),
		" ",
# ifdef WANT_SPLASH
		N_("  -s    don't show splash/init screen on startup"),
# endif
		N_("  -l    load previous session"),
		NULL
	};
#else
	char *usage_msg[] = {
		N_("Options:"),
		N_("  --help        shows this screen"),
		N_("  --version     prints app name and version"),
		N_("  --info        prints compile/link options used"),
		" ",
# ifdef WANT_SPLASH
		N_("  --nosplash    don't show splash/init screen on startup"),
# endif
# ifdef WANT_SESSION
		N_("  --ldsession   load previous session"),
# endif
		NULL
	};
#endif
#ifdef ENABLE_NLS
	gtk_set_locale();
	bindtextdomain("gnotepad+", LOCALEDIR);
	textdomain("gnotepad+");
#endif

	printf(_("\nUsage: %s [options] [files...]\n\n"), argv[0]);
	for (i = 0; usage_msg[i]; i++)
		printf("%s\n", gettext(usage_msg[i]));
	printf("\n");
	exit(0);
} /* app_show_help */

#endif	/* USE_GNOME */


/*
 * PRIVATE: app_print_appinfo
 *
 * duh
 */
static void
app_print_appinfo(void)
{
	printf("Compile/Link info:\n");
	printf("\tGTK version   = %s\n", gtkver);
#ifdef USE_GNOME
	printf("\tGNOME version = %s\n", gnomever);
#endif
	printf("\tCC            = %s\n", compiler);
	printf("\tGTK_CFLAGS    = %s\n", gtk_cflags);
	printf("\tCFLAGS        = %s\n", cflags);
	printf("\tGTK_LDFLAGS   = %s\n", gtk_ldflags);
	printf("\tLDFLAGS       = %s\n", ldflags);
	printf("\tUNAME         = %s\n", unamenfo);
} /* app_print_appinfo */


static void
print_version(void)
{
	printf("%s %s (%s)\n", APP_NAME, APP_VERSION, APP_DATE);
	printf("Compiled by %s@%s on %s\n", build_user, build_host, build_date);
} /* print_version */


/*
 * PRIVATE: app_show_info
 *
 * duh
 */
static void
app_show_info(void)
{
	print_version();
	app_print_appinfo();
	exit(0);
} /* app_show_help */


/*
 * PRIVATE: app_show_version
 *
 * duh
 */
static void
app_show_version(void)
{
	print_version();
	exit(0);
} /* app_show_version */


/* the end */
