/*  Gtk+ User Interface Builder
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#ifdef ENABLE_BONOBO

#include <gtk/gtkhbox.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkspinbutton.h>

#include "../gb.h"
#include "../palette.h"

#include <bonobo.h>
#include <bonobo/bonobo-object-directory.h>

/* Include the 21x21 icon pixmap for this widget, to be used in the palette */
#include "../graphics/custom.xpm"

/*
 * This is the GbWidget struct for this widget (see ../gbwidget.h).
 * It is initialized in the init() function at the end of this file
 */

#define PBC_KEY   "PROPERTY_BAG_CLIENT"

typedef struct {
	GbWidget gbwidget;
	char    *obj_id;
	char    *descr;
} BonoboGbWidget;

static GtkWidget *
control_create (char *obj_id)
{
  GtkWidget *widget;
  BonoboControlFrame *cf;
  BonoboPropertyBagClient *pbc;

  g_return_val_if_fail (obj_id != NULL, NULL);

  widget = bonobo_widget_new_control (obj_id, CORBA_OBJECT_NIL);
  
  g_return_val_if_fail (widget != NULL, NULL);
  cf = bonobo_widget_get_control_frame (BONOBO_WIDGET (widget));

  if (!cf) {
    g_warning ("Control has no frame!");
    gtk_widget_destroy (widget);

    return NULL;
  }

  pbc = bonobo_control_frame_get_control_property_bag (cf);

  gtk_object_set_data (GTK_OBJECT (widget), PBC_KEY, pbc);

  return widget;
}

static BonoboPropertyBagClient *
control_get_pbc (GtkWidget *widget)
{
  return gtk_object_get_data (GTK_OBJECT (widget),
			      PBC_KEY);
}

GtkWidget *
gb_bonobo_control_new (GbWidgetNewData * data)
{
  GbWidget     *gbwidget;
  GbWidgetData *w;

  w = data->widget_data;
  g_return_val_if_fail (w != NULL, NULL);

  gbwidget = w->gbwidget;
  g_return_val_if_fail (gbwidget != NULL, NULL);
  
  return control_create (gbwidget->class_id);
}

inline static char *
create_prop_name (const char *txt)
{
  return g_strconcat ("BonoboWidget::", txt, NULL);
}

/*
 * Creates the components needed to edit the extra properties of this widget.
 */
static void
gb_bonobo_control_create_properties (GtkWidget * widget, GbWidgetCreateArgData * data)
{
  BonoboPropertyBagClient *pbc = control_get_pbc (widget);
  GList *names;

  if (!pbc)
	  return;

  for (names = bonobo_property_bag_client_get_property_names (pbc);
       names; names = names->next) {
    CORBA_TypeCode tc;
    char          *doc;
    char          *prop = create_prop_name (names->data);

    tc  = bonobo_property_bag_client_get_property_type (pbc, names->data);
    doc = bonobo_property_bag_client_get_docstring (pbc, names->data);
    switch (tc->kind) {
    case CORBA_tk_boolean:
      property_add_bool (prop, names->data, doc);
      break;
    case CORBA_tk_string:
      property_add_string (prop, names->data, doc);
      break;
    case CORBA_tk_short:
      property_add_int (prop, names->data, doc);
      break;
    case CORBA_tk_double:
    case CORBA_tk_float:
      property_add_float (prop, names->data, doc);
      break;
    case CORBA_tk_long:
      property_add_int (prop, names->data, doc);
      break;
    default:
      g_warning ("Unhandled type %d", tc->kind);
      break;
    }
    g_free (prop);
  }
}

/*
 * Gets the properties of the widget. This is used for both displaying the
 * properties in the property editor, and also for saving the properties.
 */
static void
gb_bonobo_control_get_properties (GtkWidget * widget, GbWidgetGetArgData * data)
{
  BonoboPropertyBagClient *pbc = control_get_pbc (widget);
  GList      *names;
  GbWidget   *gbwidget;

  g_return_if_fail (pbc != NULL);

  gbwidget = gb_widget_lookup (widget);

  for (names = bonobo_property_bag_client_get_property_names (pbc);
       names; names = names->next) {
    CORBA_TypeCode tc;
    char          *prop = create_prop_name (names->data);

    tc  = bonobo_property_bag_client_get_property_type (pbc, names->data);
    switch (tc->kind) {
    case CORBA_tk_boolean:
      gb_widget_output_bool (data, prop,
			     bonobo_property_bag_client_get_value_gboolean (pbc, names->data));
      break;
    case CORBA_tk_string:
    {
      char *str = bonobo_property_bag_client_get_value_string (pbc, names->data);

      gb_widget_output_string (data, prop, str);

      g_free (str);
      break;
    }
    case CORBA_tk_long:
      gb_widget_output_int (data, prop,
			    bonobo_property_bag_client_get_value_glong (pbc, names->data));
      break;
    case CORBA_tk_float:
      gb_widget_output_float (data, prop,
			      bonobo_property_bag_client_get_value_gfloat (pbc, names->data));
      break;
    case CORBA_tk_double:
      gb_widget_output_float (data, prop,
			      bonobo_property_bag_client_get_value_gdouble (pbc, names->data));
      break;
    default:
      g_warning ("Unhandled type %d", tc->kind);
      break;
    }
    g_free (prop);
  }
}


/*
 * Sets the properties of the widget. This is used for both applying the
 * properties changed in the property editor, and also for loading.
 */
static void
gb_bonobo_control_set_properties (GtkWidget * widget, GbWidgetSetArgData * data)
{
  BonoboPropertyBagClient *pbc = control_get_pbc (widget);

  GList *names;

  g_return_if_fail (pbc != NULL);

  for (names = bonobo_property_bag_client_get_property_names (pbc);
       names; names = names->next) {
    CORBA_TypeCode tc;
    char          *prop = create_prop_name (names->data);

    tc  = bonobo_property_bag_client_get_property_type (pbc, names->data);
    switch (tc->kind) {
    case CORBA_tk_boolean:
    {
      gboolean val;

      val = gb_widget_input_bool (data, prop);
      if (data->apply)
        bonobo_property_bag_client_set_value_gboolean (pbc, names->data, val);
      break;
    }
    case CORBA_tk_string:
    {
      const char *str;

      str = gb_widget_input_string (data, prop);
      if (data->apply)
        bonobo_property_bag_client_set_value_string (pbc, names->data, str);

      break;
    }
    case CORBA_tk_float:
    {
      gfloat val;

      val = gb_widget_input_float (data, prop);
      if (data->apply)
        bonobo_property_bag_client_set_value_gfloat (pbc, names->data, val);
      break;
    }
    case CORBA_tk_double:
    {
      gdouble val;

      val = gb_widget_input_float (data, prop);
      if (data->apply)
        bonobo_property_bag_client_set_value_gdouble (pbc, names->data, val);
      break;
    }
    case CORBA_tk_long:
    {
      glong val;

      val = gb_widget_input_int (data, prop);
      if (data->apply)
        bonobo_property_bag_client_set_value_glong (pbc, names->data, val);
      break;
    }
    default:
      g_warning ("Unhandled type %d", tc->kind);
      break;
    }
    g_free (prop);
  }
}

GbWidget *
bonobo_gb_widget_new (const char *obj_id, const char *descr)
{
	BonoboGbWidget *w = g_new (BonoboGbWidget, 1);

	w->obj_id = g_strdup (obj_id);
	w->descr   = g_strdup (descr);

	gb_widget_init_struct (&w->gbwidget);

	/* Fill in the pixmap struct & tooltip */
	w->gbwidget.pixmap_struct = custom_xpm;
	w->gbwidget.tooltip       = w->descr;
	
	/* Fill in any functions that this Gbwidget has */
	w->gbwidget.gb_widget_new = gb_bonobo_control_new;
	w->gbwidget.gb_widget_create_properties = gb_bonobo_control_create_properties;
	w->gbwidget.gb_widget_get_properties = gb_bonobo_control_get_properties;
	w->gbwidget.gb_widget_set_properties = gb_bonobo_control_set_properties;

	MSG2 ("goad id '%s' '%s'\n", w->obj_id, w->descr);
	gb_widget_register_gbwidget (w->obj_id, (GbWidget *)w);
	palette_add_gbwidget ((GbWidget *)w, "Controls", w->obj_id);
	
	return (GbWidget *)w;
}

void
bonobo_gb_widget_destroy (GbWidget *gbwidget)
{
	BonoboGbWidget *w = (BonoboGbWidget *)gbwidget;

	if (w->obj_id)
		g_free (w->obj_id);
	w->obj_id = NULL;

	if (w->descr)
		g_free (w->descr);
	w->descr = NULL;
}

/*
 * Initializes the GbWidget structure.
 * I've placed this at the end of the file so we don't have to include
 * declarations of all the functions.
 */
void
gb_bonobo_control_init ()
{
	const char *ids [] = { "IDL:Bonobo/Control:1.0", NULL };
	GList *controls, *l;

	/* Initialise the GTK type */
	bonobo_widget_get_type ();

	controls = od_get_server_list (ids);
	for (l = controls; l; l = l->next) {
		static GbWidget *gbwidget;

		const char *obj_id = od_server_info_get_id (l->data);
		const char *desc   = od_server_info_get_description (l->data);

		gbwidget = bonobo_gb_widget_new (obj_id, desc);
	}
}

#endif
