/****************************************************************************
** $Id: qt/src/tools/qeuckrcodec.cpp   2.1.1   edited 2000-04-04 $
**
** Implementation of QEucKrCodec class
**
** Created : 990225
**
** Copyright (C) 1992-2000 Troll Tech AS.  All rights reserved.
**
** This file is part of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Troll Tech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** Licensees holding valid Qt Professional Edition licenses may use this
** file in accordance with the Qt Professional Edition License Agreement
** provided with the Qt Professional Edition.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
** information about the Professional Edition licensing, or see
** http://www.trolltech.com/qpl/ for QPL licensing information.
**
*****************************************************************************/

/*! \class QEucKrCodec qeuckrcodec.h

  \brief Provides conversion to and from EUC-KR character sets

  The QEucKrCodec class subclasses QTextCodec to provide support for
  EUC-JP, the main legacy encoding for UNIX machines in Korea

  It was margely written by Mizi Research Inc.; here is the copyright
  statement for the code as it was at the point of contribution.
  (Troll Tech's subsequent modifications are covered by the usual
  copyright for Qt.)

  \mustquote

  Copyright (c) 1999 Mizi Research Inc., All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met: <ol>
  <li> Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  <li> Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  </ol>

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.
*/

/* these must be made \internal
    virtual int mibEnum() const;
    const char* name() const;

    QTextDecoder* makeDecoder() const;

    QCString fromUnicode(const QString& uc, int& len_in_out) const;
    QString toUnicode(const char* chars, int len) const;

    int heuristicContentMatch(const char* chars, int len) const;
    int heuristicNameMatch(const char* hint) const;
*/


#include "qeuckrcodec.h"

unsigned int qt_Ksc5601ToUnicode(unsigned int code);
unsigned int qt_JohabToUnicode(unsigned int code);

unsigned int qt_UnicodeToKsc5601(unsigned int unicode);
unsigned int qt_UnicodeToJohab(unsigned int unicode);

#define	IsEucChar(c)	(((c) >= 0xa1) && ((c) <= 0xfe))
#define	QValidChar(u)	((u) ? QChar((ushort)(u)) : QChar::replacement)

/*!
  \reimp
*/

int QEucKrCodec::mibEnum() const
{
  /*
   * Name: EUC-KR  (preferred MIME name)                     [RFC1557,Choi]
   * MIBenum: 38
   * Source: RFC-1557 (see also KS_C_5861-1992)
   * Alias: csEUCKR
   */
  /* mibEnum for other codeset related with Korean.
     KS_C_5601-1987 36, ISO2022-KRi 37 */

  return 38;
}

/*!
  \reimp
*/

QCString QEucKrCodec::fromUnicode(const QString& uc, int& len_in_out) const
{
  int l = QMIN((int)uc.length(),len_in_out);
  int rlen = l*3+1;
  QCString rstr(rlen);
  uchar* cursor = (uchar*)rstr.data();
  for (int i=0; i<l; i++) {
    QChar ch = uc[i];
    uint j;
    if ( ch.row() == 0x00 && ch.cell() < 0x80 ) {
      // ASCII
      *cursor++ = ch.cell();
    } else if ((j = qt_UnicodeToKsc5601((ch.row() << 8) | ch.cell())) ) {
      // KSC 5601
      *cursor++ = (j >> 8)   | 0x80;
      *cursor++ = (j & 0xff) | 0x80;
    } else {
      // Error
      *cursor++ = '?';	// unknown char
    }
  }
  len_in_out = cursor - (uchar*)rstr.data();
  *cursor = 0;
  return rstr;
}

/*!
  \reimp
*/

QString QEucKrCodec::toUnicode(const char* chars, int len) const
{
  QString result;
  for (int i=0; i<len; i++) {
    uchar ch = chars[i];
    if ( ch < 0x80 ) {
      // ASCII
      result += QChar(ch);
    } else if ( IsEucChar(ch) ) {
      // KSC 5601
      if ( i < len-1 ) {
	uchar c2 = chars[++i];
	if ( IsEucChar(c2) ) {
	  uint u = qt_Ksc5601ToUnicode((ch << 8) | c2);
	  result += QValidChar(u);
	} else {
	  i--;
	  result += QChar::replacement;
	}
      } else {
	result += QChar::replacement;
      }
    } else {
      // Invalid
      result += QChar::replacement;
    }
  }
  return result;
}

/*!
  \reimp
*/

const char* QEucKrCodec::name() const
{
  return "eucKR";
}

/*!
  \reimp
*/

int QEucKrCodec::heuristicNameMatch(const char* hint) const
{
  int score = 0;
  bool ko = FALSE;
  if (strnicmp(hint, "ko_KR", 5) == 0 ||
      strnicmp(hint, "korean", 5) == 0) {
    score += 3;
    ko = TRUE;
  } else if (strnicmp(hint, "ko", 2) == 0) {
    score += 2;
    ko = TRUE;
  }
  const char *p;
  if (ko) {
    p = strchr(hint, '.');
    if (p == 0) {
      return score;
    }
    p++;
  } else {
    p = hint;
  }
  if (p) {
    if (stricmp(p, "eucKR") == 0) {
      return score + 4;
    }
    else if (stricmp(p, "euc") == 0) {
      return ko ? score + 4 : 1;
    }
  }
  return QTextCodec::heuristicNameMatch(hint);
}

/*!
  \reimp
*/

int QEucKrCodec::heuristicContentMatch(const char* chars, int len) const
{
  int score = 0;
  for (int i=0; i<len; i++) {
    uchar ch = chars[i];
    // No nulls allowed.
    if ( !ch )
      return -1;
    if ( ch < 32 && ch != '\t' && ch != '\n' && ch != '\r' ) {
      // Suspicious
      if ( score )
	score--;
    } else if ( ch < 0x80 ) {
      // Inconclusive
    } else if ( IsEucChar(ch) ) {
      // KSC 5601
      if ( i < len-1 ) {
	uchar c2 = chars[++i];
	if ( !IsEucChar(c2) )
	  return -1;
	score++;
      }
      score++;
    } else {
      // Invalid
      return -1;
    }
  }
  return score;
}

class QEucKrDecoder : public QTextDecoder {
  uchar buf[2];
  int nbuf;
public:
  QEucKrDecoder() : nbuf(0)
  {
  }

  QString toUnicode(const char* chars, int len)
  {
    QString result;
    for (int i=0; i<len; i++) {
      uchar ch = chars[i];
      switch (nbuf) {
      case 0:
	if ( ch < 0x80 ) {
	  // ASCII
	  result += QChar(ch);
	} else if ( IsEucChar(ch) ) {
	  // KSC 5601
	  buf[0] = ch;
	  nbuf = 1;
	} else {
	  // Invalid
	  result += QChar::replacement;
	}
	break;
      case 1:
	// KSC 5601
	if ( IsEucChar(ch) ) {
	  uint u = qt_Ksc5601ToUnicode((buf[0] << 8) |  ch);
	  result += QValidChar(u);
	} else {
	  // Error
	  result += QChar::replacement;
	}
	nbuf = 0;
	break;
      }
    }
    return result;
  }
};

/*!
  \reimp
*/

QTextDecoder* QEucKrCodec::makeDecoder() const
{
  return new QEucKrDecoder;
}

// code converter wrapper

static unsigned short ksc2kssmc ( unsigned short code );
static unsigned short kssmc2ksc ( unsigned short code );

static unsigned short unicode2kssmc ( unsigned short code );
static unsigned short kssmc2unicode ( unsigned short code );

unsigned int qt_Ksc5601ToUnicode(unsigned int code)
{
#if 0
  printf("qt_Ksc5601ToUnicode : code = %x, unicode = %x\n",
      code, kssmc2unicode(ksc2kssmc((unsigned short)code)));
#endif
  return kssmc2unicode(ksc2kssmc((unsigned short)code));
}

unsigned int qt_JohabToUnicode(unsigned int code)
{
  return kssmc2unicode((unsigned short)code);
}

unsigned int qt_UnicodeToKsc5601(unsigned int unicode)
{
#if 0
  printf("qt_UnicodeToKsc5601 : unicode = %x, %x\n",
      unicode, kssmc2ksc(unicode2kssmc((unsigned short)unicode)));
#endif
  return kssmc2ksc(unicode2kssmc((unsigned short)unicode));
}

unsigned int qt_UnicodeToJohab(unsigned int unicode)
{
  return unicode2kssmc((unsigned short)unicode);
}

/* Table including TG(johab) codes */
static const unsigned short KS_table[2350] = {

  0x8861,0x8862,0x8865,0x8868,0x8869,0x886a,0x886b,0x8871,
  0x8873,0x8874,0x8875,0x8876,0x8877,0x8878,0x8879,0x887b,
  0x887c,0x887d,0x8881,0x8882,0x8885,0x8889,0x8891,0x8893,
  0x8895,0x8896,0x8897,0x88a1,0x88a2,0x88a5,0x88a9,0x88b5,
  0x88b7,0x88c1,0x88c5,0x88c9,0x88e1,0x88e2,0x88e5,0x88e8,
  0x88e9,0x88eb,0x88f1,0x88f3,0x88f5,0x88f6,0x88f7,0x88f8,
  0x88fb,0x88fc,0x88fd,0x8941,0x8945,0x8949,0x8951,0x8953,
  0x8955,0x8956,0x8957,0x8961,0x8962,0x8963,0x8965,0x8968,
  0x8969,0x8971,0x8973,0x8975,0x8976,0x8977,0x897b,0x8981,
  0x8985,0x8989,0x8993,0x8995,0x89a1,0x89a2,0x89a5,0x89a8,
  0x89a9,0x89ab,0x89ad,0x89b0,0x89b1,0x89b3,0x89b5,0x89b7,
  0x89b8,0x89c1,0x89c2,0x89c5,0x89c9,0x89cb,

  0x89d1,0x89d3,0x89d5,0x89d7,0x89e1,0x89e5,0x89e9,0x89f1,
  0x89f6,0x89f7,0x8a41,0x8a42,0x8a45,0x8a49,0x8a51,0x8a53,
  0x8a55,0x8a57,0x8a61,0x8a65,0x8a69,0x8a73,0x8a75,0x8a81,
  0x8a82,0x8a85,0x8a88,0x8a89,0x8a8a,0x8a8b,0x8a90,0x8a91,
  0x8a93,0x8a95,0x8a97,0x8a98,0x8aa1,0x8aa2,0x8aa5,0x8aa9,
  0x8ab6,0x8ab7,0x8ac1,0x8ad5,0x8ae1,0x8ae2,0x8ae5,0x8ae9,
  0x8af1,0x8af3,0x8af5,0x8b41,0x8b45,0x8b49,0x8b61,0x8b62,
  0x8b65,0x8b68,0x8b69,0x8b6a,0x8b71,0x8b73,0x8b75,0x8b77,
  0x8b81,0x8ba1,0x8ba2,0x8ba5,0x8ba8,0x8ba9,0x8bab,0x8bb1,
  0x8bb3,0x8bb5,0x8bb7,0x8bb8,0x8bbc,0x8c61,0x8c62,0x8c63,
  0x8c65,0x8c69,0x8c6b,0x8c71,0x8c73,0x8c75,0x8c76,0x8c77,
  0x8c7b,0x8c81,0x8c82,0x8c85,0x8c89,0x8c91,

  0x8c93,0x8c95,0x8c96,0x8c97,0x8ca1,0x8ca2,0x8ca9,0x8ce1,
  0x8ce2,0x8ce3,0x8ce5,0x8ce9,0x8cf1,0x8cf3,0x8cf5,0x8cf6,
  0x8cf7,0x8d41,0x8d42,0x8d45,0x8d51,0x8d55,0x8d57,0x8d61,
  0x8d65,0x8d69,0x8d75,0x8d76,0x8d7b,0x8d81,0x8da1,0x8da2,
  0x8da5,0x8da7,0x8da9,0x8db1,0x8db3,0x8db5,0x8db7,0x8db8,
  0x8db9,0x8dc1,0x8dc2,0x8dc9,0x8dd6,0x8dd7,0x8de1,0x8de2,
  0x8df7,0x8e41,0x8e45,0x8e49,0x8e51,0x8e53,0x8e57,0x8e61,
  0x8e81,0x8e82,0x8e85,0x8e89,0x8e90,0x8e91,0x8e93,0x8e95,
  0x8e97,0x8e98,0x8ea1,0x8ea9,0x8eb6,0x8eb7,0x8ec1,0x8ec2,
  0x8ec5,0x8ec9,0x8ed1,0x8ed3,0x8ed6,0x8ee1,0x8ee5,0x8ee9,
  0x8ef1,0x8ef3,0x8f41,0x8f61,0x8f62,0x8f65,0x8f67,0x8f69,
  0x8f6b,0x8f70,0x8f71,0x8f73,0x8f75,0x8f77,

  0x8f7b,0x8fa1,0x8fa2,0x8fa5,0x8fa9,0x8fb1,0x8fb3,0x8fb5,
  0x8fb7,0x9061,0x9062,0x9063,0x9065,0x9068,0x9069,0x906a,
  0x906b,0x9071,0x9073,0x9075,0x9076,0x9077,0x9078,0x9079,
  0x907b,0x907d,0x9081,0x9082,0x9085,0x9089,0x9091,0x9093,
  0x9095,0x9096,0x9097,0x90a1,0x90a2,0x90a5,0x90a9,0x90b1,
  0x90b7,0x90e1,0x90e2,0x90e4,0x90e5,0x90e9,0x90eb,0x90ec,
  0x90f1,0x90f3,0x90f5,0x90f6,0x90f7,0x90fd,0x9141,0x9142,
  0x9145,0x9149,0x9151,0x9153,0x9155,0x9156,0x9157,0x9161,
  0x9162,0x9165,0x9169,0x9171,0x9173,0x9176,0x9177,0x917a,
  0x9181,0x9185,0x91a1,0x91a2,0x91a5,0x91a9,0x91ab,0x91b1,
  0x91b3,0x91b5,0x91b7,0x91bc,0x91bd,0x91c1,0x91c5,0x91c9,
  0x91d6,0x9241,0x9245,0x9249,0x9251,0x9253,

  0x9255,0x9261,0x9262,0x9265,0x9269,0x9273,0x9275,0x9277,
  0x9281,0x9282,0x9285,0x9288,0x9289,0x9291,0x9293,0x9295,
  0x9297,0x92a1,0x92b6,0x92c1,0x92e1,0x92e5,0x92e9,0x92f1,
  0x92f3,0x9341,0x9342,0x9349,0x9351,0x9353,0x9357,0x9361,
  0x9362,0x9365,0x9369,0x936a,0x936b,0x9371,0x9373,0x9375,
  0x9377,0x9378,0x937c,0x9381,0x9385,0x9389,0x93a1,0x93a2,
  0x93a5,0x93a9,0x93af,0x93b1,0x93b3,0x93b5,0x93b7,0x93bc,
  0x9461,0x9462,0x9463,0x9465,0x9468,0x9469,0x946a,0x946b,
  0x946c,0x9470,0x9471,0x9473,0x9475,0x9476,0x9477,0x9478,
  0x9479,0x947d,0x9481,0x9482,0x9485,0x9489,0x9491,0x9493,
  0x9495,0x9496,0x9497,0x94a1,0x94e1,0x94e2,0x94e3,0x94e5,
  0x94e8,0x94e9,0x94eb,0x94ec,0x94f1,0x94f3,

  0x94f5,0x94f7,0x94f9,0x94fc,0x9541,0x9542,0x9545,0x9549,
  0x9551,0x9553,0x9555,0x9556,0x9557,0x9561,0x9565,0x9569,
  0x9576,0x9577,0x9581,0x9585,0x95a1,0x95a2,0x95a5,0x95a8,
  0x95a9,0x95ab,0x95ad,0x95b1,0x95b3,0x95b5,0x95b7,0x95b9,
  0x95bb,0x95c1,0x95c5,0x95c9,0x95e1,0x95f6,0x9641,0x9645,
  0x9649,0x9651,0x9653,0x9655,0x9661,0x9681,0x9682,0x9685,
  0x9689,0x9691,0x9693,0x9695,0x9697,0x96a1,0x96b6,0x96c1,
  0x96d7,0x96e1,0x96e5,0x96e9,0x96f3,0x96f5,0x96f7,0x9741,
  0x9745,0x9749,0x9751,0x9757,0x9761,0x9762,0x9765,0x9768,
  0x9769,0x976b,0x9771,0x9773,0x9775,0x9777,0x9781,0x97a1,
  0x97a2,0x97a5,0x97a8,0x97a9,0x97b1,0x97b3,0x97b5,0x97b6,
  0x97b7,0x97b8,0x9861,0x9862,0x9865,0x9869,

  0x9871,0x9873,0x9875,0x9876,0x9877,0x987d,0x9881,0x9882,
  0x9885,0x9889,0x9891,0x9893,0x9895,0x9896,0x9897,0x98e1,
  0x98e2,0x98e5,0x98e9,0x98eb,0x98ec,0x98f1,0x98f3,0x98f5,
  0x98f6,0x98f7,0x98fd,0x9941,0x9942,0x9945,0x9949,0x9951,
  0x9953,0x9955,0x9956,0x9957,0x9961,0x9976,0x99a1,0x99a2,
  0x99a5,0x99a9,0x99b7,0x99c1,0x99c9,0x99e1,0x9a41,0x9a45,
  0x9a81,0x9a82,0x9a85,0x9a89,0x9a90,0x9a91,0x9a97,0x9ac1,
  0x9ae1,0x9ae5,0x9ae9,0x9af1,0x9af3,0x9af7,0x9b61,0x9b62,
  0x9b65,0x9b68,0x9b69,0x9b71,0x9b73,0x9b75,0x9b81,0x9b85,
  0x9b89,0x9b91,0x9b93,0x9ba1,0x9ba5,0x9ba9,0x9bb1,0x9bb3,
  0x9bb5,0x9bb7,0x9c61,0x9c62,0x9c65,0x9c69,0x9c71,0x9c73,
  0x9c75,0x9c76,0x9c77,0x9c78,0x9c7c,0x9c7d,

  0x9c81,0x9c82,0x9c85,0x9c89,0x9c91,0x9c93,0x9c95,0x9c96,
  0x9c97,0x9ca1,0x9ca2,0x9ca5,0x9cb5,0x9cb7,0x9ce1,0x9ce2,
  0x9ce5,0x9ce9,0x9cf1,0x9cf3,0x9cf5,0x9cf6,0x9cf7,0x9cfd,
  0x9d41,0x9d42,0x9d45,0x9d49,0x9d51,0x9d53,0x9d55,0x9d57,
  0x9d61,0x9d62,0x9d65,0x9d69,0x9d71,0x9d73,0x9d75,0x9d76,
  0x9d77,0x9d81,0x9d85,0x9d93,0x9d95,0x9da1,0x9da2,0x9da5,
  0x9da9,0x9db1,0x9db3,0x9db5,0x9db7,0x9dc1,0x9dc5,0x9dd7,
  0x9df6,0x9e41,0x9e45,0x9e49,0x9e51,0x9e53,0x9e55,0x9e57,
  0x9e61,0x9e65,0x9e69,0x9e73,0x9e75,0x9e77,0x9e81,0x9e82,
  0x9e85,0x9e89,0x9e91,0x9e93,0x9e95,0x9e97,0x9ea1,0x9eb6,
  0x9ec1,0x9ee1,0x9ee2,0x9ee5,0x9ee9,0x9ef1,0x9ef5,0x9ef7,
  0x9f41,0x9f42,0x9f45,0x9f49,0x9f51,0x9f53,

  0x9f55,0x9f57,0x9f61,0x9f62,0x9f65,0x9f69,0x9f71,0x9f73,
  0x9f75,0x9f77,0x9f78,0x9f7b,0x9f7c,0x9fa1,0x9fa2,0x9fa5,
  0x9fa9,0x9fb1,0x9fb3,0x9fb5,0x9fb7,0xa061,0xa062,0xa065,
  0xa067,0xa068,0xa069,0xa06a,0xa06b,0xa071,0xa073,0xa075,
  0xa077,0xa078,0xa07b,0xa07d,0xa081,0xa082,0xa085,0xa089,
  0xa091,0xa093,0xa095,0xa096,0xa097,0xa098,0xa0a1,0xa0a2,
  0xa0a9,0xa0b7,0xa0e1,0xa0e2,0xa0e5,0xa0e9,0xa0eb,0xa0f1,
  0xa0f3,0xa0f5,0xa0f7,0xa0f8,0xa0fd,0xa141,0xa142,0xa145,
  0xa149,0xa151,0xa153,0xa155,0xa156,0xa157,0xa161,0xa162,
  0xa165,0xa169,0xa175,0xa176,0xa177,0xa179,0xa181,0xa1a1,
  0xa1a2,0xa1a4,0xa1a5,0xa1a9,0xa1ab,0xa1b1,0xa1b3,0xa1b5,
  0xa1b7,0xa1c1,0xa1c5,0xa1d6,0xa1d7,0xa241,

  0xa245,0xa249,0xa253,0xa255,0xa257,0xa261,0xa265,0xa269,
  0xa273,0xa275,0xa281,0xa282,0xa283,0xa285,0xa288,0xa289,
  0xa28a,0xa28b,0xa291,0xa293,0xa295,0xa297,0xa29b,0xa29d,
  0xa2a1,0xa2a5,0xa2a9,0xa2b3,0xa2b5,0xa2c1,0xa2e1,0xa2e5,
  0xa2e9,0xa341,0xa345,0xa349,0xa351,0xa355,0xa361,0xa365,
  0xa369,0xa371,0xa375,0xa3a1,0xa3a2,0xa3a5,0xa3a8,0xa3a9,
  0xa3ab,0xa3b1,0xa3b3,0xa3b5,0xa3b6,0xa3b7,0xa3b9,0xa3bb,
  0xa461,0xa462,0xa463,0xa464,0xa465,0xa468,0xa469,0xa46a,
  0xa46b,0xa46c,0xa471,0xa473,0xa475,0xa477,0xa47b,0xa481,
  0xa482,0xa485,0xa489,0xa491,0xa493,0xa495,0xa496,0xa497,
  0xa49b,0xa4a1,0xa4a2,0xa4a5,0xa4b3,0xa4e1,0xa4e2,0xa4e5,
  0xa4e8,0xa4e9,0xa4eb,0xa4f1,0xa4f3,0xa4f5,

  0xa4f7,0xa4f8,0xa541,0xa542,0xa545,0xa548,0xa549,0xa551,
  0xa553,0xa555,0xa556,0xa557,0xa561,0xa562,0xa565,0xa569,
  0xa573,0xa575,0xa576,0xa577,0xa57b,0xa581,0xa585,0xa5a1,
  0xa5a2,0xa5a3,0xa5a5,0xa5a9,0xa5b1,0xa5b3,0xa5b5,0xa5b7,
  0xa5c1,0xa5c5,0xa5d6,0xa5e1,0xa5f6,0xa641,0xa642,0xa645,
  0xa649,0xa651,0xa653,0xa661,0xa665,0xa681,0xa682,0xa685,
  0xa688,0xa689,0xa68a,0xa68b,0xa691,0xa693,0xa695,0xa697,
  0xa69b,0xa69c,0xa6a1,0xa6a9,0xa6b6,0xa6c1,0xa6e1,0xa6e2,
  0xa6e5,0xa6e9,0xa6f7,0xa741,0xa745,0xa749,0xa751,0xa755,
  0xa757,0xa761,0xa762,0xa765,0xa769,0xa771,0xa773,0xa775,
  0xa7a1,0xa7a2,0xa7a5,0xa7a9,0xa7ab,0xa7b1,0xa7b3,0xa7b5,
  0xa7b7,0xa7b8,0xa7b9,0xa861,0xa862,0xa865,

  0xa869,0xa86b,0xa871,0xa873,0xa875,0xa876,0xa877,0xa87d,
  0xa881,0xa882,0xa885,0xa889,0xa891,0xa893,0xa895,0xa896,
  0xa897,0xa8a1,0xa8a2,0xa8b1,0xa8e1,0xa8e2,0xa8e5,0xa8e8,
  0xa8e9,0xa8f1,0xa8f5,0xa8f6,0xa8f7,0xa941,0xa957,0xa961,
  0xa962,0xa971,0xa973,0xa975,0xa976,0xa977,0xa9a1,0xa9a2,
  0xa9a5,0xa9a9,0xa9b1,0xa9b3,0xa9b7,0xaa41,0xaa61,0xaa77,
  0xaa81,0xaa82,0xaa85,0xaa89,0xaa91,0xaa95,0xaa97,0xab41,
  0xab57,0xab61,0xab65,0xab69,0xab71,0xab73,0xaba1,0xaba2,
  0xaba5,0xaba9,0xabb1,0xabb3,0xabb5,0xabb7,0xac61,0xac62,
  0xac64,0xac65,0xac68,0xac69,0xac6a,0xac6b,0xac71,0xac73,
  0xac75,0xac76,0xac77,0xac7b,0xac81,0xac82,0xac85,0xac89,
  0xac91,0xac93,0xac95,0xac96,0xac97,0xaca1,

  0xaca2,0xaca5,0xaca9,0xacb1,0xacb3,0xacb5,0xacb7,0xacc1,
  0xacc5,0xacc9,0xacd1,0xacd7,0xace1,0xace2,0xace3,0xace4,
  0xace5,0xace8,0xace9,0xaceb,0xacec,0xacf1,0xacf3,0xacf5,
  0xacf6,0xacf7,0xacfc,0xad41,0xad42,0xad45,0xad49,0xad51,
  0xad53,0xad55,0xad56,0xad57,0xad61,0xad62,0xad65,0xad69,
  0xad71,0xad73,0xad75,0xad76,0xad77,0xad81,0xad85,0xad89,
  0xad97,0xada1,0xada2,0xada3,0xada5,0xada9,0xadab,0xadb1,
  0xadb3,0xadb5,0xadb7,0xadbb,0xadc1,0xadc2,0xadc5,0xadc9,
  0xadd7,0xade1,0xade5,0xade9,0xadf1,0xadf5,0xadf6,0xae41,
  0xae45,0xae49,0xae51,0xae53,0xae55,0xae61,0xae62,0xae65,
  0xae69,0xae71,0xae73,0xae75,0xae77,0xae81,0xae82,0xae85,
  0xae88,0xae89,0xae91,0xae93,0xae95,0xae97,

  0xae99,0xae9b,0xae9c,0xaea1,0xaeb6,0xaec1,0xaec2,0xaec5,
  0xaec9,0xaed1,0xaed7,0xaee1,0xaee2,0xaee5,0xaee9,0xaef1,
  0xaef3,0xaef5,0xaef7,0xaf41,0xaf42,0xaf49,0xaf51,0xaf55,
  0xaf57,0xaf61,0xaf62,0xaf65,0xaf69,0xaf6a,0xaf71,0xaf73,
  0xaf75,0xaf77,0xafa1,0xafa2,0xafa5,0xafa8,0xafa9,0xafb0,
  0xafb1,0xafb3,0xafb5,0xafb7,0xafbc,0xb061,0xb062,0xb064,
  0xb065,0xb069,0xb071,0xb073,0xb076,0xb077,0xb07d,0xb081,
  0xb082,0xb085,0xb089,0xb091,0xb093,0xb096,0xb097,0xb0b7,
  0xb0e1,0xb0e2,0xb0e5,0xb0e9,0xb0eb,0xb0f1,0xb0f3,0xb0f6,
  0xb0f7,0xb141,0xb145,0xb149,0xb157,0xb1a1,0xb1a2,0xb1a5,
  0xb1a8,0xb1a9,0xb1ab,0xb1b1,0xb1b3,0xb1b7,0xb1c1,0xb1c2,
  0xb1c5,0xb1d6,0xb1e1,0xb1f6,0xb241,0xb245,

  0xb249,0xb251,0xb253,0xb261,0xb281,0xb282,0xb285,0xb289,
  0xb291,0xb293,0xb297,0xb2a1,0xb2b6,0xb2c1,0xb2e1,0xb2e5,
  0xb357,0xb361,0xb362,0xb365,0xb369,0xb36b,0xb370,0xb371,
  0xb373,0xb381,0xb385,0xb389,0xb391,0xb3a1,0xb3a2,0xb3a5,
  0xb3a9,0xb3b1,0xb3b3,0xb3b5,0xb3b7,0xb461,0xb462,0xb465,
  0xb466,0xb467,0xb469,0xb46a,0xb46b,0xb470,0xb471,0xb473,
  0xb475,0xb476,0xb477,0xb47b,0xb47c,0xb481,0xb482,0xb485,
  0xb489,0xb491,0xb493,0xb495,0xb496,0xb497,0xb4a1,0xb4a2,
  0xb4a5,0xb4a9,0xb4ac,0xb4b1,0xb4b3,0xb4b5,0xb4b7,0xb4bb,
  0xb4bd,0xb4c1,0xb4c5,0xb4c9,0xb4d3,0xb4e1,0xb4e2,0xb4e5,
  0xb4e6,0xb4e8,0xb4e9,0xb4ea,0xb4eb,0xb4f1,0xb4f3,0xb4f4,
  0xb4f5,0xb4f6,0xb4f7,0xb4f8,0xb4fa,0xb4fc,

  0xb541,0xb542,0xb545,0xb549,0xb551,0xb553,0xb555,0xb557,
  0xb561,0xb562,0xb563,0xb565,0xb569,0xb56b,0xb56c,0xb571,
  0xb573,0xb574,0xb575,0xb576,0xb577,0xb57b,0xb57c,0xb57d,
  0xb581,0xb585,0xb589,0xb591,0xb593,0xb595,0xb596,0xb5a1,
  0xb5a2,0xb5a5,0xb5a9,0xb5aa,0xb5ab,0xb5ad,0xb5b0,0xb5b1,
  0xb5b3,0xb5b5,0xb5b7,0xb5b9,0xb5c1,0xb5c2,0xb5c5,0xb5c9,
  0xb5d1,0xb5d3,0xb5d5,0xb5d6,0xb5d7,0xb5e1,0xb5e2,0xb5e5,
  0xb5f1,0xb5f5,0xb5f7,0xb641,0xb642,0xb645,0xb649,0xb651,
  0xb653,0xb655,0xb657,0xb661,0xb662,0xb665,0xb669,0xb671,
  0xb673,0xb675,0xb677,0xb681,0xb682,0xb685,0xb689,0xb68a,
  0xb68b,0xb691,0xb693,0xb695,0xb697,0xb6a1,0xb6a2,0xb6a5,
  0xb6a9,0xb6b1,0xb6b3,0xb6b6,0xb6b7,0xb6c1,

  0xb6c2,0xb6c5,0xb6c9,0xb6d1,0xb6d3,0xb6d7,0xb6e1,0xb6e2,
  0xb6e5,0xb6e9,0xb6f1,0xb6f3,0xb6f5,0xb6f7,0xb741,0xb742,
  0xb745,0xb749,0xb751,0xb753,0xb755,0xb757,0xb759,0xb761,
  0xb762,0xb765,0xb769,0xb76f,0xb771,0xb773,0xb775,0xb777,
  0xb778,0xb779,0xb77a,0xb77b,0xb77c,0xb77d,0xb781,0xb785,
  0xb789,0xb791,0xb795,0xb7a1,0xb7a2,0xb7a5,0xb7a9,0xb7aa,
  0xb7ab,0xb7b0,0xb7b1,0xb7b3,0xb7b5,0xb7b6,0xb7b7,0xb7b8,
  0xb7bc,0xb861,0xb862,0xb865,0xb867,0xb868,0xb869,0xb86b,
  0xb871,0xb873,0xb875,0xb876,0xb877,0xb878,0xb881,0xb882,
  0xb885,0xb889,0xb891,0xb893,0xb895,0xb896,0xb897,0xb8a1,
  0xb8a2,0xb8a5,0xb8a7,0xb8a9,0xb8b1,0xb8b7,0xb8c1,0xb8c5,
  0xb8c9,0xb8e1,0xb8e2,0xb8e5,0xb8e9,0xb8eb,

  0xb8f1,0xb8f3,0xb8f5,0xb8f7,0xb8f8,0xb941,0xb942,0xb945,
  0xb949,0xb951,0xb953,0xb955,0xb957,0xb961,0xb965,0xb969,
  0xb971,0xb973,0xb976,0xb977,0xb981,0xb9a1,0xb9a2,0xb9a5,
  0xb9a9,0xb9ab,0xb9b1,0xb9b3,0xb9b5,0xb9b7,0xb9b8,0xb9b9,
  0xb9bd,0xb9c1,0xb9c2,0xb9c9,0xb9d3,0xb9d5,0xb9d7,0xb9e1,
  0xb9f6,0xb9f7,0xba41,0xba45,0xba49,0xba51,0xba53,0xba55,
  0xba57,0xba61,0xba62,0xba65,0xba77,0xba81,0xba82,0xba85,
  0xba89,0xba8a,0xba8b,0xba91,0xba93,0xba95,0xba97,0xbaa1,
  0xbab6,0xbac1,0xbae1,0xbae2,0xbae5,0xbae9,0xbaf1,0xbaf3,
  0xbaf5,0xbb41,0xbb45,0xbb49,0xbb51,0xbb61,0xbb62,0xbb65,
  0xbb69,0xbb71,0xbb73,0xbb75,0xbb77,0xbba1,0xbba2,0xbba5,
  0xbba8,0xbba9,0xbbab,0xbbb1,0xbbb3,0xbbb5,

  0xbbb7,0xbbb8,0xbbbb,0xbbbc,0xbc61,0xbc62,0xbc65,0xbc67,
  0xbc69,0xbc6c,0xbc71,0xbc73,0xbc75,0xbc76,0xbc77,0xbc81,
  0xbc82,0xbc85,0xbc89,0xbc91,0xbc93,0xbc95,0xbc96,0xbc97,
  0xbca1,0xbca5,0xbcb7,0xbce1,0xbce2,0xbce5,0xbce9,0xbcf1,
  0xbcf3,0xbcf5,0xbcf6,0xbcf7,0xbd41,0xbd57,0xbd61,0xbd76,
  0xbda1,0xbda2,0xbda5,0xbda9,0xbdb1,0xbdb3,0xbdb5,0xbdb7,
  0xbdb9,0xbdc1,0xbdc2,0xbdc9,0xbdd6,0xbde1,0xbdf6,0xbe41,
  0xbe45,0xbe49,0xbe51,0xbe53,0xbe77,0xbe81,0xbe82,0xbe85,
  0xbe89,0xbe91,0xbe93,0xbe97,0xbea1,0xbeb6,0xbeb7,0xbee1,
  0xbf41,0xbf61,0xbf71,0xbf75,0xbf77,0xbfa1,0xbfa2,0xbfa5,
  0xbfa9,0xbfb1,0xbfb3,0xbfb7,0xbfb8,0xbfbd,0xc061,0xc062,
  0xc065,0xc067,0xc069,0xc071,0xc073,0xc075,

  0xc076,0xc077,0xc078,0xc081,0xc082,0xc085,0xc089,0xc091,
  0xc093,0xc095,0xc096,0xc097,0xc0a1,0xc0a5,0xc0a7,0xc0a9,
  0xc0b1,0xc0b7,0xc0e1,0xc0e2,0xc0e5,0xc0e9,0xc0f1,0xc0f3,
  0xc0f5,0xc0f6,0xc0f7,0xc141,0xc142,0xc145,0xc149,0xc151,
  0xc153,0xc155,0xc157,0xc161,0xc165,0xc176,0xc181,0xc185,
  0xc197,0xc1a1,0xc1a2,0xc1a5,0xc1a9,0xc1b1,0xc1b3,0xc1b5,
  0xc1b7,0xc1c1,0xc1c5,0xc1c9,0xc1d7,0xc241,0xc245,0xc249,
  0xc251,0xc253,0xc255,0xc257,0xc261,0xc271,0xc281,0xc282,
  0xc285,0xc289,0xc291,0xc293,0xc295,0xc297,0xc2a1,0xc2b6,
  0xc2c1,0xc2c5,0xc2e1,0xc2e5,0xc2e9,0xc2f1,0xc2f3,0xc2f5,
  0xc2f7,0xc341,0xc345,0xc349,0xc351,0xc357,0xc361,0xc362,
  0xc365,0xc369,0xc371,0xc373,0xc375,0xc377,

  0xc3a1,0xc3a2,0xc3a5,0xc3a8,0xc3a9,0xc3aa,0xc3b1,0xc3b3,
  0xc3b5,0xc3b7,0xc461,0xc462,0xc465,0xc469,0xc471,0xc473,
  0xc475,0xc477,0xc481,0xc482,0xc485,0xc489,0xc491,0xc493,
  0xc495,0xc496,0xc497,0xc4a1,0xc4a2,0xc4b7,0xc4e1,0xc4e2,
  0xc4e5,0xc4e8,0xc4e9,0xc4f1,0xc4f3,0xc4f5,0xc4f6,0xc4f7,
  0xc541,0xc542,0xc545,0xc549,0xc551,0xc553,0xc555,0xc557,
  0xc561,0xc565,0xc569,0xc571,0xc573,0xc575,0xc576,0xc577,
  0xc581,0xc5a1,0xc5a2,0xc5a5,0xc5a9,0xc5b1,0xc5b3,0xc5b5,
  0xc5b7,0xc5c1,0xc5c2,0xc5c5,0xc5c9,0xc5d1,0xc5d7,0xc5e1,
  0xc5f7,0xc641,0xc649,0xc661,0xc681,0xc682,0xc685,0xc689,
  0xc691,0xc693,0xc695,0xc697,0xc6a1,0xc6a5,0xc6a9,0xc6b7,
  0xc6c1,0xc6d7,0xc6e1,0xc6e2,0xc6e5,0xc6e9,

  0xc6f1,0xc6f3,0xc6f5,0xc6f7,0xc741,0xc745,0xc749,0xc751,
  0xc761,0xc762,0xc765,0xc769,0xc771,0xc773,0xc777,0xc7a1,
  0xc7a2,0xc7a5,0xc7a9,0xc7b1,0xc7b3,0xc7b5,0xc7b7,0xc861,
  0xc862,0xc865,0xc869,0xc86a,0xc871,0xc873,0xc875,0xc876,
  0xc877,0xc881,0xc882,0xc885,0xc889,0xc891,0xc893,0xc895,
  0xc896,0xc897,0xc8a1,0xc8b7,0xc8e1,0xc8e2,0xc8e5,0xc8e9,
  0xc8eb,0xc8f1,0xc8f3,0xc8f5,0xc8f6,0xc8f7,0xc941,0xc942,
  0xc945,0xc949,0xc951,0xc953,0xc955,0xc957,0xc961,0xc965,
  0xc976,0xc981,0xc985,0xc9a1,0xc9a2,0xc9a5,0xc9a9,0xc9b1,
  0xc9b3,0xc9b5,0xc9b7,0xc9bc,0xc9c1,0xc9c5,0xc9e1,0xca41,
  0xca45,0xca55,0xca57,0xca61,0xca81,0xca82,0xca85,0xca89,
  0xca91,0xca93,0xca95,0xca97,0xcaa1,0xcab6,

  0xcac1,0xcae1,0xcae2,0xcae5,0xcae9,0xcaf1,0xcaf3,0xcaf7,
  0xcb41,0xcb45,0xcb49,0xcb51,0xcb57,0xcb61,0xcb62,0xcb65,
  0xcb68,0xcb69,0xcb6b,0xcb71,0xcb73,0xcb75,0xcb81,0xcb85,
  0xcb89,0xcb91,0xcb93,0xcba1,0xcba2,0xcba5,0xcba9,0xcbb1,
  0xcbb3,0xcbb5,0xcbb7,0xcc61,0xcc62,0xcc63,0xcc65,0xcc69,
  0xcc6b,0xcc71,0xcc73,0xcc75,0xcc76,0xcc77,0xcc7b,0xcc81,
  0xcc82,0xcc85,0xcc89,0xcc91,0xcc93,0xcc95,0xcc96,0xcc97,
  0xcca1,0xcca2,0xcce1,0xcce2,0xcce5,0xcce9,0xccf1,0xccf3,
  0xccf5,0xccf6,0xccf7,0xcd41,0xcd42,0xcd45,0xcd49,0xcd51,
  0xcd53,0xcd55,0xcd57,0xcd61,0xcd65,0xcd69,0xcd71,0xcd73,
  0xcd76,0xcd77,0xcd81,0xcd89,0xcd93,0xcd95,0xcda1,0xcda2,
  0xcda5,0xcda9,0xcdb1,0xcdb3,0xcdb5,0xcdb7,

  0xcdc1,0xcdd7,0xce41,0xce45,0xce61,0xce65,0xce69,0xce73,
  0xce75,0xce81,0xce82,0xce85,0xce88,0xce89,0xce8b,0xce91,
  0xce93,0xce95,0xce97,0xcea1,0xceb7,0xcee1,0xcee5,0xcee9,
  0xcef1,0xcef5,0xcf41,0xcf45,0xcf49,0xcf51,0xcf55,0xcf57,
  0xcf61,0xcf65,0xcf69,0xcf71,0xcf73,0xcf75,0xcfa1,0xcfa2,
  0xcfa5,0xcfa9,0xcfb1,0xcfb3,0xcfb5,0xcfb7,0xd061,0xd062,
  0xd065,0xd069,0xd06e,0xd071,0xd073,0xd075,0xd077,0xd081,
  0xd082,0xd085,0xd089,0xd091,0xd093,0xd095,0xd096,0xd097,
  0xd0a1,0xd0b7,0xd0e1,0xd0e2,0xd0e5,0xd0e9,0xd0eb,0xd0f1,
  0xd0f3,0xd0f5,0xd0f7,0xd141,0xd142,0xd145,0xd149,0xd151,
  0xd153,0xd155,0xd157,0xd161,0xd162,0xd165,0xd169,0xd171,
  0xd173,0xd175,0xd176,0xd177,0xd181,0xd185,

  0xd189,0xd193,0xd1a1,0xd1a2,0xd1a5,0xd1a9,0xd1ae,0xd1b1,
  0xd1b3,0xd1b5,0xd1b7,0xd1bb,0xd1c1,0xd1c2,0xd1c5,0xd1c9,
  0xd1d5,0xd1d7,0xd1e1,0xd1e2,0xd1e5,0xd1f5,0xd1f7,0xd241,
  0xd242,0xd245,0xd249,0xd253,0xd255,0xd257,0xd261,0xd265,
  0xd269,0xd273,0xd275,0xd281,0xd282,0xd285,0xd289,0xd28e,
  0xd291,0xd295,0xd297,0xd2a1,0xd2a5,0xd2a9,0xd2b1,0xd2b7,
  0xd2c1,0xd2c2,0xd2c5,0xd2c9,0xd2d7,0xd2e1,0xd2e2,0xd2e5,
  0xd2e9,0xd2f1,0xd2f3,0xd2f5,0xd2f7,0xd341,0xd342,0xd345,
  0xd349,0xd351,0xd355,0xd357,0xd361,0xd362,0xd365,0xd367,
  0xd368,0xd369,0xd36a,0xd371,0xd373,0xd375,0xd377,0xd37b,
  0xd381,0xd385,0xd389,0xd391,0xd393,0xd397,0xd3a1,0xd3a2,
  0xd3a5,0xd3a9,0xd3b1,0xd3b3,0xd3b5,0xd3b7
};

/* ѱ   */
#define SingleNum 51

/* ѱ  51  ڵ */
static unsigned short single[SingleNum] = {
  0x8841,0x8c41,0x8444,0x9041,0x8446,0x8447,0x9441,0x9841,
  0x9c41,0x844a,0x844b,0x844c,0x844d,0x844e,0x844f,0x8450,
  0xa041,0xa441,0xa841,0x8454,0xac41,0xb041,0xb441,0xb841,
  0xbc41,0xc041,0xc441,0xc841,0xcc41,0xd041,0x8461,0x8481,
  0x84a1,0x84c1,0x84e1,0x8541,0x8561,0x8581,0x85a1,0x85c1,
  0x85e1,0x8641,0x8661,0x8681,0x86a1,0x86c1,0x86e1,0x8741,
  0x8761,0x8781,0x87a1
};

#define ChongSungNum 16
static unsigned short single_chongsung[ChongSungNum][2] = {
  {0x8458, 0xa4b8},   /* */
  {0x845d, 0xa4be},   /* */
  {0x8456, 0xa4b6},   /* */
  {0x8453, 0xa4b2},   /* */
  {0x845c, 0xa4bd},   /* */
  {0x8455, 0xa4b5},   /* */
  {0x845b, 0xa4bc},   /* */
  {0x8449, 0xa4a9},   /* */
  {0x845a, 0xa4bb},   /* */
  {0x8448, 0xa4a7},   /* */
  {0x8457, 0xa4b7},   /* */
  {0x8445, 0xa4a4},   /* */
  {0x8443, 0xa4a2},   /* */
  {0x8459, 0xa4ba},   /* */
  {0x8451, 0xa4b1},   /* */
  {0x8442, 0xa4a1}    /* */
};

// ϼ ڸ  ȯѴ.
static unsigned short ksc2kssmc( unsigned short code )
{
  unsigned int c1, c2, val1 = 0, val2 = 0;

  if ( (0x8080 & code) == 0 )
    return code;

  c1 = code >> 8;
  c2 = code & 0x00ff;
  if (c1 == 0xa4)
    return (c2 >= 0xa1 && c2 <= 0xd3) ?
      single[c2-0xa1] : (0xda00 +  c2);

  else if (c1 >= 0xb0 && c1 <= 0xc8)
  {
    if (c2 < 0xa1 || c2 > 0xfe)
      return 0;
    else
    {
      val1 = (c1 - 0xb0)*94 + c2 - 0xa1;
      return KS_table[val1];
    }
  }
  else if (c1 >= 0xa1 && c1 <= 0xac)
  {
    val1 = (c1 - 0xa1) / 2 + 0xd9;
    /* hanemacs װ ־µ Ͽ. */
    val2 = c1 & 0x01 ?  /* 0xa1, 0xa3, 0xa5, ... */
      ((c2-0xa1 < 0x7f-0x31) ? c2-0xa1+0x31 : c2-0xef+0x91)
      : c2;
  }
  else if (c1 >= 0xad && c1 <= 0xaf) /*  ȵ  */
  {
    val1 = 0xd9;      /* ڷ ä */
    val2 = 0x31;
  }
  else if (c1 == 0xc9 || c1 == 0xfe)
  {
    val1 = 0xd8;
    val2 = (c1 & 1 ?
	    ((c2 - 0xa1 < 0x7f - 0x31) ?
	     c2 - 0xa1+0x31 : c2 - 0xa1+0x91) : c2);
  }
  else if (c1 > 0xc9 && c1 < 0xfe) /*  */
  {
    val1 = (c1 - 0xca) / 2 + 0xe0;
    val2 = (c1 & 1 ? c2 :
	    (c2 - 0xa1 < 0x7f - 0x31) ?
	    c2 - 0xa1 + 0x31 : c2 - 0xa1 + 0x91);
  }
  return ((val1 << 8) + val2);
}

//
//  ϼ ٲٴ κ
//
#ifdef QT_EUCKR_USE_BSEARCH

static int codecmp( const void *p, const void *q )
{
  unsigned short *p1 = (unsigned short *) p;
  unsigned short *q1 = (unsigned short *) q;
  return ( *p1 - *q1 );
}

#else /* QT_EUCKR_USE_BSEARCH */

static const unsigned short * KsSearch( unsigned short code )
{
  unsigned short  c2;
  int		lo, hi, mid;

  lo = mid = 0;
  hi = sizeof(KS_table) / sizeof(KS_table[1]);

  while (lo <= hi) {
    mid = (lo + hi) / 2;
    c2  = KS_table[mid];
    if (code == c2)
      break;
    if (code < c2)
      hi = mid - 1;
    else
      lo = mid + 1;
  }
  /* bsearch()  return  ߱  */
  if( code == KS_table[mid] )
    return KS_table + mid;
  else
    return 0;
}

#endif /* QT_EUCKR_USE_BSEARCH */

// ϼ  ȯѴ.
static unsigned short kssmc2ksc ( unsigned short code )
{
  int		index;
  unsigned short c1, c2, val1 = 0, val2 = 0;
  unsigned short const *p;

  if ( (0x8000 & code) == 0 )
    return code;
  c1 = code >> 8;
  c2 = code & 0x00ff;

  if (c1 >= 0x84 && c1 <= 0xd3)		/* ѱ */
  {
#ifdef QT_EUCKR_USE_BSEARCH
    p = (const unsigned short *) bsearch (&code, KS_table,
	sizeof(KS_table)/sizeof(unsigned short),
	sizeof(unsigned short), codecmp);
#else
    p = KsSearch( code );
#endif
    if( p ) {
      index = p - KS_table;
      return (((index / 94) + 0xb0) << 8) + (index % 94) + 0xa1;
    }
    for( index = 0; index < SingleNum; index++ )
      if( single[index] == code )
	return (0xa4a1 + index);
    if( c1 == 0xda && c2 >= 0xd4 )	/* ѱ   */
      return (0x8400 + c2);
    for( index = 0; index < ChongSungNum; index++ )
      if( single_chongsung[index][0] == code )
	return (single_chongsung[index][1]);
  }
  else if( 0xd9 <= c1 && c1 <= 0xde )	/* ѱ ڸ   */
  {
    val1 = (c1 - 0xd9) * 2 + ((c2 < 0xa1) ? 0xa1 : 0xa2);
    val2 = (c2 < 0x91) ? (c2 - 0x31 + 0xa1) :
      ((c2 < 0xa1) ? c2 - 0x91 + 0xef : c2);
  }
  else if( 0xe0 <= c1 && c1 <= 0xf9 )	/*  */
  {
    val1 = (c1 - 0xe0) * 2 + ((c2 < 0xa1) ? 0xca : 0xcb);
    val2 = (c2 < 0x91) ? (c2 - 0x31 + 0xa1) :
      ((c2 < 0xa1) ? c2 - 0x91 + 0xef : c2);
  }
  else if( c1 == 0xd8 )			/*    */
  {
    val1 = (c2 < 0xa1) ? 0xc9 : 0xfe;
    val2 = (c2 < 0x91) ? (c2 - 0x31 + 0xa1) :
      ((c2 < 0xa1) ? c2 - 0x91 + 0xef : c2);
  }

  return ((val1 << 8) | val2);
}

static unsigned short unicode2kssmc(unsigned short unicode)
{
  unsigned short	f, m, l;

  unicode -= 0xac00;
  f = unicode / (21 * 28);
  unicode %= (21 * 28);
  m = unicode / 28;
  l = unicode % 28;

  // LATER rewrite with array
  f += 2;

  if( 16 < m)		m += 9;
  else if( 10 < m)	m += 7;
  else if( 4 < m)	m += 5;
  else			m += 3;

  if( 16 < l )		l += 2;
  else			l += 1;

#if 0
  fprintf(stderr, "unicode = %x, kssm = %d\n", unicode,
      f * (21 * 28) + m * 28 + l + 0xac00);
#endif
  return 0x8000 | (f << 10) | (m << 5) | l;
}

static unsigned short kssmc2unicode(unsigned short code)
{
  unsigned short	f, m, l;

  f = (code >> 10) & 0x1f;
  m = (code >> 5) & 0x1f;
  l = code & 0x1f;

  // LATER rewrite with array
  f -= 2;

  if( 25 < m)		m -= 9;
  else if( 17 < m)	m -= 7;
  else if( 9 < m)	m -= 5;
  else			m -= 3;

  if( 18 < l )		l -= 2;
  else			l -= 1;

#if 0
  fprintf(stderr, "code = %x, kssm = %d\n",
      code, f * (21 * 28) + m * 28 + l + 0xac00);
#endif
  return f * (21 * 28) + m * 28 + l + 0xac00;
}

#if 0 // for hanja

static unsigned short unicode_hanja[] = {
  0x4F3D,0x4F73,0x5047,0x50F9,0x52A0,0x53EF,0x5475,0x54E5,
  0x5609,0x5AC1,0x5BB6,0x6687,0x67B6,0x67B7,0x67EF,0x6B4C,
  0x73C2,0x75C2,0x7A3C,0x82DB,0x8304,0x8857,0x8888,0x8A36,
  0x8CC8,0x8DCF,0x8EFB,0x8FE6,0x99D5,0x523B,0x5374,0x5404,
  0x606A,0x6164,0x6BBC,0x73CF,0x811A,0x89BA,0x89D2,0x95A3,
  0x4F83,0x520A,0x58BE,0x5978,0x59E6,0x5E72,0x5E79,0x61C7,
  0x63C0,0x6746,0x67EC,0x687F,0x6F97,0x764E,0x770B,0x78F5,
  0x7A08,0x7AFF,0x7C21,0x809D,0x826E,0x8271,0x8AEB,0x9593,
  0x4E6B,0x559D,0x66F7,0x6E34,0x78A3,0x7AED,0x845B,0x8910,
  0x874E,0x97A8,0x52D8,0x574E,0x582A,0x5D4C,0x611F,0x61BE,
  0x6221,0x6562,0x67D1,0x6A44,0x6E1B,0x7518,0x75B3,0x76E3,
  0x77B0,0x7D3A,0x90AF,0x9451,0x9452,0x9F95,0x5323,0x5CAC,
  0x7532,0x80DB,0x9240,0x9598,0x525B,0x5808,0x59DC,0x5CA1,
  0x5D17,0x5EB7,0x5F3A,0x5F4A,0x6177,0x6C5F,0x757A,0x7586,
  0x7CE0,0x7D73,0x7DB1,0x7F8C,0x8154,0x8221,0x8591,0x8941,
  0x8B1B,0x92FC,0x964D,0x9C47,0x4ECB,0x4EF7,0x500B,0x51F1,
  0x584F,0x6137,0x613E,0x6168,0x6539,0x69EA,0x6F11,0x75A5,
  0x7686,0x76D6,0x7B87,0x82A5,0x84CB,0xF900,0x93A7,0x958B,
  0x5580,0x5BA2,0x5751,0xF901,0x7CB3,0x7FB9,0x91B5,0x5028,
  0x53BB,0x5C45,0x5DE8,0x62D2,0x636E,0x64DA,0x64E7,0x6E20,
  0x70AC,0x795B,0x8DDD,0x8E1E,0xF902,0x907D,0x9245,0x92F8,
  0x4E7E,0x4EF6,0x5065,0x5DFE,0x5EFA,0x6106,0x6957,0x8171,
  0x8654,0x8E47,0x9375,0x9A2B,0x4E5E,0x5091,0x6770,0x6840,
  0x5109,0x528D,0x5292,0x6AA2,0x77BC,0x9210,0x9ED4,0x52AB,
  0x602F,0x8FF2,0x5048,0x61A9,0x63ED,0x64CA,0x683C,0x6A84,
  0x6FC0,0x8188,0x89A1,0x9694,0x5805,0x727D,0x72AC,0x7504,
  0x7D79,0x7E6D,0x80A9,0x898B,0x8B74,0x9063,0x9D51,0x6289,
  0x6C7A,0x6F54,0x7D50,0x7F3A,0x8A23,0x517C,0x614A,0x7B9D,
  0x8B19,0x9257,0x938C,0x4EAC,0x4FD3,0x501E,0x50BE,0x5106,
  0x52C1,0x52CD,0x537F,0x5770,0x5883,0x5E9A,0x5F91,0x6176,
  0x61AC,0x64CE,0x656C,0x666F,0x66BB,0x66F4,0x6897,0x6D87,
  0x7085,0x70F1,0x749F,0x74A5,0x74CA,0x75D9,0x786C,0x78EC,
  0x7ADF,0x7AF6,0x7D45,0x7D93,0x8015,0x803F,0x811B,0x8396,
  0x8B66,0x8F15,0x9015,0x93E1,0x9803,0x9838,0x9A5A,0x9BE8,
  0x4FC2,0x5553,0x583A,0x5951,0x5B63,0x5C46,0x60B8,0x6212,
  0x6842,0x68B0,0x68E8,0x6EAA,0x754C,0x7678,0x78CE,0x7A3D,
  0x7CFB,0x7E6B,0x7E7C,0x8A08,0x8AA1,0x8C3F,0x968E,0x9DC4,
  0x53E4,0x53E9,0x544A,0x5471,0x56FA,0x59D1,0x5B64,0x5C3B,
  0x5EAB,0x62F7,0x6537,0x6545,0x6572,0x66A0,0x67AF,0x69C1,
  0x6CBD,0x75FC,0x7690,0x777E,0x7A3F,0x7F94,0x8003,0x80A1,
  0x818F,0x82E6,0x82FD,0x83F0,0x85C1,0x8831,0x88B4,0x8AA5,
  0xF903,0x8F9C,0x932E,0x96C7,0x9867,0x9AD8,0x9F13,0x54ED,
  0x659B,0x66F2,0x688F,0x7A40,0x8C37,0x9D60,0x56F0,0x5764,
  0x5D11,0x6606,0x68B1,0x68CD,0x6EFE,0x7428,0x889E,0x9BE4,
  0x6C68,0xF904,0x9AA8,0x4F9B,0x516C,0x5171,0x529F,0x5B54,
  0x5DE5,0x6050,0x606D,0x62F1,0x63A7,0x653B,0x73D9,0x7A7A,
  0x86A3,0x8CA2,0x978F,0x4E32,0x5BE1,0x6208,0x679C,0x74DC,
  0x79D1,0x83D3,0x8A87,0x8AB2,0x8DE8,0x904E,0x934B,0x9846,
  0x5ED3,0x69E8,0x85FF,0x90ED,0xF905,0x51A0,0x5B98,0x5BEC,
  0x6163,0x68FA,0x6B3E,0x704C,0x742F,0x74D8,0x7BA1,0x7F50,
  0x83C5,0x89C0,0x8CAB,0x95DC,0x9928,0x522E,0x605D,0x62EC,
  0x9002,0x4F8A,0x5149,0x5321,0x58D9,0x5EE3,0x66E0,0x6D38,
  0x709A,0x72C2,0x73D6,0x7B50,0x80F1,0x945B,0x5366,0x639B,
  0x7F6B,0x4E56,0x5080,0x584A,0x58DE,0x602A,0x6127,0x62D0,
  0x69D0,0x9B41,0x5B8F,0x7D18,0x80B1,0x8F5F,0x4EA4,0x50D1,
  0x54AC,0x55AC,0x5B0C,0x5DA0,0x5DE7,0x652A,0x654E,0x6821,
  0x6A4B,0x72E1,0x768E,0x77EF,0x7D5E,0x7FF9,0x81A0,0x854E,
  0x86DF,0x8F03,0x8F4E,0x90CA,0x9903,0x9A55,0x9BAB,0x4E18,
  0x4E45,0x4E5D,0x4EC7,0x4FF1,0x5177,0x52FE,0x5340,0x53E3,
  0x53E5,0x548E,0x5614,0x5775,0x57A2,0x5BC7,0x5D87,0x5ED0,
  0x61FC,0x62D8,0x6551,0x67B8,0x67E9,0x69CB,0x6B50,0x6BC6,
  0x6BEC,0x6C42,0x6E9D,0x7078,0x72D7,0x7396,0x7403,0x77BF,
  0x77E9,0x7A76,0x7D7F,0x8009,0x81FC,0x8205,0x820A,0x82DF,
  0x8862,0x8B33,0x8CFC,0x8EC0,0x9011,0x90B1,0x9264,0x92B6,
  0x99D2,0x9A45,0x9CE9,0x9DD7,0x9F9C,0x570B,0x5C40,0x83CA,
  0x97A0,0x97AB,0x9EB4,0x541B,0x7A98,0x7FA4,0x88D9,0x8ECD,
  0x90E1,0x5800,0x5C48,0x6398,0x7A9F,0x5BAE,0x5F13,0x7A79,
  0x7AAE,0x828E,0x8EAC,0x5026,0x5238,0x52F8,0x5377,0x5708,
  0x62F3,0x6372,0x6B0A,0x6DC3,0x7737,0x53A5,0x7357,0x8568,
  0x8E76,0x95D5,0x673A,0x6AC3,0x6F70,0x8A6D,0x8ECC,0x994B,
  0xF906,0x6677,0x6B78,0x8CB4,0x9B3C,0xF907,0x53EB,0x572D,
  0x594E,0x63C6,0x69FB,0x73EA,0x7845,0x7ABA,0x7AC5,0x7CFE,
  0x8475,0x898F,0x8D73,0x9035,0x95A8,0x52FB,0x5747,0x7547,
  0x7B60,0x83CC,0x921E,0xF908,0x6A58,0x514B,0x524B,0x5287,
  0x621F,0x68D8,0x6975,0x9699,0x50C5,0x52A4,0x52E4,0x61C3,
  0x65A4,0x6839,0x69FF,0x747E,0x7B4B,0x82B9,0x83EB,0x89B2,
  0x8B39,0x8FD1,0x9949,0xF909,0x4ECA,0x5997,0x64D2,0x6611,
  0x6A8E,0x7434,0x7981,0x79BD,0x82A9,0x887E,0x887F,0x895F,
  0xF90A,0x9326,0x4F0B,0x53CA,0x6025,0x6271,0x6C72,0x7D1A,
  0x7D66,0x4E98,0x5162,0x77DC,0x80AF,0x4F01,0x4F0E,0x5176,
  0x5180,0x55DC,0x5668,0x573B,0x57FA,0x57FC,0x5914,0x5947,
  0x5993,0x5BC4,0x5C90,0x5D0E,0x5DF1,0x5E7E,0x5FCC,0x6280,
  0x65D7,0x65E3,0x671E,0x671F,0x675E,0x68CB,0x68C4,0x6A5F,
  0x6B3A,0x6C23,0x6C7D,0x6C82,0x6DC7,0x7398,0x7426,0x742A,
  0x7482,0x74A3,0x7578,0x757F,0x7881,0x78EF,0x7941,0x7947,
  0x7948,0x797A,0x7B95,0x7D00,0x7DBA,0x7F88,0x8006,0x802D,
  0x808C,0x8A18,0x8B4F,0x8C48,0x8D77,0x9321,0x9324,0x98E2,
  0x9951,0x9A0E,0x9A0F,0x9A65,0x9E92,0x7DCA,0x4F76,0x5409,
  0x62EE,0x6854,0x91D1,0x55AB,0x513A,0xF90B,0xF90C,0x5A1C,
  0x61E6,0xF90D,0x62CF,0x62FF,0xF90E,0xF90F,0xF910,0xF911,
  0xF912,0xF913,0x90A3,0xF914,0xF915,0xF916,0xF917,0xF918,
  0x8AFE,0xF919,0xF91A,0xF91B,0xF91C,0x6696,0xF91D,0x7156,
  0xF91E,0xF91F,0x96E3,0xF920,0x634F,0x637A,0x5357,0xF921,
  0x678F,0x6960,0x6E73,0xF922,0x7537,0xF923,0xF924,0xF925,
  0x7D0D,0xF926,0xF927,0x8872,0x56CA,0x5A18,0xF928,0xF929,
  0xF92A,0xF92B,0xF92C,0x4E43,0xF92D,0x5167,0x5948,0x67F0,
  0x8010,0xF92E,0x5973,0x5E74,0x649A,0x79CA,0x5FF5,0x606C,
  0x62C8,0x637B,0x5BE7,0x5BD7,0x52AA,0xF92F,0x5974,0x5F29,
  0x6012,0xF930,0xF931,0xF932,0x7459,0xF933,0xF934,0xF935,
  0xF936,0xF937,0xF938,0x99D1,0xF939,0xF93A,0xF93B,0xF93C,
  0xF93D,0xF93E,0xF93F,0xF940,0xF941,0xF942,0xF943,0x6FC3,
  0xF944,0xF945,0x81BF,0x8FB2,0x60F1,0xF946,0xF947,0x8166,
  0xF948,0xF949,0x5C3F,0xF94A,0xF94B,0xF94C,0xF94D,0xF94E,
  0xF94F,0xF950,0xF951,0x5AE9,0x8A25,0x677B,0x7D10,0xF952,
  0xF953,0xF954,0xF955,0xF956,0xF957,0x80FD,0xF958,0xF959,
  0x5C3C,0x6CE5,0x533F,0x6EBA,0x591A,0x8336,0x4E39,0x4EB6,
  0x4F46,0x55AE,0x5718,0x58C7,0x5F56,0x65B7,0x65E6,0x6A80,
  0x6BB5,0x6E4D,0x77ED,0x7AEF,0x7C1E,0x7DDE,0x86CB,0x8892,
  0x9132,0x935B,0x64BB,0x6FBE,0x737A,0x75B8,0x9054,0x5556,
  0x574D,0x61BA,0x64D4,0x66C7,0x6DE1,0x6E5B,0x6F6D,0x6FB9,
  0x75F0,0x8043,0x81BD,0x8541,0x8983,0x8AC7,0x8B5A,0x931F,
  0x6C93,0x7553,0x7B54,0x8E0F,0x905D,0x5510,0x5802,0x5858,
  0x5E62,0x6207,0x649E,0x68E0,0x7576,0x7CD6,0x87B3,0x9EE8,
  0x4EE3,0x5788,0x576E,0x5927,0x5C0D,0x5CB1,0x5E36,0x5F85,
  0x6234,0x64E1,0x73B3,0x81FA,0x888B,0x8CB8,0x968A,0x9EDB,
  0x5B85,0x5FB7,0x60B3,0x5012,0x5200,0x5230,0x5716,0x5835,
  0x5857,0x5C0E,0x5C60,0x5CF6,0x5D8B,0x5EA6,0x5F92,0x60BC,
  0x6311,0x6389,0x6417,0x6843,0x68F9,0x6AC2,0x6DD8,0x6E21,
  0x6ED4,0x6FE4,0x71FE,0x76DC,0x7779,0x79B1,0x7A3B,0x8404,
  0x89A9,0x8CED,0x8DF3,0x8E48,0x9003,0x9014,0x9053,0x90FD,
  0x934D,0x9676,0x97DC,0x6BD2,0x7006,0x7258,0x72A2,0x7368,
  0x7763,0x79BF,0x7BE4,0x7E9B,0x8B80,0x58A9,0x60C7,0x6566,
  0x65FD,0x66BE,0x6C8C,0x711E,0x71C9,0x8C5A,0x9813,0x4E6D,
  0x7A81,0x4EDD,0x51AC,0x51CD,0x52D5,0x540C,0x61A7,0x6771,
  0x6850,0x68DF,0x6D1E,0x6F7C,0x75BC,0x77B3,0x7AE5,0x80F4,
  0x8463,0x9285,0x515C,0x6597,0x675C,0x6793,0x75D8,0x7AC7,
  0x8373,0xF95A,0x8C46,0x9017,0x982D,0x5C6F,0x81C0,0x829A,
  0x9041,0x906F,0x920D,0x5F97,0x5D9D,0x6A59,0x71C8,0x767B,
  0x7B49,0x85E4,0x8B04,0x9127,0x9A30,0x5587,0x61F6,0xF95B,
  0x7669,0x7F85,0x863F,0x87BA,0x88F8,0x908F,0xF95C,0x6D1B,
  0x70D9,0x73DE,0x7D61,0x843D,0xF95D,0x916A,0x99F1,0xF95E,
  0x4E82,0x5375,0x6B04,0x6B12,0x703E,0x721B,0x862D,0x9E1E,
  0x524C,0x8FA3,0x5D50,0x64E5,0x652C,0x6B16,0x6FEB,0x7C43,
  0x7E9C,0x85CD,0x8964,0x89BD,0x62C9,0x81D8,0x881F,0x5ECA,
  0x6717,0x6D6A,0x72FC,0x7405,0x746F,0x8782,0x90DE,0x4F86,
  0x5D0D,0x5FA0,0x840A,0x51B7,0x63A0,0x7565,0x4EAE,0x5006,
  0x5169,0x51C9,0x6881,0x6A11,0x7CAE,0x7CB1,0x7CE7,0x826F,
  0x8AD2,0x8F1B,0x91CF,0x4FB6,0x5137,0x52F5,0x5442,0x5EEC,
  0x616E,0x623E,0x65C5,0x6ADA,0x6FFE,0x792A,0x85DC,0x8823,
  0x95AD,0x9A62,0x9A6A,0x9E97,0x9ECE,0x529B,0x66C6,0x6B77,
  0x701D,0x792B,0x8F62,0x9742,0x6190,0x6200,0x6523,0x6F23,
  0x7149,0x7489,0x7DF4,0x806F,0x84EE,0x8F26,0x9023,0x934A,
  0x51BD,0x5217,0x52A3,0x6D0C,0x70C8,0x88C2,0x5EC9,0x6582,
  0x6BAE,0x6FC2,0x7C3E,0x7375,0x4EE4,0x4F36,0x56F9,0xF95F,
  0x5CBA,0x5DBA,0x601C,0x73B2,0x7B2D,0x7F9A,0x7FCE,0x8046,
  0x901E,0x9234,0x96F6,0x9748,0x9818,0x9F61,0x4F8B,0x6FA7,
  0x79AE,0x91B4,0x96B7,0x52DE,0xF960,0x6488,0x64C4,0x6AD3,
  0x6F5E,0x7018,0x7210,0x76E7,0x8001,0x8606,0x865C,0x8DEF,
  0x8F05,0x9732,0x9B6F,0x9DFA,0x9E75,0x788C,0x797F,0x7DA0,
  0x83C9,0x9304,0x9E7F,0x9E93,0x8AD6,0x58DF,0x5F04,0x6727,
  0x7027,0x74CF,0x7C60,0x807E,0x5121,0x7028,0x7262,0x78CA,
  0x8CC2,0x8CDA,0x8CF4,0x96F7,0x4E86,0x50DA,0x5BEE,0x5ED6,
  0x6599,0x71CE,0x7642,0x77AD,0x804A,0x84FC,0x907C,0x9B27,
  0x9F8D,0x58D8,0x5A41,0x5C62,0x6A13,0x6DDA,0x6F0F,0x763B,
  0x7D2F,0x7E37,0x851E,0x8938,0x93E4,0x964B,0x5289,0x65D2,
  0x67F3,0x69B4,0x6D41,0x6E9C,0x700F,0x7409,0x7460,0x7559,
  0x7624,0x786B,0x8B2C,0x985E,0x516D,0x622E,0x9678,0x4F96,
  0x502B,0x5D19,0x6DEA,0x7DB8,0x8F2A,0x5F8B,0x6144,0x6817,
  0xF961,0x9686,0x52D2,0x808B,0x51DC,0x51CC,0x695E,0x7A1C,
  0x7DBE,0x83F1,0x9675,0x4FDA,0x5229,0x5398,0x540F,0x550E,
  0x5C65,0x60A7,0x674E,0x68A8,0x6D6C,0x7281,0x72F8,0x7406,
  0x7483,0xF962,0x75E2,0x7C6C,0x7F79,0x7FB8,0x8389,0x88CF,
  0x88E1,0x91CC,0x91D0,0x96E2,0x9BC9,0x541D,0x6F7E,0x71D0,
  0x7498,0x85FA,0x8EAA,0x96A3,0x9C57,0x9E9F,0x6797,0x6DCB,
  0x7433,0x81E8,0x9716,0x782C,0x7ACB,0x7B20,0x7C92,0x6469,
  0x746A,0x75F2,0x78BC,0x78E8,0x99AC,0x9B54,0x9EBB,0x5BDE,
  0x5E55,0x6F20,0x819C,0x83AB,0x9088,0x4E07,0x534D,0x5A29,
  0x5DD2,0x5F4E,0x6162,0x633D,0x6669,0x66FC,0x6EFF,0x6F2B,
  0x7063,0x779E,0x842C,0x8513,0x883B,0x8F13,0x9945,0x9C3B,
  0x551C,0x62B9,0x672B,0x6CAB,0x8309,0x896A,0x977A,0x4EA1,
  0x5984,0x5FD8,0x5FD9,0x671B,0x7DB2,0x7F54,0x8292,0x832B,
  0x83BD,0x8F1E,0x9099,0x57CB,0x59B9,0x5A92,0x5BD0,0x6627,
  0x679A,0x6885,0x6BCF,0x7164,0x7F75,0x8CB7,0x8CE3,0x9081,
  0x9B45,0x8108,0x8C8A,0x964C,0x9A40,0x9EA5,0x5B5F,0x6C13,
  0x731B,0x76F2,0x76DF,0x840C,0x51AA,0x8993,0x514D,0x5195,
  0x52C9,0x68C9,0x6C94,0x7704,0x7720,0x7DBF,0x7DEC,0x9762,
  0x9EB5,0x6EC5,0x8511,0x51A5,0x540D,0x547D,0x660E,0x669D,
  0x6927,0x6E9F,0x76BF,0x7791,0x8317,0x84C2,0x879F,0x9169,
  0x9298,0x9CF4,0x8882,0x4FAE,0x5192,0x52DF,0x59C6,0x5E3D,
  0x6155,0x6478,0x6479,0x66AE,0x67D0,0x6A21,0x6BCD,0x6BDB,
  0x725F,0x7261,0x7441,0x7738,0x77DB,0x8017,0x82BC,0x8305,
  0x8B00,0x8B28,0x8C8C,0x6728,0x6C90,0x7267,0x76EE,0x7766,
  0x7A46,0x9DA9,0x6B7F,0x6C92,0x5922,0x6726,0x8499,0x536F,
  0x5893,0x5999,0x5EDF,0x63CF,0x6634,0x6773,0x6E3A,0x732B,
  0x7AD7,0x82D7,0x9328,0x52D9,0x5DEB,0x61AE,0x61CB,0x620A,
  0x62C7,0x64AB,0x65E0,0x6959,0x6B66,0x6BCB,0x7121,0x73F7,
  0x755D,0x7E46,0x821E,0x8302,0x856A,0x8AA3,0x8CBF,0x9727,
  0x9D61,0x58A8,0x9ED8,0x5011,0x520E,0x543B,0x554F,0x6587,
  0x6C76,0x7D0A,0x7D0B,0x805E,0x868A,0x9580,0x96EF,0x52FF,
  0x6C95,0x7269,0x5473,0x5A9A,0x5C3E,0x5D4B,0x5F4C,0x5FAE,
  0x672A,0x68B6,0x6963,0x6E3C,0x6E44,0x7709,0x7C73,0x7F8E,
  0x8587,0x8B0E,0x8FF7,0x9761,0x9EF4,0x5CB7,0x60B6,0x610D,
  0x61AB,0x654F,0x65FB,0x65FC,0x6C11,0x6CEF,0x739F,0x73C9,
  0x7DE1,0x9594,0x5BC6,0x871C,0x8B10,0x525D,0x535A,0x62CD,
  0x640F,0x64B2,0x6734,0x6A38,0x6CCA,0x73C0,0x749E,0x7B94,
  0x7C95,0x7E1B,0x818A,0x8236,0x8584,0x8FEB,0x96F9,0x99C1,
  0x4F34,0x534A,0x53CD,0x53DB,0x62CC,0x642C,0x6500,0x6591,
  0x69C3,0x6CEE,0x6F58,0x73ED,0x7554,0x7622,0x76E4,0x76FC,
  0x78D0,0x78FB,0x792C,0x7D46,0x822C,0x87E0,0x8FD4,0x9812,
  0x98EF,0x52C3,0x62D4,0x64A5,0x6E24,0x6F51,0x767C,0x8DCB,
  0x91B1,0x9262,0x9AEE,0x9B43,0x5023,0x508D,0x574A,0x59A8,
  0x5C28,0x5E47,0x5F77,0x623F,0x653E,0x65B9,0x65C1,0x6609,
  0x678B,0x699C,0x6EC2,0x78C5,0x7D21,0x80AA,0x8180,0x822B,
  0x82B3,0x84A1,0x868C,0x8A2A,0x8B17,0x90A6,0x9632,0x9F90,
  0x500D,0x4FF3,0xF963,0x57F9,0x5F98,0x62DC,0x6392,0x676F,
  0x6E43,0x7119,0x76C3,0x80CC,0x80DA,0x88F4,0x88F5,0x8919,
  0x8CE0,0x8F29,0x914D,0x966A,0x4F2F,0x4F70,0x5E1B,0x67CF,
  0x6822,0x767D,0x767E,0x9B44,0x5E61,0x6A0A,0x7169,0x71D4,
  0x756A,0xF964,0x7E41,0x8543,0x85E9,0x98DC,0x4F10,0x7B4F,
  0x7F70,0x95A5,0x51E1,0x5E06,0x68B5,0x6C3E,0x6C4E,0x6CDB,
  0x72AF,0x7BC4,0x8303,0x6CD5,0x743A,0x50FB,0x5288,0x58C1,
  0x64D8,0x6A97,0x74A7,0x7656,0x78A7,0x8617,0x95E2,0x9739,
  0xF965,0x535E,0x5F01,0x8B8A,0x8FA8,0x8FAF,0x908A,0x5225,
  0x77A5,0x9C49,0x9F08,0x4E19,0x5002,0x5175,0x5C5B,0x5E77,
  0x661E,0x663A,0x67C4,0x68C5,0x70B3,0x7501,0x75C5,0x79C9,
  0x7ADD,0x8F27,0x9920,0x9A08,0x4FDD,0x5821,0x5831,0x5BF6,
  0x666E,0x6B65,0x6D11,0x6E7A,0x6F7D,0x73E4,0x752B,0x83E9,
  0x88DC,0x8913,0x8B5C,0x8F14,0x4F0F,0x50D5,0x5310,0x535C,
  0x5B93,0x5FA9,0x670D,0x798F,0x8179,0x832F,0x8514,0x8907,
  0x8986,0x8F39,0x8F3B,0x99A5,0x9C12,0x672C,0x4E76,0x4FF8,
  0x5949,0x5C01,0x5CEF,0x5CF0,0x6367,0x68D2,0x70FD,0x71A2,
  0x742B,0x7E2B,0x84EC,0x8702,0x9022,0x92D2,0x9CF3,0x4E0D,
  0x4ED8,0x4FEF,0x5085,0x5256,0x526F,0x5426,0x5490,0x57E0,
  0x592B,0x5A66,0x5B5A,0x5B75,0x5BCC,0x5E9C,0xF966,0x6276,
  0x6577,0x65A7,0x6D6E,0x6EA5,0x7236,0x7B26,0x7C3F,0x7F36,
  0x8150,0x8151,0x819A,0x8240,0x8299,0x83A9,0x8A03,0x8CA0,
  0x8CE6,0x8CFB,0x8D74,0x8DBA,0x90E8,0x91DC,0x961C,0x9644,
  0x99D9,0x9CE7,0x5317,0x5206,0x5429,0x5674,0x58B3,0x5954,
  0x596E,0x5FFF,0x61A4,0x626E,0x6610,0x6C7E,0x711A,0x76C6,
  0x7C89,0x7CDE,0x7D1B,0x82AC,0x8CC1,0x96F0,0xF967,0x4F5B,
  0x5F17,0x5F7F,0x62C2,0x5D29,0x670B,0x68DA,0x787C,0x7E43,
  0x9D6C,0x4E15,0x5099,0x5315,0x532A,0x5351,0x5983,0x5A62,
  0x5E87,0x60B2,0x618A,0x6249,0x6279,0x6590,0x6787,0x69A7,
  0x6BD4,0x6BD6,0x6BD7,0x6BD8,0x6CB8,0xF968,0x7435,0x75FA,
  0x7812,0x7891,0x79D5,0x79D8,0x7C83,0x7DCB,0x7FE1,0x80A5,
  0x813E,0x81C2,0x83F2,0x871A,0x88E8,0x8AB9,0x8B6C,0x8CBB,
  0x9119,0x975E,0x98DB,0x9F3B,0x56AC,0x5B2A,0x5F6C,0x658C,
  0x6AB3,0x6BAF,0x6D5C,0x6FF1,0x7015,0x725D,0x73AD,0x8CA7,
  0x8CD3,0x983B,0x6191,0x6C37,0x8058,0x9A01,0x4E4D,0x4E8B,
  0x4E9B,0x4ED5,0x4F3A,0x4F3C,0x4F7F,0x4FDF,0x50FF,0x53F2,
  0x53F8,0x5506,0x55E3,0x56DB,0x58EB,0x5962,0x5A11,0x5BEB,
  0x5BFA,0x5C04,0x5DF3,0x5E2B,0x5F99,0x601D,0x6368,0x659C,
  0x65AF,0x67F6,0x67FB,0x68AD,0x6B7B,0x6C99,0x6CD7,0x6E23,
  0x7009,0x7345,0x7802,0x793E,0x7940,0x7960,0x79C1,0x7BE9,
  0x7D17,0x7D72,0x8086,0x820D,0x838E,0x84D1,0x86C7,0x88DF,
  0x8A50,0x8A5E,0x8B1D,0x8CDC,0x8D66,0x8FAD,0x90AA,0x98FC,
  0x99DF,0x9E9D,0x524A,0xF969,0x6714,0xF96A,0x5098,0x522A,
  0x5C71,0x6563,0x6C55,0x73CA,0x7523,0x759D,0x7B97,0x849C,
  0x9178,0x9730,0x4E77,0x6492,0x6BBA,0x715E,0x85A9,0x4E09,
  0xF96B,0x6749,0x68EE,0x6E17,0x829F,0x8518,0x886B,0x63F7,
  0x6F81,0x9212,0x98AF,0x4E0A,0x50B7,0x50CF,0x511F,0x5546,
  0x55AA,0x5617,0x5B40,0x5C19,0x5CE0,0x5E38,0x5E8A,0x5EA0,
  0x5EC2,0x60F3,0x6851,0x6A61,0x6E58,0x723D,0x7240,0x72C0,
  0x76F8,0x7965,0x7BB1,0x7FD4,0x88F3,0x89F4,0x8A73,0x8C61,
  0x8CDE,0x971C,0x585E,0x74BD,0x8CFD,0x55C7,0xF96C,0x7A61,
  0x7D22,0x8272,0x7272,0x751F,0x7525,0xF96D,0x7B19,0x5885,
  0x58FB,0x5DBC,0x5E8F,0x5EB6,0x5F90,0x6055,0x6292,0x637F,
  0x654D,0x6691,0x66D9,0x66F8,0x6816,0x68F2,0x7280,0x745E,
  0x7B6E,0x7D6E,0x7DD6,0x7F72,0x80E5,0x8212,0x85AF,0x897F,
  0x8A93,0x901D,0x92E4,0x9ECD,0x9F20,0x5915,0x596D,0x5E2D,
  0x60DC,0x6614,0x6673,0x6790,0x6C50,0x6DC5,0x6F5F,0x77F3,
  0x78A9,0x84C6,0x91CB,0x932B,0x4ED9,0x50CA,0x5148,0x5584,
  0x5B0B,0x5BA3,0x6247,0x657E,0x65CB,0x6E32,0x717D,0x7401,
  0x7444,0x7487,0x74BF,0x766C,0x79AA,0x7DDA,0x7E55,0x7FA8,
  0x817A,0x81B3,0x8239,0x861A,0x87EC,0x8A75,0x8DE3,0x9078,
  0x9291,0x9425,0x994D,0x9BAE,0x5368,0x5C51,0x6954,0x6CC4,
  0x6D29,0x6E2B,0x820C,0x859B,0x893B,0x8A2D,0x8AAA,0x96EA,
  0x9F67,0x5261,0x66B9,0x6BB2,0x7E96,0x87FE,0x8D0D,0x9583,
  0x965D,0x651D,0x6D89,0x71EE,0xF96E,0x57CE,0x59D3,0x5BAC,
  0x6027,0x60FA,0x6210,0x661F,0x665F,0x7329,0x73F9,0x76DB,
  0x7701,0x7B6C,0x8056,0x8072,0x8165,0x8AA0,0x9192,0x4E16,
  0x52E2,0x6B72,0x6D17,0x7A05,0x7B39,0x7D30,0xF96F,0x8CB0,
  0x53EC,0x562F,0x5851,0x5BB5,0x5C0F,0x5C11,0x5DE2,0x6240,
  0x6383,0x6414,0x662D,0x68B3,0x6CBC,0x6D88,0x6EAF,0x701F,
  0x70A4,0x71D2,0x7526,0x758F,0x758E,0x7619,0x7B11,0x7BE0,
  0x7C2B,0x7D20,0x7D39,0x852C,0x856D,0x8607,0x8A34,0x900D,
  0x9061,0x90B5,0x92B7,0x97F6,0x9A37,0x4FD7,0x5C6C,0x675F,
  0x6D91,0x7C9F,0x7E8C,0x8B16,0x8D16,0x901F,0x5B6B,0x5DFD,
  0x640D,0x84C0,0x905C,0x98E1,0x7387,0x5B8B,0x609A,0x677E,
  0x6DDE,0x8A1F,0x8AA6,0x9001,0x980C,0x5237,0xF970,0x7051,
  0x788E,0x9396,0x8870,0x91D7,0x4FEE,0x53D7,0x55FD,0x56DA,
  0x5782,0x58FD,0x5AC2,0x5B88,0x5CAB,0x5CC0,0x5E25,0x6101,
  0x620D,0x624B,0x6388,0x641C,0x6536,0x6578,0x6A39,0x6B8A,
  0x6C34,0x6D19,0x6F31,0x71E7,0x72E9,0x7378,0x7407,0x74B2,
  0x7626,0x7761,0x79C0,0x7A57,0x7AEA,0x7CB9,0x7D8F,0x7DAC,
  0x7E61,0x7F9E,0x8129,0x8331,0x8490,0x84DA,0x85EA,0x8896,
  0x8AB0,0x8B90,0x8F38,0x9042,0x9083,0x916C,0x9296,0x92B9,
  0x968B,0x96A7,0x96A8,0x96D6,0x9700,0x9808,0x9996,0x9AD3,
  0x9B1A,0x53D4,0x587E,0x5919,0x5B70,0x5BBF,0x6DD1,0x6F5A,
  0x719F,0x7421,0x74B9,0x8085,0x83FD,0x5DE1,0x5F87,0x5FAA,
  0x6042,0x65EC,0x6812,0x696F,0x6A53,0x6B89,0x6D35,0x6DF3,
  0x73E3,0x76FE,0x77AC,0x7B4D,0x7D14,0x8123,0x821C,0x8340,
  0x84F4,0x8563,0x8A62,0x8AC4,0x9187,0x931E,0x9806,0x99B4,
  0x620C,0x8853,0x8FF0,0x9265,0x5D07,0x5D27,0x5D69,0x745F,
  0x819D,0x8768,0x6FD5,0x62FE,0x7FD2,0x8936,0x8972,0x4E1E,
  0x4E58,0x50E7,0x52DD,0x5347,0x627F,0x6607,0x7E69,0x8805,
  0x965E,0x4F8D,0x5319,0x5636,0x59CB,0x5AA4,0x5C38,0x5C4E,
  0x5C4D,0x5E02,0x5F11,0x6043,0x65BD,0x662F,0x6642,0x67BE,
  0x67F4,0x731C,0x77E2,0x793A,0x7FC5,0x8494,0x84CD,0x8996,
  0x8A66,0x8A69,0x8AE1,0x8C55,0x8C7A,0x57F4,0x5BD4,0x5F0F,
  0x606F,0x62ED,0x690D,0x6B96,0x6E5C,0x7184,0x7BD2,0x8755,
  0x8B58,0x8EFE,0x98DF,0x98FE,0x4F38,0x4F81,0x4FE1,0x547B,
  0x5A20,0x5BB8,0x613C,0x65B0,0x6668,0x71FC,0x7533,0x795E,
  0x7D33,0x814E,0x81E3,0x8398,0x85AA,0x85CE,0x8703,0x8A0A,
  0x8EAB,0x8F9B,0xF971,0x8FC5,0x5931,0x5BA4,0x5BE6,0x6089,
  0x5BE9,0x5C0B,0x5FC3,0x6C81,0xF972,0x6DF1,0x700B,0x751A,
  0x82AF,0x8AF6,0x4EC0,0x5341,0xF973,0x96D9,0x6C0F,0x4E9E,
  0x4FC4,0x5152,0x555E,0x5A25,0x5CE8,0x6211,0x7259,0x82BD,
  0x83AA,0x86FE,0x8859,0x8A1D,0x963F,0x96C5,0x9913,0x9D09,
  0x9D5D,0x580A,0x5CB3,0x5DBD,0x5E44,0x60E1,0x6115,0x63E1,
  0x6A02,0x6E25,0x9102,0x9354,0x984E,0x9C10,0x9F77,0x5B89,
  0x5CB8,0x6309,0x664F,0x6848,0x773C,0x96C1,0x978D,0x9854,
  0x9B9F,0x65A1,0x8B01,0x8ECB,0x95BC,0x5535,0x5CA9,0x5DD6,
  0x5EB5,0x6697,0x764C,0x83F4,0x95C7,0x58D3,0x62BC,0x72CE,
  0x9D28,0x4EF0,0x592E,0x600F,0x663B,0x6B83,0x79E7,0x9D26,
  0x5393,0x54C0,0x57C3,0x5D16,0x611B,0x66D6,0x6DAF,0x788D,
  0x827E,0x9698,0x9744,0x5384,0x627C,0x6396,0x6DB2,0x7E0A,
  0x814B,0x984D,0x6AFB,0x7F4C,0x9DAF,0x9E1A,0x4E5F,0x503B,
  0x51B6,0x591C,0x60F9,0x63F6,0x6930,0x723A,0x8036,0xF974,
  0x91CE,0x5F31,0xF975,0xF976,0x7D04,0x82E5,0x846F,0x84BB,
  0x85E5,0x8E8D,0xF977,0x4F6F,0xF978,0xF979,0x58E4,0x5B43,
  0x6059,0x63DA,0x6518,0x656D,0x6698,0xF97A,0x694A,0x6A23,
  0x6D0B,0x7001,0x716C,0x75D2,0x760D,0x79B3,0x7A70,0xF97B,
  0x7F8A,0xF97C,0x8944,0xF97D,0x8B93,0x91C0,0x967D,0xF97E,
  0x990A,0x5704,0x5FA1,0x65BC,0x6F01,0x7600,0x79A6,0x8A9E,
  0x99AD,0x9B5A,0x9F6C,0x5104,0x61B6,0x6291,0x6A8D,0x81C6,
  0x5043,0x5830,0x5F66,0x7109,0x8A00,0x8AFA,0x5B7C,0x8616,
  0x4FFA,0x513C,0x56B4,0x5944,0x63A9,0x6DF9,0x5DAA,0x696D,
  0x5186,0x4E88,0x4F59,0xF97F,0xF980,0xF981,0x5982,0xF982,
  0xF983,0x6B5F,0x6C5D,0xF984,0x74B5,0x7916,0xF985,0x8207,
  0x8245,0x8339,0x8F3F,0x8F5D,0xF986,0x9918,0xF987,0xF988,
  0xF989,0x4EA6,0xF98A,0x57DF,0x5F79,0x6613,0xF98B,0xF98C,
  0x75AB,0x7E79,0x8B6F,0xF98D,0x9006,0x9A5B,0x56A5,0x5827,
  0x59F8,0x5A1F,0x5BB4,0xF98E,0x5EF6,0xF98F,0xF990,0x6350,
  0x633B,0xF991,0x693D,0x6C87,0x6CBF,0x6D8E,0x6D93,0x6DF5,
  0x6F14,0xF992,0x70DF,0x7136,0x7159,0xF993,0x71C3,0x71D5,
  0xF994,0x784F,0x786F,0xF995,0x7B75,0x7DE3,0xF996,0x7E2F,
  0xF997,0x884D,0x8EDF,0xF998,0xF999,0xF99A,0x925B,0xF99B,
  0x9CF6,0xF99C,0xF99D,0xF99E,0x6085,0x6D85,0xF99F,0x71B1,
  0xF9A0,0xF9A1,0x95B1,0x53AD,0xF9A2,0xF9A3,0xF9A4,0x67D3,
  0xF9A5,0x708E,0x7130,0x7430,0x8276,0x82D2,0xF9A6,0x95BB,
  0x9AE5,0x9E7D,0x66C4,0xF9A7,0x71C1,0x8449,0xF9A8,0xF9A9,
  0x584B,0xF9AA,0xF9AB,0x5DB8,0x5F71,0xF9AC,0x6620,0x668E,
  0x6979,0x69AE,0x6C38,0x6CF3,0x6E36,0x6F41,0x6FDA,0x701B,
  0x702F,0x7150,0x71DF,0x7370,0xF9AD,0x745B,0xF9AE,0x74D4,
  0x76C8,0x7A4E,0x7E93,0xF9AF,0xF9B0,0x82F1,0x8A60,0x8FCE,
  0xF9B1,0x9348,0xF9B2,0x9719,0xF9B3,0xF9B4,0x4E42,0x502A,
  0xF9B5,0x5208,0x53E1,0x66F3,0x6C6D,0x6FCA,0x730A,0x777F,
  0x7A62,0x82AE,0x85DD,0x8602,0xF9B6,0x88D4,0x8A63,0x8B7D,
  0x8C6B,0xF9B7,0x92B3,0xF9B8,0x9713,0x9810,0x4E94,0x4F0D,
  0x4FC9,0x50B2,0x5348,0x543E,0x5433,0x55DA,0x5862,0x58BA,
  0x5967,0x5A1B,0x5BE4,0x609F,0xF9B9,0x61CA,0x6556,0x65FF,
  0x6664,0x68A7,0x6C5A,0x6FB3,0x70CF,0x71AC,0x7352,0x7B7D,
  0x8708,0x8AA4,0x9C32,0x9F07,0x5C4B,0x6C83,0x7344,0x7389,
  0x923A,0x6EAB,0x7465,0x761F,0x7A69,0x7E15,0x860A,0x5140,
  0x58C5,0x64C1,0x74EE,0x7515,0x7670,0x7FC1,0x9095,0x96CD,
  0x9954,0x6E26,0x74E6,0x7AA9,0x7AAA,0x81E5,0x86D9,0x8778,
  0x8A1B,0x5A49,0x5B8C,0x5B9B,0x68A1,0x6900,0x6D63,0x73A9,
  0x7413,0x742C,0x7897,0x7DE9,0x7FEB,0x8118,0x8155,0x839E,
  0x8C4C,0x962E,0x9811,0x66F0,0x5F80,0x65FA,0x6789,0x6C6A,
  0x738B,0x502D,0x5A03,0x6B6A,0x77EE,0x5916,0x5D6C,0x5DCD,
  0x7325,0x754F,0xF9BA,0xF9BB,0x50E5,0x51F9,0x582F,0x592D,
  0x5996,0x59DA,0x5BE5,0xF9BC,0xF9BD,0x5DA2,0x62D7,0x6416,
  0x6493,0x64FE,0xF9BE,0x66DC,0xF9BF,0x6A48,0xF9C0,0x71FF,
  0x7464,0xF9C1,0x7A88,0x7AAF,0x7E47,0x7E5E,0x8000,0x8170,
  0xF9C2,0x87EF,0x8981,0x8B20,0x9059,0xF9C3,0x9080,0x9952,
  0x617E,0x6B32,0x6D74,0x7E1F,0x8925,0x8FB1,0x4FD1,0x50AD,
  0x5197,0x52C7,0x57C7,0x5889,0x5BB9,0x5EB8,0x6142,0x6995,
  0x6D8C,0x6E67,0x6EB6,0x7194,0x7462,0x7528,0x752C,0x8073,
  0x8338,0x84C9,0x8E0A,0x9394,0x93DE,0xF9C4,0x4E8E,0x4F51,
  0x5076,0x512A,0x53C8,0x53CB,0x53F3,0x5B87,0x5BD3,0x5C24,
  0x611A,0x6182,0x65F4,0x725B,0x7397,0x7440,0x76C2,0x7950,
  0x7991,0x79B9,0x7D06,0x7FBD,0x828B,0x85D5,0x865E,0x8FC2,
  0x9047,0x90F5,0x91EA,0x9685,0x96E8,0x96E9,0x52D6,0x5F67,
  0x65ED,0x6631,0x682F,0x715C,0x7A36,0x90C1,0x980A,0x4E91,
  0xF9C5,0x6A52,0x6B9E,0x6F90,0x7189,0x8018,0x82B8,0x8553,
  0x904B,0x9695,0x96F2,0x97FB,0x851A,0x9B31,0x4E90,0x718A,
  0x96C4,0x5143,0x539F,0x54E1,0x5713,0x5712,0x57A3,0x5A9B,
  0x5AC4,0x5BC3,0x6028,0x613F,0x63F4,0x6C85,0x6D39,0x6E72,
  0x6E90,0x7230,0x733F,0x7457,0x82D1,0x8881,0x8F45,0x9060,
  0xF9C6,0x9662,0x9858,0x9D1B,0x6708,0x8D8A,0x925E,0x4F4D,
  0x5049,0x50DE,0x5371,0x570D,0x59D4,0x5A01,0x5C09,0x6170,
  0x6690,0x6E2D,0x7232,0x744B,0x7DEF,0x80C3,0x840E,0x8466,
  0x853F,0x875F,0x885B,0x8918,0x8B02,0x9055,0x97CB,0x9B4F,
  0x4E73,0x4F91,0x5112,0x516A,0xF9C7,0x552F,0x55A9,0x5B7A,
  0x5BA5,0x5E7C,0x5E7D,0x5EBE,0x60A0,0x60DF,0x6108,0x6109,
  0x63C4,0x6538,0x6709,0xF9C8,0x67D4,0x67DA,0xF9C9,0x6961,
  0x6962,0x6CB9,0x6D27,0xF9CA,0x6E38,0xF9CB,0x6FE1,0x7336,
  0x7337,0xF9CC,0x745C,0x7531,0xF9CD,0x7652,0xF9CE,0xF9CF,
  0x7DAD,0x81FE,0x8438,0x88D5,0x8A98,0x8ADB,0x8AED,0x8E30,
  0x8E42,0x904A,0x903E,0x907A,0x9149,0x91C9,0x936E,0xF9D0,
  0xF9D1,0x5809,0xF9D2,0x6BD3,0x8089,0x80B2,0xF9D3,0xF9D4,
  0x5141,0x596B,0x5C39,0xF9D5,0xF9D6,0x6F64,0x73A7,0x80E4,
  0x8D07,0xF9D7,0x9217,0x958F,0xF9D8,0xF9D9,0xF9DA,0xF9DB,
  0x807F,0x620E,0x701C,0x7D68,0x878D,0xF9DC,0x57A0,0x6069,
  0x6147,0x6BB7,0x8ABE,0x9280,0x96B1,0x4E59,0x541F,0x6DEB,
  0x852D,0x9670,0x97F3,0x98EE,0x63D6,0x6CE3,0x9091,0x51DD,
  0x61C9,0x81BA,0x9DF9,0x4F9D,0x501A,0x5100,0x5B9C,0x610F,
  0x61FF,0x64EC,0x6905,0x6BC5,0x7591,0x77E3,0x7FA9,0x8264,
  0x858F,0x87FB,0x8863,0x8ABC,0x8B70,0x91AB,0x4E8C,0x4EE5,
  0x4F0A,0xF9DD,0xF9DE,0x5937,0x59E8,0xF9DF,0x5DF2,0x5F1B,
  0x5F5B,0x6021,0xF9E0,0xF9E1,0xF9E2,0xF9E3,0x723E,0x73E5,
  0xF9E4,0x7570,0x75CD,0xF9E5,0x79FB,0xF9E6,0x800C,0x8033,
  0x8084,0x82E1,0x8351,0xF9E7,0xF9E8,0x8CBD,0x8CB3,0x9087,
  0xF9E9,0xF9EA,0x98F4,0x990C,0xF9EB,0xF9EC,0x7037,0x76CA,
  0x7FCA,0x7FCC,0x7FFC,0x8B1A,0x4EBA,0x4EC1,0x5203,0x5370,
  0xF9ED,0x54BD,0x56E0,0x59FB,0x5BC5,0x5F15,0x5FCD,0x6E6E,
  0xF9EE,0xF9EF,0x7D6A,0x8335,0xF9F0,0x8693,0x8A8D,0xF9F1,
  0x976D,0x9777,0xF9F2,0xF9F3,0x4E00,0x4F5A,0x4F7E,0x58F9,
  0x65E5,0x6EA2,0x9038,0x93B0,0x99B9,0x4EFB,0x58EC,0x598A,
  0x59D9,0x6041,0xF9F4,0xF9F5,0x7A14,0xF9F6,0x834F,0x8CC3,
  0x5165,0x5344,0xF9F7,0xF9F8,0xF9F9,0x4ECD,0x5269,0x5B55,
  0x82BF,0x4ED4,0x523A,0x54A8,0x59C9,0x59FF,0x5B50,0x5B57,
  0x5B5C,0x6063,0x6148,0x6ECB,0x7099,0x716E,0x7386,0x74F7,
  0x75B5,0x78C1,0x7D2B,0x8005,0x81EA,0x8328,0x8517,0x85C9,
  0x8AEE,0x8CC7,0x96CC,0x4F5C,0x52FA,0x56BC,0x65AB,0x6628,
  0x707C,0x70B8,0x7235,0x7DBD,0x828D,0x914C,0x96C0,0x9D72,
  0x5B71,0x68E7,0x6B98,0x6F7A,0x76DE,0x5C91,0x66AB,0x6F5B,
  0x7BB4,0x7C2A,0x8836,0x96DC,0x4E08,0x4ED7,0x5320,0x5834,
  0x58BB,0x58EF,0x596C,0x5C07,0x5E33,0x5E84,0x5F35,0x638C,
  0x66B2,0x6756,0x6A1F,0x6AA3,0x6B0C,0x6F3F,0x7246,0xF9FA,
  0x7350,0x748B,0x7AE0,0x7CA7,0x8178,0x81DF,0x81E7,0x838A,
  0x846C,0x8523,0x8594,0x85CF,0x88DD,0x8D13,0x91AC,0x9577,
  0x969C,0x518D,0x54C9,0x5728,0x5BB0,0x624D,0x6750,0x683D,
  0x6893,0x6E3D,0x6ED3,0x707D,0x7E21,0x88C1,0x8CA1,0x8F09,
  0x9F4B,0x9F4E,0x722D,0x7B8F,0x8ACD,0x931A,0x4F47,0x4F4E,
  0x5132,0x5480,0x59D0,0x5E95,0x62B5,0x6775,0x696E,0x6A17,
  0x6CAE,0x6E1A,0x72D9,0x732A,0x75BD,0x7BB8,0x7D35,0x82E7,
  0x83F9,0x8457,0x85F7,0x8A5B,0x8CAF,0x8E87,0x9019,0x90B8,
  0x96CE,0x9F5F,0x52E3,0x540A,0x5AE1,0x5BC2,0x6458,0x6575,
  0x6EF4,0x72C4,0xF9FB,0x7684,0x7A4D,0x7B1B,0x7C4D,0x7E3E,
  0x7FDF,0x837B,0x8B2B,0x8CCA,0x8D64,0x8DE1,0x8E5F,0x8FEA,
  0x8FF9,0x9069,0x93D1,0x4F43,0x4F7A,0x50B3,0x5168,0x5178,
  0x524D,0x526A,0x5861,0x587C,0x5960,0x5C08,0x5C55,0x5EDB,
  0x609B,0x6230,0x6813,0x6BBF,0x6C08,0x6FB1,0x714E,0x7420,
  0x7530,0x7538,0x7551,0x7672,0x7B4C,0x7B8B,0x7BAD,0x7BC6,
  0x7E8F,0x8A6E,0x8F3E,0x8F49,0x923F,0x9293,0x9322,0x942B,
  0x96FB,0x985A,0x986B,0x991E,0x5207,0x622A,0x6298,0x6D59,
  0x7664,0x7ACA,0x7BC0,0x7D76,0x5360,0x5CBE,0x5E97,0x6F38,
  0x70B9,0x7C98,0x9711,0x9B8E,0x9EDE,0x63A5,0x647A,0x8776,
  0x4E01,0x4E95,0x4EAD,0x505C,0x5075,0x5448,0x59C3,0x5B9A,
  0x5E40,0x5EAD,0x5EF7,0x5F81,0x60C5,0x633A,0x653F,0x6574,
  0x65CC,0x6676,0x6678,0x67FE,0x6968,0x6A89,0x6B63,0x6C40,
  0x6DC0,0x6DE8,0x6E1F,0x6E5E,0x701E,0x70A1,0x738E,0x73FD,
  0x753A,0x775B,0x7887,0x798E,0x7A0B,0x7A7D,0x7CBE,0x7D8E,
  0x8247,0x8A02,0x8AEA,0x8C9E,0x912D,0x914A,0x91D8,0x9266,
  0x92CC,0x9320,0x9706,0x9756,0x975C,0x9802,0x9F0E,0x5236,
  0x5291,0x557C,0x5824,0x5E1D,0x5F1F,0x608C,0x63D0,0x68AF,
  0x6FDF,0x796D,0x7B2C,0x81CD,0x85BA,0x88FD,0x8AF8,0x8E44,
  0x918D,0x9664,0x969B,0x973D,0x984C,0x9F4A,0x4FCE,0x5146,
  0x51CB,0x52A9,0x5632,0x5F14,0x5F6B,0x63AA,0x64CD,0x65E9,
  0x6641,0x66FA,0x66F9,0x671D,0x689D,0x68D7,0x69FD,0x6F15,
  0x6F6E,0x7167,0x71E5,0x722A,0x74AA,0x773A,0x7956,0x795A,
  0x79DF,0x7A20,0x7A95,0x7C97,0x7CDF,0x7D44,0x7E70,0x8087,
  0x85FB,0x86A4,0x8A54,0x8ABF,0x8D99,0x8E81,0x9020,0x906D,
  0x91E3,0x963B,0x96D5,0x9CE5,0x65CF,0x7C07,0x8DB3,0x93C3,
  0x5B58,0x5C0A,0x5352,0x62D9,0x731D,0x5027,0x5B97,0x5F9E,
  0x60B0,0x616B,0x68D5,0x6DD9,0x742E,0x7A2E,0x7D42,0x7D9C,
  0x7E31,0x816B,0x8E2A,0x8E35,0x937E,0x9418,0x4F50,0x5750,
  0x5DE6,0x5EA7,0x632B,0x7F6A,0x4E3B,0x4F4F,0x4F8F,0x505A,
  0x59DD,0x80C4,0x546A,0x5468,0x55FE,0x594F,0x5B99,0x5DDE,
  0x5EDA,0x665D,0x6731,0x67F1,0x682A,0x6CE8,0x6D32,0x6E4A,
  0x6F8D,0x70B7,0x73E0,0x7587,0x7C4C,0x7D02,0x7D2C,0x7DA2,
  0x821F,0x86DB,0x8A3B,0x8A85,0x8D70,0x8E8A,0x8F33,0x9031,
  0x914E,0x9152,0x9444,0x99D0,0x7AF9,0x7CA5,0x4FCA,0x5101,
  0x51C6,0x57C8,0x5BEF,0x5CFB,0x6659,0x6A3D,0x6D5A,0x6E96,
  0x6FEC,0x710C,0x756F,0x7AE3,0x8822,0x9021,0x9075,0x96CB,
  0x99FF,0x8301,0x4E2D,0x4EF2,0x8846,0x91CD,0x537D,0x6ADB,
  0x696B,0x6C41,0x847A,0x589E,0x618E,0x66FE,0x62EF,0x70DD,
  0x7511,0x75C7,0x7E52,0x84B8,0x8B49,0x8D08,0x4E4B,0x53EA,
  0x54AB,0x5730,0x5740,0x5FD7,0x6301,0x6307,0x646F,0x652F,
  0x65E8,0x667A,0x679D,0x67B3,0x6B62,0x6C60,0x6C9A,0x6F2C,
  0x77E5,0x7825,0x7949,0x7957,0x7D19,0x80A2,0x8102,0x81F3,
  0x829D,0x82B7,0x8718,0x8A8C,0xF9FC,0x8D04,0x8DBE,0x9072,
  0x76F4,0x7A19,0x7A37,0x7E54,0x8077,0x5507,0x55D4,0x5875,
  0x632F,0x6422,0x6649,0x664B,0x686D,0x699B,0x6B84,0x6D25,
  0x6EB1,0x73CD,0x7468,0x74A1,0x755B,0x75B9,0x76E1,0x771E,
  0x778B,0x79E6,0x7E09,0x7E1D,0x81FB,0x852F,0x8897,0x8A3A,
  0x8CD1,0x8EEB,0x8FB0,0x9032,0x93AD,0x9663,0x9673,0x9707,
  0x4F84,0x53F1,0x59EA,0x5AC9,0x5E19,0x684E,0x74C6,0x75BE,
  0x79E9,0x7A92,0x81A3,0x86ED,0x8CEA,0x8DCC,0x8FED,0x659F,
  0x6715,0xF9FD,0x57F7,0x6F57,0x7DDD,0x8F2F,0x93F6,0x96C6,
  0x5FB5,0x61F2,0x6F84,0x4E14,0x4F98,0x501F,0x53C9,0x55DF,
  0x5D6F,0x5DEE,0x6B21,0x6B64,0x78CB,0x7B9A,0xF9FE,0x8E49,
  0x8ECA,0x906E,0x6349,0x643E,0x7740,0x7A84,0x932F,0x947F,
  0x9F6A,0x64B0,0x6FAF,0x71E6,0x74A8,0x74DA,0x7AC4,0x7C12,
  0x7E82,0x7CB2,0x7E98,0x8B9A,0x8D0A,0x947D,0x9910,0x994C,
  0x5239,0x5BDF,0x64E6,0x672D,0x7D2E,0x50ED,0x53C3,0x5879,
  0x6158,0x6159,0x61FA,0x65AC,0x7AD9,0x8B92,0x8B96,0x5009,
  0x5021,0x5275,0x5531,0x5A3C,0x5EE0,0x5F70,0x6134,0x655E,
  0x660C,0x6636,0x66A2,0x69CD,0x6EC4,0x6F32,0x7316,0x7621,
  0x7A93,0x8139,0x8259,0x83D6,0x84BC,0x50B5,0x57F0,0x5BC0,
  0x5BE8,0x5F69,0x63A1,0x7826,0x7DB5,0x83DC,0x8521,0x91C7,
  0x91F5,0x518A,0x67F5,0x7B56,0x8CAC,0x51C4,0x59BB,0x60BD,
  0x8655,0x501C,0xF9FF,0x5254,0x5C3A,0x617D,0x621A,0x62D3,
  0x64F2,0x65A5,0x6ECC,0x7620,0x810A,0x8E60,0x965F,0x96BB,
  0x4EDF,0x5343,0x5598,0x5929,0x5DDD,0x64C5,0x6CC9,0x6DFA,
  0x7394,0x7A7F,0x821B,0x85A6,0x8CE4,0x8E10,0x9077,0x91E7,
  0x95E1,0x9621,0x97C6,0x51F8,0x54F2,0x5586,0x5FB9,0x64A4,
  0x6F88,0x7DB4,0x8F1F,0x8F4D,0x9435,0x50C9,0x5C16,0x6CBE,
  0x6DFB,0x751B,0x77BB,0x7C3D,0x7C64,0x8A79,0x8AC2,0x581E,
  0x59BE,0x5E16,0x6377,0x7252,0x758A,0x776B,0x8ADC,0x8CBC,
  0x8F12,0x5EF3,0x6674,0x6DF8,0x807D,0x83C1,0x8ACB,0x9751,
  0x9BD6,0xFA00,0x5243,0x66FF,0x6D95,0x6EEF,0x7DE0,0x8AE6,
  0x902E,0x905E,0x9AD4,0x521D,0x527F,0x54E8,0x6194,0x6284,
  0x62DB,0x68A2,0x6912,0x695A,0x6A35,0x7092,0x7126,0x785D,
  0x7901,0x790E,0x79D2,0x7A0D,0x8096,0x8278,0x82D5,0x8349,
  0x8549,0x8C82,0x8D85,0x9162,0x918B,0x91AE,0x4FC3,0x56D1,
  0x71ED,0x77D7,0x8700,0x89F8,0x5BF8,0x5FD6,0x6751,0x90A8,
  0x53E2,0x585A,0x5BF5,0x60A4,0x6181,0x6460,0x7E3D,0x8070,
  0x8525,0x9283,0x64AE,0x50AC,0x5D14,0x6700,0x589C,0x62BD,
  0x63A8,0x690E,0x6978,0x6A1E,0x6E6B,0x76BA,0x79CB,0x82BB,
  0x8429,0x8ACF,0x8DA8,0x8FFD,0x9112,0x914B,0x919C,0x9310,
  0x9318,0x939A,0x96DB,0x9A36,0x9C0D,0x4E11,0x755C,0x795D,
  0x7AFA,0x7B51,0x7BC9,0x7E2E,0x84C4,0x8E59,0x8E74,0x8EF8,
  0x9010,0x6625,0x693F,0x7443,0x51FA,0x672E,0x9EDC,0x5145,
  0x5FE0,0x6C96,0x87F2,0x885D,0x8877,0x60B4,0x81B5,0x8403,
  0x8D05,0x53D6,0x5439,0x5634,0x5A36,0x5C31,0x708A,0x7FE0,
  0x805A,0x8106,0x81ED,0x8DA3,0x9189,0x9A5F,0x9DF2,0x5074,
  0x4EC4,0x53A0,0x60FB,0x6E2C,0x5C64,0x4F88,0x5024,0x55E4,
  0x5CD9,0x5E5F,0x6065,0x6894,0x6CBB,0x6DC4,0x71BE,0x75D4,
  0x75F4,0x7661,0x7A1A,0x7A49,0x7DC7,0x7DFB,0x7F6E,0x81F4,
  0x86A9,0x8F1C,0x96C9,0x99B3,0x9F52,0x5247,0x52C5,0x98ED,
  0x89AA,0x4E03,0x67D2,0x6F06,0x4FB5,0x5BE2,0x6795,0x6C88,
  0x6D78,0x741B,0x7827,0x91DD,0x937C,0x87C4,0x79E4,0x7A31,
  0x5FEB,0x4ED6,0x54A4,0x553E,0x58AE,0x59A5,0x60F0,0x6253,
  0x62D6,0x6736,0x6955,0x8235,0x9640,0x99B1,0x99DD,0x502C,
  0x5353,0x5544,0x577C,0xFA01,0x6258,0xFA02,0x64E2,0x666B,
  0x67DD,0x6FC1,0x6FEF,0x7422,0x7438,0x8A17,0x9438,0x5451,
  0x5606,0x5766,0x5F48,0x619A,0x6B4E,0x7058,0x70AD,0x7DBB,
  0x8A95,0x596A,0x812B,0x63A2,0x7708,0x803D,0x8CAA,0x5854,
  0x642D,0x69BB,0x5B95,0x5E11,0x6E6F,0xFA03,0x8569,0x514C,
  0x53F0,0x592A,0x6020,0x614B,0x6B86,0x6C70,0x6CF0,0x7B1E,
  0x80CE,0x82D4,0x8DC6,0x90B0,0x98B1,0xFA04,0x64C7,0x6FA4,
  0x6491,0x6504,0x514E,0x5410,0x571F,0x8A0E,0x615F,0x6876,
  0xFA05,0x75DB,0x7B52,0x7D71,0x901A,0x5806,0x69CC,0x817F,
  0x892A,0x9000,0x9839,0x5078,0x5957,0x59AC,0x6295,0x900F,
  0x9B2A,0x615D,0x7279,0x95D6,0x5761,0x5A46,0x5DF4,0x628A,
  0x64AD,0x64FA,0x6777,0x6CE2,0x6D3E,0x722C,0x7436,0x7834,
  0x7F77,0x82AD,0x8DDB,0x9817,0x5224,0x5742,0x677F,0x7248,
  0x74E3,0x8CA9,0x8FA6,0x9211,0x962A,0x516B,0x53ED,0x634C,
  0x4F69,0x5504,0x6096,0x6557,0x6C9B,0x6D7F,0x724C,0x72FD,
  0x7A17,0x8987,0x8C9D,0x5F6D,0x6F8E,0x70F9,0x81A8,0x610E,
  0x4FBF,0x504F,0x6241,0x7247,0x7BC7,0x7DE8,0x7FE9,0x904D,
  0x97AD,0x9A19,0x8CB6,0x576A,0x5E73,0x67B0,0x840D,0x8A55,
  0x5420,0x5B16,0x5E63,0x5EE2,0x5F0A,0x6583,0x80BA,0x853D,
  0x9589,0x965B,0x4F48,0x5305,0x530D,0x530F,0x5486,0x54FA,
  0x5703,0x5E03,0x6016,0x629B,0x62B1,0x6355,0xFA06,0x6CE1,
  0x6D66,0x75B1,0x7832,0x80DE,0x812F,0x82DE,0x8461,0x84B2,
  0x888D,0x8912,0x900B,0x92EA,0x98FD,0x9B91,0x5E45,0x66B4,
  0x66DD,0x7011,0x7206,0xFA07,0x4FF5,0x527D,0x5F6A,0x6153,
  0x6753,0x6A19,0x6F02,0x74E2,0x7968,0x8868,0x8C79,0x98C7,
  0x98C4,0x9A43,0x54C1,0x7A1F,0x6953,0x8AF7,0x8C4A,0x98A8,
  0x99AE,0x5F7C,0x62AB,0x75B2,0x76AE,0x88AB,0x907F,0x9642,
  0x5339,0x5F3C,0x5FC5,0x6CCC,0x73CC,0x7562,0x758B,0x7B46,
  0x82FE,0x999D,0x4E4F,0x903C,0x4E0B,0x4F55,0x53A6,0x590F,
  0x5EC8,0x6630,0x6CB3,0x7455,0x8377,0x8766,0x8CC0,0x9050,
  0x971E,0x9C15,0x58D1,0x5B78,0x8650,0x8B14,0x9DB4,0x5BD2,
  0x6068,0x608D,0x65F1,0x6C57,0x6F22,0x6FA3,0x701A,0x7F55,
  0x7FF0,0x9591,0x9592,0x9650,0x97D3,0x5272,0x8F44,0x51FD,
  0x542B,0x54B8,0x5563,0x558A,0x6ABB,0x6DB5,0x7DD8,0x8266,
  0x929C,0x9677,0x9E79,0x5408,0x54C8,0x76D2,0x86E4,0x95A4,
  0x95D4,0x965C,0x4EA2,0x4F09,0x59EE,0x5AE6,0x5DF7,0x6052,
  0x6297,0x676D,0x6841,0x6C86,0x6E2F,0x7F38,0x809B,0x822A,
  0xFA08,0xFA09,0x9805,0x4EA5,0x5055,0x54B3,0x5793,0x595A,
  0x5B69,0x5BB3,0x61C8,0x6977,0x6D77,0x7023,0x87F9,0x89E3,
  0x8A72,0x8AE7,0x9082,0x99ED,0x9AB8,0x52BE,0x6838,0x5016,
  0x5E78,0x674F,0x8347,0x884C,0x4EAB,0x5411,0x56AE,0x73E6,
  0x9115,0x97FF,0x9909,0x9957,0x9999,0x5653,0x589F,0x865B,
  0x8A31,0x61B2,0x6AF6,0x737B,0x8ED2,0x6B47,0x96AA,0x9A57,
  0x5955,0x7200,0x8D6B,0x9769,0x4FD4,0x5CF4,0x5F26,0x61F8,
  0x665B,0x6CEB,0x70AB,0x7384,0x73B9,0x73FE,0x7729,0x774D,
  0x7D43,0x7D62,0x7E23,0x8237,0x8852,0xFA0A,0x8CE2,0x9249,
  0x986F,0x5B51,0x7A74,0x8840,0x9801,0x5ACC,0x4FE0,0x5354,
  0x593E,0x5CFD,0x633E,0x6D79,0x72F9,0x8105,0x8107,0x83A2,
  0x92CF,0x9830,0x4EA8,0x5144,0x5211,0x578B,0x5F62,0x6CC2,
  0x6ECE,0x7005,0x7050,0x70AF,0x7192,0x73E9,0x7469,0x834A,
  0x87A2,0x8861,0x9008,0x90A2,0x93A3,0x99A8,0x516E,0x5F57,
  0x60E0,0x6167,0x66B3,0x8559,0x8E4A,0x91AF,0x978B,0x4E4E,
  0x4E92,0x547C,0x58D5,0x58FA,0x597D,0x5CB5,0x5F27,0x6236,
  0x6248,0x660A,0x6667,0x6BEB,0x6D69,0x6DCF,0x6E56,0x6EF8,
  0x6F94,0x6FE0,0x6FE9,0x705D,0x72D0,0x7425,0x745A,0x74E0,
  0x7693,0x795C,0x7CCA,0x7E1E,0x80E1,0x82A6,0x846B,0x84BF,
  0x864E,0x865F,0x8774,0x8B77,0x8C6A,0x93AC,0x9800,0x9865,
  0x60D1,0x6216,0x9177,0x5A5A,0x660F,0x6DF7,0x6E3E,0x743F,
  0x9B42,0x5FFD,0x60DA,0x7B0F,0x54C4,0x5F18,0x6C5E,0x6CD3,
  0x6D2A,0x70D8,0x7D05,0x8679,0x8A0C,0x9D3B,0x5316,0x548C,
  0x5B05,0x6A3A,0x706B,0x7575,0x798D,0x79BE,0x82B1,0x83EF,
  0x8A71,0x8B41,0x8CA8,0x9774,0xFA0B,0x64F4,0x652B,0x78BA,
  0x78BB,0x7A6B,0x4E38,0x559A,0x5950,0x5BA6,0x5E7B,0x60A3,
  0x63DB,0x6B61,0x6665,0x6853,0x6E19,0x7165,0x74B0,0x7D08,
  0x9084,0x9A69,0x9C25,0x6D3B,0x6ED1,0x733E,0x8C41,0x95CA,
  0x51F0,0x5E4C,0x5FA8,0x604D,0x60F6,0x6130,0x614C,0x6643,
  0x6644,0x69A5,0x6CC1,0x6E5F,0x6EC9,0x6F62,0x714C,0x749C,
  0x7687,0x7BC1,0x7C27,0x8352,0x8757,0x9051,0x968D,0x9EC3,
  0x532F,0x56DE,0x5EFB,0x5F8A,0x6062,0x6094,0x61F7,0x6666,
  0x6703,0x6A9C,0x6DEE,0x6FAE,0x7070,0x736A,0x7E6A,0x81BE,
  0x8334,0x86D4,0x8AA8,0x8CC4,0x5283,0x7372,0x5B96,0x6A6B,
  0x9404,0x54EE,0x5686,0x5B5D,0x6548,0x6585,0x66C9,0x689F,
  0x6D8D,0x6DC6,0x723B,0x80B4,0x9175,0x9A4D,0x4FAF,0x5019,
  0x539A,0x540E,0x543C,0x5589,0x55C5,0x5E3F,0x5F8C,0x673D,
  0x7166,0x73DD,0x9005,0x52DB,0x52F3,0x5864,0x58CE,0x7104,
  0x718F,0x71FB,0x85B0,0x8A13,0x6688,0x85A8,0x55A7,0x6684,
  0x714A,0x8431,0x5349,0x5599,0x6BC1,0x5F59,0x5FBD,0x63EE,
  0x6689,0x7147,0x8AF1,0x8F1D,0x9EBE,0x4F11,0x643A,0x70CB,
  0x7566,0x8667,0x6064,0x8B4E,0x9DF8,0x5147,0x51F6,0x5308,
  0x6D36,0x80F8,0x9ED1,0x6615,0x6B23,0x7098,0x75D5,0x5403,
  0x5C79,0x7D07,0x8A16,0x6B20,0x6B3D,0x6B46,0x5438,0x6070,
  0x6D3D,0x7FD5,0x8208,0x50D6,0x51DE,0x559C,0x566B,0x56CD,
  0x59EC,0x5B09,0x5E0C,0x6199,0x6198,0x6231,0x665E,0x66E6,
  0x7199,0x71B9,0x71BA,0x72A7,0x79A7,0x7A00,0x7FB2,0x8A70,
};

static unsigned short *linear_search( unsigned short code )
{
  int i, n = sizeof(unicode_hanja) / sizeof(unicode_hanja[1]);
  for( i = 0; i < n; i++)
    if( unicode_hanja[i] == code )
      return unicode_hanja +i;
  return 0;
}

static unsigned short uc_hanja2hwpc(unsigned short code)
{
  unsigned short *p, hanja;
  p = linear_search( code );
  if( ! p )
    return 0;
  hanja =  p - unicode_hanja;
  return (hanja + 0x4000);
}

static unsigned short uc_hangul2hwpc(unsigned short code)
{
  unsigned short	f, m, l;
  code -= 0xac00;
  f = code / (21 * 28);	code %= (21 * 28);
  m = code / 28;	l = code % 28;

  f += 2;
  if( 16 < m)		m += 9;
  else if( 10 < m)	m += 7;
  else if( 4 < m)	m += 5;
  else			m += 3;
  if( 16 < l )		l += 2;
  else			l += 1;

  return 0x8000 | (f << 10) | (m << 5) | l;
}

#endif
