/*  This file is part of the KDE project
    Copyright (C) 1999 Simon Hausmann <hausmann@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "kview_view.h"
#include "canvas.h"
#include "numdlg.h"
#include "kview_asyncio.h"

#include <qiconset.h>
#include <qimage.h>
#include <qdragobject.h>
#include <qevent.h>
#include <qprinter.h>

#include <kaction.h>
#include <klocale.h>
#include <kio/job.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kimageio.h>
#include <kparts/partmanager.h>

extern "C"
{
  void *init_libkview()
  {
    KImageIO::registerFormats();
    return new KViewFactory;
  }
};

KInstance *KViewFactory::s_instance = 0L;

KViewFactory::KViewFactory()
{
}

KViewFactory::~KViewFactory()
{
  if ( s_instance )
    delete s_instance;

  s_instance = 0;
}

QObject* KViewFactory::create( QObject *parent , const char *name , const char*, const QStringList & )
{
  QObject *obj = new KViewPart( (QWidget *)parent, name );
  emit objectCreated( obj );
  return obj;
}

KInstance *KViewFactory::instance()
{
  if ( !s_instance )
    s_instance = new KInstance( "kview" );
  return s_instance;
}

void KViewKonqExtension::print()
{
  QPrinter printer;

  if ( !printer.setup( ((KViewPart *)parent())->widget() ) )
    return;

  QPainter painter;
  painter.begin( &printer );

  for ( int i = 0; i < printer.numCopies(); i++ )
  {
    painter.drawPixmap( 0, 0, *(canvas()->client()->imagePix()) );
    if ( i < printer.numCopies() - 1 )
      printer.newPage();
  }

  painter.end();
}

KViewPart::KViewPart( QWidget *parent, const char *name )
 : KParts::ReadOnlyPart( parent, name )
{
  m_job = 0L;

  setInstance( KViewFactory::instance() );

  m_extension = new KViewKonqExtension( this );

  zoomFactor = 0;
  m_pCanvas = new KImageCanvas( parent );
  m_pCanvas->setFocusPolicy( QWidget::StrongFocus );
  setWidget( m_pCanvas );

  KVImageHolder *img = m_pCanvas->client();
  img->move( 0, 0 );
  img->show();
  m_pCanvas->show();

  m_paZoomIn = new KAction( i18n( "Zoom in 10%" ), QIconSet( BarIcon( "viewmag+", KViewFactory::instance() ) ) , 0, this, SLOT( slotZoomIn() ), actionCollection(), "zoomin" );
  m_paZoomOut = new KAction( i18n( "Zoom out 10%" ), QIconSet( BarIcon( "viewmag-", KViewFactory::instance() ) ), 0, this, SLOT( slotZoomOut() ), actionCollection(), "zoomout" );
  m_paRotate = new KAction( i18n( "Rotate clockwise" ), QIconSet( UserIcon( "rotate", KViewFactory::instance() ) ), 0, this, SLOT( slotRotate() ), actionCollection(), "rotate" );
  m_paReset = new KAction( i18n( "Reset" ), QIconSet( BarIcon( "undo", KViewFactory::instance() ) ), 0, this, SLOT( slotReset() ), actionCollection(), "reset" );

  setXMLFile( "kview_part.rc" );
}

KViewPart::~KViewPart()
{
  closeURL();
}

bool KViewPart::openURL( const KURL &url )
{
  closeURL();

  m_job = KIO::get( url, m_extension->urlArgs().reload, false );

  //m_job->setGUImode( KIO::Job::NONE );

  QObject::connect( m_job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotResult( KIO::Job * ) ) );
  //QObject::connect( job, SIGNAL( sigRedirection( int, const char * ) ), this, SLOT( slotRedirection( int, const char * ) ) );

  (void)new KIOImageLoader( m_job, m_pCanvas->client() );

  m_url = url;

  emit started( m_job );

  return true;
}

bool KViewPart::closeURL()
{
  if ( m_job )
  {
    m_job->kill();
    m_job = 0;
  }
  return true;
}

void KViewPart::slotZoomIn()
{
  matrix.scale( 1.1, 1.1 );
  m_pCanvas->transformImage( matrix );
}

void KViewPart::slotZoomOut()
{
  matrix.scale( 0.9, 0.9 );
  m_pCanvas->transformImage( matrix );
}

void KViewPart::slotRotate()
{
  matrix.rotate( -90 );
  m_pCanvas->transformImage( matrix );
}

void KViewPart::slotReset()
{
  matrix.reset();
  m_pCanvas->transformImage( matrix );
}

void KViewPart::keyPressEvent( QKeyEvent *ev )
{
  switch ( ev->key() )
  {
    case Key_Down:
      ev->accept();
      m_pCanvas->lineDown();
      break;
    case Key_Up:
      ev->accept();
      m_pCanvas->lineUp();
      break;
    case Key_Left:
      ev->accept();
      m_pCanvas->lineLeft();
      break;
    case Key_Right:
      ev->accept();
      m_pCanvas->lineRight();
      break;
    case Key_PageUp:
      ev->accept();
      m_pCanvas->pageUp();
      break;
    case Key_PageDown:
      ev->accept();
      m_pCanvas->pageDown();
      break;
    default:
      ev->ignore();
      break;
  }
}

void KViewPart::slotResult( KIO::Job * job )
{
  if (job->error())
  {
    job->showErrorDialog();
    closeURL();
    emit canceled( QString( job->errorString() ) );
  } else
  {
    m_pCanvas->updateScrollBars();
    emit completed();
  }
  m_job = 0;
}

/*
void KViewPart::slotRedirection( int, const char *url )
{
  QString sUrl ( url );
  m_url = KURL( sUrl );
  emit m_extension->setLocationBarURL( sUrl );
  emit setWindowCaption( m_url.prettyURL() );
}
*/

KViewKonqExtension::KViewKonqExtension( KViewPart *parent,
                                        const char *name )
: KParts::BrowserExtension( parent, name )
{
}

void KViewKonqExtension::setXYOffset( int x, int y )
{
  canvas()->updateScrollBars();
  canvas()->setContentsPos( x, y );
}

int KViewKonqExtension::xOffset()
{
  return canvas()->contentsX();
}

int KViewKonqExtension::yOffset()
{
  return canvas()->contentsY();
}

/*
void KViewKonqView::resizeEvent( QResizeEvent * )
{
  m_pCanvas->setGeometry( 0, 0, width(), height() );
}
*/

#include "kview_view.moc"


