//////////////////////////////////////////////////////////////
//     $Id: rpmInstall.cpp,v 1.20 2000/05/29 16:17:21 toivo Exp $
#include "../config.h"

#ifdef HAVE_RPM

#include <fcntl.h>
#include <stdio.h>  
#include <unistd.h>

#include "rpmMessages.h"

#include <rpm/rpmlib.h>
#include "rpmInstall.h"

#include <klocale.h>

#include "kpackage.h"
#include "rpmutils.h"

extern "C"
{
#include <rpm/rpmmacro.h>
}

static void printHash(const unsigned long amount, const unsigned long total);
static void * showProgress(const Header h, const rpmCallbackType what, 
			   const unsigned long amount, 
			   const unsigned long total,
			   const void * pkgKey, void * data);

static int hashesPrinted = 0;

static void printHash(const unsigned long amount, const unsigned long total)
{
    int hashesNeeded;

    if (hashesPrinted != 50) {
	hashesNeeded = (int)( 50 * (total ? (((float) amount) / total) : 1));
    }
    kpackage->setPercent(hashesNeeded*2);   
}

static void * showProgress(const Header h, const rpmCallbackType what, 
			   const unsigned long amount, 
			   const unsigned long total,
			   const void * pkgKey, void * data) {
    char * s;
    int flags = (int) data;
    void * rc = NULL;
    const char * filename = (const char *)pkgKey;
    static FD_t fd;

    switch (what) {
      case RPMCALLBACK_INST_OPEN_FILE:
	fd = fdOpen(filename, O_RDONLY, 0);
	return fd;

      case RPMCALLBACK_INST_CLOSE_FILE:
	fdClose(fd);
	break;

      case RPMCALLBACK_INST_START:
	hashesPrinted = 0;
	if (flags & INSTALL_LABEL) {
	    if (flags & INSTALL_HASH) {
		s = headerSprintf(h, "%{NAME}",
				  rpmTagTable, rpmHeaderFormats, NULL);
		KpMsgE(QString("%1").arg(s,28), FALSE);
		fflush(stdout);
	    } else {
		s = headerSprintf(h, "%{NAME}-%{VERSION}-%{RELEASE}", 
				  rpmTagTable, rpmHeaderFormats, NULL);
		KpMsgE(QString("%1\n").arg(s), FALSE);
	    }
	    free(s);
	}
	break;

      case RPMCALLBACK_INST_PROGRESS:
	printHash(amount, total);
	break;

      case RPMCALLBACK_TRANS_PROGRESS:
      case RPMCALLBACK_TRANS_START:
      case RPMCALLBACK_TRANS_STOP:
      case RPMCALLBACK_UNINST_PROGRESS:
      case RPMCALLBACK_UNINST_START:
      case RPMCALLBACK_UNINST_STOP:
	/* ignore */
	break;
    }

    return rc;
}	

int doInstal(const char * rootdir, const QStringList &argv, int transFlags, 
	      int interfaceFlags, int probFilter, 
	      rpmRelocation * relocations) {
    rpmdb db = NULL;
    FD_t fd;
    int i = 0;
    int mode, rc, major;
    const char ** packages, ** tmpPackages;
    const char ** filename;
    int numPackages;
    int numTmpPackages = 0, numBinaryPackages = 0, numSourcePackages = 0;
    int numFailed = 0;
    Header h;
    int isSource;
    rpmTransactionSet rpmdep = NULL;
    struct rpmDependencyConflict * conflicts;
    int numConflicts;
    int stopInstall = 0;
    size_t nb;
    int notifyFlags = interfaceFlags | (rpmIsVerbose() ? INSTALL_LABEL : 0 );
    int dbIsOpen = 0;
    const char ** sourcePackages;
    rpmRelocation * defaultReloc;

    if (transFlags & RPMTRANS_FLAG_TEST) 
	mode = O_RDONLY;
    else
	mode = O_RDWR | O_CREAT;

    for (defaultReloc = relocations; defaultReloc && defaultReloc->oldPath;
	 defaultReloc++);
    if (defaultReloc && !defaultReloc->newPath) defaultReloc = NULL;

    rpmMessage(RPMMESS_DEBUG, (char *) "%s\n", i18n("counting packages to install").ascii());

    numPackages = argv.count();

    rpmMessage(RPMMESS_DEBUG, (char *) "%s\n", i18n("found %1 packages").arg(numPackages).ascii() );

    nb = (numPackages + 1) * sizeof(char *);
    packages = (const char **) alloca(nb);
    memset(packages, 0, nb);
    tmpPackages = (const char **)alloca(nb);
    memset(tmpPackages, 0, nb);
    nb = (numPackages + 1) * sizeof(Header);

    for (QStringList::ConstIterator it = argv.begin();
         (it != argv.end());
         it++) {
       packages[i++] = (*it).ascii();
    }

    sourcePackages = (const char **)alloca(sizeof(*sourcePackages) * i);

    rpmMessage(RPMMESS_DEBUG, (char *) "%s\n", i18n("retrieved %1 packages").arg(numTmpPackages).ascii());

    /* Build up the transaction set. As a special case, v1 source packages
       are installed right here, only because they don't have headers and
       would create all sorts of confusion later. */

    for (filename = packages; *filename; filename++) {
	fd = fdOpen(*filename, O_RDONLY, 0);
	if (fdFileno(fd) < 0) {
	    rpmMessage(RPMMESS_ERROR, (char *) "%s\n", i18n("cannot open file %1").arg(*filename).ascii());
	    numFailed++;
	    packages[i] = NULL;
	    continue;
	}

	rc = rpmReadPackageHeader(fd, &h, &isSource, &major, NULL);

	switch (rc) {
	case 1:
	    fdClose(fd);
	    rpmMessage(RPMMESS_ERROR, 
		        (char *) "%s\n", i18n("%1 does not appear to be a RPM package").arg(*filename).ascii());
	    break;
	default:
	    rpmMessage(RPMMESS_ERROR, (char *) "%s\n", i18n("%1 cannot be installed").arg(*filename).ascii());
	    numFailed++;
	    packages[i] = NULL;
	    break;
	case 0:
	    if (isSource) {
		sourcePackages[numSourcePackages++] = *filename;
		fdClose(fd);
	    } else {
		if (!dbIsOpen) {
		    if (rpmdbOpen(rootdir, &db, mode, 0644)) {
		      //	const char *dn;
		      //	dn = rpmGetPath( (rootdir ? rootdir : ""), 
		      //			"%{_dbpath}", NULL);
		      if (getuid() != 0)
			rpmMessage(RPMMESS_ERROR, 
				    (char *) "%s\n", i18n("KPACKAGE has to run as ROOT").ascii());
			//			xfree(dn);
			return(EXIT_FAILURE);
		    }
		    rpmdep = rpmtransCreateSet(db, rootdir);
		    dbIsOpen = 1;
		}

		if (defaultReloc) {
		    char ** paths;
		    char * name;
		    int c;

		    if (headerGetEntry(h, RPMTAG_PREFIXES, NULL,
				       (void **) &paths, &c) && (c == 1)) {
			defaultReloc->oldPath = paths[0];
			free(paths);
		    } else {
			headerGetEntry(h, RPMTAG_NAME, NULL, (void **) &name,
				       NULL);
			rpmMessage(RPMMESS_ERROR, 
			        (char *) "%s\n", i18n("package %1 is not relocateable").arg(name).ascii());

			return numPackages;
		    }
		}

		rc = rpmtransAddPackage(rpmdep, h, NULL, *filename,
			       (interfaceFlags & INSTALL_UPGRADE) != 0,
			       relocations);
		if (rc) {
		    if (rc == 1)
			rpmMessage(RPMMESS_ERROR, 
			     (char *) "%s\n", i18n("error reading from file %1").arg(*filename).ascii());
		    else if (rc == 2)
			rpmMessage(RPMMESS_ERROR, 
			     (char *) "%s\n", i18n("file %1 requires a newer version of RPM").arg(*filename).ascii());
		    return numPackages;
		}

		if (defaultReloc)
		    defaultReloc->oldPath = NULL;

		fdClose(fd);
		numBinaryPackages++;
	    }
	    break;
	}
    }

    rpmMessage(RPMMESS_DEBUG,  (char *) "%s\n", i18n("found %1 source and %2 binary packages").arg(numSourcePackages).arg(numBinaryPackages).ascii());

    if (numBinaryPackages && !(interfaceFlags & INSTALL_NODEPS)) {
	if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) {
	    numFailed = numPackages;
	    stopInstall = 1;
	}

	if (!stopInstall && conflicts) {
	  //	    rpmMessage(RPMMESS_ERROR, (char *) "%s\n", i18n("failed dependencies:").ascii());
	    KprintDepProblems(stderr, conflicts, numConflicts);
	    rpmdepFreeConflicts(conflicts, numConflicts);
	    numFailed = numPackages;
	    stopInstall = 1;
	}
    }

    if (numBinaryPackages && !(interfaceFlags & INSTALL_NOORDER)) {
	if (rpmdepOrder(rpmdep)) {
	    numFailed = numPackages;
	    stopInstall = 1;
	}
    }

    if (numBinaryPackages && !stopInstall) {
	rpmProblemSet probs = NULL;
;
	rpmMessage(RPMMESS_DEBUG,  (char *) "%s\n", i18n("installing binary packages").ascii());
	rc = rpmRunTransactions(rpmdep, showProgress, (void *) notifyFlags, 
				    NULL, &probs, transFlags, probFilter);

	if (rc < 0) {
	    numFailed += numBinaryPackages;
	} else if (rc) {
	    numFailed += rc;
	    for (i = 0; i < probs->numProblems; i++) {
		if (!probs->probs[i].ignoreProblem) {
		    char *msg = (char *) rpmProblemString(probs->probs[i]);
		    rpmMessage(RPMMESS_ERROR, (char *) "%s\n", "%s", msg);
		    free(msg);
		}
	    }
	}

	if (probs) rpmProblemSetFree(probs);
    }

    if (numBinaryPackages) rpmtransFree(rpmdep);


    if (numSourcePackages && !stopInstall) {
	for (i = 0; i < numSourcePackages; i++) {
	    fd = fdOpen(sourcePackages[i], O_RDONLY, 0);
	    if (fdFileno(fd) < 0) {
		rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("cannot open file %1").arg(sourcePackages[i]).ascii());
		continue;
	    }

	    if (!(transFlags & RPMTRANS_FLAG_TEST))
		numFailed += rpmInstallSourcePackage(rootdir, fd, NULL,
				showProgress, (void *) notifyFlags, NULL);

	    fdClose(fd);
	}
    }

    for (i = 0; i < numTmpPackages; i++) {
	unlink(tmpPackages[i]);
	xfree(tmpPackages[i]);
    }

    /* FIXME how do we close our various fd's? */

    if (dbIsOpen) rpmdbClose(db);

    return numFailed;
}

int doUninstal(const char * rootdir, const QStringList &argv, int transFlags,
		 int interfaceFlags) {
    rpmdb db;
    dbiIndexSet matches;
    int i, j;
    int mode;
    int rc;
    int count;
    int numFailed = 0;
    rpmTransactionSet rpmdep;
    struct rpmDependencyConflict * conflicts;
    int numConflicts;
    int stopUninstall = 0;
    int numPackages = 0;
    rpmProblemSet probs;

    if (transFlags & RPMTRANS_FLAG_TEST) 
	mode = O_RDONLY;
    else
	mode = O_RDWR | O_EXCL;
	
    if (rpmdbOpen(rootdir, &db, mode, 0644)) {
	const char *dn;
	dn = rpmGetPath( (rootdir ? rootdir : ""), "%{_dbpath}", NULL);
        if (getuid() != 0)
          rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("KPACKAGE has to run as ROOT").ascii(), dn);
	xfree(dn);
	return(EXIT_FAILURE);
    }

    j = 0;
    rpmdep = rpmtransCreateSet(db, rootdir);
    for(QStringList::ConstIterator it = argv.begin();
        (it != argv.end());
        it++)
    {
        const char *arg = (*it).ascii();
        rc = rpmdbFindByLabel(db, arg, &matches);
	switch (rc) {
	case 1:
            rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("package %1 is not installed").arg(arg).ascii());
	    numFailed++;
	    break;
	case 2:
            rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("searching for package %1").arg(arg).ascii());
	    numFailed++;
	    break;
	default:
	    count = 0;
	    for (i = 0; i < dbiIndexSetCount(matches); i++)
		if (dbiIndexRecordOffset(matches, i)) count++;

	    if (count > 1 && !(interfaceFlags & UNINSTALL_ALLMATCHES)) {
		rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("\"%1\" specifies multiple packages").arg(arg).ascii());
		numFailed++;
	    }
	    else { 
		for (i = 0; i < dbiIndexSetCount(matches); i++) {
		    unsigned int recOffset = dbiIndexRecordOffset(matches, i);
		    if (recOffset) {
			rpmtransRemovePackage(rpmdep, recOffset);
			numPackages++;
		    }
		}
	    }

	    dbiFreeIndexRecord(matches);
	    break;
	}
    }

    if (!(interfaceFlags & UNINSTALL_NODEPS)) {
	if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) {
	    numFailed = numPackages;
	    stopUninstall = 1;
	}

	if (!stopUninstall && conflicts) {
	  //	    rpmMessage(RPMMESS_ERROR, (char *) "%s\n", i18n("removing these packages would break "
	  //			      "dependencies:"));
	    KprintDepProblems(stderr, conflicts, numConflicts);
	    rpmdepFreeConflicts(conflicts, numConflicts);
	    numFailed += numPackages;
	    stopUninstall = 1;
	}
    }

    if (!stopUninstall) {
	numFailed += rpmRunTransactions(rpmdep, NULL, NULL, NULL, &probs,
					transFlags, 0);
    }

    rpmtransFree(rpmdep);
    rpmdbClose(db);

    return numFailed;
}

int doSourceInstall(const char * rootdir, const char * arg, const char ** specFile,
		    char ** cookie) {
    FD_t fd;
    int rc;

    fd = fdOpen(arg, O_RDONLY, 0);
    if (fdFileno(fd) < 0) {
	rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("cannot open %1").arg(arg).ascii());
	return 1;
    }

    if (rpmIsVerbose())
      KpMsgE(i18n("Installing %1\n").arg(arg), FALSE);

    rc = rpmInstallSourcePackage(rootdir, fd, specFile, NULL, NULL, 
				 cookie);
    if (rc == 1) {
	rpmMessage(RPMMESS_ERROR,  (char *) "%s\n", i18n("%1 cannot be installed").arg(arg).ascii());
	if (specFile) FREE(*specFile);
	if (cookie) FREE(*cookie);
    }

    fdClose(fd);

    return rc;
}

void printDepFlags(FILE * f, QString s, char * version, int flags) {
 
  if (flags) {
    fprintf(f, " ");
    s += " ";
  }
 
  if (flags & RPMSENSE_LESS) {
    fprintf(f, "<");
    s += "<";
  }
  if (flags & RPMSENSE_GREATER) {
    fprintf(f, ">");
    s += ">";
  }
  if (flags & RPMSENSE_EQUAL) {
    fprintf(f, "=");
    s += "=";
  }
  if (flags & RPMSENSE_SERIAL) {
    fprintf(f, "S");
    s += "S";
  }
  if (flags) {
    fprintf(f, " %s", version);
    s += " ";
    s += version;
  }
}
 
void KprintDepProblems(FILE * f, struct rpmDependencyConflict * conflicts,
			     int numConflicts) {
    int i;
    QString s, st;
 
     st += i18n("Dependency Problem:\n");
     for (i = 0; i < numConflicts; i++) {
	fprintf(f, "\t%s", conflicts[i].needsName);
	st += " ";
	st += conflicts[i].needsName;
	if (conflicts[i].needsFlags) {
	    printDepFlags(stderr, st, conflicts[i].needsVersion, 
			  conflicts[i].needsFlags);
	}

	if (conflicts[i].sense == (rpmDependencyConflict::RPMDEP_SENSE_REQUIRES)) { 
	    fprintf(f, i18n(" is needed by %s-%s-%s\n"), conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
            st += s = i18n(" is needed by %1-%2-%3\n")
                             .arg(conflicts[i].byName)
                    .arg(conflicts[i].byVersion).arg(conflicts[i].byRelease);
	} else {
	    fprintf(f, i18n(" conflicts with %s-%s-%s\n"), conflicts[i].byName, 
		    conflicts[i].byVersion, conflicts[i].byRelease);
            st += s = i18n(" conflicts with %1-%2-%3\n")
                            .arg(conflicts[i].byName)
                    .arg(conflicts[i].byVersion).arg(conflicts[i].byRelease);
 	}
    }
     if (st.length() > 1000) {
       st.truncate(1000);
       st+="...";
     }
    rpmMessage(RPMMESS_ERROR, (char *) "%s\n",st.ascii());	
}

extern "C"
{

  void rpmBuff(char *buff) {
    kpackage->setStatus(buff);
  }

  void rpmMess(char *buff) {
    KpMsgE(buff,TRUE);
  } 

  QString *rpmEBuf = 0;

  void rpmErr() {
    if (rpmEBuf) {
      *rpmEBuf += rpmErrorString();
      *rpmEBuf += "\n";
    } else {
      KpMsgE(rpmErrorString(),TRUE);
    }
  }

}

#endif

