/*
 *   kmoon - a moon phase indicator
 *   $Id: kmoon.cpp,v 1.15 2000/02/04 10:19:19 kulow Exp $
 *   Copyright (C) 1998  Stephan Kulow
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <kapp.h>
#include <qwidget.h>
#include "kmoon.moc"
#include <kwm.h>
#include <qwmatrix.h>
#include <qbitmap.h>
#include <qtooltip.h>
#include <kmessagebox.h>
#include <kaboutdata.h>
#include <qimage.h>
#include <stdlib.h>
#include <klocale.h>
#include <qpopupmenu.h>
#include <kglobal.h>
#include <kstddirs.h>
#include <kcmdlineargs.h>
#include "version.h"

double moonphasebylunation(int lun, int phi);
time_t JDtoDate(double jd, struct tm *event_date);

MoonWidget::MoonWidget(QWidget *parent, const char *name)
    : QWidget(parent, name)
{
    struct tm * t;
    time_t clock;

    counter = -1;
    old_w = old_h = old_counter = -1;
    startTimer(100000);

    time(&clock);
    t = localtime(&clock);
    calcStatus(mktime(t));
    renderGraphic();
    KWM::setDockWindow(winId());
    kapp->setTopWidget(new QWidget());
    popup = new QPopupMenu();
    id_about = popup->insertItem(i18n("About"));
    popup->insertSeparator();
    id_quit = popup->insertItem(i18n("Quit"));

    connect(popup, SIGNAL(activated(int)), SLOT(reactOn(int)));

}

void MoonWidget::reactOn(int index)
{
  if (index == id_about)
    showAbout();

  if (index == id_quit)
    kapp->quit();
}

void MoonWidget::calcStatus( time_t time )
{
    double JDE;
    uint lun = 0;
    struct tm event_date;
    struct tm last_event;
    time_t last_new, next_new;

    do {
	last_event = event_date;
	JDE = moonphasebylunation(lun, 0);
	next_new = last_new;
	last_new = JDtoDate(JDE, &event_date);
	lun++;
    } while (last_new < time);

    counter = (time - next_new) / ( 60 * 60 * 24);
    if (counter >= 29)
	counter -= 29;
}

QImage MoonWidget::loadMoon(int index)
{
  QString filename = QString("kmoon/pics/moon%1.png").arg(index);
  filename = locate("data", filename);
  return QImage(filename);
}

const char *description = I18N_NOOP("Moon Phase Indicator for KDE");

void MoonWidget::showAbout()
{
    KMessageBox::about(0,
		       i18n(description) + QString::fromLatin1("\n\n") + 
		       i18n("Written by Stephan Kulow <coolo@kde.org>\n"
			    "\n"
			    "Lunar code by Chris Osburn "
			    "<chris@speakeasy.org>\n"
			    "\n"
			    "Moon graphics by Tim Beauchamp "
			    "<timb@googol.com>"),
		       i18n("About Moon Phase Indicator"));
}

void MoonWidget::timerEvent( QTimerEvent * )
{
    struct tm * t;
    time_t clock;
    time(&clock);
    t = localtime(&clock);
    calcStatus(mktime(t));
    renderGraphic();
    repaint( false );
}
	
void MoonWidget::paintEvent( QPaintEvent * )
{
    renderGraphic();
    bitBlt(this, 0, 0, &pixmap, 0, 0);
}

void MoonWidget::resizeEvent( QResizeEvent *)
{
    renderGraphic();
    repaint();
}

void MoonWidget::renderGraphic()
{
    if (old_counter == counter && old_w == width() && old_h == height())
	return;
    old_counter = counter;
    old_w = width();
    old_h = height();
    QPixmap t;
    t.convertFromImage(loadMoon(counter), AvoidDither);
    QWMatrix m;
    m.scale(float(width()) / t.width(),
	    float(height()) / t.height());
    QImage im = t.xForm(m).convertToImage().convertDepth(8, AvoidDither);

    QColor col1 = yellow;
    QColor col2 = QApplication::palette().normal().background();

    int r1 = col1.red(), r2 = col2.red();
    int g1 = col1.green(), g2 = col2.green();
    int b1 = col1.blue(), b2 = col2.blue();

    for (int i = 0; i < im.numColors(); i++) {
	int grey = qRed(im.color(i));
	im.setColor(i, qRgb(int(float(r1 - r2) / 255 * grey) + r2,
			    int(float(g1 - g2) / 255 * grey) + g2,
			    int(float(b1 - b2) / 255 * grey) + b2));
    }
    pixmap.convertFromImage(im, AvoidDither);

    QToolTip::remove(this);
    tooltip = i18n("Moon is %1 days old.").arg(counter + 1);
    QToolTip::add(this, tooltip);
}

void MoonWidget::mousePressEvent( QMouseEvent *e)
{
    if (e->button() == RightButton) {
	popup->popup(mapToGlobal(e->pos()));
	popup->exec();
    }
    if (e->button() == LeftButton) {
	showAbout();
    }
}

static KCmdLineOptions options[] =
{
    { "o", 0, 0 },
    { "offset <days>", I18N_NOOP("Set the moon some days off"), "0" },
    { 0, 0, 0}
};

int main( int argc, char **argv)
{
    KAboutData aboutData( "kmoon", I18N_NOOP("KMoon"),
        version, description, KAboutData::License_GPL,
        "(c) 1998, Stephan Kulow");
    aboutData.addAuthor("Stephan Kulow",0, "coolo@kde.org");
    KCmdLineArgs::init( argc, argv, &aboutData );
    KCmdLineArgs::addCmdLineOptions( options );

    KApplication a;
    MoonWidget *moon = new MoonWidget();
#warning fix session management
#if 0
    kapp->enableSessionManagement( TRUE );
#endif
    kapp->setTopWidget(new QWidget);
    a.setMainWidget(moon);
    moon->show();
    KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
    int offset = args->getOption("offset").toInt();
    args->clear();
    if (offset)
      moon->calcStatus(time(0) + offset * 24 * 60 * 60);
    return a.exec();
}
