/* minicli.cpp
*
* Copyright (C) 1997 Matthias Ettrich <ettrich@kde.org>
* Copyright (c) 1999 Preston Brown <pbrown@kde.org>
* Complete re-write:
* Copyright (C) 1999 - 2000 Dawit Alemayehu <adawit@kde.org>
* Copyright (C) 2000 Malte Starostik <starosti@zedat.fu-berlin.de>
*/

#include <qlabel.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qvbox.h>
#include <qdir.h>

#include <kdialog.h>
#include <klocale.h>
#include <kconfig.h>
#include <kiconloader.h>
#include <kbuttonbox.h>
#include <kmessagebox.h>
#include <kmimetype.h>
#include <kservice.h>
#include <kprocess.h>
#include <kcombobox.h>
#include <kcompletion.h>
#include <kwm.h>
#include <kapp.h>

#include "minicli.moc"

Minicli::Minicli( QWidget *parent, const char *name)
        :QDialog( parent, name )
{
    m_pCompletion = new KCompletion();
    m_strCWD = QString::null;
    loadConfig();
    // TODO: Should be invoked from the loadConfig.  Need to
    // make it a DCOP client as well so that GUI changes from
    // control panel can be reflected immediately!!
    loadStandardGUI();
}

Minicli::~Minicli()
{
    delete m_pCompletion;
    m_pCompletion = 0; // Set it to proper NULL pointer! We play nice :))
}

void Minicli::loadStandardGUI()
{
    QVBoxLayout* vbox = new QVBoxLayout( this );
    setIcon(SmallIcon("exec"));

    setCaption( i18n("Run Command") );
    vbox->setResizeMode( QLayout::Fixed );

    vbox->setSpacing( KDialog::spacingHint() );
    vbox->setMargin( KDialog::marginHint() );

    QHBox *hBox = new QHBox( this );
    CHECK_PTR(hBox);
    vbox->addWidget( hBox );
    hBox->setSpacing( KDialog::marginHint() );

    m_runIcon = new QLabel( hBox );
    CHECK_PTR(m_runIcon);
    m_runIcon->setPixmap(DesktopIcon("go"));
    m_runIcon->setFixedSize(m_runIcon->pixmap()->size()); // avoid layout changes when only a non-medium sized icon is found
    m_runIcon->setAlignment(AlignCenter);

    QLabel *label = new QLabel( i18n("Enter the name of the program or the command you wish to execute.\n"
                                     "Check the \"Run in Terminal\" box if program is not graphical."), hBox);
    CHECK_PTR(label);

    hBox = new QHBox( this );
    vbox->addWidget( hBox );
    CHECK_PTR(hBox);
    hBox->setSpacing( KDialog::marginHint() );

    label = new QLabel(i18n("&Command:"), hBox);
    CHECK_PTR(label);
    label->setFixedSize(label->sizeHint());

    m_runCombo = new KComboBox( true, hBox );
    CHECK_PTR(m_runCombo);
    if( m_pCompletion )
    {
        m_runCombo->setCompletionObject( m_pCompletion );
        m_runCombo->setCompletionMode( m_pCompletion->completionMode() ); //sync completion modes.
        m_runCombo->insertStringList( m_pCompletion->items() ); // Also add it into completion object.
    }
    m_runCombo->setDuplicatesEnabled( false );    // Avoid duplicate entries.
    m_runCombo->setInsertionPolicy( QComboBox::AtTop ); // Insert at the new enteries at the top.
    m_runCombo->setMaxCount( m_iMaxHistory ); // Limit entry into box to defined max.
    connect( m_runCombo, SIGNAL( textChanged( const QString& ) ), SLOT( slotCmdChanged( const QString& ) ) );
    connect( m_runCombo, SIGNAL( returnPressed( const QString& ) ), SLOT( slotReturnPressed( const QString& ) ) );
    connect( this, SIGNAL( aboutToHide() ), m_runCombo, SLOT( clearEdit() ) );
    label->setBuddy(m_runCombo);

    m_parseTimer = new QTimer(this);
    CHECK_PTR(m_parseTimer);
    connect(m_parseTimer, SIGNAL(timeout()), SLOT(slotParseTimer()));

    hBox = new QHBox(this);
    CHECK_PTR(hBox);
    vbox->addWidget( hBox );
    hBox->setSpacing( KDialog::marginHint() );
    m_terminalBox = new QCheckBox(i18n("Run in &Terminal"), hBox);
    CHECK_PTR(m_terminalBox);
    connect(m_terminalBox, SIGNAL(toggled(bool)), SLOT(slotTerminalToggled(bool)));

    KButtonBox *bbox = new KButtonBox(hBox);
    CHECK_PTR(bbox);
    bbox->addStretch();
    QPushButton *runButton = bbox->addButton(i18n("&Run"), false);
    CHECK_PTR(runButton);
    runButton->setDefault(true);
    connect(runButton, SIGNAL(clicked()), this, SLOT(run_command()));

    QPushButton *cancelButton = bbox->addButton(i18n("Cancel"), false);
    CHECK_PTR(cancelButton);
    connect(cancelButton, SIGNAL(clicked()), this, SLOT(cleanup()));
    connect(cancelButton, SIGNAL(clicked()), m_runCombo, SLOT(clearEdit()));

    m_runCombo->clearEdit();
    m_runCombo->setFocus();
    bbox->layout();
    move( QApplication::desktop()->width()/2 - width()/2,
          QApplication::desktop()->height()/2 - height()/2);
}

void Minicli::loadConfig()
{
    KConfig *config = KGlobal::config();
    config->setGroup("MiniCli");
    QStringList hist = config->readListEntry("History");
    m_iMaxHistory = config->readNumEntry("HistoryLength", 50);
    if ( hist.count() > m_iMaxHistory )
        for ( QStringList::Iterator it = hist.at( m_iMaxHistory ); it != hist.end(); it = hist.remove(it) );
    m_pCompletion->setItems( hist ); // load the list into the completion object.
    KGlobalSettings::Completion comp = ( KGlobalSettings::Completion) config->readNumEntry("CompletionMode", KGlobalSettings::completionMode());
    m_pCompletion->setCompletionMode( comp );
}

void Minicli::saveConfig()
{
    if( m_pCompletion )
    {
        // save history list (M.H.)
        KConfig *config = KGlobal::config();
        config->setGroup("MiniCli");
        config->writeEntry( "History", m_pCompletion->items() );
        config->writeEntry( "CompletionMode", m_pCompletion->completionMode() );
        config->sync();
    }
}

void Minicli::hideEvent( QHideEvent* ev )
{
    emit aboutToHide();
    QWidget::hideEvent( ev );
}

bool Minicli::setCWD( const QString& cwd )
{
    QDir d( cwd );
    if( d.exists() )
    {
        m_strCWD = cwd;
        if( m_strCWD.left(1) != '/' ) m_strCWD += '/';
        return true;
    }
    return false;
}


void Minicli::keyPressEvent(QKeyEvent *kev)
{
    int a = ((QKeyEvent*)kev)->ascii();
    // ^C, EOF, ESC
    if (a == 3 || a == 4 || a == 27)
    {
        cleanup();
        kev->accept();
        return;
    }
    if (kev->key() == Key_Return || kev->key() == Key_Enter)
    {
        kev->accept();
        run_command();
        return;
    }
    kev->ignore();
}

void Minicli::run_command()
{
    kapp->propagateSessionManager();
    QString s;
    if( m_strCWD.isNull() ) s = m_runCombo->currentText().simplifyWhiteSpace();
    else s = m_strCWD + m_runCombo->currentText().simplifyWhiteSpace();
    // TODO: krun, klauncher, whatever will implement launching feedback
    if (s == "logout")
    {
        bool shutdown = kapp->requestShutDown();
        if( !shutdown )
            KMessageBox::error( 0, i18n("Could not logout properly.  The sesion manager cannot\n"
                                        "be contacted.  You can try to force a shutdown by pressing\n"
                                        "the CTRL, SHIFT and BACKSPACE keys at the same time.  Note\n"
                                        "however that your current session will not be saved with a\n"
                                        "forced shutdown." ) );
    }
    else if ( !s.isEmpty() )
    {
        KShellProcess proc;
        if (m_terminalBox->isChecked()) proc << "konsole -e \"" + s + '"';
        else
        {
            KURIFilterData data ( s );
            KURIFilter::self()->filterURI( data, "ShortURIFilter" );
            debug( "Original url : %s\nFiltered url : %s", s.latin1(), data.uri().url().latin1() );	
            switch( data.uriType() )
            {
                case KURIFilterData::LOCAL_FILE:
                case KURIFilterData::LOCAL_DIR:
                case KURIFilterData::NET_PROTOCOL:
                    proc << "kfmclient exec \"" + data.uri().url() + '"';
                    break;
                case KURIFilterData::EXECUTABLE:
                    proc << data.uri().url();
                    break;
                case KURIFilterData::SHELL:
                    proc << data.uri().url();
                    break;
                case KURIFilterData::HELP:
                    // FIXME parent should be this but somehow that screws our layout (malte)
                    KMessageBox::sorry( 0, i18n("This is currently not supported") );
                    return;
                case KURIFilterData::ERROR:
                    KMessageBox::error( 0, data.uri().url() );
                    return;
                default:
                    KMessageBox::sorry( 0, i18n("The program name or command: %1\ncannot be found.").arg( data.uri().url() ) );
                    return;
            }
        }
        proc.start(KShellProcess::DontCare);
    }
    cleanup();
}


void Minicli::cleanup()
{
    hide();
	saveConfig();
	if( m_runCombo )
	{
    	m_terminalBox->setChecked( false );
	    m_runIcon->setPixmap( DesktopIcon("go") );
    	m_runCombo->setFocus();
 	}
}

void Minicli::closeEvent ( QCloseEvent* e )
{
    e->accept();
    cleanup();
}

void Minicli::slotCmdChanged( const QString& )
{
    m_parseTimer->stop();
    m_parseTimer->start(250, true);
}

void Minicli::slotTerminalToggled( bool )
{
    slotParseTimer();
}

void Minicli::slotReturnPressed( const QString& text )
{
    if( m_pCompletion )
    {
        uint count = m_pCompletion->items().count();
        if( count < m_iMaxHistory )
            m_pCompletion->addItem( text );
    }
}

void Minicli::slotParseTimer()
{
    // Change the icon according to the command type
    QPixmap icon;
    if (m_terminalBox->isChecked()) icon = DesktopIcon("konsole");
    else
    {
        QString cmd = m_runCombo->currentText().simplifyWhiteSpace();
        if ( !cmd.isEmpty() )
        {
            KURIFilterData data( cmd );	    
            KURIFilter::self()->filterURI( data, "ShortURIFilter" );
	    debug( "Original url : %s\nFiltered url : %s", cmd.latin1(), data.uri().url().latin1() );
	    if( data.hasBeenFiltered() ) cmd = data.uri().url(); // get the new URL
            switch ( data.uriType() )
            {
                case KURIFilterData::LOCAL_FILE:
                case KURIFilterData::LOCAL_DIR:
                case KURIFilterData::NET_PROTOCOL:
                    icon = KMimeType::pixmapForURL(cmd, 0, KIcon::SizeMedium);
                    break;
                case KURIFilterData::EXECUTABLE:
                {
                    KService::Ptr service = KService::serviceByDesktopName(cmd);
                    if( service ) icon = service->pixmap(KIcon::SizeMedium);
                    else icon = DesktopIcon("exec");
                    break;
                }
                case KURIFilterData::HELP:
                    icon = DesktopIcon("khelpcenter");
                    break;
                case KURIFilterData::ERROR:
                    icon = DesktopIcon("error");
                default:
                    break;
            }
        }
    }
    m_runIcon->setPixmap(icon.isNull() ? DesktopIcon("go") : icon);
}
