/* vi:set ts=8 sts=0 sw=8:
 * $Id: win.c,v 1.30 1999/01/21 21:38:18 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdarg.h>
#include <string.h>
#include "win.h"
#include "doc.h"
#include "file.h"
#include "menu.h"
#include "toolbar.h"
#include "msgbar.h"
#include "msgbox.h"
#include "prefs.h"
#include "recent.h"
#ifdef APP_GNP
#include "prjbar.h"
#include "print.h"
#include "clipboard.h"
#include "search.h"
#include "htmltags.h"
#endif

/*** external declarations ***/
extern void	win_set_title(doc_t *d);


/*** local declarations ***/
#define AUTOSAVE_MSG	"Autosaving changed files..."
#define DOC_TABS_OFF	"Document tabs OFF"
#define DOC_TABS_ON	"Document tabs ON"
#define DOC_TABS_TOP	"Document tabs on the TOP"
#define DOC_TABS_BOTTOM	"Document tabs on the BOTTOM"
#define DOC_TABS_LEFT	"Document tabs on the LEFT"
#define DOC_TABS_RIGHT	"Document tabs on the RIGHT"
#define DOC_WWRAP_OFF	"Wordwrap OFF"
#define DOC_WWRAP_ON	"Wordwrap ON"


/*** local variables ***/
static GSList *winlist = NULL;		/* list of win_t's */
static unsigned nextid = 10000;		/* internal window id for each win_t */

static GtkWidget *wlw = NULL;		/* these three are used for the */
static GtkWidget *wlw_data = NULL;	/* winlist popup window */
static unsigned curwinrow = 0;		/* current window rownum */

static toolbar_data_t main_tbdata[] = {
	{ " New ",	"New File",	"new",		"tb_new.xpm",
			(GtkSignalFunc)doc_new_cb },
	{ " Open ",	"Open File",	"open",		"tb_open.xpm",
			(GtkSignalFunc)doc_open_cb },
	{ " Close ",	"Close File",	"close",	"tb_cancel.xpm",
			(GtkSignalFunc)doc_close_cb },
#ifdef APP_GNP
	{ " Save ",	"Save File",	"save",		"tb_save.xpm",
			(GtkSignalFunc)doc_save_cb },
	{ " Print ",	"Print File",	"print",	"tb_print.xpm",
			(GtkSignalFunc)print_cb },
	{ " SPACE " },
	{ " Cut ",	"Cut Text",	"cut",		"tb_cut.xpm",
			(GtkSignalFunc)cut_cb },
	{ " Copy ",	"Copy Text",	"copy",		"tb_copy.xpm",
			(GtkSignalFunc)copy_cb },
	{ " Paste ",	"Paste Text",	"paste",	"tb_paste.xpm",
			(GtkSignalFunc)paste_cb	},
	{ " Find ",	"Find Text",	"Find",		"tb_search.xpm",
			(GtkSignalFunc)search_cb },
#endif
	{ " SPACE " },
	{ " Prefs ",	"Preferences",	"prefs",	"tb_prefs.xpm",
			(GtkSignalFunc)prefs_cb },
	{ " Exit ",	"Exit App",	"exit",		"tb_exit.xpm",
			(GtkSignalFunc)win_close_all_cb },
	{ NULL }
};


/*** local function prototypes ***/
static void	notebook_init(win_t *w);
static void	notebook_switch_page(GtkWidget *wgt, GtkNotebookPage *page,
				     int num, gpointer cbdata);
static bool_t	win_close_common(win_t *w);
static void	win_list_destroy(GtkWidget *wgt, gpointer cbdata);
static void	win_list_remove(win_t *w);
static void	win_list_add(win_t *w, char *title);
static int	win_list_select(GtkWidget *wgt, unsigned row, int column,
				GdkEventButton *event, gpointer cbdata);
static void	win_list_close_selected(GtkWidget *wgt, gpointer cbdata);


/*** global function definitions ***/
#ifdef APP_GNP
/*
 * PUBLIC: win_autosave
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
int
win_autosave(gpointer cbdata)
{
	GSList *wlp, *dlp;
	win_t *w;
	doc_t *d;
	int num = 0;

	msgbox_printf("Autosave in progress...");
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		msgbar_printf(w, AUTOSAVE_MSG);
		dlp = (GSList *)(w->doclist);
		while (dlp) {
			d = (doc_t *)(dlp->data);
			if (d->changed && strcmp(d->fname, UNTITLED)) {
				(void)file_save_execute(d, FALSE);
				num++;
			}
			dlp = dlp->next;
		}
		wlp = wlp->next;
	}
	msgbox_printf("Autosave completed (%d files saved)", num);

	return 1;	/* for GTK, must return 1 */
} /* win_new_with_new_doc */
#endif


/*
 * PUBLIC: win_foreach
 *
 * traver window list and call a function on each window.
 */
void
win_foreach(void (*func)(void *))
{
	GSList *wlp;
	win_t *wp;

	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		func(wp);
		wlp = wlp->next;
	}
} /* win_foreach */


/*
 * PUBLIC: win_doc_foreach
 *
 * traverse window list, and traverse all docs in each window.
 */
void
win_doc_foreach(void (*func)(void *))
{
	GSList *wlp;
	win_t *wp;

	for (wlp = winlist; wlp; wlp = wlp->next) {
		wp = (win_t *)(wlp->data);
		doc_foreach(wp, func);
	}
} /* win_doc_foreach */


/*
 * PUBLIC: win_new_with_doc
 *
 * create a new window with a document.  currently, only creates an "Untitled"
 * document.
 */
void
win_new_with_doc(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w;

	w = win_new();
	doc_new(w, UNTITLED, TRUE);
	gtk_widget_show(w->toplev);
} /* win_new_with_new_doc */


/*
 * PUBLIC: win_new
 *
 * creates a new window and adds it to the winlist.
 */
win_t *
win_new(void)
{
	win_t *w;

	w	= (win_t *)g_malloc0(sizeof(win_t));
	w->id	= nextid++;

	/* toplevel */
	w->toplev = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_widget_set_name(w->toplev, "gnp+");
	gtk_window_set_wmclass(GTK_WINDOW(w->toplev), "gnp+", "gnp+");
	gtk_window_set_title(GTK_WINDOW(w->toplev), "gnotepad+");
	gtk_window_set_policy(GTK_WINDOW(w->toplev), TRUE, TRUE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(w->toplev), 0);
	gtk_signal_connect(GTK_OBJECT(w->toplev), "destroy",
			   GTK_SIGNAL_FUNC(win_close_cb), w);

	gtk_widget_set_usize(GTK_WIDGET(w->toplev), prefs.win_width, prefs.win_height);
	if (IS_SAVE_WIN_POS())
		gtk_widget_set_uposition(GTK_WIDGET(w->toplev),
					 prefs.win_xpos, prefs.win_ypos);

	/* mainbox */
	w->mainbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(w->toplev), w->mainbox);
	gtk_widget_show(w->mainbox);

	/* msgbox can be initialized anywhere before a file is opened */
	msgbox_init();

	/* these need to be in order */
	menu_main_init(w);
	recent_list_init(w);
	gtk_widget_realize(w->toplev);
	toolbar_create(w, w->mainbox, main_tbdata, &(w->main_tb),
		       &(w->main_tb_h));
#ifdef APP_GNP
	html_tb_create(w, w->mainbox, &(w->html_tb), &(w->html_tb_h));
#endif

	notebook_init(w);
#ifdef WANT_PROJECT
	prjbar_init(w);
#endif	/* WANT_PROJECT */

	/* msgbar and docinfo button go together */
	w->hbox_bot = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->hbox_bot, FALSE, FALSE, 0);
	gtk_widget_show(w->hbox_bot);
	msgbar_init(w, w->hbox_bot);
	doc_info_init(w, w->hbox_bot);

#ifdef APP_GNP
	quickmenu_init(w);
#endif

	/* add to winlist and winlist popup */
	winlist = g_slist_append(winlist, (gpointer)w);
	win_list_add(w, GTK_WINDOW(w->toplev)->title);

	return w;
} /* win_new */


/*
 * PUBLIC: win_close_cb
 *
 * window close callback
 */
void
win_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_close_common((win_t *)cbdata);
} /* win_close_cb */


/*
 * PUBLIC: win_close_all_cb
 *
 * close all windows (e.g., quit) callback
 */
void
win_close_all_cb(GtkWidget *wgt, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;
	bool_t changed = FALSE;

	msgbox_close();
	if (wlw)
		gtk_widget_hide(wlw);

	for (wlp = winlist; wlp; wlp = wlp->next) {
		w = (win_t *)(wlp->data);
#ifdef APP_GNP
		if (doc_check_if_any_changed(w)) {
			changed = TRUE;
			break;
		}
#endif
	}

	if (!changed) {
		for (wlp = winlist; wlp; wlp = wlp->next) {
			w = (win_t *)(wlp->data);
			gtk_widget_hide(w->toplev);
#ifdef APP_GNP
			if (w->saveas)
				gtk_widget_hide(w->saveas);
			if (w->search)
				gtk_widget_hide(w->search);
#endif
			if (w->filesel)
				gtk_widget_hide(w->filesel);
			if (w->dlw)
				gtk_widget_hide(w->dlw);
		}
	}

	while (winlist) {
		if (win_close_common((win_t *)(winlist->data)) == FALSE)
			break;
	}

	g_assert(winlist != NULL);
} /* win_close_all_cb */


/*
 * PUBLIC: win_dtab_toggle
 *
 * callback: toggles word wrap
 */
void
win_dtab_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	if (IS_SHOW_TABS()) {
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), FALSE);
		CLEAR_SHOW_TABS();
		msgbar_printf(w, DOC_TABS_OFF);
	} else {
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), TRUE);
		SET_SHOW_TABS();
		msgbar_printf(w, DOC_TABS_ON);
	}
} /* win_dtab_toggle */


/*
 * PUBLIC: win_dtab_top
 *
 * callback: toggles word wrap
 */
void
win_dtab_top(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), GTK_POS_TOP);
	prefs.tabpos = GTK_POS_TOP;
	msgbar_printf(w, DOC_TABS_TOP);
} /* win_dtab_top */


/*
 * PUBLIC: win_dtab_bot
 *
 * callback: toggles word wrap
 */
void
win_dtab_bot(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), GTK_POS_BOTTOM);
	prefs.tabpos = GTK_POS_BOTTOM;
	msgbar_printf(w, DOC_TABS_BOTTOM);
} /* win_dtab_bot */


/*
 * PUBLIC: win_dtab_left
 *
 * callback: toggles word wrap
 */
void
win_dtab_left(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), GTK_POS_LEFT);
	prefs.tabpos = GTK_POS_LEFT;
	msgbar_printf(w, DOC_TABS_LEFT);
} /* win_dtab_left */


/*
 * PUBLIC: win_dtab_right
 *
 * callback: toggles word wrap
 */
void
win_dtab_right(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), GTK_POS_RIGHT);
	prefs.tabpos = GTK_POS_RIGHT;
	msgbar_printf(w, DOC_TABS_RIGHT);
} /* win_dtab_right */


/*
 * PUBLIC: win_redraw_doc_tab
 *
 * redraw the document tabs for a specified window.  called from prefs_save,
 * when the preferences may have changed.
 */
void
win_redraw_doc_tab(void *data)
{
	win_t *w = (win_t *)data;

	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), IS_SHOW_TABS());
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), prefs.tabpos);
} /* win_redraw_doc_tab */


#ifdef APP_GNP
/*
 * PUBLIC: win_toggle_wwrap
 *
 * callback: toggles word wrap
 */
void
win_wwrap_toggle(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d;

	g_assert(w != NULL);
	d = doc_current(w);
	g_assert(d != NULL);
	if (IS_USE_WORDWRAP()) {
		gtk_text_set_word_wrap(GTK_TEXT(d->data), FALSE);
		CLEAR_USE_WORDWRAP();
		msgbar_printf(w, DOC_WWRAP_OFF);
	} else {
		gtk_text_set_word_wrap(GTK_TEXT(d->data), TRUE);
		SET_USE_WORDWRAP();
		msgbar_printf(w, DOC_WWRAP_ON);
	}
} /* win_wwrap_toggle */
#endif	/* APP_GNP */


/*** local function definitions ***/
/*
 * PRIVATE: win_close_common
 *
 * common routine for actually closing a window.  called from win_close_cb()
 * and win_close_all_cb().
 */
static bool_t
win_close_common(win_t *w)
{
	doc_new_name_t *dnnp;
	GSList *dlp, *wlp, *rlp;

	doc_list_destroy(NULL, w);
	win_list_destroy(NULL, NULL);

	dlp = (GSList *)w->doclist;
	if (dlp == NULL || w->numdoc < 1)
		return TRUE;

#ifdef APP_GNP
	if (doc_check_if_any_changed(w) == FALSE)
		gtk_widget_hide(w->toplev);
#endif

	if (doc_close_all_common(w) == FALSE)
		return FALSE;

	dlp = (GSList *)w->doclist;
	g_assert(dlp == NULL && w->numdoc == 0);

	/* update preferences */
	if (IS_SAVE_WIN_POS())
		gdk_window_get_position(w->toplev->window,
					&prefs.win_xpos, &prefs.win_ypos);

	/* remove from the win list popup window */
	win_list_remove(w);

	/* remove from the linked list of win_t's */
	wlp = g_slist_find(winlist, w);
	winlist = g_slist_remove_link(winlist, wlp);

	/* cleanup */
	while (w->recentlist) {
		rlp = w->recentlist;
		w->recentlist = g_slist_remove_link(w->recentlist, rlp);
		dnnp = (doc_new_name_t *)(rlp->data);
		g_free(dnnp->fname);
		g_free(dnnp);
#if GLIST_MAGIC
		g_free(rlp);
#endif
	}
	g_free(w->lastmsg);
	gtk_timeout_remove(w->timeout_id);
	gtk_widget_destroy(w->toplev);
	g_free(w);
#if GLIST_MAGIC
	g_free(wlp);
#endif

	/* save preferences */
	if (IS_SAVE_WIN_WIDTH() || IS_SAVE_WIN_HEIGHT() || IS_SAVE_WIN_POS())
		prefs_save();

	if (winlist == NULL)
		gtk_exit(0);

	return TRUE;
} /* win_close_common */


/*
 * PRIVATE: notebook_init
 *
 * creates the notebook to contain all the documents.
 */
static void
notebook_init(win_t *w)
{
	w->nb = gtk_notebook_new();
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w->nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(w->nb));
	gtk_signal_connect_after(GTK_OBJECT(w->nb), "switch_page",
				 GTK_SIGNAL_FUNC(notebook_switch_page), w);
	gtk_box_pack_start(GTK_BOX(w->mainbox), w->nb, TRUE, TRUE, 0);

	/* set according to preferences */
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->nb), prefs.tabpos);
	if (IS_SHOW_TABS())
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), TRUE);
	else
		gtk_notebook_set_show_tabs(GTK_NOTEBOOK(w->nb), FALSE);

	gtk_widget_show(w->nb);
} /* notebook_init */


/*
 * PRIVATE: notebook_switch_page
 *
 * callback invoked when a switch page has occured on the notebook.
 */
static void
notebook_switch_page(GtkWidget *wgt, GtkNotebookPage *page,
	int num, gpointer cbdata)
{
	doc_t *d;
	win_t *w = (win_t *)cbdata;

	g_assert(w != NULL);

	if (w->doclist == NULL)
		return;

	if (!GTK_WIDGET_REALIZED(w->toplev))
		return;

	w->curdoc = g_slist_nth_data(w->doclist, num);
	if (w->dlw_data)
		gtk_clist_select_row(GTK_CLIST(w->dlw_data), num, FnumCol);
	if ((d = doc_current(w)) != NULL) {
		gtk_widget_grab_focus(d->data);
		win_set_title(d);
		win_list_set_curdoc(w);
		doc_info_label_update(w);
	}
} /* notebook_switch_page */


/*
 * PUBLIC: win_set_title
 *
 * sets the window title.
 */
void
win_set_title(doc_t *d)
{
	char *title;
	int len;

	len = strlen(doc_basefname(d)) + strlen(APP_NAME) + 5;
	title = (char *)g_malloc(len);
	g_snprintf(title, len, "%s - %s", APP_NAME, doc_basefname(d));
	gtk_window_set_title(GTK_WINDOW(d->w->toplev), title);
} /* win_set_title */


/*
 * PUBLIC: win_list_set_curdoc
 *
 * set the current document column/field for the window specified.
 */
void
win_list_set_curdoc(win_t *w)
{
	GtkCList *clist;
	doc_t *d;

	if (wlw_data == NULL)
		return;

	d = doc_current(w);
	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	gtk_clist_set_text(clist, w->rownum, FnameCol, doc_basefname(d));
	gtk_clist_thaw(clist);
} /* win_list_set_curdoc */


/*
 * PUBLIC: win_list_set_numdoc
 *
 * set the # of docs column in the window list popup.
 */
void
win_list_set_numdoc(win_t *w)
{
	GtkCList *clist;
	char buf[8];

	if (wlw_data == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	gtk_clist_freeze(clist);
	g_snprintf(buf, 16, "%d", w->numdoc);
	gtk_clist_set_text(clist, w->rownum, FsizeCol, buf);
	gtk_clist_thaw(clist);
} /* win_list_set_numdoc */


/*
 * PUBLIC: win_list_show
 *
 * creates a new popup window containing a list of open files/documents
 * for the window from which it was invoked.
 */
#define WIN_LIST_NUM_COLUMNS	3
#define INSERT			0
#define APPEND			1
void
win_list_show(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget *tmp, *vbox, *hbox, *scrolled_win;
	GSList *wlp;
	win_t *wp;

	char *titles[] = { " # ", " # Docs ", " Current Document " };
	win_t *w = (win_t *)cbdata;

	if (wlw) {
		gdk_window_raise(wlw->window);
		return;
	}
	g_assert(w != NULL);

	/* create top level and vbox to hold everything */
	wlw = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(wlw), "Window List");
	gtk_signal_connect(GTK_OBJECT(wlw), "destroy",
			   GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(wlw), vbox);
	gtk_container_border_width(GTK_CONTAINER(vbox), 5);

	/* create clist */
	wlw_data = gtk_clist_new_with_titles(WIN_LIST_NUM_COLUMNS, titles);
	gtk_clist_set_column_width(GTK_CLIST(wlw_data), FnumCol, 25);
	gtk_clist_set_column_justification(GTK_CLIST(wlw_data),
					   FnumCol, GTK_JUSTIFY_LEFT);
	gtk_clist_set_column_width(GTK_CLIST(wlw_data), FsizeCol, 55);
	gtk_clist_set_column_justification(GTK_CLIST(wlw_data),
					   FsizeCol, GTK_JUSTIFY_CENTER);
	gtk_clist_column_titles_passive(GTK_CLIST(wlw_data));
	scrolled_win = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_container_add(GTK_CONTAINER(scrolled_win), wlw_data);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_win, TRUE, TRUE, 0);
	gtk_widget_show(scrolled_win);
	gtk_signal_connect(GTK_OBJECT(wlw_data), "select_row",
			   GTK_SIGNAL_FUNC(win_list_select), NULL);

	/* for each window, add info into the clist */
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		win_list_add(wp, doc_basefname(doc_current(wp)));
		wlp = wlp->next;
	}

	/* create a separator and then some handy buttons */
	tmp = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, TRUE, 0);
	hbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	tmp = gtk_button_new_with_label("Close Selected");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			   GTK_SIGNAL_FUNC(win_list_close_selected), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 2);

	tmp = gtk_button_new_with_label("Close All");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			   GTK_SIGNAL_FUNC(win_close_all_cb), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), tmp, TRUE, TRUE, 2);

	tmp = gtk_button_new_with_label("Exit Window List");
	gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
			   GTK_SIGNAL_FUNC(win_list_destroy), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), tmp, FALSE, FALSE, 2);
	GTK_WIDGET_SET_FLAGS(tmp, GTK_CAN_DEFAULT);
	gtk_widget_grab_default(tmp);

	/*
	 * there should be a way to auto-detect how wide to make these widgets
	 * instead of having to use arbitrary values
	 */
	gtk_widget_set_usize(wlw, 300, 200);

	gtk_widget_show_all(wlw);
} /* win_list_show */


/*
 * PUBLIC: win_list_destroy
 *
 * zap!
 */
static void
win_list_destroy(GtkWidget *wgt, gpointer cbdata)
{
	if (wlw) {
		gtk_widget_destroy(wlw);
		wlw = NULL;
		wlw_data = NULL;
	}
} /* win_list_destroy */


/*
 * PRIVATE: win_list_close_selected
 *
 * close the window that is highlighted in the win list popup.
 */
static void
win_list_close_selected(GtkWidget *wgt, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;

	if (wlw == NULL)
		return;

	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == curwinrow) {
			if (win_close_common(w) == TRUE)
				break;
		}
		wlp = wlp->next;
	}
} /* win_list_close_selected */


/*
 * PUBLIC: win_list_remove
 *
 * removes an entry from the files list window.
 */
static void
win_list_remove(win_t *w)
{
	char buf[4];
	int i;
	GtkCList *clist;
	GSList *wlp;
	win_t *wp;
	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);
	g_assert(clist != NULL);
	gtk_clist_freeze(clist);
	gtk_clist_remove(clist, w->rownum);

	/* renumber any entries starting with the row just removed */
	if (clist->rows > 1) {
		for (i = w->rownum; i < clist->rows; i++) {
			g_snprintf(buf, 4, "%d", i + 1);
			gtk_clist_set_text(clist, i, FnumCol, buf);
		}
	}

	/* renumber rownum field for remaining windows */
	wlp = winlist;
	while (wlp) {
		wp = (win_t *)(wlp->data);
		if (wp->rownum > w->rownum)
			wp->rownum--;
		wlp = wlp->next;
	}

	/* pick "next most recent" row to highlight */
	if (w->rownum == (unsigned)(clist->rows))
		gtk_clist_select_row(clist, w->rownum - 1, FnumCol);
	else
		gtk_clist_select_row(clist, w->rownum, FnumCol);
	gtk_clist_thaw(clist);
} /* win_list_remove */


/*
 * PRIVATE: win_list_add
 *
 * appends an entry from the files list window.
 */
static void
win_list_add(win_t *w, char *title)
{
	char numstr[4], numdocstr[8];
	char *rownfo[WIN_LIST_NUM_COLUMNS];
	GtkCList *clist;

	
	if (wlw == NULL)
		return;
	g_assert(w != NULL);

	clist = GTK_CLIST(wlw_data);

	g_snprintf(numstr, 4, "%d", clist->rows + 1);
	rownfo[0] = numstr;
	g_snprintf(numdocstr, 8, "%4d", w->numdoc);
	rownfo[1] = numdocstr;
	rownfo[2] = title;
	gtk_clist_freeze(clist);
	gtk_clist_append(clist, rownfo);
	gtk_clist_select_row(clist, clist->rows - 1, FnumCol);
	gtk_clist_thaw(clist);
	w->rownum = clist->rows - 1;
} /* win_list_add */


/*
 * PRIVATE: win_list_select
 *
 * callback routine when selecting a window in the winlist popup.  this does
 * two things:
 *
 * 1. sets the curwinrow variable to the row that was selected.  this is used
 * for the "close" button on the winlist popup, so we know which window to
 * close.  we need this because the winlist itself is our own creation, and
 * there is no association at all with the close button's context.
 *
 * 2. raises the window which was selected.
 */
static int
win_list_select(GtkWidget *wgt, unsigned row, int column,
	GdkEventButton *event, gpointer cbdata)
{
	GSList *wlp;
	win_t *w;

	curwinrow = row;
	wlp = winlist;
	while (wlp) {
		w = (win_t *)(wlp->data);
		if (w->rownum == row) {
			gdk_window_raise(w->toplev->window);
			break;
		}
		wlp = wlp->next;
	}

	return TRUE;
} /* win_list_select */


/* the end */
