;;; emacspeak-realaudio.el --- Play realaudio from Emacs
;;; $Id: emacspeak-realaudio.el,v 9.0 1998/10/24 05:17:28 raman Exp $
;;; $Author: raman $
;;; Description: Single click access to RealAudio from emacspeak
;;; Keywords: Emacspeak, RealAudio
;;{{{  LCD Archive entry: 

;;; LCD Archive Entry:
;;; emacspeak| T. V. Raman |raman@adobe.com 
;;; A speech interface to Emacs |
;;; $Date: 1998/10/24 05:17:28 $ |
;;;  $Revision: 9.0 $ | 
;;; Location undetermined
;;;

;;}}}
;;{{{  Copyright:

;;; Copyright (c) 1997 by T. V. Raman Adobe Systems Incorporated 
;;; All Rights Reserved. 
;;;
;;; This file is not part of GNU Emacs, but the same permissions apply.
;;;
;;; GNU Emacs is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2, or (at your option)
;;; any later version.
;;;
;;; GNU Emacs is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Emacs; see the file COPYING.  If not, write to
;;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;}}}
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;{{{  Required modules

(require 'cl)
(declaim  (optimize  (safety 0) (speed 3)))
(eval-when (compile)
  (require 'emacspeak-fix-interactive))
(require 'emacspeak-sounds)
;;}}}
;;{{{ Introduction:

;;; Commentary:

;;; Assuming you have a correctly configured RealAudio
;;; player, this package provides single click access to
;;; starting and stopping a RealAudio stream from anywhere
;;; on the Emacspeak desktop.  Before using this package,
;;; make sure that your realaudio player works outside
;;; Emacs. Then set variable Emacspeak-relaudio-player to
;;; point to the program you use to play RealAudio streams.

;;; Emacs 19 note:
;;; This module does not work on Emacs 19 due to an
;;; incompatibility in function completing-read

;;; Code:


;;}}}
;;{{{ variables

(defvar emacspeak-realaudio-player
  (if (eq window-system 'w32)
      "shelex"
    "rap")
  "*Executable that plays relaudio")

(defvar emacspeak-realaudio-process nil
  "Process handle to running player")
(defvar emacspeak-realaudio-last-url nil
  "Records the last RealAudio resource we played")
(defvar emacspeak-realaudio-history nil
  "History list holding resources we played recently")
(defvar emacspeak-realaudio-shortcuts-directory 
(expand-file-name "realaudio" emacspeak-dir)
"*Directory where we keep realaudio shortcuts.
I typically keep .ram --RealAudio metafiles-- in this
directory.
Realaudio metafiles typically contain a single line that
specifies the actual location of the realaudio stream
--typically the .ra file.")

;;}}}
;;{{{ commands


(defun emacspeak-realaudio-play (resource)
  "Play a realaudio stream.  Uses files from your Realaudio
shortcuts directory for completion.  See documentation for
user configurable variable
emacspeak-realaudio-shortcuts-directory"
  (interactive
   (list
    (let ((completion-ignore-case t))
    (completing-read
     "Realaudio Resource:"
     (loop for f in
           (cddr (directory-files  emacspeak-realaudio-shortcuts-directory t))
           collect (cons f f ))
     nil
     nil
     ""                                 ;initial content
     'emacspeak-realaudio-history
     emacspeak-realaudio-last-url))))
  (declare (special emacspeak-realaudio-player emacspeak-realaudio-process
                    emacspeak-realaudio-shortcuts-directory
                    emacspeak-realaudio-history
                    emacspeak-use-auditory-icons))
  (unless (string= resource (car emacspeak-realaudio-history))
    (push resource emacspeak-realaudio-history))
  (when (and emacspeak-realaudio-process
             (eq 'run (process-status emacspeak-realaudio-process)))
    (kill-process emacspeak-realaudio-process))
  (setq emacspeak-realaudio-process
        (start-process"realaudio" " realaudio"
                      emacspeak-realaudio-player
                      resource ))
  (unless (eq 'run (process-status
                    emacspeak-realaudio-process))
    (message "Failed to start RealAudio"))
  (message "Playing  Realaudio stream %s" resource)
  (setq emacspeak-realaudio-last-url resource)
  (when emacspeak-use-auditory-icons
(emacspeak-toggle-auditory-icons t)))


(defun emacspeak-realaudio-play-url-at-point ()
  "Play url under point as realaudio"
  (interactive)
  (let ((url (w3-view-this-url 'no-show)))
    (cond
     ((string-match ".ram?$" url)
      (message "Playing Realaudio URL under point")
        (emacspeak-realaudio-play url))
      (t (message "%s does not look like realaudio"
             url)))))

(defun emacspeak-realaudio-stop ()
  "Stop playing realaudio"
  (interactive)
  (declare (special emacspeak-realaudio-process))
  (kill-process emacspeak-realaudio-process)
  (message "Stopped RealAudio")
  (emacspeak-toggle-auditory-icons t))

(defun emacspeak-realaudio  ()
  "Start or stop playing RealAudio.  Stops playing realaudio
if there is a stream currently playing.  Otherwise, prompts
for a realaudio resource.  Realaudio resources can be
specified either as a Realaudio URL, the location of a local
Realaudio file, or as the name of a local Realaudio
metafile. Realaudio resources you have played in this
session are available in the minibuffer history.  The
default is to play the resource you played most
recently. Emacspeak uses the contents of the directory
specified by variable
emacspeak-realaudio-shortcuts-directory to offer a set of
completions. Hit tab to use this completion list."
  (interactive)
  (declare (special emacspeak-realaudio-process))
  (cond
   ((and emacspeak-realaudio-process
         (eq 'run (process-status emacspeak-realaudio-process)))
    (emacspeak-realaudio-stop))
   (t  (call-interactively 'emacspeak-realaudio-play))))

;;}}}
;;{{{ W3 hook

(add-hook 'w3-mode-hook
          (function
           (lambda nil
             (declare (special w3-mode-map))
             (define-key w3-mode-map "\M-r" 'emacspeak-realaudio-play-url-at-point)
             (define-key w3-mode-map "\M-d" 'emacspeak-realaudio-stop))))

;;}}}
(provide 'emacspeak-realaudio)
;;{{{ end of file 

;;; local variables:
;;; folded-file: t
;;; byte-compile-dynamic: t
;;; end: 

;;}}}
