/**
 *
 * $Id: Visual.c,v 1.10 1998/03/24 23:01:13 rwscott Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: Visual.c,v 1.10 1998/03/24 23:01:13 rwscott Exp $";

#include <LTconfig.h>
#include <XmI/XmI.h>
#include <stdio.h>
#include <limits.h>

#include <Xm/XmP.h>
#include <Xm/PrimitiveP.h>
#include <Xm/ManagerP.h>
#include <Xm/ExtObjectP.h>
#include <Xm/GadgetP.h>

#include <XmI/DebugUtil.h>

static XmColorProc _color_proc = NULL;

#define COLORED_SHADOWS	1

/*
 *  Note: JAC speaks:
 *  When displaying remotely, profile data indicates that the vast
 *  majority of the time spent in _XtCreateWidget is spent getting colors
 *  from here.  Therefore, I've added a cache to each of these routines so
 *  that it will remember the data from the last call and avoid server round
 *  trips if they are called again under the same conditions.  These
 *  changes speeded up _XtCreateWidget by 10 *times* or 90%.
 */

static void _XmColorProcDefaultProc(XColor *bg_color, XColor *fg_color,
				    XColor *sel_color, XColor *ts_color,
				    XColor *bs_color);

/*
 * forward declarations
 */
void __XmRGB2HSV(unsigned short r,
		 unsigned short g,
		 unsigned short b,
		 double *hr,
		 double *sr,
		 double *vr);

XmColorProc
XmSetColorCalculation(XmColorProc proc)
{
    XmColorProc ret_val;

    ret_val = _color_proc;

    if (proc == NULL)
    {
	_color_proc = _XmColorProcDefaultProc;
    }
    else
    {
	_color_proc = proc;
    }

    return ret_val;
}

XmColorProc
XmGetColorCalculation()
{
    return _color_proc;
}

void
XmGetColors(Screen *screen,
	    Colormap color_map,
	    Pixel background,
	    Pixel *foreground_ret,
	    Pixel *top_shadow_ret,
	    Pixel *bottom_shadow_ret,
	    Pixel *select_ret)
{
    static XColor background_xcolor;
    static XColor foreground_xcolor;
    static XColor top_shadow_xcolor;
    static XColor bottom_shadow_xcolor;
    static XColor select_xcolor;
    static int inited = False;
    static Pixel last_background;
    static Screen *last_screen;
    static Colormap last_color_map;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *5* server round trips if we are repeating the last
     * request which is actually the most common use here--most widgets
     * have the same foreground/background/tsc/bsc.
     */
    /* check cache */
    if (inited && 
	background == last_background && 
	color_map == last_color_map && 
	screen == last_screen)
    {
	if (foreground_ret)
	{
	    *foreground_ret = foreground_xcolor.pixel;
	}
	if (select_ret)
	{
	    *select_ret = select_xcolor.pixel;
	}
	if (top_shadow_ret)
	{
	    *top_shadow_ret = top_shadow_xcolor.pixel;
	}
	if (bottom_shadow_ret)
	{
	    *bottom_shadow_ret = bottom_shadow_xcolor.pixel;
	}

	return;
    }
    else
    {
	inited = True;
	last_background = background;
	last_color_map = color_map;
	last_screen = screen;
    }

    background_xcolor.pixel = background;

    XQueryColor(XDisplayOfScreen(screen),
		color_map,
		&background_xcolor);

    _color_proc(&background_xcolor,
		&foreground_xcolor,
		&select_xcolor,
		&top_shadow_xcolor,
		&bottom_shadow_xcolor);

    if (foreground_ret)
    {
	XAllocColor(XDisplayOfScreen(screen),
		    color_map,
		    &foreground_xcolor);

	*foreground_ret = foreground_xcolor.pixel;
    }
    if (select_ret)
    {
	XAllocColor(XDisplayOfScreen(screen),
		    color_map,
		    &select_xcolor);

	*select_ret = select_xcolor.pixel;
    }
    if (top_shadow_ret)
    {
	XAllocColor(XDisplayOfScreen(screen),
		    color_map,
		    &top_shadow_xcolor);

	*top_shadow_ret = top_shadow_xcolor.pixel;
    }
    if (bottom_shadow_ret)
    {
	XAllocColor(XDisplayOfScreen(screen),
		    color_map,
		    &bottom_shadow_xcolor);

	*bottom_shadow_ret = bottom_shadow_xcolor.pixel;
    }
}

void
XmChangeColor(Widget widget,
	      Pixel background)
{
    Pixel foreground;
    Pixel top_shadow;
    Pixel bottom_shadow;
    Pixel select;
    Colormap color_map;

    color_map = ColormapOfObject(widget);

    XmGetColors(XtScreenOfObject(widget),
		color_map,
		background,
		&foreground,
		&top_shadow,
		&bottom_shadow,
		&select);

    XtVaSetValues(widget,
		  XmNbackground, background,
		  XmNforeground, foreground,
		  XmNtopShadowColor, top_shadow,
		  XmNbottomShadowColor, bottom_shadow,
		  XmNarmColor, select,
		  NULL);
}

/* the default color calculation */
static void
_XmColorProcDefaultProc(XColor *bg_color,
			XColor *fg_color,
			XColor *sel_color,
			XColor *ts_color,
			XColor *bs_color)
{
    double h, s, v;
    XColor _widgetBackground;

    _widgetBackground = *bg_color;

    __XmRGB2HSV(_widgetBackground.red,
		_widgetBackground.green,
		_widgetBackground.blue,
		&h, &s, &v);

    if (v < 0.5)
    {
	fg_color->blue = 0xFFFF;
	fg_color->green = 0xFFFF;
	fg_color->red = 0xFFFF;
    }
    else
    {
	fg_color->blue = 0x0;
	fg_color->green = 0x0;
	fg_color->red = 0x0;
    }

    /* FIX THESE -- they're the same as the bottom shadow color */
    sel_color->blue = _widgetBackground.blue * .7;
    sel_color->green = _widgetBackground.green * .7;
    sel_color->red = _widgetBackground.red * .7;

    if (_widgetBackground.red > 65535 / 1.5)
    {
	ts_color->red = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	ts_color->red = _widgetBackground.red * 1.5;
#else
	ts_color->red = 65535 * 0.75;
#endif
    }

    if (_widgetBackground.green > 65535 / 1.5)
    {
	ts_color->green = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	ts_color->green = _widgetBackground.green * 1.5;
#else
	ts_color->green = 65535 * 0.75;
#endif
    }

    if (_widgetBackground.blue > 65535 / 1.5)
    {
	ts_color->blue = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	ts_color->blue = _widgetBackground.blue * 1.5;
#else
	ts_color->blue = 65535 * 0.75;
#endif
    }

    bs_color->blue = _widgetBackground.blue * .5;
    bs_color->green = _widgetBackground.green * .5;
    bs_color->red = _widgetBackground.red * .5;
}

/****************************************/
/* private functions for default colors */
/****************************************/

#define FLOOR(x) ((int)((x) - 0.5) > (x) ? (x) : (x)-1)

/* Convert between RGB and HSV */
void
__XmRGB2HSV(unsigned short r,
	    unsigned short g,
	    unsigned short b,
	    double *hr,
	    double *sr,
	    double *vr)
{
    double rd, gd, bd, max, min, del;
    double rc, gc, bc;
    double v, h, s;

    /* convert RGB to HSV */
    rd = r / 65535.0;		/* rd,gd,bd range 0-1 instead of 0-65535 */
    gd = g / 65535.0;
    bd = b / 65535.0;

    /* compute maximum of rd,gd,bd */
    if (rd >= gd)
    {
	if (rd >= bd)
	{
	    max = rd;
	}
	else
	{
	    max = bd;
	}
    }
    else
    {
	if (gd >= bd)
	{
	    max = gd;
	}
	else
	{
	    max = bd;
	}
    }

    /* compute minimum of rd,gd,bd */
    if (rd <= gd)
    {
	if (rd <= bd)
	{
	    min = rd;
	}
	else
	{
	    min = bd;
	}
    }
    else
    {
	if (gd <= bd)
	{
	    min = gd;
	}
	else
	{
	    min = bd;
	}
    }


    del = max - min;
    v = max;
    if (max != 0.0)
    {
	s = (del) / max;
    }
    else
    {
	s = 0.0;
    }

    h = -1;
    if (s != 0.0)
    {
	rc = (max - rd) / del;
	gc = (max - gd) / del;
	bc = (max - bd) / del;

	if (rd == max)
	{
	    h = bc - gc;
	}
	else if (gd == max)
	{
	    h = 2 + rc - bc;
	}
	else if (bd == max)
	{
	    h = 4 + gc - rc;
	}

	h = h * 60;
	if (h < 0)
	    h += 360;
    }

    *hr = h;
    *sr = s;
    *vr = v;
}

/* Convert between HSV and RGB */
void
_XmHSV2RGB(double h,
	   double s,
	   double v,
	   unsigned short *rr,
	   unsigned short *gr,
	   unsigned short *br)
{
    int j;
    double rd, gd, bd;
    double f, p, q, t;

    /* convert HSV back to RGB */
    if (h == -1 || s == 0.0)
    {
	rd = v;
	gd = v;
	bd = v;
    }
    else
    {
	if (h == 360.0)
	{
	    h = 0.0;
	}
	h = h / 60.0;
	j = (int)FLOOR(h);
	if (j < 0)
	{
	    j = 0;		/* either h or floor seem to go neg on some sys */
	}
	f = h - j;
	p = v * (1 - s);
	q = v * (1 - (s * f));
	t = v * (1 - (s * (1 - f)));

	switch (j)
	{
	case 0:
	    rd = v;
	    gd = t;
	    bd = p;
	    break;
	case 1:
	    rd = q;
	    gd = v;
	    bd = p;
	    break;
	case 2:
	    rd = p;
	    gd = v;
	    bd = t;
	    break;
	case 3:
	    rd = p;
	    gd = q;
	    bd = v;
	    break;
	case 4:
	    rd = t;
	    gd = p;
	    bd = v;
	    break;
	case 5:
	    rd = v;
	    gd = p;
	    bd = q;
	    break;
	default:
	    rd = v;
	    gd = t;
	    bd = p;
	    break;		/* never happen */
	}
    }


    *rr = (unsigned short)FLOOR((rd * 65535.0) + 0.5);
    *gr = (unsigned short)FLOOR((gd * 65535.0) + 0.5);
    *br = (unsigned short)FLOOR((bd * 65535.0) + 0.5);
}


void
_XmForegroundColorDefault(Widget w,
			  int offset,
			  XrmValue *val)
{
    double h, s, v;
    XColor _widgetBackground;
    Colormap colormap;
    static XColor foregroundColor;
    static int inited = False;
    static Pixel last_background;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *2* server round trips if we are repeating the last
     * request which is actually the most common use here--most widgets
     * have the same foreground.
     */
    if (XmIsGadget(w))
	_widgetBackground.pixel = XmParentBackground(w);
    else
	_widgetBackground.pixel = XtBackground(w);
    colormap = ColormapOfObject(w);

    /* check cache */
    if (inited && 
	_widgetBackground.pixel == last_background &&
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&foregroundColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_background = _widgetBackground.pixel;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&_widgetBackground);

    __XmRGB2HSV(_widgetBackground.red,
		_widgetBackground.green,
		_widgetBackground.blue,
		&h, &s, &v);

    if (v < 0.5)
	foregroundColor.pixel = _XmWhitePixelOfObject(w);
    else
	foregroundColor.pixel = _XmBlackPixelOfObject(w);

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&foregroundColor);

    val->addr = (XtPointer)&foregroundColor.pixel;
}

void
_XmHighlightColorDefault(Widget w,
			 int offset,
			 XrmValue *val)
{
    double h, s, v;
    XColor _widgetBackground;
    Colormap colormap;
    Screen *screen;
    static XColor highlightColor;
    static int inited = False;
    static Pixel last_background;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *2* server round trips if we are repeating the last
     * request which is actually the most common use here--most widgets
     * have the same highlight color.
     */
    if (XmIsGadget(w))
	_widgetBackground.pixel = XtBackground(XtParent(w));
    else
	_widgetBackground.pixel = XtBackground(w);
    colormap = ColormapOfObject(w);

    /* check cache */
    if (inited && 
	_widgetBackground.pixel == last_background &&
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&highlightColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_background = _widgetBackground.pixel;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&_widgetBackground);

    __XmRGB2HSV(_widgetBackground.red,
		_widgetBackground.green,
		_widgetBackground.blue,
		&h, &s, &v);

    screen = XtScreenOfObject(w);
    if (v < 0.5)
	highlightColor.pixel = _XmWhitePixelOfObject(w);
    else
	highlightColor.pixel = _XmBlackPixelOfObject(w);
       
    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&highlightColor);

    val->addr = (XtPointer)&highlightColor.pixel;
}

void
_XmBackgroundColorDefault(Widget w,
			  int offset,
			  XrmValue *val)
{
    Colormap colormap = CoreColormap(w);
    static XColor backgroundColor;
    static int inited = False;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid up to 2 server round trips if we are repeating the
     * last request which is actually the most common use here--most widgets
     * have the same background.
     */
    /* check cache */
    if (inited && 
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&backgroundColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    if (!XParseColor(XtDisplayOfObject(w),
		     colormap,
		     "rgb:72/9f/ff",
		     &backgroundColor))
    {
	backgroundColor.pixel = _XmWhitePixelOfObject(w);
    }
    else
    {
	XAllocColor(XtDisplayOfObject(w),
		    colormap,
		    &backgroundColor);
    }

    val->addr = (XtPointer)&backgroundColor.pixel;
}


void
_XmTopShadowColorDefault(Widget w,
			 int offset,
			 XrmValue *val)
{
    XColor _widgetBackground;
    Colormap colormap;
    static XColor topShadowColor;
    static int inited = False;
    static Pixel last_background;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *2* server round trips if we are repeating the last
     * request which is actually the most common use here--most widgets
     * have the same tsc.
     */
    if (XmIsGadget(w))
	_widgetBackground.pixel = XtBackground(XtParent(w));
    else
	_widgetBackground.pixel = XtBackground(w);
    colormap = ColormapOfObject(w);

    /* check cache */
    if (inited && 
	_widgetBackground.pixel == last_background &&
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&topShadowColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_background = _widgetBackground.pixel;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&_widgetBackground);

    if (_widgetBackground.red > 65535 / 1.5)
    {
	topShadowColor.red = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	topShadowColor.red = _widgetBackground.red * 1.5;
#else
	topShadowColor.red = 65535 * 0.75;
#endif
    }

    if (_widgetBackground.green > 65535 / 1.5)
    {
	topShadowColor.green = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	topShadowColor.green = _widgetBackground.green * 1.5;
#else
	topShadowColor.green = 65535 * 0.75;
#endif
    }

    if (_widgetBackground.blue > 65535 / 1.5)
    {
	topShadowColor.blue = 65535;
    }
    else
    {
#if COLORED_SHADOWS
	topShadowColor.blue = _widgetBackground.blue * 1.5;
#else
	topShadowColor.blue = 65535 * 0.75;
#endif
    }


    if (!XAllocColor(XtDisplayOfObject(w),
		     colormap,
		     &topShadowColor))
    {
	topShadowColor.pixel = _XmWhitePixelOfObject(w);
    }

    val->addr = (XtPointer)&topShadowColor.pixel;
}


void
_XmBottomShadowColorDefault(Widget w,
			    int offset,
			    XrmValue *val)
{
    XColor _widgetBackground;
    Colormap colormap;
    static XColor bottomShadowColor;
    static int inited = False;
    static Pixel last_background;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *2* server round trips if we are repeating the last
     * request which is actually the most common use here--most widgets
     * have the same bsc.
     */
    if (XmIsGadget(w))
	_widgetBackground.pixel = XtBackground(XtParent(w));
    else
	_widgetBackground.pixel = XtBackground(w);
    colormap = ColormapOfObject(w);

    /* check cache */
    if (inited && 
	_widgetBackground.pixel == last_background &&
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&bottomShadowColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_background = _widgetBackground.pixel;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&_widgetBackground);

#if COLORED_SHADOWS
    bottomShadowColor.blue = _widgetBackground.blue * .5;
    bottomShadowColor.green = _widgetBackground.green * .5;
    bottomShadowColor.red = _widgetBackground.red * .5;
#else
    bottomShadowColor.blue = 65535 * 0.25;
    bottomShadowColor.green = 65535 * 0.25;
    bottomShadowColor.red = 65535 * 0.25;
#endif

    if (!XAllocColor(XtDisplayOfObject(w),
		     colormap,
		     &bottomShadowColor))
    {
	bottomShadowColor.pixel = _XmWhitePixelOfObject(w);
    }

    val->addr = (XtPointer)&bottomShadowColor.pixel;
}

/*
 * conversion args for the pixmap converters.  Thanks to Gilles Seguin
 * for pointing out the lack.
 */
#define POFFSET(field) \
    (XtPointer)XtOffsetOf(XmPrimitiveRec, field)
#define MOFFSET(field) \
    (XtPointer)XtOffsetOf(XmManagerRec, field)

/* background for everyone */
static XtConvertArgRec backgroundArgs[] =
{
    {
	XtBaseOffset, (XtPointer)XtOffsetOf(CoreRec, core.self),
	sizeof(WidgetRec *)
    }
};


/* primitive foreground */
static XtConvertArgRec PrimForegroundPixmapArgs[] =
{
    {
	XtBaseOffset, POFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtBaseOffset, POFFSET(primitive.foreground), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.depth), sizeof(Cardinal)
    }
};

/* primitive highlight */
static XtConvertArgRec PrimHighlightPixmapArgs[] =
{
    {
	XtBaseOffset, POFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtBaseOffset, POFFSET(primitive.highlight_color), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.depth), sizeof(Cardinal)
    }
};

/* primitive topShadow */
static XtConvertArgRec PrimTopShadowPixmapArgs[] =
{
    {
	XtBaseOffset, POFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtBaseOffset, POFFSET(primitive.highlight_color), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.depth), sizeof(Cardinal)
    }
};

/* primitive bottomShadow */
static XtConvertArgRec PrimBottomShadowPixmapArgs[] =
{
    {
	XtBaseOffset, POFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtBaseOffset, POFFSET(primitive.bottom_shadow_color), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, POFFSET(core.depth), sizeof(Cardinal)
    }
};

/* manager foreground */
static XtConvertArgRec ManForegroundPixmapArgs[] =
{
    {
	XtWidgetBaseOffset, MOFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtWidgetBaseOffset, MOFFSET(manager.foreground), sizeof(Pixel)
    },
    {
	XtWidgetBaseOffset, MOFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtWidgetBaseOffset, MOFFSET(core.depth), sizeof(Cardinal)
    }
};

/* manager highlight */
static XtConvertArgRec ManHighlightPixmapArgs[] =
{
    {
	XtWidgetBaseOffset, MOFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtWidgetBaseOffset, MOFFSET(manager.highlight_color), sizeof(Pixel)
    },
    {
	XtWidgetBaseOffset, MOFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, MOFFSET(core.depth), sizeof(Cardinal)
    }
};

/* manager topShadow */
static XtConvertArgRec ManTopShadowPixmapArgs[] =
{
    {
	XtWidgetBaseOffset, MOFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtWidgetBaseOffset, MOFFSET(manager.top_shadow_color), sizeof(Pixel)
    },
    {
	XtWidgetBaseOffset, MOFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, MOFFSET(core.depth), sizeof(Cardinal)
    }
};

/* manager bottomShadow */
static XtConvertArgRec ManBottomShadowPixmapArgs[] =
{
    {
	XtWidgetBaseOffset, MOFFSET(core.screen), sizeof(Screen *)
    },
    {
	XtWidgetBaseOffset, MOFFSET(manager.bottom_shadow_color), sizeof(Pixel)
    },
    {
	XtWidgetBaseOffset, MOFFSET(core.background_pixel), sizeof(Pixel)
    },
    {
	XtBaseOffset, MOFFSET(core.depth), sizeof(Cardinal)
    }
};

/* gadget pixmap */
static XtConvertArgRec gadgetPixmapArgs[] =
{
    {
	XtBaseOffset, (XtPointer)XtOffsetOf(XmGadgetRec, object.parent),
	sizeof(WidgetRec *)
    }
};

static char *background_pixmap_name = NULL;

char *
_XmGetBGPixmapName()
{
    return background_pixmap_name;
}

void
_XmClearBGPixmap( void )
{
    background_pixmap_name = NULL;
}

static Boolean
CvtStringToBackgroundPixmap(Display *display,
			    XrmValuePtr args, Cardinal *num_args,
			    XrmValuePtr fromVal, XrmValuePtr toVal,
			    XtPointer *converter_data)
{
    static Pixmap buf;

    DEBUGOUT(XdbDebug(__FILE__, NULL,
		      "CvtStringToBackgroundPixmap: %s\n",
		      fromVal->addr));

    if (!XtIsShell((Widget)args[0].addr))
    {
	background_pixmap_name = (String)fromVal->addr;
    }

    if (toVal->addr != NULL)
    {
	if( toVal->size < sizeof(Pixmap))
	{
	    toVal->size = sizeof(Pixmap);

	    return False;
	}
	else
	{
	    *((Pixmap *)(toVal->addr)) = XmUNSPECIFIED_PIXMAP;
	    toVal->size = sizeof(Pixmap);

	    return True;
	}
    }
    else
    {
	buf = XmUNSPECIFIED_PIXMAP;
	toVal->addr = (XtPointer)&buf;
	toVal->size = sizeof(Pixmap);

	return True;
    }
}

static void
BuildPixmap(int ncargs, char *which,
	    XrmValuePtr args, Cardinal *num_args,
	    XrmValuePtr fromVal, XrmValuePtr toVal)
{
    static Pixmap pixmap;
    Screen *screen;
    Pixel foreground;
    Pixel background;
    Cardinal depth;
    char *name = fromVal->addr;

    pixmap = XmUNSPECIFIED_PIXMAP;

    toVal->addr = (char *)&pixmap;
    toVal->size = sizeof(Pixmap);

    if (*num_args != ncargs)
    {
	XtWarningMsg("wrongParameters", which, "XmToolkitError",
		     "String to Pixmap conversion needs four argument",
		     (String *)NULL, (Cardinal *)NULL);

	return;
    }

    if (strcmp(name, "unspecified_pixmap") == 0)
    {
	return;
    }

    screen = *((Screen **)args[0].addr);
    foreground = *((Pixel *)args[1].addr);
    background = *((Pixel *)args[2].addr);
    depth = *((Cardinal *)args[3].addr);

    pixmap = XmGetPixmapByDepth(screen, name,
				foreground, background, depth);
}

void
_XmCvtStringToPrimForegroundPixmap(XrmValuePtr args, Cardinal *num_args,
				   XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToPrimForegroundPixmap\n"));

    BuildPixmap(XtNumber(PrimForegroundPixmapArgs),
		"_XmCvtStringToPrimForegroundPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToPrimHighlightPixmap(XrmValuePtr args, Cardinal *num_args,
				  XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToPrimHighlightPixmap\n"));

    BuildPixmap(XtNumber(PrimHighlightPixmapArgs),
		"_XmCvtStringToPrimHighlightPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToPrimTopShadowPixmap(XrmValuePtr args, Cardinal *num_args,
				  XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToPrimTopShadowPixmap\n"));

    BuildPixmap(XtNumber(PrimTopShadowPixmapArgs),
		"_XmCvtStringToPrimTopShadowPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToPrimBottomShadowPixmap(XrmValuePtr args, Cardinal *num_args,
				     XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToPrimBottomShadowPixmap\n"));

    BuildPixmap(XtNumber(PrimBottomShadowPixmapArgs),
		"_XmCvtStringToPrimBottomShadowPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToManForegroundPixmap(XrmValuePtr args, Cardinal *num_args,
				  XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToManForegroundPixmap\n"));

    BuildPixmap(XtNumber(ManForegroundPixmapArgs),
		"_XmCvtStringToManForegroundPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToManHighlightPixmap(XrmValuePtr args, Cardinal *num_args,
				 XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToManHighlightPixmap\n"));

    BuildPixmap(XtNumber(ManHighlightPixmapArgs),
		"_XmCvtStringToManHighlightPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToManTopShadowPixmap(XrmValuePtr args, Cardinal *num_args,
				 XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToManTopShadowPixmap\n"));

    BuildPixmap(XtNumber(ManTopShadowPixmapArgs),
		"_XmCvtStringToManTopShadowPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToManBottomShadowPixmap(XrmValuePtr args, Cardinal *num_args,
				    XrmValuePtr fromVal, XrmValuePtr toVal)
{
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToManBottomShadowPixmap\n"));

    BuildPixmap(XtNumber(ManBottomShadowPixmapArgs),
		"_XmCvtStringToManBottomShadowPixmap",
		args, num_args, fromVal, toVal);
}

void
_XmCvtStringToGadgetPixmap(XrmValuePtr args, Cardinal *num_args,
			   XrmValuePtr fromVal, XrmValuePtr toVal)
{
    XrmValue pargs[4];
    Cardinal pnargs = 4;
    Widget parent;

    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmCvtStringToGadgetPixmap\n"));

    parent = *((Widget *)args[0].addr);

    pargs[0].addr = (XPointer)&CoreScreen(parent);
    pargs[1].addr = (XPointer)&CoreBackground(parent);
    pargs[2].addr = (XPointer)&MGR_Foreground(parent);
    pargs[3].addr = (XPointer)&CoreDepth(parent);

    BuildPixmap(pnargs, "_XmCvtStringToGadgetPixmap",
		pargs, &pnargs, fromVal, toVal);
}

void
_XmRegisterPixmapConverters(void)
{
    static inited = False;
  
    DEBUGOUT(XdbDebug(__FILE__, NULL, "_XmRegisterPixmapConverters\n"));

    if (inited)
    {
	return;
    }

    inited = True;

    XtSetTypeConverter(XtRString, XmRXmBackgroundPixmap,
                       CvtStringToBackgroundPixmap, backgroundArgs,
		       XtNumber(backgroundArgs), XtCacheNone, NULL);
    XtSetTypeConverter(XtRString, XmRPixmap,
                       CvtStringToBackgroundPixmap, backgroundArgs,
		       XtNumber(backgroundArgs), XtCacheNone, NULL);

    XtAddConverter(XtRString, XmRPrimForegroundPixmap,
                   _XmCvtStringToPrimForegroundPixmap, PrimForegroundPixmapArgs,
		   XtNumber(PrimForegroundPixmapArgs));
    XtAddConverter(XtRString,  XmRHighlightPixmap,
                   _XmCvtStringToPrimHighlightPixmap, PrimHighlightPixmapArgs,
		   XtNumber(PrimHighlightPixmapArgs));
    XtAddConverter(XtRString, XmRTopShadowPixmap,
                   _XmCvtStringToPrimTopShadowPixmap, PrimTopShadowPixmapArgs,
		   XtNumber(PrimTopShadowPixmapArgs));
    XtAddConverter(XtRString,  XmRBottomShadowPixmap,
                   _XmCvtStringToPrimBottomShadowPixmap,
		   PrimBottomShadowPixmapArgs,
		   XtNumber(PrimBottomShadowPixmapArgs));

    XtAddConverter(XtRString,  XmRManForegroundPixmap,
                   _XmCvtStringToManForegroundPixmap, ManForegroundPixmapArgs,
		   XtNumber(ManForegroundPixmapArgs));
    XtAddConverter(XtRString,  XmRManHighlightPixmap,
                   _XmCvtStringToManHighlightPixmap, ManHighlightPixmapArgs,
		   XtNumber(ManHighlightPixmapArgs));
    XtAddConverter(XtRString,  XmRManTopShadowPixmap,
                   _XmCvtStringToManTopShadowPixmap, ManTopShadowPixmapArgs,
		   XtNumber(ManTopShadowPixmapArgs));
    XtAddConverter(XtRString,  XmRManBottomShadowPixmap,
                   _XmCvtStringToManBottomShadowPixmap,
		   ManBottomShadowPixmapArgs,
		   XtNumber(ManBottomShadowPixmapArgs));

    XtAddConverter(XtRString,  XmRGadgetPixmap,
                   _XmCvtStringToGadgetPixmap, gadgetPixmapArgs,
		   XtNumber(gadgetPixmapArgs));

#if 0
    XtSetTypeConverter(XtRString,  XmRAnimationMask,
		       CvtStringToAnimationMask,
                       backgroundArgs, XtNumber(backgroundArgs),
		       XtCacheNone, NULL);

    XtSetTypeConverter(XtRString, XmRAnimationPixmap,
		       CvtStringToAnimationPixmap,
                       backgroundArgs, XtNumber(backgroundArgs),
		       XtCacheNone, NULL);
#endif
}

/*
 * XmP.h says this is here
 */
void
_XmSelectColorDefault(Widget w,
		      int offset,
		      XrmValue *val)
{
    XColor _widgetBackground;
    Colormap colormap;
    static XColor selectColor;
    static int inited = False;
    static Pixel last_background;
    static Colormap last_colormap;
    static Display *last_display;

    /* 
     * JAC has modified this routine so that it will remember the previous
     * colors and avoid *2* server round trips if we are repeating the last
     * request which is actually the most common use here--most toggles
     * have the same select color.
     */
    if (XmIsGadget(w))
	_widgetBackground.pixel = XmParentBackground(w);
    else if (XmIsExtObject(w))
	_widgetBackground.pixel = XmParentBackground(w);
    else
	_widgetBackground.pixel = XtBackground(w);
    colormap = ColormapOfObject(w);

    /* check cache */
    if (inited && 
	_widgetBackground.pixel == last_background &&
	last_colormap == colormap &&
	last_display == XtDisplayOfObject(w))
    {
	val->addr = (XtPointer)&selectColor.pixel;
	return;
    }
    else
    {
	inited = True;
	last_background = _widgetBackground.pixel;
	last_colormap = colormap;
	last_display = XtDisplayOfObject(w);
    }

    XQueryColor(XtDisplayOfObject(w),
		colormap,
		&_widgetBackground);

    /* FIX THESE -- they're the same as the bottom shadow color */
#if COLORED_SHADOWS
    selectColor.blue = _widgetBackground.blue * .7;
    selectColor.green = _widgetBackground.green * .7;
    selectColor.red = _widgetBackground.red * .7;
#else
    selectColor.blue = 65535 * .5;
    selectColor.green = 65535 * .7;
    selectColor.red = 65535 * .7;
#endif

    if (!XAllocColor(XtDisplayOfObject(w),
		     colormap,
		     &selectColor))
    {
	selectColor.pixel = _XmWhitePixelOfObject(w);
    }


    val->addr = (XtPointer)&selectColor.pixel;
}

/*
 * These functions (_XmBlackPixel and _XmWhitePixel) appear in
 * libXm:Obso2_0.o in the Motif 2 libraries.  The prototypes are in
 * Xm/XmP.h but there were no implementations.
 *
 * The final XColor parameter makes no sense to me at all but the Motif
 * headers on HP-UX 9.05 specify a raw XColor (as opposed to the more
 * sensible XColor*) too so the signatures are correct.
 *
 * I don't know the proper behavior of these functions so I'm guessing.
 *
 *	--mu@echo-on.net, 1998.03.15
 */
Pixel
_XmWhitePixel(Screen *screen, Colormap colormap, XColor whitecolor)
{
    whitecolor.red   =
    whitecolor.green =
    whitecolor.blue  = USHRT_MAX;

    if (colormap == DefaultColormapOfScreen(screen) ||
	!XAllocColor(DisplayOfScreen(screen), colormap, &whitecolor))
    {
	whitecolor.pixel = WhitePixelOfScreen(screen);
    }
    return whitecolor.pixel;
}

Pixel
_XmBlackPixel(Screen *screen, Colormap colormap, XColor blackcolor)
{
    blackcolor.red   =
    blackcolor.green =
    blackcolor.blue  = 0;

    if (colormap == DefaultColormapOfScreen(screen) ||
	!XAllocColor(DisplayOfScreen(screen), colormap, &blackcolor))
    {
	blackcolor.pixel = BlackPixelOfScreen(screen);
    }
    return blackcolor.pixel;
}

/*
 * More convenient forms of _XmWhitePixel and _XmBlackPixel.
 * These can't be macros because the stupid last parameter is a raw structure
 * rather than a pointer.
 */
Pixel
_XmWhitePixelOfObject(Widget w)
{
    XColor unused;
    return _XmWhitePixel(XtScreenOfObject(w), ColormapOfObject(w), unused);
}

Pixel
_XmBlackPixelOfObject(Widget w)
{
    XColor unused;
    return _XmBlackPixel(XtScreenOfObject(w), ColormapOfObject(w), unused);
}
