/*
 * METALBASE 5.1
 *
 * Released January 1st, 1993 by Huan-Ti [ t-richj@microsoft.com ]
 *
 *
 */

#include <mbase.h>
#include "internal.h"


/*
 * PROTOTYPES -----------------------------------------------------------------
 *
 */

   void     main           XARGS( (int, char **) );
   bool     parseArgs      XARGS( (int, char **) );
   bool     GetYN          XARGS( (char *) );
   bool     checkLock      XARGS( (char *) );
   bool     checkDamage    XARGS( (void) );
   bool     checkIndex     XARGS( (void) );

   bool     listGenerate   XARGS( (void) );
   bool     listSort       XARGS( (void) );
   bool     treeBuild      XARGS( (void) );
   void     fillBuffer     XARGS( (dataptr, long) );
   void     sectionSort    XARGS( (dataptr, dataptr, long, long, long, bool) );
   long     nodeBuild      XARGS( (long, long, long, int) );

   bool     dataGenerate   XARGS( (void) );
   bool     fieldCopy      XARGS( (long, int) );
   bool     dataCopy       XARGS( (file, file, long) );
   bool     finalCopy      XARGS( (file, file) );
   void     finalData      XARGS( (file, file) );


/*
 * VARIABLES ------------------------------------------------------------------
 *
 */

   relation *rel = RNULL;

   bool      fAsk      = TRUE;
   bool      fFix      = FALSE;
   bool      fIndex    = FALSE;
   bool      fUnlock   = FALSE;
   bool      fRelink   = FALSE;
   bool      fOverride = FALSE;
   bool      fDatafile = FALSE;

   file      fhList = 0;
   file      fhTemp = 0;
   char      tempname[128];
   char      listname[128];
   bool      fUseLow;

   int       idx;
   long      nRec;
   long      nFail;
   long      posNext;

   int       nPass, nPassMax;


/*
 * PRIMARY ROUTINES -----------------------------------------------------------
 *
 */

void
main  (argc, argv)
int    argc;
char **argv;
{
   char  stat;
   long  i;
   int   fld, per, last;
   bool  rc;

   setbuf (stdout, NULL);
   setbuf (stderr, NULL);

/*
 * Parse the command-line
 *
 */

   if (! parseArgs (argc, argv))
      {
      fprintf (stderr, "format: mbdiag [-u,-i | -r | -f] [-k key] relation\n");
      fprintf (stderr, "   -u or -r will unlock a relation\n");
      fprintf (stderr, "   -i or -r will index queued additions\n");
      fprintf (stderr, "   -r will fix a damaged relation\n");
      fprintf (stderr, "   use -f to force a rebuild of all indices\n");
      }
   if (rel == RNULL)
      {
      mb_exit (1);
      }

/*
 * Determine what needs to be done; unlock relation if necessary
 *
 */

   if (! checkDamage ())
      {
      mb_exit (2);
      }
   if (! checkIndex ())
      {
      mb_exit (2);
      }
   if (! fRelink && ! fDatafile)
      {
      if (fAsk || fFix)
         {
         fprintf (stderr, "relation is not in need of repair.\n");
         }
      mb_exit (0);
      }


/*
 * If we get here, we might have to rebuild the tree...
 *
 */

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   readx (rel->fhRel, &stat, 1);

   if (fRelink)
      {
      if (! listGenerate ())
         {
         mb_exit (3);
         }

      for (idx = 0; idx < rel->nIdx; idx++)
         {
         printf ("working on index %d of %d...", idx+1, rel->nIdx);

         if (! listSort ())
            {
            break;
            }
         if (! treeBuild ())
            {
            break;
            }
         }

      close (fhList);
      unlink (listname);

      if (idx < rel->nIdx)
         {
         mb_exit (3);
         }

      lseek (rel->fhRel, POS_NUMREC, 0);
      writx (rel->fhRel, &nRec, 4);

      posNext = 0L;
      lseek (rel->fhRel, POS_NUMQUE, 0);
      writx (rel->fhRel, &posNext, 4);

      stat &= 10;  /* Turn off the corrupt-indices bits */
      }

/*
 * And we might have to regenerate the data file...
 *
 */

   if (fDatafile)
      {
      if (! dataGenerate ())
         {
         mb_exit (3);
         }

      nFail = 0L;
      last  = -1;
      rc = TRUE;

      printf ("rebuilding datafile.......complete: %3d%%", 0);

      for (i = 0L; rc && i < nRec; i++)
         {
         per = (int)((100L * (i+1)) / nRec);
         if (per != last)
            {
            printf ("\b\b\b\b%3d%%", per);  /* Backup over % complete */
            last = per;
            }
         for (fld = 0; fld < rel->nFld; fld++)
            {
            if (rel->fldType[fld] != T_MCHAR && rel->fldType[fld] != T_MBYTE)
               continue;

            if ((rc = fieldCopy (i, fld)) == FALSE)
               {
               break;
               }
            }
         }

      i = 0L;
      lseek (fhList, 1L, 0);
      writx (fhList, &posNext, 4);
      lseek (fhList, posNext, 0);
      writx (fhList, &i, 4);       /* Terminate the free-space chain here */
      writx (fhList, &i, 4);

      if (! rc)
         {
         printf ("...aborted\n");
         fprintf (stderr, "out of free space\n");
         fDatafile = FALSE;
         close (fhTemp);
         close (fhList);
         unlink (tempname);
         unlink (listname);
         }
      }

   if (fDatafile)
      {
      printf ("\b\b\b\b%3d%%\n", 100);
      printf ("replacing new datafile....");

           finalData (rel->fhRel, fhTemp);
      rc = finalCopy (rel->fhDat, fhList);

      printf ("\b\b\b\b%3d%%\n", 100);

      close (fhTemp);
      close (fhList);
      unlink (tempname);
      unlink (listname);

      if (! rc)
         {
         fprintf (stderr, "could not finish repair--check permissions and\n");
         fprintf (stderr, "replace %s.dat with %s.\n", rel->relname, listname);
         fDatafile = FALSE;
         }
      else if (nFail == 1)
         fprintf (stderr, "datafile repaired--one record's data lost\n");
      else if (nFail > 1)
         fprintf (stderr, "datafile repaired--%d records' data lost\n", nFail);

      stat &= 7;  /* Turn off the corrupt-datafile bit */
      }

   if (fRelink && fDatafile)
      fprintf (stderr, "relation repaired.\n");
   else if (fRelink)
      fprintf (stderr, "indices repaired.\n");
   else if (fDatafile)
      fprintf (stderr, "datafile repaired.\n");

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   writx (rel->fhRel, &stat, 1);

   mb_exit (0);
}


/*
 * SERVICE ROUTINES -----------------------------------------------------------
 *
 */

bool
parseArgs (argc, argv)
int        argc;
char     **argv;
{
   file     fh = 0;
   charptr  key = NULL;
   charptr  pch, arg;
   char     buffer[128];
   long     temp;


   for (--argc,++argv; argc; --argc,++argv)
      {
      if (*(pch = *argv) != '-' || *(pch+1) == 0)
         break;

      for (pch++; pch && *pch; pch++)
         {
         switch (*pch)
            {
            case '?':  return FALSE;
                      break;
            case 'f':  fOverride = TRUE;
                      break;
            case 'r':  fFix    = TRUE;
                       fIndex  = TRUE;
                       fUnlock = TRUE;
                      break;
            case 'i':  fIndex  = TRUE;
                      break;
            case 'u':  fUnlock = TRUE;
                      break;
            case 'k':  if (*(1+pch))
                          key = 1+pch;
                       else
                          {
                          if (! (--argc))
                             return FALSE;
                          key = *(++argv);
                          }
                       pch = NULL;
                      break;

            default:   fprintf (stderr, "unrecognized option '%c'\n", *pch);
                       return FALSE;
                      break;
            }

         if (pch == NULL)
            break;
         }
      }

   if (! argc)
      {
      return FALSE;
      }

   if (fFix || fIndex || fUnlock)
      {
      fAsk = FALSE;
      }


   if (! checkLock (arg = pch))
      {
      mb_exit (2);
      }

   strcpy (buffer, pch);
   if (strcmp (&buffer[strlen(buffer)-4], ".rel"))
      strcat (buffer, ".rel");

   if ((fh = openx (buffer, OPENMODE)) == -1)
      {
      fprintf (stderr, "cannot open relation '%s'\n", buffer);
      return TRUE;
      }


/*
 * Open the relation, and gather data about it... to do that, though, we'll
 * have to pretend that it's not damaged.  So overwrite the WORKFLAG bit
 * and remove any indicators of damage--then open the thing and rewrite the
 * old damage indicators.
 *
 */

   lseek (fh, POS_WORKFLAG, 0);
   readx (fh, &buffer[0], 1);
   buffer[1] = (char)( (int)buffer[0] & 2 );
   lseek (fh, POS_WORKFLAG, 0);
   writx (fh, &buffer[1], 1);

   close (fh);

   if ((rel = mb_inc (arg, key)) == RNULL)
      {
      fprintf (stderr, "%s\n", mb_error);
      return FALSE;
      }

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   writx (rel->fhRel, &buffer[0], 1);


   lseek (rel->fhRel, POS_NUMREC, 0);
   readx (rel->fhRel, &temp, 4);
   lseek (rel->fhRel, POS_NUMQUE, 0);
   readx (rel->fhRel, &nRec, 4);
   nRec += temp;

   return TRUE;
}

bool
GetYN (prompt)
char  *prompt;
{
   char buf[40];

   if (fOverride)
      return TRUE;

   printf ("%s [Y/n] ? ", prompt);

   gets (buf);
   if (buf[0] == 'n' || buf[0] == 'N')
      return FALSE;

   return TRUE;
}

bool
checkLock (arg)
char      *arg;
{
   file    fh;
   char    name[128], buf[128], *pch;
   bool    fLocked = FALSE;
   int     i;


   if ((pch = strrchr (arg, DIRSEP)) != NULL)
      strcpy (buf, 1+pch);
   else
      strcpy (buf, arg);
   if (! strcmp ((pch = &buf[ strlen(buf)-4 ]), ".rel"))
      *pch = 0;

   if (! *(GetTmpDir (name)) )
      {
      SetError (MB_TMPDIR);
      fprintf (stderr, "%s\n", mb_error);
      return FALSE;
      }
   strcat (name, buf);
   strcat (name, ".lck");


   if ((fh = openx (name, OPENMODE)) == -1)
      {
      if (fUnlock)
         {
         fprintf (stderr, "relation was not locked\n");
         }
      return TRUE;
      }


   lseek (fh, lckPOS_ELOCK, 0);
   readx (fh, &buf[0],  2);

   lseek (fh, lckPOS_HLOCK, 0);
   readx (fh, &buf[2],  6);

   lseek (fh, lckPOS_QUEUE, 0);
   readx (fh, &buf[8], 60);

   close (fh);

   unlink (name);


   for (i = 0; i < 34; i++)
      if (*(short *)&buf[2*i] != 0)
         {
         fLocked = TRUE;
         break;
         }

   if (fLocked)
      {
      fprintf (stderr, "locks broken\n");
      }
   else if (fUnlock)
      {
      fprintf (stderr, "relation was not locked\n");
      }

   return TRUE;
}

bool
checkDamage ()
{
   char    buf;

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   readx (rel->fhRel, &buf, 1);

   if (fOverride)
      fDatafile = TRUE;
   else
      {
      if (buf & 8)
         {
         if (fFix || GetYN ("datafile is damaged -- repair"))
            {
            fDatafile = TRUE;
            }
         }
      }

   if (! (buf & 5))  /* If niether damaged nor interrupted, return==okay */
      {
      if (! fOverride)
         {
         if ((buf & 8) && !fDatafile)
            return FALSE;  /* Damaged but told not to repair?  Return FALSE */
         return TRUE;
         }
      }

   if (fAsk)
      {
      if (! GetYN ("relation is damaged -- repair indices"))
         {
         return FALSE;
         }
      fFix = TRUE;
      }

   if (! fFix)
      {
      fprintf (stderr, "relation is damaged.  Use -f to repair it.\n");
      return FALSE;
      }

   fRelink = TRUE;

   return TRUE;
}

bool
checkIndex ()
{
   long  num;

   lseek (rel->fhRel, POS_NUMQUE, 0);
   readx (rel->fhRel, &num, 4);

   if ((!fOverride) && (! num))
      {
      if (fIndex && !fFix && !fUnlock)
         {
         fprintf (stderr, "there are no unindexed records\n");
         return FALSE;
         }
      return TRUE;
      }

   if (fAsk)
      {
      if (! GetYN ("relation has queued additions -- index them"))
         {
         return FALSE;
         }
      fIndex = TRUE;
      }

   if (! fIndex)
      {
      fprintf (stderr, "there are records in queue.  Use -i to index them.\n");
      return FALSE;
      }

   fRelink = TRUE;

   return TRUE;
}

void
fillBuffer (buf, rcd)
dataptr     buf;
long             rcd;
{
   GO_RECID (rel, rcd);
   readx (rel->fhRel, buf, rel->cbRecord);
}


/*
 * INDEX REGENERATION ROUTINES ------------------------------------------------
 *
 */

bool
listGenerate ()
{
   char  temp;

   GetTmpDir (listname);

   strcat (listname, rel->relname);
   strcat (listname, ".tmp");

   if (access (listname, 0) != -1)
      {
      unlink (listname);
      }

   if ((fhList = creatx (listname)) < 0)
      {
      fprintf (stderr, "cannot open temporary file '%s'\n", listname);
      return FALSE;
      }
   close (fhList);
   if ((fhList = openx (listname, OPENMODE)) < 0)
      {
      fprintf (stderr, "cannot open temporary file '%s'\n", listname);
      return FALSE;
      }

   temp  = (char)( (rel->fMulti) ? 2 : 0 );
   temp |= (char)1;

   lseek (rel->fhRel, POS_WORKFLAG, 0);
   writx (rel->fhRel, &temp, 1);

   return TRUE;
}

/*
 * Tests with 790 random records:
 *      shellsort:   sort takes .62 seconds per record
 *      mb_add:      sort takes .31 seconds per record
 *      qsort:       sort takes .27 seconds per record
 *      merge sort:  sort takes .02 seconds per record
 *
 * My thanks to Tennessee State Governor's School, who presented this sort
 * algorithm to me many, many moons ago... I _think_ "merge sort" is its real
 * name, but can't be sure.  Anyway, it's O(n*lg(n)) every time, best- and
 * worst-case.  And because it does much less I/O than qsort, it's perfect for
 * the on-disk sort we have to do here...
 *
 */

bool
listSort ()
{
   dataptr  buf_a, buf_b;
   long     ptr_a;
   long     size,  i;


   if ((buf_a = (char *)malloc (1 + rel->cbRecord)) == NULL)
      {
      fprintf (stderr, "out of memory--aborted\n");
      return FALSE;
      }
   if ((buf_b = (char *)malloc (1 + rel->cbRecord)) == NULL)
      {
      fprintf (stderr, "out of memory--aborted\n");
      return FALSE;
      }

   lseek (fhList, 0L, 0);
   for (i = 1L; i <= nRec; i++)
      {
      writx (fhList, &i, 4);
      }
   for (i = 1L; i <= nRec; i++)
      {
      writx (fhList, &i, 4);
      }

   fUseLow = TRUE;  /* Use second half as target */


   nPass = nPassMax = 1;
   for (size = 1; size < nRec; size *= 2)
      nPassMax ++;

   nPassMax ++;  /* Account for final build of tree--takes 1 extra pass */

   printf ("complete: ");
   for (size = 1; size < nRec; size *= 2, nPass++)
      {
      printf ("%3d%%", (100*(nPass-1)) / (nPassMax-1));

      for (ptr_a = 0; ptr_a < nRec; ptr_a += 2*size)
         {
         sectionSort (buf_a, buf_b, ptr_a, ptr_a +size, size, fUseLow);
         }
      fUseLow = (fUseLow) ? FALSE : TRUE;    /* Use other half as target */

      printf ("\b\b\b\b");  /* Backup over % complete */
      }

   free (buf_b);
   free (buf_a);

   return TRUE;
}

void
sectionSort (buf_a, buf_b, ptr_a, ptr_b, size, fForward)
dataptr      buf_a, buf_b;
long                       ptr_a, ptr_b, size;
bool                                           fForward;
{
   long  org,   trg;
   long  num_a, num_b;
   long  max_a, max_b;
   bool  bad_a, bad_b;


   org = (fForward) ? 0L : (nRec * 4L);
   trg = (fForward) ? (nRec * 4L) : 0L;

   max_a = min (nRec, ptr_a + size);    /* ptr_a < max_a always */
   max_b = min (nRec, ptr_b + size);    /* ptr_b < max_b always */

   bad_a = bad_b = TRUE;

   trg += (ptr_a * 4L);


   for ( ; (ptr_a < max_a) || (ptr_b < max_b); )
      {
      if (ptr_a >= max_a)  goto ss_useb;
      if (ptr_b >= max_b)  goto ss_usea;

      if (bad_a)
         {
         lseek (fhList, org+ 4*ptr_a, 0);
         readx (fhList, &num_a, 4);
         fillBuffer (buf_a, num_a);
         bad_a = FALSE;
         }
      if (bad_b)
         {
         lseek (fhList, org+ 4*ptr_b, 0);
         readx (fhList, &num_b, 4);
         fillBuffer (buf_b, num_b);
         bad_b = FALSE;
         }

      if (_compare (rel, buf_a, buf_b, idx) == 1)
         goto ss_useb;

ss_usea:
      if (bad_a)
         {
         lseek (fhList, org+ 4*ptr_a, 0);
         readx (fhList, &num_a, 4);
         }

      lseek (fhList, trg, 0);
      writx (fhList, &num_a, 4);

      trg += 4;  /* Next target is four bytes forward */
      ptr_a++;
      bad_a = TRUE;
      continue;

ss_useb:
      if (bad_b)
         {
         lseek (fhList, org+ 4*ptr_b, 0);
         readx (fhList, &num_b, 4);
         }

      lseek (fhList, trg, 0);
      writx (fhList, &num_b, 4);

      trg += 4;  /* Next target is four bytes forward */
      ptr_b++;
      bad_b = TRUE;
      continue;
      }
}

bool
treeBuild ()
{
   long  top;

   printf ("%3d%%", (100*(nPass-1)) / (nPassMax-1));

   top = nodeBuild (0L, nRec-1L, 0L, PARDIR);

   GO_TOP (rel, idx);
   writx  (rel->fhRel, &top, 4);

   printf ("\b\b\b\b100%%\n");

   return TRUE;
}

long
nodeBuild (low, high, par, pardir)
long       low, high, par;
int                        pardir;
{
   long  top, num, x;
   char  bal;
   long l,r;

   if (high < low)
      {
      return 0L;
      }

   top = low+ ((1+ high-low) / 2L);

   lseek (fhList, (top *4L) + ((fUseLow) ? 0L : (nRec *4L)), 0);
   readx (fhList, &num, 4);

   if (top-low  < high-top)  bal = (char)(pardir | (int)BAL_RT);
   if (top-low == high-top)  bal = (char)(pardir | (int)BAL_EV);
   if (top-low >  high-top)  bal = (char)(pardir | (int)BAL_LT);

   GO_BAL (rel, num, idx);
   writx (rel->fhRel, &bal, 1);

   GO_POINT (rel, num, idx, 0);
   writx (rel->fhRel, &par, 4);

   x = nodeBuild (low, top-1, num, 0);
l=x;
   GO_POINT (rel, num, idx, -1);
   writx (rel->fhRel, &x, 4);

   x = nodeBuild (top+1, high, num, PARDIR);
r=x;
   GO_POINT (rel, num, idx, 1);
   writx (rel->fhRel, &x, 4);

   return num;
}


/*
 * DATAFILE REGENERATION ROUTINES ---------------------------------------------
 *
 */

bool
dataGenerate ()
{
   char  ch;
   long  tlong;

/*
 * First, make a file for the new .DAT file...
 *
 */

   GetTmpDir (listname);

   strcat (listname, rel->relname);
   strcat (listname, ".tmp");

   if (access (listname, 0) != -1)
      {
      unlink (listname);
      }

   if ((fhList = creatx (listname)) < 0)
      {
      fprintf (stderr, "cannot open temporary file '%s'\n", listname);
      return FALSE;
      }
   close (fhList);
   if ((fhList = openx (listname, OPENMODE)) < 0)
      {
      fprintf (stderr, "cannot open temporary file '%s'\n", listname);
      return FALSE;
      }

/*
 * Then a file for the records...
 *
 */

   GetTmpName (tempname);

   if ((fhTemp = creatx (tempname)) < 0)
      {
      fprintf (stderr, "cannot open temporary file '%s'\n", tempname);
      close (fhList);
      unlink (listname);
      return FALSE;
      }
   close (fhTemp);
   if ((fhTemp = openx (tempname, OPENMODE)) < 0)
      {
      close (fhList);
      unlink (listname);
      unlink (tempname);
      fprintf (stderr, "cannot open temporary file '%s'\n", tempname);
      return FALSE;
      }

/*
 * Now that we've created a temporary file, create a header for it.
 *
 */

   ch = verCURRENT;
   tlong = 0L;
   writx (fhList, &ch, 1);      /* .DAT-file signature                     */
   writx (fhList, &tlong, 4);   /* First free-chain link (temporary value) */

   posNext = 5L;

   return TRUE;
}

bool
fieldCopy (rcd, fld)
long       rcd;
int             fld;
{
   char  ch;
   long  cbTotal;
   long  posField, posData;

/*
 * The first step here is to find the offset within the .REL file which
 * will indicate where the data exists in the .DAT file... put that in
 * posField.  Note that 'rcd', as passed in, is zero-based.
 *
 */

   posField  = rel->posRecZ;
   posField += rcd * (rel->cbRecord + ((long)cbINDEX*rel->nIdx));
   posField += (long)cbINDEX*rel->nIdx;
   posField += rel->cbStart[fld];

   lseek (rel->fhRel, posField, 0);
   readx (rel->fhRel, &posData, 4);

   if (! posData)    /* No data to begin with?     */
      goto lblDONE;  /* Then we're done right now. */

/*
 * Before we copy data for field, find out if the data exists and is intact.
 * The format should be:
 *    4 bytes--overall length of data, including these 4 bytes
 *    sig:     "----" to end chain, or "+" to continue chain
 *             if "+":  4 bytes--sizeof page
 *
 */

   if (lseek (rel->fhDat, posData, 0) != posData)
      {
      goto lblHOSED;
      }

   if (readx (rel->fhDat, &cbTotal, 4) != 4)
      {
      goto lblHOSED;
      }

   if (readx (rel->fhDat, &ch, 1) != 1)
      {
      goto lblHOSED;
      }

   if (ch != '+' && ch != '-')   /* Not a valid compression signature? */
      {
      goto lblHOSED;
      }

   lseek (rel->fhDat, posData, 0);
   lseek (fhList,     posNext, 0);

   if (! dataCopy (fhList, rel->fhDat, cbTotal))
      return FALSE;

/*
 * If we were successful in copying the data, update the record's pointer to
 * the new location, and move posNext up to the end of the new .DAT file...
 *
 */

   lseek (fhTemp, rcd * 4,  0);
   writx (fhTemp, &posNext, 4);
   posNext += cbTotal;

   return TRUE;


/*
 * If we couldn't recover the data for a field, point the field into 0L ==
 * no data available.
 *
 */

lblHOSED:
   nFail++;
   posData = 0L;

lblDONE:
   lseek (fhTemp, rcd * 4,  0);
   writx (fhTemp, &posData, 4);

   return TRUE;
}

bool
dataCopy (fhOut, fhIn, cbTotal)
file      fhOut, fhIn;
long                   cbTotal;
{
   long  n;

   for ( ; cbTotal; cbTotal -= n)
      {
      n = min (cbTotal, MAX_CBUF);

      if (_fhCopy (fhOut, fhIn, n) != n)
         {
         return FALSE;
         }
      }

   return TRUE;
}

bool
finalCopy (fhOut, fhIn)
file       fhOut, fhIn;
{
   long  cbTotal, cbOrig;
   int   per, last;
   long  n;

   cbOrig = cbTotal = lseek (fhIn, 0L, 2);

   lseek (fhOut, 0L, 0);
   lseek (fhIn,  0L, 0);

   printf ("complete: %3d%%", (last = 0));

   for ( ; cbTotal; cbTotal -= n)
      {
      per = (int)((100L * (cbOrig - cbTotal)) / cbOrig);
      if (per != last)
         {
         printf ("\b\b\b\b%3d%%", per);
         }

      n = min (cbTotal, MAX_CBUF);

      if (_fhCopy (fhOut, fhIn, n) != n)
         {
         return FALSE;
         }
      }

   return TRUE;
}

void
finalData (fhOut, fhIn)
file       fhOut, fhIn;
{
   long  rcd;
   int   fld;
   long  posField;
   long  posData;

   lseek (fhIn, 0L, 0);

   for (rcd = 0; rcd < nRec; rcd++)
      {
      for (fld = 0; fld < rel->nFld; fld++)
         {
         if (rel->fldType[fld] != T_MCHAR && rel->fldType[fld] != T_MBYTE)
            continue;

         posField  = rel->posRecZ;
         posField += rcd * (rel->cbRecord + ((long)cbINDEX*rel->nIdx));
         posField += (long)cbINDEX*rel->nIdx;
         posField += rel->cbStart[fld];

         lseek (fhOut, posField, 0);
         readx (fhIn,  &posData, 4);
         writx (fhOut, &posData, 4);
         }
      }
}

