#ifndef _QIC80_VTBL_H
#define _QIC80_VTBL_H
/*
 *
 *      Copyright (c) 1995  Claus-Justus Heine (cH)
 *
 *      This file defines a volume table as defined in the QIC-80 development 
 *      standards defined.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * claus
 *
 * 1.6
 * 1995/11/16 22:35:40
 * Exp
 *
 */

#include "ftape.h"
#include "ftape-eof.h"
#include "qic80-compress.h"
#include "ftape-ctl.h"

struct qic80_vtbl 
{
  unsigned long signature;            /* must be "VTBL" */
  unsigned short first_segment; /* first segment of this volume (2 bytes) */
  unsigned short last_segment;  /* last segment of this volume (2 bytes)  */
  char description[44];         /* any string */
  unsigned long date;           /* encoded date */
  union {
    struct {
      byte vendorspecific     : 1; /* rem. of vtbl entry is vendor specific */
      byte multicartridge     : 1; /* self explanatory */
      byte not_verified       : 1; /* set if file set wasn't verified */
      byte redirection_inhibit: 1; /* file set redirection is not allowed??? */
      byte segment_spanning   : 1; /* til QIC-80 Rev. H: always 0
                                    * Rev. I til ...   : 0 if segment spanning 
                                    *                      wasn't used
                                    *                    1 if segment spanning 
                                    *                       was used
                                    */
      byte unused5 : 1;
      byte unused6 : 1;
      byte unused7 : 1;
    } flags;
    byte dummy;
  } flags;
  byte multi_cartridge_count;
  byte vendor_extension[26];

  char password[8];
  unsigned long dir_section_size; /* size of directory entry at begin of 
                                   * fileset 
                                   */
  unsigned long data_section_size; /* size of data section, including 
                                    * file-headers 
                                    */
  byte OS_version_major;
  byte OS_version_minor;
  char source_volume[16];
  byte logical_dev_file_set;
  byte physical_dev_file_set;
  byte QIC_compression_method;
  union {
    struct {
      byte OS_DOS   : 1;
      byte OS_UNIX  : 1;
      byte OS_OS_2  : 1;
      byte OS_MAC   : 1;
      byte OS_NOVELL: 1;
      byte OS_LAN   : 1;
      byte OS_unused6 : 1;
      byte OS_unused7 : 1;
    } flags;
    byte dummy;
  } OS_flags;
  byte OS_flag_dummy;
  byte ISO_compression_method;
  byte reserved[4];
};

/*
 *  one vtbl is 128 bytes, that results in a maximum number of
 *  29*1024/128 = 232 volumes. 
 */
#define MAX_VOLUMES (29*SECTOR_SIZE/sizeof(struct qic80_vtbl) )
#define VTBL_ID "VTBL"
#define VOL_NAME "zftape volume" /* volume label used by me */

/*
 *  global variables
 */
struct qic80_internal_vtbl
{
  unsigned short start_seg;
  unsigned short end_seg;
  unsigned long size;                 /* uncompressed size */
  unsigned short block_size;          /* block size for this volume */
  unsigned short zftape_volume   : 1; /* flag indicating that we created it */
  unsigned short use_compression : 1; /* flag indicating compressed volume  */
};
#ifdef DYN_ALLOC
#define __QIC80_VTBL_SIZE (MAX_VOLUMES*sizeof(struct qic80_internal_vtbl))
extern struct qic80_internal_vtbl *__qic80_vtbl;
#else
extern struct qic80_internal_vtbl __qic80_vtbl[MAX_VOLUMES];
#endif
extern struct qic80_internal_vtbl *__qic80_last_vtbl;
extern unsigned int __qic80_num_of_volumes;
extern unsigned __qic80_file_no;

extern int ftape_qic80_mode;
extern int ftape_old_ftape;
extern int ftape_volume_table_changed;

/*
 *  exported functions
 */
extern void qic80_init_vtbl                ( void );
extern void qic80_create_volume_headers    ( byte *buffer );
extern void qic80_extract_volume_headers   ( byte *buffer );
extern int  qic80_update_volume_table      ( unsigned segment, byte *buffer );
extern int  qic80_maybe_write_beof         ( unsigned segment,
					    unsigned no_new_one );
extern int  qic80_check_for_eof            ( unsigned segment, 
					    unsigned long data_pos, 
					    long *distance );
extern int  qic80_skip_files               ( unsigned segment, int count );
extern void qic80_skip_to_eom              ( void );
extern void qic80_fake_volume_headers      ( eof_mark_union *eof_map, 
					    int num_failed_sectgors );
extern void qic80_fake_single_volume_header( unsigned start_seg, 
					    unsigned end_seg, 
					    unsigned long size );
extern int  qic80_weof                     ( unsigned count );

static inline int  qic80_check_for_eom            ( unsigned segment );
static inline int  qic80_find_file_no             ( unsigned segment );
static inline int  qic80_get_volume_block_size    ( unsigned segment );
static inline int  qic80_get_volume_size          ( unsigned segment );
static inline int  qic80_get_volume_cmpr_flag     ( unsigned segment );
static inline int  qic80_get_volume_boundaries    ( unsigned segment,
						   unsigned *end_seg );
static inline void qic80_maybe_write_eof          ( unsigned segment, 
						   unsigned long data_pos );

/*
 *  checks for EOM. easy
 *
 */

static inline int
qic80_check_for_eom( unsigned seg_pos )
{ 
  TRACE_FUN( 8, "qic80_check_for_eom");
  if ( ftape_qic80_mode ) {
    TRACE_EXIT;
    return ( __qic80_num_of_volumes == 0 || seg_pos > __qic80_last_vtbl->end_seg) ? 1 : 0;
  } else {
    TRACE_EXIT;
    return ( seg_pos > ftape_last_data_segment );
  }
}

/*
 *  find the file (volume) seg_pos belongs to
 */       

static inline int
qic80_find_file_no( unsigned seg_pos )
{
  TRACE_FUN( 8, "qic80_find_file_no");

  TRACEi( 8, "called with seg_pos ",seg_pos);
  if ( ftape_qic80_mode ) {
    if ( __qic80_file_no > __qic80_num_of_volumes        ||
	seg_pos > __qic80_vtbl[__qic80_file_no].end_seg || 
	seg_pos < __qic80_vtbl[__qic80_file_no].start_seg ) {
      TRACE(5,"Searching for file ...");
      for ( __qic80_file_no = 0; __qic80_file_no < __qic80_num_of_volumes && seg_pos > __qic80_vtbl[__qic80_file_no].end_seg; __qic80_file_no++ );
    }
    TRACEi( 8, "returning ", __qic80_file_no);
    TRACE_EXIT;            
    return __qic80_file_no;
  } else {
    TRACE_EXIT;            
    return seg_pos > ftape_last_data_segment ? 1 : 0;
  }
}

/*
 *  return the block size that is valid for the volume the segment belongs to
 */

static inline int
qic80_get_volume_block_size( unsigned segment )
{
  TRACE_FUN( 8, "qic80_get_volume_block_size");
  int result;

  if ( !ftape_qic80_mode || (result = qic80_find_file_no( segment )) >= __qic80_num_of_volumes ) {
    TRACE_EXIT;
    return ftape_block_size; /* after or at EOM */
  } else {
    TRACE_EXIT;
    return __qic80_vtbl[ result ].block_size;
  }
}

/*
 *  return the size of the volume the segment belongs to
 */

static inline int
qic80_get_volume_size( unsigned segment )
{
  TRACE_FUN( 8, "qic80_get_volume_size");
  int result;

  if ( ftape_qic80_mode ) {
    result = qic80_find_file_no( segment );
    if ( result < __qic80_num_of_volumes ) {
      result = __qic80_vtbl[ result ].size;
    } else {
      result = 0;
    }
  } else {
    result = segment > ftape_last_data_segment ? 0 : ftape_total_data_amount;
  }
  TRACE_EXIT;
  return result;
}

/*
 *  return 1 if volume is compressed, else 0
 */

static inline int
qic80_get_volume_cmpr_flag( unsigned segment )
{
  TRACE_FUN( 8, "qic80_get_volume_cmpr_flag");
  int result;

  if ( ftape_qic80_mode ) {
    result = qic80_find_file_no( segment );
    if ( result < __qic80_num_of_volumes ) {
      TRACEx2(3,"use_compression (volume %d): %d", result, __qic80_vtbl[ result ].use_compression != 0 );
      result = __qic80_vtbl[ result ].use_compression;
    } else {
      result = 0;
    }
  } else {
    result = 0;
  }
  TRACE_EXIT;
  return result;
}


/*
 *  return the volume boundaries
 */

static inline int
qic80_get_volume_boundaries( unsigned segment, unsigned *end_seg )
{
  TRACE_FUN( 8, "qic80_get_volume_boundaries");
  int result;
  int vol_no;

  if ( ftape_qic80_mode ) {
    vol_no = qic80_find_file_no( segment );
    if ( vol_no < __qic80_num_of_volumes ) {
      result   = __qic80_vtbl[ vol_no ].start_seg;
      *end_seg = __qic80_vtbl[ vol_no ].end_seg;
    } else {
      result = *end_seg = 0;
    }
    TRACEx3( 6, "volume: %d, start_seg: %d, end_seg: %d", vol_no, result, *end_seg );
  } else {
    result = first_data_segment;
    *end_seg = ftape_last_data_segment;
  }
  TRACE_EXIT;
  return result;
}

/*  
 *  write an EOF-marker by setting __qic80_last_vtbl->end_seg to seg_pos.
 *  erase following volumes, as nothing but appending data is allowed
 *  NOTE: this function assumes that __qic80_last_vtbl points to a valid
 *  vtbl entry 
 */
static inline void 
qic80_maybe_write_eof( unsigned seg_pos, unsigned long size )
{ 
  TRACE_FUN( 8, "write_eof");

  if (ftape_qic80_mode) {
    if ( seg_pos < __qic80_last_vtbl->start_seg ) {
      /*
       *  we allow nothing but to apppend files to a tape. Thus if one really
       *  wants to write data to tape before the last registered volume,
       *  then the volume-entries for all following volume are deleted
       */
      while ( __qic80_last_vtbl->start_seg > seg_pos ) {
	__qic80_last_vtbl->start_seg =	__qic80_last_vtbl->end_seg = 0;
	__qic80_last_vtbl->size      = 0UL;
	__qic80_last_vtbl--;
	__qic80_num_of_volumes--;
      }
    } 
    /*
     *  easy: just set end of volume to seg_pos
     */
    __qic80_last_vtbl->end_seg = seg_pos;
    __qic80_last_vtbl->size = size;
    ftape_volume_table_changed = 1;
    TRACEx2(5,"end_seg: %d, size: %ld", seg_pos, size );
  }
  TRACE_EXIT;
}

#endif /* _QIC80_VTBL_H */

