------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                F N A M E                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.37 $                             --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

with Debug;    use Debug;
with Krunch;
with Namet;    use Namet;
with Opt;      use Opt;
with Widechar; use Widechar;

package body Fname is

   ----------------------------
   -- Get_Expected_Unit_Type --
   ----------------------------

   --  We assume that a file name whose last character is a lower case b is
   --  a body and a file name whose last character is a lower case s is a
   --  spec. If any other character is found (e.g. when we are in syntax
   --  checking only mode, where the file name conventions are not set),
   --  then we return Unknown.

   function Get_Expected_Unit_Type
     (Fname : File_Name_Type)
      return  Expected_Unit_Type
   is
   begin
      Get_Name_String (Fname);

      if Name_Buffer (Name_Len) = 'b' then
         return Expect_Body;
      elsif Name_Buffer (Name_Len) = 's' then
         return Expect_Spec;
      else
         return Unknown;
      end if;
   end Get_Expected_Unit_Type;

   -------------------
   -- Get_File_Name --
   -------------------

   function Get_File_Name (Uname : Unit_Name_Type) return File_Name_Type is
      Unit_Char   : Character;
      --  Set to 's' or 'b' for spec or body

      J : Integer;

   begin
      Get_Decoded_Name_String (Uname);

      --  Change periods to hyphens, being careful to skip past any
      --  period characters embedded in wide character escape sequences)

      J := 1;

      while J <= Name_Len loop
         if Name_Buffer (J) = '.' then
            Name_Buffer (J) := '-';
            J := J + 1;

         elsif Name_Buffer (J) = Ascii.ESC
           or else (Upper_Half_Encoding
                     and then Name_Buffer (J) in Upper_Half_Character)
         then
            Skip_Wide (Name_Buffer, J);
         else
            J := J + 1;
         end if;
      end loop;

      --  Deal with spec or body suffix

      Unit_Char := Name_Buffer (Name_Len);
      pragma Assert (Unit_Char = 'b' or else Unit_Char = 's');
      pragma Assert (Name_Len >= 3 and then Name_Buffer (Name_Len - 1) = '%');
      Name_Len := Name_Len - 2;

      --  The file name (minus the extension) to be used is stored in
      --  Name_Buffer (1 .. Name_Buffer). If it's too long then crunch it.

      Krunch
        (Name_Buffer,
         Name_Len,
         Integer (Maximum_File_Name_Length),
         Debug_Flag_4);

      --  Here with the file name set and of OK length, add the extension

      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := '.';
      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := 'a';
      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := 'd';
      Name_Len := Name_Len + 1;
      Name_Buffer (Name_Len) := Unit_Char;

      return File_Name_Type (Name_Find);
   end Get_File_Name;

   ------------------------------
   -- Is_Language_Defined_Unit --
   ------------------------------

   function Is_Language_Defined_Unit (Fname : File_Name_Type) return Boolean is
      subtype Str8 is String (1 .. 8);

      Predef_Names : array (1 .. 12) of Str8 :=
         ("ada     ",       -- Ada
          "calendar",       -- Calendar
          "direc_io",       -- Direct_IO
          "gnat    ",       -- GNAT
          "interfac",       -- Interfaces
          "ioexcept",       -- IO_Exceptions
          "machcode",       -- Machine_Code
          "sequenio",       -- Sequential_IO
          "system  ",       -- System
          "text_io ",       -- Text_IO
          "unchconv",       -- Unchecked_Conversion
          "unchdeal");      -- Unchecked_Deallocation

   begin
      --  Get file name, removing the extension (if any)

      Get_Name_String (Fname);

      if Name_Len > 4 and then Name_Buffer (Name_Len - 3) = '.' then
         Name_Len := Name_Len - 4;
      end if;

      --  Definitely false if longer than 8 characters

      if Name_Len > 8 then
         return False;
      end if;

      --  Definitely predefined if prefix is a- g- i- or s-

      if Name_Len > 2
        and then Name_Buffer (2) = '-'
        and then (Name_Buffer (1) = 'a' or else
                  Name_Buffer (1) = 'g' or else
                  Name_Buffer (1) = 'i' or else
                  Name_Buffer (1) = 's')
      then
         return True;
      end if;

      --  Otherwise check against special list, first padding to 8 characters

      while Name_Len < 8 loop
         Name_Len := Name_Len + 1;
         Name_Buffer (Name_Len) := ' ';
      end loop;

      for J in 1 .. 12 loop
         if Name_Buffer (1 .. 8) = Predef_Names (J) then
            return True;
         end if;
      end loop;

      return False;

   end Is_Language_Defined_Unit;

   ------------------
   -- Is_File_Name --
   ------------------

   function Is_File_Name (Name : Name_Id) return Boolean is
   begin
      Get_Name_String (Name);
      return
        Name_Len > 4
          and then Name_Buffer (Name_Len - 3 .. Name_Len - 1) = ".ad"
          and then (Name_Buffer (Name_Len) = 'b'
                      or else Name_Buffer (Name_Len) = 's');
   end Is_File_Name;

   -----------------------
   -- File_Name_Of_Spec --
   -----------------------

   function File_Name_Of_Spec (Name : Name_Id) return File_Name_Type is
   begin
      Get_Name_String (Name);
      Name_Buffer (Name_Len + 1 .. Name_Len + 2) := "%s";
      Name_Len := Name_Len + 2;
      return Get_File_Name (Name_Enter);
   end File_Name_Of_Spec;

   -----------------------
   -- File_Name_Of_Body --
   -----------------------

   function File_Name_Of_Body (Name : Name_Id) return File_Name_Type is
   begin
      Get_Name_String (Name);
      Name_Buffer (Name_Len + 1 .. Name_Len + 2) := "%b";
      Name_Len := Name_Len + 2;
      return Get_File_Name (Name_Enter);
   end File_Name_Of_Body;

end Fname;
