/* File managing
   Copyright (C) 1994, 1995 Miguel de Icaza, Fred Leeflang, Janne Kukonlehto.

   The copy code is based in GNU's cp, and was written by:
   Torbjorn Granlund, David MacKenzie, and Jim Meyering.

   The move code is based in GNU's mv, and was written by:
   Mike Parker and David MacKenzie.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <errno.h>
#include <ncurses.h>
#include <malloc.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
    #include <unistd.h>
#endif
#include <sys/stat.h>
#include <sys/param.h>
#include <fcntl.h>
#include <time.h>
#include "mad.h"
#include "regex.h"
#include "dialog.h"
#include "global.h"
#include "main.h"
#include "util.h"
#include "input.h"
#include "file.h"

/* Needed for current_panel and other_panel */
#include "dir.h"
#include "panel.h"

/* Needed by query_replace */
#include "color.h"
#include "win.h"
#include "dlg.h"
#include "widget.h"
#include "key.h"

static char rcsid [] = "$Id: file.c,v 1.11 1995/01/27 02:34:51 miguel Exp $";
int verbose = 1;

#define WX 60
#define WY 10
#define BY 10

static WINDOW *op_win;
static char *op_titles [] = { " Copying ", " Moving ", " Deleting " };
static char *op_names [] = { " Copy ", " Move ", " Delete " };
static int selected_button;
static int last_percentage [3];

enum { REPLACE_YES = B_USER, REPLACE_NO, REPLACE_ALWAYS,
	   REPLACE_UPDATE, REPLACE_NEVER, REPLACE_ABORT };
static chtype replace_colors [4];
static WINDOW *replace_box;
static Dlg_head *replace_dlg;
static char *replace_filename;
static struct stat *s_stat, *d_stat;
static int replace_result;

enum { RECURSIVE_YES, RECURSIVE_NO, RECURSIVE_ALWAYS,
	   RECURSIVE_NEVER, RECURSIVE_ABORT };
static int recursive_result;
static int recursive_erase (char *s);

static void update_buttons (void)
{
    wattron (op_win, A_REVERSE);
    if (selected_button == FILE_SKIP)
	wattroff (op_win, A_REVERSE);
    mvwaddstr (op_win, BY, 14, "[ Skip ]");
    if (selected_button == FILE_ABORT)
	wattroff (op_win, A_REVERSE);
    else
	wattron (op_win, A_REVERSE);
    mvwaddstr (op_win, BY, WX - 19, "[ Abort ]");
    wattron (op_win, A_REVERSE);
    wrefresh (op_win);
}

static int check_buttons (void)
{
    int c;

    rotate_dash ();
    nodelay (stdscr, TRUE);
#if 0
    /* It doesn't make any sense yet, since the abort/skip buttons are  */
    /* fake buttons.  Have to adjust this to use the dialog manager one */
    /* of this days.  In the meantime, we stick to xgetch */
    c = mouse_getch (0);
#else
    c = xgetch ();
#endif
    nodelay (stdscr, FALSE);
    switch (c){
    case 's':
	selected_button = FILE_SKIP;
	update_buttons ();
	return selected_button;
    case 'a':
    case KEY_F(10):
    case XCTRL('g'):
    case XCTRL('c'):
	selected_button = FILE_ABORT;
	update_buttons ();
	return selected_button;
    case '\n':
	return selected_button;
    case KEY_LEFT:
    case KEY_RIGHT:
	selected_button = (selected_button == FILE_SKIP) ? FILE_ABORT : FILE_SKIP;
	update_buttons ();
	/* Fall through */
    default:
	return FILE_CONT;
    }
}

void create_op_win (int op)
{
    int i;

    replace_result = 0;
    recursive_result = 0;
    create_dialog (WX, WY, op_titles [op], "", 0);
    op_win = get_top_text ();
    leaveok (op_win, TRUE);
    selected_button = FILE_SKIP;
    update_buttons ();
    for (i = 0; i < 3; i++)
	last_percentage [i] = -99;
}

void destroy_op_win (void)
{
    destroy_dialog ();
}

void refresh_op_win (void)
{
    touchwin (op_win);
    wrefresh (op_win);
}

static int show_no_bar (int n)
{
    if (n >= 0){
	if (last_percentage [n] == -99)
	    return check_buttons ();
	last_percentage [n] = -99;
    }
    wprintw (op_win, "%*s", WX - 4, "");
    wrefresh (op_win);
    return check_buttons ();
}

static int show_source (char *s)
{
    wmove (op_win, 3, 4);
    if (s != NULL){
	wprintw (op_win, "Source: %-*s", WX - 12, name_trunc (s, WX - 13));
	/*wrefresh (op_win);*/
	return check_buttons ();
    } else
	return show_no_bar (-1);
}

static int show_target (char *s)
{
    wmove (op_win, 4, 4);
    if (s != NULL){
	wprintw (op_win, "Target: %-*s", WX - 12, name_trunc (s, WX - 13));
	wrefresh (op_win);
	return check_buttons ();
    } else
	return show_no_bar (-1);
}

static int show_deleting (char *s)
{
    mvwprintw (op_win, 3, 4, "%-*s", WX - 4, "Deleting: ");
    mvwprintw (op_win, 4, 4, "  %-*s", WX - 6, name_trunc (s, WX - 6));
    wrefresh (op_win);
    return check_buttons ();
}

static int show_bar (int n, long done, long total)
{
    long percentage, columns;
    int  diff, limit;

    if (total <= 0 || done < 0 || done > total)
	return check_buttons ();
    if (total > 65535){
	total /= 256;
	done /= 256;
    }
    percentage = (200 * done / total + 1) / 2;
    columns = (2 * (WX - 17) * done / total + 1) / 2;
    diff = last_percentage [n] - percentage;
    limit = 5 * (1 - verbose);
    if (diff >= -limit && diff <= limit)
	return check_buttons ();
    last_percentage [n] = percentage;

    waddch (op_win, '[');
    wattroff (op_win, A_REVERSE);
    wprintw (op_win, "%*s", columns, "");
    wattron (op_win, A_REVERSE);
    wprintw (op_win, "%*s", WX - 17 - columns, "");
    wprintw (op_win, "] %3d%%", percentage);
    wrefresh (op_win);
    return check_buttons ();
}

static int show_file_progress (long done, long total)
{
    if (!verbose)
	return check_buttons ();
    wmove (op_win, 6, 4);
    if (total > 0){
	waddstr (op_win, "File  ");
	return show_bar (0, done, total);
    } else
	return show_no_bar (0);
}

static int show_count_progress (long done, long total)
{
    wmove (op_win, 7, 4);
    if (total > 0){
	waddstr (op_win, "Count ");
	return show_bar (1, done, total);
    } else
	return show_no_bar (1);
}

static int show_bytes_progress (long done, long total)
{
    if (!verbose)
	return check_buttons ();
    wmove (op_win, 8, 4);
    if (total > 0){
	waddstr (op_win, "Bytes ");
	return show_bar (2, done, total);
    } else
	return show_no_bar (2);
}

static int do_file_error (char *error)
{
    int result;

    result = query_dialog (" Error ", error, 3, 3, " Skip ", " Retry ", " Abort ");

    switch (result){
    case 0:
	refresh_op_win ();
	return FILE_SKIP;
    case 1:
	refresh_op_win ();
	return FILE_RETRY;
    case 2:
    default:
	return FILE_ABORT;
    }
}

/* Report error with one file */
static int file_error (char *format, char *file)
{
    sprintf (cmd_buf, format,
	     name_trunc (file, 30), unix_error_string (errno));
    return do_file_error (cmd_buf);
}

/* Report error with two files */
static int files_error (char *format, char *file1, char *file2)
{
    sprintf (cmd_buf, format, name_trunc (file1, 15),
	     name_trunc (file2, 15), unix_error_string (errno));
    return do_file_error (cmd_buf);
}

static int replace_callback (struct Dlg_head *h, int Id, int Msg)
{
    char *format = "Target file \"%s\" already exists!";

    switch (Msg){
    case DLG_DRAW:
	wattrset (h->window, ERROR_COLOR);
	wclr (replace_box);
	draw_box (h->window, 1, 2, 12, 56);

	mvwaddstr (h->window, 1, 24, " File exists ");
	mvwprintw (h->window, 3, 5, format,
		   name_trunc (replace_filename, 52 - strlen (format)));
	mvwprintw (h->window, 5, 5, "Source date: %s, size %d",
		   file_date (s_stat->st_mtime), s_stat->st_size);
	mvwprintw (h->window, 6, 5, "Target date: %s, size %d",
		   file_date (d_stat->st_mtime), d_stat->st_size);
	mvwaddstr (h->window, BY - 2, 5, "Overwrite this target?");
	mvwaddstr (h->window, BY - 1, 5, "Overwrite all targets?");
	break;
    }
    return 0;
}

static void init_replace (void)
{
    replace_colors [0] = ERROR_COLOR;
    replace_colors [1] = REVERSE_COLOR;
    replace_colors [2] = ERROR_COLOR;
    replace_colors [3] = REVERSE_COLOR;
    replace_box = centerwin (14, 60);
    replace_dlg = dlg_new (replace_box, replace_colors, replace_callback,
			  winpos (14, 60), "[Replace]");

    add_widget (replace_dlg,
		button_new (BY + 1, 46, REPLACE_ABORT, "[ Abort ]", 'a', 2, 0, 0));
    add_widget (replace_dlg,
		button_new (BY - 1, 47, REPLACE_NEVER, "[ nonE ]", 'e', 5, 0, 0));
    add_widget (replace_dlg,
		button_new (BY - 1, 36, REPLACE_UPDATE, "[ Update ]", 'u', 2, 0, 0));
    add_widget (replace_dlg,
		button_new (BY - 1, 28, REPLACE_ALWAYS, "[ alL ]", 'l', 4, 0, 0));
    add_widget (replace_dlg,
		button_new (BY - 2, 36, REPLACE_NO, "[ No ]", 'n', 2, 0, 0));
    add_widget (replace_dlg,
		button_new (BY - 2, 28, REPLACE_YES, "[ Yes ]", 'y', 2, 0, 0));
}

static int query_replace (char *destname, struct stat *_s_stat,
			  struct stat *_d_stat)
{
    if (replace_result < REPLACE_ALWAYS){
	replace_filename = destname;
	s_stat = _s_stat;
	d_stat = _d_stat;
	init_replace ();
	run_dlg (replace_dlg);
	replace_result = replace_dlg->ret_value;
	if (replace_result == B_CANCEL)
	    replace_result = REPLACE_ABORT;
	destroy_dlg (replace_dlg);
	delwin (replace_box);
    }

    switch (replace_result){
    case REPLACE_UPDATE:
	refresh_op_win ();
	if (s_stat->st_mtime > d_stat->st_mtime)
	    return FILE_CONT;
	else
	    return FILE_SKIP;
    case REPLACE_YES:
    case REPLACE_ALWAYS:
	refresh_op_win ();
	return FILE_CONT;
    case REPLACE_NO:
    case REPLACE_NEVER:
	refresh_op_win ();
	return FILE_SKIP;
    case REPLACE_ABORT:
    default:
	return FILE_ABORT;
    }
}

int query_recursive (char *s)
{
    char *confirm, *text;

    if (recursive_result < RECURSIVE_ALWAYS){
	text = copy_strings (" Delete: ", name_trunc (s, 30), " ", 0);
	recursive_result = query_dialog (text, "\n   Directory not empty.   "
					 "\n   Delete it recursively? ",
					 3 | WITH_HOTKEYS, 5,
					 "y Yes ", "n No ", "l alL ", "e nonE ", "a Abort ");
	if (recursive_result != RECURSIVE_ABORT)
	    refresh_op_win ();
	free (text);
	if (recursive_result == RECURSIVE_YES
	    || recursive_result == RECURSIVE_ALWAYS){
	    text = copy_strings (" Please type 'yes' to confirm recursive delete of ",
				 recursive_result == RECURSIVE_YES
				 ? name_trunc (s, 20) : "all the directories", " ", 0);
	    confirm = input_dialog (" Recursive Delete ",
				    text, "no");
	    refresh_op_win ();
	    if (!confirm || strcmp (confirm, "yes"))
		recursive_result = RECURSIVE_NEVER;
	    free (confirm);
	    free (text);
	}
    }
    switch (recursive_result){
    case RECURSIVE_YES:
    case RECURSIVE_ALWAYS:
	return FILE_CONT;
    case RECURSIVE_NO:
    case RECURSIVE_NEVER:
	return FILE_SKIP;
    case RECURSIVE_ABORT:
    default:
	return FILE_ABORT;
    }
}

int copy_file_file (char *src_path, char *dst_path)
{
    char *buf = 0;
    int  buf_size;
    int  dest_desc = 0;
    int  source_desc;
    int  n_read;
    int  n_written;
    int  src_mode;		/* The mode of the source file */
    struct stat sb, sb2;
    int  dst_exists = 0;
    long n_read_total = 0;
    long file_size;
    int  return_status, temp_status;

    if (show_source (src_path) == FILE_ABORT
	|| show_target (dst_path) == FILE_ABORT)
	return FILE_ABORT;

 retry_dst_stat:
    if (!stat (dst_path, &sb2)){
	if (S_ISDIR (sb2.st_mode)){
	    return_status = file_error (" Cannot overwrite directory \"%s\" ", dst_path);
	    if (return_status == FILE_RETRY)
		goto retry_dst_stat;
	    return return_status;
	}
	dst_exists = 1;
    }

 retry_src_lstat:
    if (lstat (src_path, &sb)){
	return_status = file_error (" Cannot lstat source file \"%s\" \n %s ", src_path);
	if (return_status == FILE_RETRY)
	    goto retry_src_lstat;
	return return_status;
    }

    if (dst_exists){
	return_status = query_replace (dst_path, &sb, &sb2);
	if (return_status != FILE_CONT)
	    return return_status;
    }

    if (S_ISLNK (sb.st_mode)){
	char link_target [MAXPATHLEN];
	int len;

    retry_src_readlink:
	len = readlink (src_path, link_target, MAXPATHLEN);
	if (len < 0){
	    return_status = file_error (" Cannot read source link \"%s\" \n %s ", src_path);
	    if (return_status == FILE_RETRY)
		goto retry_src_readlink;
	    return return_status;
	}
	link_target [len] = 0;
    retry_dst_symlink:
	if (symlink (link_target, dst_path) != 0){
	    return_status = file_error (" Cannot create target symlink \"%s\" \n %s ", dst_path);
	    if (return_status == FILE_RETRY)
		goto retry_dst_symlink;
	    return return_status;
	}
	/* Success */
	return FILE_CONT;
    }

 retry_src_open:
    if ((source_desc = open (src_path, O_RDONLY)) < 0){
	return_status = file_error (" Cannot open source file \"%s\" \n %s ", src_path);
	if (return_status == FILE_RETRY)
	    goto retry_src_open;
	return return_status;
    }

 retry_src_fstat:
    if (fstat (source_desc, &sb)){
	return_status = file_error (" Cannot fstat source file \"%s\" \n %s ", src_path);
	if (return_status == FILE_RETRY)
	    goto retry_src_fstat;
        goto ret;
    }
    src_mode = sb.st_mode;
    file_size = sb.st_size;

    /* Create the new regular file with small permissions initially,
       to not create a security hole.  */

 retry_dst_open:
    if ((dest_desc = open (dst_path, O_WRONLY | O_CREAT | O_TRUNC, 0600)) < 0){
	return_status = file_error (" Cannot create target file \"%s\" \n %s ", dst_path);
	if (return_status == FILE_RETRY)
	    goto retry_dst_open;
	goto ret2;
    }

 retry_dst_fstat:
    /* Find out the optimal buffer size.  */
    if (fstat (dest_desc, &sb)){
	return_status = file_error (" Cannot fstat target file \"%s\" \n %s ", dst_path);
	if (return_status == FILE_RETRY)
	    goto retry_dst_fstat;
	goto ret2;
    }

    buf_size = 8*1024;

    buf = (char *) xmalloc (buf_size, "copy_file_file");

    return_status = show_file_progress (0, file_size);
    if (return_status != FILE_CONT)
	goto ret3;

    for (;;){
    retry_src_read:
	n_read = read (source_desc, buf, buf_size);
	if (n_read < 0){
	    return_status = file_error (" Cannot read source file \"%s\" \n %s ", src_path);
	    if (return_status == FILE_RETRY)
		goto retry_src_read;
	    goto ret3;
	}
	if (n_read == 0)
	    break;

	n_read_total += n_read;

    retry_dst_write:
	n_written = write (dest_desc, buf, n_read);
	if (n_written < n_read){
	    return_status = file_error (" Cannot write target file \"%s\" \n %s ", dst_path);
	    if (return_status == FILE_RETRY)
		goto retry_dst_write;
	    goto ret3;
	}
	return_status = show_file_progress (n_read_total, file_size);
	if (return_status != FILE_CONT)
	    goto ret3;
    }

 retry_dst_chmod:
    if (chmod (dst_path, src_mode)){
	temp_status = file_error (" Cannot chmod target file \"%s\" \n %s ", dst_path);
	if (temp_status == FILE_RETRY)
	    goto retry_dst_chmod;
	return_status = temp_status;
    }

 ret:
    free (buf);
 retry_dst_close:
    if (close (dest_desc) < 0){
	temp_status = file_error (" Cannot close target file \"%s\" \n %s ", dst_path);
	if (temp_status == FILE_RETRY)
	    goto retry_dst_close;
	return_status = temp_status;
    }

 ret2:
 retry_src_close:
    if (close (source_desc) < 0){
	temp_status = file_error (" Cannot close source file \"%s\" \n %s ", src_path);
	if (temp_status == FILE_RETRY)
	    goto retry_src_close;
	if (temp_status == FILE_ABORT)
	    return_status = temp_status;
    }

    return return_status;

 ret3:
    /* Remove short file */
    unlink (dst_path);
    goto ret;
}

#if 0
static int copy_file_dir (char *s, char *d)
{
    int status;
    
    char *dest_file = get_full_name (d, s);
    status = copy_file_file (s, dest_file);
    free (dest_file);
    return status;
}
#endif

/*
 * Me thinks these copy_*_* functions should have a return type.
 * anyway, this function *must* have two directories as arguments.
 */
/* FIXME: This function needs to check the return values of the
   function calls */
int copy_dir_dir (char *s, char *d, int toplevel)
{
    struct dirent *next;
    struct stat   buf, cbuf;
    DIR    *reading;
    char   *path, *mdpath, *dest_file, *dest_dir;
    int    return_status = FILE_CONT;

    /* First get the mode of the source dir */
 retry_src_stat:
    if (stat (s, &cbuf)){
	return_status = file_error (" Cannot stat source directory \"%s\" \n %s ", s);
	if (return_status == FILE_RETRY)
	    goto retry_src_stat;
	return return_status;
    } else if (!S_ISDIR (cbuf.st_mode)){
	return_status = file_error (" Source directory \"%s\" is not a directory ", s);
	if (return_status == FILE_RETRY)
	    goto retry_src_stat;
	return return_status;
    }

    /* Now, check if the dest dir exists, if not, create it. */
    if (stat (d, &buf)){
    	/* Here the dir doesn't exist : make it !*/

    	dest_dir = copy_strings (d, 0);
    } else {
        /*
         * If the destination directory exists, we want to copy the whole
         * directory, but we only want this to happen once.
	 *
	 * Escape sequences added to the * to avoid compiler warnings.
         * so, say /bla exists, if we copy /tmp/\* to /bla, we get /bla/tmp/\*
         * or ( /bla doesn't exist )       /tmp/\* to /bla     ->  /bla/\*
         */
        if (toplevel){
            dest_dir = copy_strings (d, "/", basename (s), 0);
	} else {
	    dest_dir = copy_strings (d, 0);
	    goto dont_mkdir;
	}
    }
 retry_dst_mkdir:
    if (my_mkdir (dest_dir, cbuf.st_mode)){
	return_status = file_error (" Cannot create target directory \"%s\" \n %s ", dest_dir);
	if (return_status == FILE_RETRY)
	    goto retry_dst_mkdir;
	return return_status;
    }

 dont_mkdir:
    /* open the source dir for reading */
    reading = opendir (s);

    while ((next = readdir (reading)) && return_status != FILE_ABORT){
        /*
         * Now, we don't want '.' and '..' to be created / copied at any time 
         */
        if (!strcmp (next->d_name, "."))
            continue;
        if (!strcmp (next->d_name, ".."))
           continue;

        /* get the filename and add it to the src directory */
        path = copy_strings (s, "/", next->d_name, 0);
        stat (path, &buf);

        if (S_ISDIR (buf.st_mode)){
            mdpath = copy_strings (dest_dir, "/", next->d_name, 0);
            /*
             * From here, we just intend to recursively copy subdirs, not
             * the double functionality of copying different when the target
             * dir already exists. So, we give the recursive call the flag 0
             * meaning no toplevel.
             */
            return_status = copy_dir_dir (path, mdpath, 0);
	    free (mdpath);
	} else {
	    dest_file = copy_strings (dest_dir, "/", basename (path), 0);
            return_status = copy_file_file (path, dest_file);
	    free (dest_file);
	}
        free (path);
    }
    closedir (reading);
    free (dest_dir);
    return return_status;
}


static int move_file_file (char *s, char *d)
{
    struct stat src_stats, dst_stats;
    int return_status = FILE_CONT;

    if (show_source (s) == FILE_ABORT
	|| show_target (d) == FILE_ABORT)
	return FILE_ABORT;

 retry_src_lstat:
    if (lstat (s, &src_stats) != 0){
	/* Source doesn't exist */
	return_status = file_error (" Cannot stat file \"%s\" \n %s ", s);
	if (return_status == FILE_RETRY)
	    goto retry_src_lstat;
	return return_status;
    }

    if (lstat (d, &dst_stats) == 0){
	/* Destination already exists */
	if (src_stats.st_dev == dst_stats.st_dev
	    && src_stats.st_ino == dst_stats.st_ino){
	    message (1, " Error ", " `%s' and `%s' are the same file ", s, d);
	    refresh_op_win ();
	    return FILE_SKIP;
	}
	if (S_ISDIR (dst_stats.st_mode)){
	    message (1, " Error ", " Cannot overwrite directory `%s' ", d);
	    refresh_op_win ();
	    return FILE_SKIP;
	}

/* Perhaps this should be configurable? See also main.c */
#define confirm_overwrite 1

	if (confirm_overwrite){
	    return_status = query_replace (d, &src_stats, &dst_stats);
	    if (return_status != FILE_CONT)
		return return_status;
	}
	/* Ok to overwrite */
    }

 retry_rename:
    if (rename (s, d) == 0)
	return FILE_CONT;

    if (errno != EXDEV){
	return_status = files_error (" Cannot move file \"%s\" to \"%s\" \n %s ", s, d);
	if (return_status == FILE_RETRY)
	    goto retry_rename;
	return return_status;
    }

    /* Failed because filesystem boundary -> copy the file instead */
    if ((return_status = copy_file_file (s, d)) != FILE_CONT)
	return return_status;
    if ((return_status = show_source (NULL)) != FILE_CONT
	|| (return_status = show_file_progress (0, 0)) != FILE_CONT)
	return return_status;

 retry_src_remove:
    if (unlink (s)){
	return_status = file_error (" Cannot remove file \"%s\" \n %s ", s);
	if (return_status == FILE_RETRY)
	    goto retry_src_remove;
	return return_status;
    }

    return FILE_CONT;
}

#if 0
static int move_file_dir (char *s, char *d)
{
    int status;
    
    char *dest_file = get_full_name (d, s);
    status = move_file_file (s, dest_file);
    free (dest_file);
    return status;
}
#endif

int move_dir_dir (char *s, char *d)
{
    struct stat sbuf, dbuf, destbuf;
    char *destdir;
    int return_status;

    if (show_source (s) == FILE_ABORT
	|| show_target (d) == FILE_ABORT)
	return FILE_ABORT;

    stat (s, &sbuf);
    if (stat (d, &dbuf))
	destdir = copy_strings (d, 0);
    else
	destdir = copy_strings (d, "/", basename (s), 0);

    /* Check if the user inputted an existing dir */
 retry_dst_stat:
    if (!stat (destdir, &destbuf)){
	if (S_ISDIR (destbuf.st_mode))
	    return_status = file_error (" Cannot overwrite directory \"%s\" ", destdir);
	else
	    return_status = file_error (" Cannot overwrite file \"%s\" ", destdir);
	if (return_status == FILE_RETRY)
	    goto retry_dst_stat;
        free (destdir);
        return return_status;
    }

 retry_rename:
    if (rename (s, destdir) == 0){
	return_status = FILE_CONT;
	goto ret;
    }

    if (errno != EXDEV){
	return_status = files_error (" Cannot move directory \"%s\" to \"%s\" \n %s ", s, d);
	if (return_status == FILE_RETRY)
	    goto retry_rename;
	goto ret;
    }

    /* Failed because of filesystem boundary -> copy dir instead */
    if ((return_status = copy_dir_dir (s, destdir, 0)) != FILE_CONT)
	goto ret;
    if ((return_status = show_source (NULL)) != FILE_CONT
	|| (return_status = show_file_progress (0, 0)) != FILE_CONT)
	goto ret;

    return_status = recursive_erase (s);

 ret:
    free (destdir);
    return return_status;
}


static int erase_file (char *s)
{
    int return_status;

    if (show_deleting (s) == FILE_ABORT)
	return FILE_ABORT;

 retry_unlink:
    if (unlink (s)){
	return_status = file_error (" Cannot delete file \"%s\" \n %s ", s);
	if (return_status == FILE_RETRY)
	    goto retry_unlink;
	return return_status;
    }
    return FILE_CONT;
}

static int recursive_erase (char *s)
{
    struct dirent *next;
    struct stat	buf;
    DIR    *reading;
    char   *path;
    int    return_status = FILE_CONT;

    if (!strcmp (s, ".."))
	return 1;
    
    reading = opendir (s);
    
    if (!reading)
	return 1;

    while ((next = readdir (reading)) && return_status == FILE_CONT){
	if (!strcmp (next->d_name, "."))
	    continue;
   	if (!strcmp (next->d_name, ".."))
	    continue;
	path = copy_strings (s, "/", next->d_name, 0);
   	if (lstat (path, &buf)){
	    free (path);
	    return 1;
	} 
	if (S_ISDIR (buf.st_mode))
	    return_status = (recursive_erase (path) != FILE_CONT);
	else
	    return_status = erase_file (path);
	free (path);
    }
    closedir (reading);
    if (return_status != FILE_CONT)
	return return_status;
    if (show_deleting (s) == FILE_ABORT)
	return FILE_ABORT;
 retry_rmdir:
    if (my_rmdir (s)){
	return_status = file_error (" Cannot remove directory \"%s\" \n %s ", s);
	if (return_status == FILE_RETRY)
	    goto retry_rmdir;
	return return_status;
    }
    return FILE_CONT;
}

int erase_dir (char *s)
{
    int error;

    if (strcmp (s, "..") == 0)
	return FILE_SKIP;

    if (strcmp (s, ".") == 0)
	return FILE_SKIP;

    if (show_deleting (s) == FILE_ABORT)
	return FILE_ABORT;

 retry_rmdir:
    error = my_rmdir (s);
    if (error && (errno == ENOTEMPTY || errno == EEXIST)){
	error = query_recursive (s);
	if (error == FILE_CONT)
	    return recursive_erase (s);
	else
	    return error;
    }
    if (error){
	error = file_error (" Cannot remove directory \"%s\" \n %s ", s);
	if (error == FILE_RETRY)
	    goto retry_rmdir;
	return error;
    }
    return FILE_CONT;
}

#if 0
static FILE *mopen (char *filename, char *mode)
{
    FILE *f;
    
    if ((f = fopen (filename, mode)) == NULL){
	message (1, " Error ", " Can't open '%s' \n %s ",
		 filename, unix_error_string (errno));
	return 0;
    }
    return f;
}
#endif

int mask_rename (char *source_mask)
{
    char ren_file [] = " Rename file \"%s\" to \"%s\"? ";
    char *orig_mask, *target_mask;
    char *fnsource, fntarget [MAXPATHLEN], *msg;
    const char *error;
    int i, j, k, l, next_reg;
    int len, answer = 0;
    int count = 0, total;
    struct re_pattern_buffer rx;
    struct re_registers regs;

 ask_source_mask:
    if (!source_mask)
	source_mask = input_dialog (" Mask rename ", "Type source mask:", "");
    if (!source_mask || !*source_mask)
	return FILE_ABORT;
    rx.buffer = (char *) malloc (256);
    rx.allocated = 256;
    rx.translate = 0;
    orig_mask = source_mask;
    source_mask = convert_pattern (source_mask, match_file, 1);
    error = re_compile_pattern (source_mask, strlen (source_mask), &rx);
    if (error){
	message (1, " Error ", " Invalid pattern \"%s\" \n %s ", orig_mask, error);
	free (rx.buffer);
	source_mask = NULL;
	goto ask_source_mask;
    }
    target_mask = input_dialog (" Mask rename ", "Type target mask:", "");
    if (!target_mask || !*target_mask){
	free (rx.buffer);
	return FILE_ABORT;
    }

    create_op_win (OP_MOVE);
    if (cpanel->marked)
	total = cpanel->marked;
    else
	total = cpanel->count;

    for (i = 0; i < cpanel->count; i++){
	file_entry *source = &cpanel->dir.list[i];

	/* Skip directories */
	if (S_ISDIR (source->buf.st_mode))
	    continue;
	/* If there is marked files skip the unmarked ones */
	if (cpanel->marked && !source->f.marked)
	    continue;

	/* Match the source mask */
	fnsource = source->fname;
	len = strlen (fnsource);
	j = re_match (&rx, fnsource, len, 0, &regs);
	if (j != len){
	    /* Didn't match */
	    continue;
	}

	/* Match the target mask */
	next_reg = 1;
	for (j = 0, k = 0; j < strlen (target_mask); j++){
	    switch (target_mask [j]){
	    case '\\':
		next_reg = target_mask [++j] - '0';
		/* Fall through */

	    case '*':
		if (next_reg < 0 || next_reg >= RE_NREGS
		    || regs.start [next_reg] < 0){
		    message (1, " Error ", " Invalid target mask ");
		    answer = 3;
		    goto break_from_loop;
		}
		for (l = regs.start [next_reg]; l < regs.end [next_reg]; l++)
		    fntarget [k++] = fnsource [l];
		next_reg ++;
		break;

	    default:
		fntarget [k++] = target_mask [j];
		break;
	    }
	}
	fntarget [k] = 0;

	msg = xmalloc (strlen (fnsource) + strlen (fntarget) +
		       sizeof (ren_file), "mask_ren_cmd, msg");
	sprintf (msg, ren_file, fnsource, fntarget);
	if (answer != 2)
	    answer = query_dialog (" Mask rename ", msg, 0, 4,
				   " Yes ", " No ", " All ", " Cancel ");
	refresh_op_win ();
	if (answer == 0 || answer == 2){
	    if (move_file_file (fnsource, fntarget) == FILE_ABORT){
		answer = 3;
		break;
	    }
	}

	free (msg);
	if (answer == 3 || answer < 0)
	    break;
	count ++;
	if (show_count_progress (count, total) == FILE_ABORT){
	    answer = 3;
	    break;
	}
    }
 break_from_loop:
    destroy_op_win ();

    free (rx.buffer);
    free (orig_mask);
    free (target_mask);

    if (answer == 1)
	return FILE_SKIP;
    if (answer == 3 || answer < 0)
	return FILE_ABORT;
    return FILE_CONT;
}

/* Returns currently selected file or the first marked file if there is one */
static char *get_file (Panel *panel, struct stat *stat_buf)
{
    int i;

    if (panel->view_type == view_tree){
	stat (panel->cwd, stat_buf);
	return panel->cwd;
    } else if (panel->marked){
	for (i = 0; i < panel->count; i++)
	    if (panel->dir.list [i].f.marked){
		*stat_buf = panel->dir.list [i].buf;
		return panel->dir.list [i].fname;
	    }
    } else {
	*stat_buf = panel->dir.list [panel->selected].buf;
	return panel->dir.list [panel->selected].fname;
    }
    fprintf (stderr, " Internal error: get_file \n");
    mi_getch ();
    return "";
}

/* Returns 1 if did change the directory structure,
   Returns 0 if user aborted */
int panel_operate (int operation)
{
    char *source = NULL;
    char *dest = NULL;
    char *temp = NULL;
    struct stat src_stat, dst_stat;
    int i, value;
    long marked, total;
    long count = 0, bytes = 0;
    int  dst_result;

    if (cpanel->view_type != view_tree
	&&!cpanel->marked && !strcmp (selection->fname, "..")){
	message (1, " Error ", " Can't operate on \"..\"! ");
	return 0;
    }

    if (operation < OP_COPY || operation > OP_DELETE)
	return 0;

    /* Generate confirmation prompt */
    if (cpanel->view_type != view_tree && cpanel->marked > 1){
	sprintf (cmd_buf, "%s%d %s%s ", op_names [operation], cpanel->marked,
		 (cpanel->marked == cpanel->dirs_marked) ? "directories" :
		 (cpanel->dirs_marked) ? "files/directories" : "files",
		 (operation == OP_DELETE) ? "?" : " to:");
    } else {
	source = get_file (cpanel, &src_stat);
	sprintf (cmd_buf,"%s%s \"%s\"%s ", op_names [operation],
	         S_ISDIR (src_stat.st_mode) ? "directory" : "file",
		 name_trunc (source, 30),
		 (operation == OP_DELETE) ? "?" : " to:");
    }

    /* Show confirmation dialog */
    if (operation == OP_DELETE){
	i = query_dialog (op_names [operation], cmd_buf,
			  3, 2, " Yes ", " No ");
	if (i != 0)
	    return 0;
    } else {
	dest = input_expand_dialog (op_names [operation],
				    cmd_buf, opanel->cwd);
	if (!dest)
	    return 0;
	if (!*dest){
	    free (dest);
	    return 0;
	}
    }

    /* Initialize things */
    create_op_win (operation);

    /* Now, let's do the job */
    if (cpanel->view_type == view_tree || cpanel->marked <= 1){
	/* One file */

	if (operation != OP_COPY && cpanel->view_type == view_tree)
	    chdir ("/");
	/* The source and src_stat variables have been initialized before */
	if (operation == OP_DELETE){
	    /* Delete operation */
	    if (S_ISDIR (src_stat.st_mode))
		value = erase_dir (source);
	    else
		value = erase_file (source);
	} else {
	    /* Copy or move operation */
	    dst_result = stat (dest, &dst_stat);
	    if (dst_result == 0 && S_ISDIR (dst_stat.st_mode)){
		/* Dest is directory */
		temp = get_full_name (dest, source);
		free (dest);
		dest = temp;
		temp = 0;
	    }
	    switch (operation){
	    case OP_COPY:
		if (S_ISDIR (src_stat.st_mode))
		    value = copy_dir_dir (source, dest, 1);
		else
		    value = copy_file_file (source, dest);
		break;
	    case OP_MOVE:
		if (S_ISDIR (src_stat.st_mode))
		    value = move_dir_dir (source, dest);
		else
		    value = move_file_file (source, dest);
		break;
	    default:
		message (1, " Internal failure ", " Unknown file operation ");
	    }
	} /* Copy or move operation */

	if (value == FILE_CONT)
	    unmark_file (cpanel);

    } else {
	/* Many files */

	if (operation != OP_DELETE){
	    /* Check destination for copy or move operation */
	retry_many_dst_stat:
	    dst_result = stat (dest, &dst_stat);
	    if (dst_result == 0 && !S_ISDIR (dst_stat.st_mode)){
		if (file_error (" Destination \"%s\" must be a directory ", dest) == FILE_RETRY)
		    goto retry_many_dst_stat;
		goto clean_up;
	    }
	}

	/* Initialize variables for progress bars */
	marked = cpanel->marked;
	total = cpanel->total;

	/* Loop for every file */
	for (i = 0; i < cpanel->count; i++){
	    if (!cpanel->dir.list [i].f.marked)
		continue;	/* Skip the unmarked ones */
	    source = cpanel->dir.list [i].fname;
	    src_stat = cpanel->dir.list [i].buf;	/* Inefficient, should we use pointers? */
	    if (operation == OP_DELETE){
		/* Delete operation */
		if (S_ISDIR (src_stat.st_mode))
		    value = erase_dir (source);
		else
		    value = erase_file (source);
	    } else {
		/* Copy or move operation */
		if (temp)
		    free (temp);
		temp = get_full_name (dest, source);
		switch (operation){
		case OP_COPY:
		    if (S_ISDIR (src_stat.st_mode))
			value = copy_dir_dir (source, temp, 1);
		    else
			value = copy_file_file (source, temp);
		    break;
		case OP_MOVE:
		    if (S_ISDIR (src_stat.st_mode))
			value = move_dir_dir (source, temp);
		    else
		        value = move_file_file (source, temp);
		    break;
		default:
		    message (1, " Internal failure ",
			     " Unknown file operation ");
		    goto clean_up;
		}
	    } /* Copy or move operation */

	    if (value == FILE_ABORT)
		goto clean_up;
	    if (value == FILE_CONT){
		cpanel->dir.list [i].f.marked = 0;
		cpanel->marked --;
		if (S_ISDIR (src_stat.st_mode))
		    cpanel->dirs_marked --;
		cpanel->total -= src_stat.st_size;
	    }
	    count ++;
	    if (show_count_progress (count, marked) == FILE_ABORT)
		goto clean_up;
	    bytes += src_stat.st_size;
	    if (show_bytes_progress (bytes, total) == FILE_ABORT)
		goto clean_up;
	    if (operation != OP_DELETE && verbose
		&& show_file_progress (0, 0) == FILE_ABORT)
		goto clean_up;
	} /* Loop for every file */
    } /* Many files */

 clean_up:
    /* Clean up */
    destroy_op_win ();
    if (dest)
	free (dest);
    if (temp)
	free (temp);
    return 1;
}
