------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                          A D A . R E A L _ T I M E                       --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.15 $                             --
--                                                                          --
--       Copyright (c) 1991,1992,1993,1994, FSU, All Rights Reserved        --
--                                                                          --
-- GNARL is free software; you can redistribute it  and/or modify it  under --
-- terms  of  the  GNU  Library General Public License  as published by the --
-- Free Software  Foundation;  either version 2, or (at  your  option)  any --
-- later  version.  GNARL is distributed  in the hope that  it will be use- --
-- ful, but but WITHOUT ANY WARRANTY;  without even the implied warranty of --
-- MERCHANTABILITY  or  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
-- eral Library Public License  for more details.  You should have received --
-- a  copy of the GNU Library General Public License along with GNARL;  see --
-- file COPYING.LIB.  If not,  write to the  Free Software Foundation,  675 --
-- Mass Ave, Cambridge, MA 02139, USA.                                      --
--                                                                          --
------------------------------------------------------------------------------

with System.Task_Clock;
--  Used for, Time definitions and operations.

with System.Task_Clock.Machine_Specifics;
--  Used for, Task_Clock.Machine_Specifics.Clock

with Unchecked_Conversion;

package body Ada.Real_Time is

   use System.Task_Clock;
   --  for time operators.

   -----------
   -- Clock --
   -----------

   function Clock return Time is
   begin
      return Time (Task_Clock.Machine_Specifics.Clock);
   end Clock;

   ---------
   -- "<" --
   ---------

   function "<" (Left, Right : Time) return Boolean is
   begin
      return Task_Clock.Stimespec (Left) < Task_Clock.Stimespec (Right);
   end "<";

   function "<" (Left, Right : Time_Span) return Boolean is
   begin
      return Task_Clock.Stimespec (Left) < Task_Clock.Stimespec (Right);
   end "<";

   ---------
   -- ">" --
   ---------

   function ">" (Left, Right : Time) return Boolean is
   begin
      return Right < Left;
   end ">";

   function ">" (Left, Right : Time_Span) return Boolean is
   begin
      return Right < Left;
   end ">";

   ----------
   -- "<=" --
   ----------

   function "<=" (Left, Right : Time) return Boolean is
   begin
      return not (Left > Right);
   end "<=";

   function "<=" (Left, Right : Time_Span) return Boolean is
   begin
      return not (Left > Right);
   end "<=";

   ----------
   -- ">=" --
   ----------

   function ">=" (Left, Right : Time) return Boolean is
   begin
      return not (Left < Right);
   end ">=";

   function ">=" (Left, Right : Time_Span) return Boolean is
   begin
      return not (Left < Right);
   end ">=";

   ---------
   -- "+" --
   ---------

   --  Note that Constraint_Error may be propagated

   function "+" (Left : Time; Right : Time_Span) return Time is
   begin
      return Time (Task_Clock.Stimespec (Left) + Task_Clock.Stimespec (Right));
   end "+";

   function "+"  (Left : Time_Span; Right : Time) return Time is
   begin
      return Right + Left;
   end "+";

   function "+"  (Left, Right : Time_Span) return Time_Span is
   begin
      return Time_Span (Time (Right) + Left);
   end "+";

   ---------
   -- "-" --
   ---------

   --  Note that Constraint_Error may be propagated

   function "-"  (Left : Time; Right : Time_Span) return Time is
   begin
      return Time (Task_Clock.Stimespec (Left) - Task_Clock.Stimespec (Right));
   end "-";

   function "-"  (Left, Right : Time) return Time_Span is
   begin
      return Time_Span (Left - Time_Span (Right));
   end "-";

   function "-"  (Left, Right : Time_Span) return Time_Span is
   begin
      return Time_Span (Time (Left) - Right);
   end "-";

   function "-"  (Right : Time_Span) return Time_Span is
   begin
      return Time_Span_Zero - Right;
   end "-";

   ---------
   -- "/" --
   ---------

   --  Note that Constraint_Error may be propagated

   function "/"  (Left, Right : Time_Span) return integer is
   begin
      return Task_Clock.Stimespec (Left) / Task_Clock.Stimespec (Right);
   end "/";

   function "/"  (Left : Time_Span; Right : Integer) return Time_Span is
   begin
      return Time_Span (Task_Clock.Stimespec (Left) / Right);
   end "/";

   ---------
   -- "*" --
   ---------

   --  Note that Constraint_Error may be propagated

   function "*"  (Left : Time_Span; Right : Integer) return Time_Span is
   begin
      return Time_Span (Task_Clock.Stimespec (Left) * Right);
   end "*";

   function "*"  (Left : Integer; Right : Time_Span) return Time_Span is
   begin
      return Right * Left;
   end "*";

   -----------
   -- "abs" --
   -----------

   --  Note that Constraint_Error may be propagated

   function "abs" (Right : Time_Span) return Time_Span is
   begin
      if Right < Time_Span_Zero then
         return -Right;
      end if;

      return Right;
   end "abs";

   -----------------
   -- To_Duration --
   -----------------

   function To_Duration (FD : Time_Span) return Duration is
   begin
      return Task_Clock.Stimespec_To_Duration (Task_Clock.Stimespec (FD));
   end To_Duration;

   ------------------
   -- To_Time_Span --
   ------------------

   function To_Time_Span (D : Duration) return Time_Span is
   begin
      return Time_Span (Task_Clock.Duration_To_Stimespec (D));
   end To_Time_Span;

   -----------------
   -- Nanoseconds --
   -----------------

   function Nanoseconds (NS : integer) return Time_Span is
   begin
      return Time_Span_Unit * NS;
   end Nanoseconds;

   ------------------
   -- Microseconds --
   ------------------

   function Microseconds  (US : integer) return Time_Span is
   begin
      return Nanoseconds (US) * 1000;
   end Microseconds;

   -------------------
   --  Milliseconds --
   -------------------

   function Milliseconds (MS : integer) return Time_Span is
   begin
      return Microseconds (MS) * 1000;
   end Milliseconds;

   -----------
   -- Split --
   -----------

   --  D is nonnegative Time_Span

   procedure Split (T : Time; S : out Seconds_Count; D : out Time_Span) is
   begin
      S := Seconds_Count
        (Task_Clock.Stimespec_Seconds (Task_Clock.Stimespec (T)));
      D := T - Time_Of (S, Time_Span_Zero);
   end Split;

   -------------
   -- Time_Of --
   -------------

   function Time_Of (S : Seconds_Count; D : Time_Span) return Time is
   begin
      return (Time (Task_Clock.Time_Of (Integer (S), 0)) + D);
   end Time_Of;

end Ada.Real_Time;
