------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              U N I X L I B                               --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.14 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains some interface functions to the Unix library

with System; use System;
with Types;  use Types;

package Unixlib is

   type Unix_FD is new Int;
   --  Unix file descriptor

   Unix_Standin  : constant Unix_FD := 0;
   Unix_Standout : constant Unix_FD := 1;
   Unix_Standerr : constant Unix_FD := 2;
   --  File descriptors for standard input output files

   function Unix_File_Length (FD : Unix_FD) return Int;
   --  Get length of unix file from descriptor

   function Unix_File_Time_Stamp (Name : Address) return Time_Stamp_Type;
   --  Get time stamp of file from name (use for unopened file)

   function Unix_File_Time_Stamp (FD : Unix_FD) return Time_Stamp_Type;
   --  Get time stamp of file from descriptor

   procedure Unix_Write (FD : Unix_FD; A : Address; N : Int);
   pragma Import (C, Unix_Write, "write");
   --  Write N bytes from address A to file referenced by FD

   procedure Unix_Exit  (Status : Int);
   pragma Import (C, Unix_Exit, "exit");
   --  Exit to Unix with given status code (program is terminated)

   procedure Unix_Abort;
   pragma Import (C, Unix_Abort, "abort");
   --  Exit to Unix signalling an abort (traceback or other appropriate
   --  diagnostic information should be given if possible, or entry made
   --  to the debugger if that is possible).

   function  Unix_Read  (FD : Unix_FD; A : Address; N : Int) return Int;
   pragma Import (C, Unix_Read, "read");
   --  Read N bytes to address A from file referenced by FD. Returned value
   --  is count of bytes actually read, which can be less than N at EOF.

   function  Unix_Open_Read (Name : Address) return Unix_FD;
   pragma Import (C, Unix_Open_Read, "open_read");
   --  Open file Name (NUL-terminated) for reading, returning file descriptor
   --  File descriptor returned is negative if file cannot be opened.

   function Unix_Create_File (Name : Address) return Unix_FD;
   pragma Import (C, Unix_Create_File, "open_create");
   --  Creates new file with given name (NUL-terminated) for writing, returning
   --  file descriptor for subsequent use in Unix_Write calls. File desctiptor
   --  returned is negative if file cannot be successfully created.

   procedure Unix_Close (FD : Unix_FD);
   pragma Import (C, Unix_Close, "close");
   --  Close file referenced by FD

   function Unix_Is_Regular_File (Name : String) return Boolean;
   --  The named file exists and is a regular file

   function Unix_Getenv (Name : String) return String;
   --  Get the value of the environment variable -- null string if it
   --  doesn't exist

end Unixlib;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.12
--  date: Sun Jan  9 11:22:12 1994;  author: dewar
--  New header with 1994 copyright
--  ----------------------------
--  revision 1.13
--  date: Wed Jan 26 18:45:46 1994;  author: porter
--  Add Unix_Is_Regular_File and Unix_Getenv
--  ----------------------------
--  revision 1.14
--  date: Tue May 31 18:08:02 1994;  author: porter
--  Change Interface/Interface_Name pragmas to Import
--  (Unix_File_Time_Stamp): new filename version added
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
