/* Dlg box features module for the Midnight Commander
   Copyright (C) 1994 Radek Doulik, Miguel de Icaza

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Known bugs: currently widgets can't refuse the WIDGET_FOCUS, this
   must be changed to allow a widget to refuse a FOCUS message.
 */

static char rcsid [] = "$Id: dlg.c,v 1.2 1995/01/27 02:34:42 miguel Exp $";
#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <ncurses.h>
#include <stdarg.h>
#include "mad.h"
#include "menu.h"
#include "global.h"
#include "win.h"
#include "color.h"
#include "util.h"
#include "mouse.h"
#include "input.h"
#include "help.h"
#include "key.h"	/* For mi_getch() */
#include "dlg.h"
#include "dialog.h"	/* For push_refresh() and pop_refresh() */

/* draw box in window */
void draw_box (WINDOW *w, int y, int x, int ys, int xs)
{
    mvwaddch (w, y, x, ACS_ULCORNER);
    whline (w, ACS_HLINE, xs - 2);
    mvwaddch (w, y + ys - 1, x, ACS_LLCORNER);
    whline (w, ACS_HLINE, xs - 2);

    mvwaddch (w, y, x + xs - 1, ACS_URCORNER);
    mvwaddch (w, y + ys - 1, x + xs - 1, ACS_LRCORNER);

    wmove (w, y + 1, x);
    wvline (w, ACS_VLINE, ys - 2);
    wmove (w, y + 1, x + xs - 1);
    wvline (w, ACS_VLINE, ys - 2);
}

void init_widget (Widget *w, int y, int x,
		  int (*callback)(Dlg_head *, int, int),
		  destroy_fn destroy)
{
    w->x = x;
    w->y = y;
    w->color = 0;
    w->callback = callback;
    w->destroy  = destroy;
}

int default_proc (Dlg_head *h, int Msg, int Par)
{
    switch (Msg){

    case WIDGET_HOTKEY:         /* Didn't use the key */
        return 0;

    case WIDGET_INIT:		/* We could tell if something went wrong */
	return 1;
	
    case WIDGET_KEY:
	return 0;		/* Didn't use the key */
	
    case WIDGET_FOCUS:		/* We accept FOCUSes */
	return 1;
	
    case WIDGET_UNFOCUS:	/* We accept loose FOCUSes */
	return 1;
	
    case WIDGET_DRAW:
	return 1;

    case WIDGET_DESTROY:
	return 1;

    case WIDGET_CHECK_HOTKEY:
	return 1;
    }
    printf ("Internal error: unhandled message: %d\n", Msg);
    return 1;
}

static int default_dlg_callback (Dlg_head *unused, int id, int msg)
{
    return 0;
}

/* creates new dialog circle buffer */
Dlg_head *dlg_new (WINDOW *w, chtype *col,
		   int (*callback) (struct Dlg_head *, int, int),
		   int starty, int startx, char *help_ctx)
{
    Dlg_head *new_d;

    new_d = (Dlg_head *) malloc (sizeof (Dlg_head));
    new_d->current = NULL;
    new_d->count = 0;
    new_d->direction = DIR_FORWARD;
    new_d->window = w;
    new_d->color = col;
    new_d->help_ctx = help_ctx;
    new_d->callback = callback ? callback : default_dlg_callback;
    new_d->send_idle_msg = 0;
    
    push_event (1, 1, COLS, LINES, (mouse_h) null_event, 0, event_absolute);
    /* Was:     push_frame (startx, starty+1, 1); */
    push_frame (startx, starty+1, 0);
    return (new_d);
}

void set_idle_proc (Dlg_head *d, int state)
{
    d->send_idle_msg = state;
}

/* add component to dialog buffer */
int add_widget (Dlg_head *where, void *what)
{
    Widget_Item *back;

    /* Don't accept 0 widgets, this could be from widgets that could not */
    /* initialize properly */
    if (!what)
	return 0;
    
    back = where->current;
    where->current = (Widget_Item *) malloc (sizeof (Widget_Item));
    if (back){
	back->prev = where->current;
	where->current->next = back;
    } else {
	where->current->next = where->current;
	where->first = where->current;
    }

    where->current->dlg_id = where->count;
    where->current->prev = where->first;
    where->last = where->current;
    where->first->next = where->last;

    where->current->widget = what;
    where->current->widget->parent = where;

    where->count++;
    return (where->count - 1);
}

/* broadcast a message to all the widgets in a dialog */
void dlg_broadcast_msg (Dlg_head *h, int message)
{
    h->current=h->last;
    do {
	send_message (h->current, h, message, 0);
	h->current = h->current->next;
    } while (h->current != h->last);
}

int dlg_FOCUS (Dlg_head *h)
{
    if (send_message (h->current, h, WIDGET_FOCUS, 0)){
	(*h->callback) (h, h->current->dlg_id, DLG_FOCUS);
	return 1;
    }
    return 0;
}

int dlg_unFOCUS (Dlg_head *h)
{
    if (send_message (h->current, h, WIDGET_UNFOCUS, 0)){
	(*h->callback) (h, h->current->dlg_id, DLG_UNFOCUS);
	return 1;
    }
    return 0;
}

void dlg_one_up (Dlg_head *h)
{
    /* If it accepts unFOCUSion */
    if (dlg_unFOCUS(h)){
	if (h->direction)
	    h->current = h->current->prev;
	else
	    h->current = h->current->next;
	
	(*h->callback) (h, h->current->dlg_id, DLG_ONE_UP);
	dlg_FOCUS (h);
    }
}

void dlg_one_down (Dlg_head *h)
{
    if (dlg_unFOCUS (h)){
	if (h->direction)
	    h->current = h->current->next;
	else
	    h->current = h->current->prev;
	
	(*h->callback) (h, h->current->dlg_id, DLG_ONE_DOWN);
	dlg_FOCUS (h);
    }
}

int dlg_select_widget (Dlg_head *h, void *w)
{
    if (dlg_unFOCUS (h)){
	while (h->current->widget != w)
	    h->current = h->current->next;
	dlg_FOCUS (h);

	wrefresh (h->window);
	return 1;
    }
    return 0;
}

static inline void dlg_redraw (Dlg_head *h)
{
    (h->callback)(h, 0, DLG_DRAW);
    dlg_broadcast_msg (h, WIDGET_DRAW);

    /* Another redraw, just to set the cursor pos */
    send_message (h->current, h, WIDGET_DRAW, 0);
}

static void dlg_refresh (void *parameter)
{
    extern void clr_scr ();

    dlg_redraw ((Dlg_head *) parameter); 
    touchwin (((Dlg_head *) parameter)->window);
    wrefresh (((Dlg_head *) parameter)->window);
}

static inline void dialog_handle_key (Dlg_head *h, int d_key)
{
    switch (d_key){
    case KEY_LEFT:
    case KEY_UP:
	dlg_one_up (h);
	break;
	
    case KEY_RIGHT:
    case KEY_DOWN:
	dlg_one_down (h);
	break;

    case KEY_F(1):
	interactive_display (LIBDIR "mc.hlp", h->help_ctx);
	do_refresh ();
	break;

    case XCTRL('l'):
	do_refresh ();
	break;
	
    case '\n':
	h->ret_value = B_ENTER;
	h->running = 0;
	break;
	
    case ESC_CHAR:
    case KEY_F (10):
    case XCTRL ('c'):
    case XCTRL ('g'):
	h->running = 0;
	h->ret_value = B_CANCEL;
	break;
    }
}

/* dialog handler */
void run_dlg (Dlg_head *h)
{
    int d_key;
    int cb_status;
    int try_hotkey;

    Widget_Item *hot_cur;

    /* Initialize dialog manager and widgets */
    (*h->callback) (h, 0, DLG_INIT);
    dlg_broadcast_msg (h, WIDGET_INIT);

    push_refresh (dlg_refresh, h, REFRESH_COVERS_PART);
    
    /* Redraw the screen */
    dlg_redraw (h);
    
    /* Initialize direction */
    if (!h->direction)
	h->current =  h->first;
    
    dlg_FOCUS (h);		/* FOCUS first (last :-)) */
    wrefresh (h->window);
    
    h->running = 1;
    
    while (h->running) {
	if (h->send_idle_msg)
	    while (is_idle ())
		(*h->callback) (h, 0, DLG_IDLE);
	
	d_key = mi_getch ();

	if (d_key == -1)
	    continue;

	/* TAB used to cycle */
	if (d_key == '\t')
	    dlg_one_down (h);
	else {
	    /* first can dlg_callback handle the key */
	    
	    cb_status = (*h->callback) (h, d_key, DLG_KEY);

	    try_hotkey = (*h->current->widget->callback)
		(h, WIDGET_CHECK_HOTKEY, 0);
	    
	    if (!cb_status && try_hotkey){ /* if not used try HOT key */
		hot_cur = h->current;
		h->current = h->last;
		
		/* send it to all widgets */
		do {
		    cb_status |= (*h->current->widget->callback)
			(h, WIDGET_HOTKEY, d_key);
		    if (cb_status)
			break;
		    h->current = h->current->next;
		} while (h->current != h->last);
		
		if (cb_status){
		    Widget_Item *defoc = hot_cur;
		    
		    hot_cur = h->current;
		    h->current = defoc;
		    dlg_unFOCUS (h);
		    h->current = hot_cur;
		    dlg_FOCUS (h);
		} else
		    h->current = hot_cur;
	    }
	    
	    /* not used - then try widget_callback */
	    if (!cb_status)
		cb_status |= (*h->current->widget->callback)
		    (h, WIDGET_KEY, d_key);
	    
	    /* unhandled key - then use standard */
	    if (!cb_status)
		dialog_handle_key (h, d_key);
	    else
		(*h->current->widget->callback) (h, WIDGET_FOCUS, 0);

	    cb_status = (*h->callback) (h, d_key, DLG_POST_KEY);
	}
	wrefresh (h->window);
    }
    (*h->callback) (h, h->current->dlg_id, DLG_END);
}

void destroy_dlg (Dlg_head *h)
{
    int i;
    Widget_Item *c;

    pop_refresh ();
    pop_frame ();
    pop_event ();

    dlg_broadcast_msg (h, WIDGET_DESTROY);
    c = h->current;
    for (i = 0; i < h->count; i++){
	if (c->widget->destroy)
	    c->widget->destroy (c->widget);
	c = h->current->next;
	free (h->current->widget);
	free (h->current);
	h->current = c;
    }

    free (h);
}

int std_callback (Dlg_head *h, int Msg, int Par)
{
    return 0;
}

