/* Windows utilities.
   Copyright (C) 1994 Miguel de Icaza, Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include "win.h"
#include "color.h"
#include "mouse.h"
#include "input.h"

static char rcsid [] = "$Id: win.c,v 1.6 1994/10/06 02:19:39 miguel Exp $";

typedef void (*fnptr)();

typedef struct Fkey_Table_List {
    fnptr actions[11];
    struct Fkey_Table_List *next;
} Fkey_Table_List;

static Fkey_Table_List *fkey_table_list = NULL;

/* Clears WINDOW *w, and leaves untouched skip spaces */
void wclrn (WINDOW *w, int skip)
{
    int x, y;
    
    for (x = skip; x <= (w->_maxx-skip); x++){
	for (y = skip; y <= (w->_maxy-skip); y++){
	    mvwaddch (w, y, x, ' ');
	}
    }
}

void set_label_text (WINDOW *fkeys, int index, char *text)
{
    wmove (fkeys, 0, 8*(index-1));
    wprintw (fkeys, "%d", index);
    wattron (fkeys, SELECTED_COLOR);
    wprintw (fkeys, "%-6s", text);
    wstandend (fkeys);
    waddch (fkeys, ' ');
}

void define_label (WINDOW *fkeys, int index, char *text, void (*cback)())
{
    fkey_table_list->actions[index] = cback;
    set_label_text (fkeys, index, text);
    if (use_mouse_p && cback)
	push_event ((index-1)*8, LINES, (index*8)-1, LINES,click,(void*)cback);
}

void define_label_quit (WINDOW *fkeys, int index, char *text, int (*cback)())
{
    fkey_table_list->actions[index] = (fnptr)cback;
    wprintw (fkeys, "%d", index);
    wattron (fkeys, SELECTED_COLOR);
    wprintw (fkeys, "%-6s", text);
    wstandend (fkeys);
    waddch (fkeys, ' ');
    if (use_mouse_p && cback){
	push_event ((index-1)*8, LINES, (index*8)-1, LINES,
		    click_may_break_loop, (void *) cback);
    }
}

/* Return values: 0 = not a fkey, other = was a fkey */
int check_fkeys (int c)
{
    int fkey;

    switch (c){
    case KEY_F(1):
	fkey = 1;
	break;
    case KEY_F(2):
	fkey = 2;
	break;
    case KEY_F(3):
	fkey = 3;
	break;
    case KEY_F(4):
	fkey = 4;
	break;
    case KEY_F(5):
	fkey = 5;
	break;
    case KEY_F(6):
	fkey = 6;
	break;
    case KEY_F(7):
	fkey = 7;
	break;
    case KEY_F(8):
	fkey = 8;
	break;
    case KEY_F(9):
	fkey = 9;
	break;
    case KEY_F(10):
	fkey = 10;
	break;
    default:
	return 0;
    }
    if (fkey_table_list->actions [fkey]){
	fkey_table_list->actions [fkey] ();
	return fkey;
    }
    else
	return 0;
}

WINDOW *push_fkey ()
{
    WINDOW *w;
    Fkey_Table_List *new;
    int i;

    new = (Fkey_Table_List*)xmalloc (sizeof (Fkey_Table_List), "push_fkey, Fkey_Table_List");
    new->next = fkey_table_list;
    fkey_table_list = new;
    for (i = 0; i < 11; i++)
	fkey_table_list->actions [i] = NULL;
    
    w = newwin (1, COLS, LINES-1, 0);
    leaveok (w, TRUE);
    
    return w;
}

void pop_fkey (WINDOW *w)
{
    Fkey_Table_List *old;

    old = fkey_table_list;
    fkey_table_list = fkey_table_list->next;
    free (old);

    delwin (w);
}

/* Return values: 0 = not a movement key, 1 = was a movement key */
int check_movement_keys (int c, int additional, int page_size,
			 movefn backfn, movefn forfn, movefn topfn, movefn bottomfn)
{
    switch (c){
    case KEY_UP:
    case XCTRL ('p'):
	backfn (1);
	return 1;
    case KEY_DOWN:
    case XCTRL ('n'):
	forfn (1);
	return 1;
    case KEY_PPAGE:
    case ALT('v'):
	backfn (page_size-1);
	return 1;
    case KEY_NPAGE:
    case XCTRL('v'):
	forfn (page_size-1);
	return 1;
    case KEY_HOME:
    case KEY_A1:
	topfn (0);
	return 1;
    case KEY_END:
    case KEY_C1:
	bottomfn (0);
	return 1;
    }
    if (additional)
        switch (c){
	case 'b':  
	case XCTRL('b'):
	case XCTRL('h'):
	case KEY_BACKSPACE:
	case 0177:
	    backfn (page_size-1);
	    return 1;
	case ' ':
	    forfn (page_size-1);
	    return 1;
	case 'u':
	    backfn (page_size / 2);
	    return 1;
	case 'd':
	    forfn (page_size / 2);
	    return 1;
	case 'g':
	    topfn (0);
	    return 1;
	case 'G':
	    bottomfn (0);
	    return 1;
	}
    return 0;
}
