/* Setup loading/saving.
   Copyright (C) 1994 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <sys/types.h>		/* Needed to include local .h files */
#include <sys/stat.h>
#include <sys/param.h>
#include <string.h>
#include <ncurses.h>

#include "dir.h"
#include "file.h"
#include "global.h"
#include "panel.h"
#include "util.h"
#include "main.h"
#include "profile.h"

static char rcsid [] = "$Id: setup.c,v 1.5 1994/10/13 04:27:58 miguel Exp $";

extern Panel *current_panel, *other_panel;
extern Panel left_panel, right_panel;
extern int max_dirt_limit;
extern int double_click_speed;

char *profile_name;

static struct {
    char *key;
    sortfn *sort_type;
} sort_names [] = {
    "name",      (sortfn *) sort_name,
    "extension", (sortfn *) sort_ext,
    "time",      (sortfn *) sort_time,
    "size",      (sortfn *) sort_size,
    "unsorted",  (sortfn *) unsorted,
    0, 0
};

static struct {
    char *key;
    int  view_type;
} view_types [] = {
    "full",  view_full,
    "brief", view_brief,
    "long",  view_long,
    "info",  view_info,
    0, 0
}; 	

void panel_save (char *profile, Panel *panel, char *section)
{
    char buffer [6];
    int  i;
    
    sprintf (buffer, "%d", panel->reverse);
    WritePrivateProfileString (section, "reverse", buffer, profile);
    for (i = 0; sort_names [i].key; i++)
	if (sort_names [i].sort_type == (sortfn *) panel->sort_type){
	    WritePrivateProfileString (section, "sort_order",
				       sort_names [i].key, profile);
	    break;
	}
    for (i = 0; view_types [i].key; i++)
	if (view_types [i].view_type == panel->view_type){
	    WritePrivateProfileString (section, "display",
				       view_types [i].key, profile);
	    break;
	}
}

static struct {
    char *opt_name;
    int  *opt_addr;
} options [] = {
    { "show_backups", &show_backups },
    { "show_dot_files", &show_dot_files },
    { "verbose", &verbose },
    { "mark_moves_down", &mark_moves_down },
    { "pause_after_run", &pause_after_run },
    { "show_mini_info", &show_mini_info },
    { "shell_patterns", &easy_patterns },
    { "auto_save_setup", &auto_save_setup },
    { "align_extensions", &align_extensions },
    { "auto_menu", &auto_menu },
    { "auto_mount", &auto_mount },
    { "use_internal_view", &use_internal_view },
    { "clear_before_exec", &clear_before_exec },
    { "mix_all_files", &mix_all_files },
    { "fast_reload", &fast_reload },
    { "fast_reload_msg_shown", &fast_reload_w },
    { "confirm_delete", &confirm_delete },
    { "mouse_repeat_rate", &mou_auto_repeat },
    { "default_list_perm", &extra_info },
    { "double_click_speed", &double_click_speed },
    { 0, 0 }
};

void save_setup ()
{
    char *profile;
    int  i;

    profile = copy_strings (home_dir, "/.mc.ini", 0);

    /* Save integer options */
    for (i = 0; options [i].opt_name; i++)
	set_int (profile, options [i].opt_name, *options [i].opt_addr);
	
    WritePrivateProfileString ("Dirs", "other_dir", opanel->cwd, profile);
    panel_save (profile, &left_panel,  "Left Panel");
    panel_save (profile, &right_panel, "Right Panel");
    sync_profiles ();
    gc_free (profile);
}

static void panel_load (char *profile, Panel *panel, char *section)
{
    int i;
    char buffer [40];
    
    panel->reverse = GetPrivateProfileInt (section, "reverse", 0, profile);
    GetPrivateProfileString (section, "sort_order", "name", buffer,
			     sizeof (buffer), profile);
    
    for (i = 0; sort_names [i].key; i++)
	if (strcasecmp (sort_names [i].key, buffer) == 0){
	    panel->sort_type = sort_names [i].sort_type;
	    break;
	}
    GetPrivateProfileString (section, "display", "full", buffer,
			     sizeof (buffer), profile);
    for (i = 0; view_types [i].key; i++)
	if (strcasecmp (view_types [i].key, buffer) == 0){
	    panel->view_type = view_types [i].view_type;
	    break;
	}
}
			  
void load_setup ()
{
    static char *buffer;
    char   *profile;
    int    i;

    /* Other modules expect load_setup to set this variables */
    left_panel.sort_type = right_panel.sort_type = (sortfn *) sort_name;
    left_panel.view_type = right_panel.view_type = view_full;

    buffer = copy_strings (home_dir, "/.mc.ini", 0);
    gc_free (buffer); /* schedule free for next garbage collect */
    
    if (exist_file (buffer)){
	profile = strdup (buffer);
    } else if (exist_file (LIBDIR "mc.ini")){
	profile = LIBDIR "mc.ini";
    } else
	return;

    profile_name = profile;

    /* Load integer boolean options */
    for (i = 0; options [i].opt_name; i++)
	*options [i].opt_addr =
	    get_int (profile, options [i].opt_name, *options [i].opt_addr);

    panel_load (profile, &left_panel, "Left Panel");
    panel_load (profile, &right_panel,"Right Panel");

    /* Only one panel can be in information mode */
    if (left_panel.view_type == view_info &&
	right_panel.view_type== view_info)
	left_panel.view_type = view_full;

    if (left_panel.view_type == view_long)
	right_panel.view_type |= VIEW_DISABLED;
    
    if (!other_dir){
	buffer = (char*) malloc (MAXPATHLEN);
	GetPrivateProfileString ("Dirs", "other_dir", ".", buffer,
			     MAXPATHLEN, profile);
	other_dir = buffer;
    }
}
