/* View file module for the MouseLess Commander
   Copyright (C) 1994 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include "win.h"
#include "color.h"
#include "util.h"
#include "input.h"
#include "dialog.h"
#include "file.h"
#include "mouse.h"

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

/* Build-in file viewer */

static WINDOW *status;
static WINDOW *fkeys;
static WINDOW *view_win;

static int  file;		/* File descriptor */
static char *last;		/* Last byte showed */
static char *last_byte;		/* Last byte of file */
static char *first;		/* First byte in file */
static char *start_display;	/* First char displayed */
static struct stat s;		/* stat for file */
static int  start_col = 0;	/* First displayed column */
static int quit = 0;		/* Quit flag */

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/view.c,v 1.11 1994/09/08 19:28:33 miguel Exp $";

static int icase_search_p (char *text, char *data)
{
    int p = icase_search (text, data) != 0; 
    return p;
}

static void view_refresh ()
{
    touchwin (fkeys);
    wrefresh (fkeys);
    touchwin (view_win);
    wrefresh (view_win);
    touchwin (status);
    wrefresh (status);
}

static void view_status ()
{
    wmove (status, 0, 26);
    wprintw (status, "Col %d", start_col);
    wmove (status, 0, COLS-5);
    wprintw (status, "%3d%%", last_byte == last ? 100 :
	     (start_display-first)*100 / s.st_size);
    wrefresh (status);
}

/* Shows the file pointed to by *start_display on view_win */
static char *display ()
{
    int col = 0;
    int row = 0;
    char *from;
    
    from = start_display;
    wclr (view_win);
    for (;row < LINES-2 && from < last_byte; from++){
	if (*from == '\n'){
	    col = 0;
	    row++;
	    continue;
	}
	if (*from == '\t'){
	    col = ((col)/8)*8 + 7;
	}
	if (!(col > COLS)){
	    int c = *from;
	    if (c < 128)
		mvwaddch (view_win, row, col, c);
	    else
		mvwaddch (view_win, row, col, '.');
	    col++;
	}
    }
    wrefresh (view_win);
    last = from;
    view_status ();
    return from;
}

static void search (char *text, int (*search)(char *, char *))
{
    char *s;
    char *p;

    if (verbose)
	message (D_INSERT, " Search ", "Searching %s", text);
    for (p = start_display; p < last_byte; ){
	for (; p < last_byte; p++)
	    if (*p == '\n'){
		p++;
		break;
	    }
	s = extract_line (p, last_byte);
	if ((*search) (text, s)){
	    start_display = p;
	    break;
	}
    }
    if (verbose)
	destroy_dialog ();
}

static void regexp_search ()
{
    char *regexp = "";
    
    refresh_fn = view_refresh;
    regexp = input_dialog (" Search ", "Enter regexp:", regexp);
    if (!regexp){
	regexp = "";
	return;
    }
    search (regexp, regexp_match);
    touchwin (view_win);
    display ();
}

static void normal_search ()
{
    char *exp = "";

    refresh_fn = view_refresh;
    exp = input_dialog (" Search ", "Enter search string:", exp);
    if (!exp){
	exp = "";
	return;
    }
    search (exp, icase_search_p);
    touchwin (view_win);
    display ();
}

static int quit_cmd ()
{
    return quit = 1;
}
    
static void init_view ()
{
    int i;
    
    status = newwin (1, COLS, 0, 0);
    view_win = newwin (LINES-2, COLS, 1, 0);
    wattron (view_win, NORMAL_COLOR);
    wattron (status, SELECTED_COLOR);
    wclr (status);

    fkeys  = new_fkey ();
    for (i = 1; i < 7; i++)
	define_label (fkeys, i, "", 0);
    define_label (fkeys, 7, "Search", normal_search);
    define_label (fkeys, 8, "RxSrch", regexp_search);
    define_label (fkeys, 9, "", 0);
    define_label_quit (fkeys, 10, "Quit", quit_cmd);
    
    /* workaround ncurses 1.8.5 bug */
    wmove (fkeys, 0, 0);
    waddch (fkeys, '1');
    clearok (view_win, TRUE);
    wrefresh (fkeys);
}

static char *move_forward (char *current, int lines)
{
    char *p;
    int  line;

    if (last == last_byte)
	return current;
    for (line = 0, p = current; p < last_byte; p++){
	if (line == lines)
	    return p;
	if (*p == '\n')
	    line++;
    }
    return current;
}

static char *move_backward (char *current, int lines)
{
    char *p;
    int line;

    if (current == first)
	return current;
    for (p = current-1; p > first; p--)
	if (*p == '\n'){
	    p--;
	    break;
	}
    /* p points to the char before the new line */
    if (p <= first)
	return current;
    
    for (line = 0; p > first; p--){
	if (*p == '\n')
	    line++;
	if (line == lines){
	    return p+1;
	}
    }
    return p;
}

void one_line_up ()
{
    start_display = move_backward (start_display, 1);
    display ();
}

void one_line_down ()
{
    start_display = move_forward (start_display, 1);
    display ();
}

void view (char *filename)
{
    int c;

    quit = 0;
    if ((file = open (filename, O_RDONLY)) < 0){
	message (1, " Error ", " Can't open file ");
	return;
    }
    if (fstat (file, &s) < 0){
	message (1, " Error ", " Can't stat file ");
	close (file);
	return;
    }
    if (s.st_size < 1) {
	message(1, " Error ", " Can't view empty file ");
	close(file);
	return;
    }
    
    first = mmap (0, s.st_size, PROT_READ, MAP_FILE | MAP_SHARED, file, 0);
    if (first == (caddr_t) -1){
	close (file);
        message (1, " Error ", " Can't mmap file ");
	return;
    }
    
    push_frame (0, 0, 0);
    push_event (1, 1, COLS, LINES/2, click_auto_repeat, one_line_up);
    push_event (1, LINES/2, COLS, LINES-1, click_auto_repeat, one_line_down);
    init_view ();
    
    wprintw (status, "File: %-20s", name_trunc (filename, 20));
    wmove   (status, 0, 42);
    wprintw (status, "%s bytes", size_trunc (s.st_size));
	     
    start_display = first;
    last_byte = first + s.st_size;
    view_refresh ();
    display ();
    do {
	c = mi_getch ();
	switch (c){
				/* Next page */
	case ' ':
	case XCTRL('v'):
	case KEY_NPAGE:
	    start_display = move_forward (start_display, LINES-3);
	    display ();
	    break;

	case KEY_HOME:
	    start_display = first;
	    display ();
	    break;

	case KEY_END:
	    start_display = move_backward (last_byte, LINES-3);
	    
	case KEY_PPAGE:
	case ALT('v'):
	case XCTRL('b'):
	case XCTRL('h'):
	    start_display = move_backward (start_display, LINES-3);
	    display ();
	    break;

	case KEY_UP:
	    one_line_up ();
	    break;

	case KEY_DOWN:
	    one_line_down ();
	    break;

	case KEY_F(8):
	case '/':
	    regexp_search ();
	    break;

	case KEY_F(7):
	    normal_search ();
	    break;
	    
	case XCTRL('l'):
	    view_refresh ();
	    break;
	    
	case KEY_F(10):
	case '\e':
	    quit = 1;
	}
    } while (!quit);
    munmap (first, s.st_size);
    close (file);

    pop_frame ();
}

