/* Hypertext file browser.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "color.h"
#include "dialog.h"
#include "util.h"
#include "input.h"
#include "win.h"

#define MAXLINKNAME 80
#define HISTORY_SIZE 20

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/help.c,v 1.8 1994/07/27 03:57:42 miguel Exp $";
static char *data;
static int help_lines = 18;
WINDOW *whelp;
static char *history [HISTORY_SIZE];
static int  history_ptr;
static char *main;
static char *last_shown = 0;

char *search_string (char *start, char *text)
{
    char *d = text;
    char *e = start;
    
    for (; *e; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

static char *search_string_node (char *start, char *text)
{
    char *d = text;
    char *e = start;

    if (!start)
	return main;
    
    for (; *e && *e != '\x4'; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

static char *move_forward (char *current, int lines)
{
    char *p;
    int  line;

    for (line = 0, p = current; *p && *p != '\x4'; p++){
	if (*p == '\n')
	    line++;
	if (line == lines)
	    return p;
    }
    return current;
}

static char *move_backward (char *current, int lines)
{
    char *p;
    int line;

    for (line = 0, p = current; *p && *p != '\x4' && p > data; p--){
	if (*p == '\n')
	    line++;
	if (line == lines)
	    return p;
    }
    return current;
}

static char *follow_link (char *start, char *selected_item)
{
    char link_name [MAXLINKNAME];
    char *p;
    int  i = 0;

    if (!selected_item)
	return start;
    
    for (p = selected_item; *p && *p != '\x4' && *p != '\002'; p++)
	;
    if (*p == '\002'){
	link_name [0] = '[';
	for (i = 1; *p != '\003' && *p && *p != '\x4' && i < MAXLINKNAME-3; )
	    link_name [i++] = *++p;
	link_name [i-1] = ']';
	link_name [i] = 0;
	p = search_string (data, link_name);
	if (p)
	    return p;
    }
    return "Help file format error\n\x4";	/*  */
}

static char *select_next_link (char *start, char *current_link)
{
    char *p;

    p = search_string_node (current_link, "\003");
    if (!p)
	p = start;
    p = search_string_node (p, "\001");
    if (p)
	return p - 1;
    p = search_string_node (start, "\001");
    if (p)
	return p - 1;
    return 0;
}

static void show (char *paint_start, char *selected_start)
{
    char *p;
    int  col, line;
    int  painting = 1;
    
    line = col = 0;
    werase (whelp);
    wclr (whelp);
    for (p = paint_start; *p != '\x4' && line < help_lines; p++){
	if (*p == 1){
	    if (p == selected_start)
		wattrset (whelp, MARKED_COLOR | A_BOLD);
	    else
		wattrset (whelp, INPUT_COLOR);
	    
	} else if (*p == 2){
	    painting = 0;
	} else if (*p == 3){
	    painting = 1;
	    wattrset (whelp, REVERSE_COLOR);
	} else if (*p == '\n'){
	    line++;
	    col = 0;
	} else {
	    if (!painting)
		continue;
	    mvwaddch (whelp, line, col, *p);
	    col++;
	}
    }
    last_shown = p;
    wrefresh (whelp);
}

void interactive_display (char *filename)
{
    int quit = 0;
    char *current, *start, *selected_item;
    
    if ((data = load_file (filename)) == 0){
	message (1, " Error ", " Can't open file %s ", filename);
	return;
    }
    if (!(main = search_string (data, "[main]"))){
	message (1, " Error ", " Can't find main node in help file ");
	return;
    }
    create_dialog (60, help_lines, " Main ", "", 0);
    whelp = get_top_text ();
    
    selected_item = search_string_node (main, "\001") - 1;
    current = start = main;
    
    for (history_ptr = 0; history_ptr < HISTORY_SIZE; history_ptr++)
	history [history_ptr] = current;
    
    do {
	show (current, selected_item);
	switch (mi_getch ()){
	case ' ':
	    current = move_forward (current, help_lines-1);
	    break;

	case KEY_BACKSPACE:
	case KEY_PPAGE:
	case 0177:
	case 8:
	    current = move_backward (current, help_lines-1);
	    break;

	case XCTRL('v'):	/* C-v like emacs */
	case KEY_NPAGE:
	case XCTRL('f'):	/* C-f */
	case KEY_DOWN:
	    current = move_forward (current, 2);
	    break;

	case XCTRL('b'):	/* C-b */
	case KEY_UP:
	    current = move_backward (current, 2);
	    break;

	case 'l':
	    current = start = history [--history_ptr % HISTORY_SIZE];
	    break;
	    
	case '\n':
	    if (!selected_item){
		current = start = history [--history_ptr % HISTORY_SIZE];
	    } else {
		history [history_ptr++ % HISTORY_SIZE] = current;
		current = start = follow_link (current, selected_item);
	    }
	    selected_item = search_string_node (current, "\001");
	    if (selected_item)
		selected_item--;
	    break;

	case '\t':
	    selected_item = select_next_link (start, selected_item);
	    if (!selected_item)
		selected_item = main;
	    else {
		if (selected_item >= last_shown ||
		    selected_item < current){
		    current = selected_item;
		}
	    }
	    break;

	case KEY_F(10):
	case '\e':
	    quit = 1;
	}
    } while (!quit);
    free (data);
    destroy_dialog ();
    refresh_screen ();
}
