/*
	This programme sets the boot partition into a LINUX kernel (zimage)

	It simply "poke" an int (major,minor numbers of the device) at offset
	508.
*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>

#if !defined(O_BINARY)
	/* Allows to compile on LINUX, even if the official rdev is */
	/* much better than this one (but won't work on DOS) */
	#define O_BINARY 0
#else
	#include <io.h>
#endif

static void usage(void)
{

    puts(
		"usage: rdev IMAGE [ major minor ]\n"
		"\n"
		"Exemple:\n"
		"To boot on primary DOS partition (C:): rdev zimage 3 3\n"
		"Here are the common major/minor paires on LINUX\n"
		"\n"
		"The 4 partition of the first disk\n"
		"/dev/hda1	3 1\n"
		"/dev/hda2	3 2\n"
		"/dev/hda3	3 3 (this is often C:)\n"
		"/dev/hda4	3 4\n"
		"The sub-partition of the extended DOS partition\n"
		"/dev/hda5	3 5 (this is often D:)\n"
		"/dev/hda6	3 6\n"
		"The second hard disk\n"
		"/dev/hdb1	3 65\n"
		"/dev/hdb2	3 66\n"
		"/dev/hdb3	3 67\n"
		"/dev/hdb4	3 68\n"
		"The sub-partition of the extended DOS partition\n"
		"/dev/hda5	3 69\n"
		"/dev/hda6	3 70\n"
		"The 4 partition of the first SCSI disk\n"
		"/dev/sda1	8 1\n"
		"/dev/sda2	8 2\n"
		"/dev/sda3	8 3 (this is often C:)\n"
		"/dev/sda4	8 4\n"
		"The sub-partition of the extended SCSI DOS partition\n"
		"/dev/sda5	8 5 (this is often D:)\n"
		"/dev/sda6	8 6\n"
		"When booting, look carefully. The kernel prints\n"
		"which partition it sees\n"
		"\n"
		"To print the current setup: rdev IMAGE\n");
    exit(-1);
}

#define DEFAULT_OFFSET 508

static int rdev_openseek(const char *fname, int mode)
{
	int fd = open(fname,mode);
	if (fd == -1){
		fprintf (stderr,"Can't open %s\n",fname);
		perror ("");
	}else if (lseek(fd,DEFAULT_OFFSET,0) < 0){
		fprintf (stderr,"Can't seek to offset %d\n",DEFAULT_OFFSET);
		perror("");
		close (fd);
		fd = -1;
	}
	return fd;
}


int main(int argc,char **argv)
{
	int ret = -1;
	char *fname = argv[1];
	if (argc == 4){
		int major = atoi(argv[2]);
		int minor = atoi(argv[3]);
		int fd = rdev_openseek(fname,O_WRONLY | O_BINARY);
		if (fd != -1){
			short int rdev = (major << 8) + minor;
			if (write(fd,&rdev,2) != 2){
				perror (fname);
			}else{
				ret = 0;
			}
			close(fd);
    	}
	}else if (argc == 2){
		int fd = rdev_openseek(fname,O_RDONLY | O_BINARY);
		if (fd != -1){
			short int rdev;
			if (read(fd,&rdev,2) != 2){
				perror (fname);
			}else{
				printf ("Major = %d minor = %d\n",rdev >> 8, rdev & 0xff);
				ret = 0;
			}
			close(fd);
		}
	}else{
		usage();
	}
	return ret;
}
