
/*
 *----------------------------------------------------------------------------
 *	
 *	WARNING****WARNING****WARNING****WARNING****WARNING
 *	
 *	
 *	This program is meant to demonstrate drag/drop options,
 *	and is not meant to be a functional file manager.  Thus,
 *	dropping the clone onto another window simply reparents the
 *	clone to that destination window.  If this was a real file
 *	manager, the drop function would do a whole lot more -- such
 *	as move, copy, etc.  To keep from corrupting your file
 *	system, I simply reparent the clone, thus no files are created,
 *	moved or destroyed.
 *
 *	Exercises left to the reader:
 *		1. Finish the subclass registration.
 *		2. If directories are moved, prevent "." and ".." from moving,
 *			and do the most appropriate action for other directories.
 *		3. When the icon is dropped into the destination window, decide
 *			on the most appropriate row/col in which to place
 *			the object.
 *		4. Only permit ascii files to be edited.
 *		5. Decide on a scheme by which the user may view other types
 *			of files.
 *
 *----------------------------------------------------------------------------
 */

/*
 *	filemgr.C -- A more complex directory listing program.
 *			which demonstrates drag/drop principles.
 *
 *	filemgr is based upon Warner Losh's complex_ls example in the
 *	glyph directory.  Besides permitting you to view directories,
 *	and view text files, this application permits you to move files
 *	between directories, as well as copy files between directories.
 *
 */

#include <OI/oi.H>
#include <X11/cursorfont.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>

	
/* Function Declarations */
		void	make_dir_box( OI_scroll_box *, const char *);
extern "C"	char	*realpath( const char *, char *);
		Cursor	movement_cursor ;
		GC	dashed_box ;	/* graphics context for drawing object outlines */

/*
 *	Directory
 *
 *	The directory class lets me store information about this directory.
 *	Thus, I can make a linked list of directories available for viewing.
 */
class Directory : public OI_app_window {
 private:
	Directory	*nxt ;
	XrmQuark	qdir ;
 public:
	Directory(char *, const char * );
	~Directory();
	Directory	*next()				{ return( nxt ); }
	XrmQuark	DirQuark()			{ return( qdir ); }
	void		set_next( Directory * n )	{ nxt = n; }
};
static	Directory	*activeDirectories = NULL ;
Directory::~Directory() { }	/* defined out of line for the compiler */

/*
 *	File
 *
 *	The file class lets me store information about the file,
 *	such as name and mode.
 */
class File {
 public:
	File( char *fn, u_short mod ) : filename( strdup( fn ) ), mode( mod ) {}
	~File()		{ free( (char *) filename ); }
	const char	*filename;
	const u_short	mode;
};

/*
 *	FileArray
 *
 *	This is just an array of File.  It lets me insert, sort, and layout
 *	the icons.  Mostly, so I do not have all this code in line.
 */
class FileArray {
 public:
				FileArray(int);
				~FileArray();
	void			sort();
	void			layout_files(OI_box *);
	void			insert(File *);
 private:
	int			num_elem;
	int			num_alloc;
	File			**files;
};

/*
 *	Icon
 *
 *	The icon class is a box that has a glyph and a static text label
 *	in it.  We use this as a base class for icons we really display.
 */
class Icon : public OI_box {
 public:
				Icon( const char * );
	virtual			~Icon();
	static	OI_class	*clsp;
	virtual	void		select();
 protected:
		void		create_icon(char*, char*, OI_number, OI_number, char*);
		char		*pathname;
};
OI_class*	Icon::clsp = NULL ;
void		Icon::select() {}	/* defined out of line for compiler */

/*
 *	Drag
 *	
 *	Everything needed to make an object draggable
 */
class Drag : public Icon {
 public:
		Drag( const char * name ) : Icon( name ) {} 
		~Drag();
	OI_bool	moved ;
	Icon	*drag_ref;		/* drag reference */
	Icon	*drag_obj;		/* object we are dragging */
	long	drag_ix;		/* initial event x loc on drag ref */
	long	drag_iy;		/* initial event y loc on drag ref */
	Time	drag_time;		/* start of action */
	void	apply_translations();

	/*
	 *	method #1, dashed outline
	 */
	void	btn1pick(OI_d_tech *, XEvent *, char **, unsigned int *);

	/*
	 *	method #2, move the object -- within the window
	 */
	void	btn2pick(OI_d_tech *, XEvent *, char **, unsigned int *);
	void	btn2move(OI_d_tech *, XEvent *, char **, unsigned int *);
	void	btn2drop(OI_d_tech *, XEvent *, char **, unsigned int *);

	/*
	 *	method #3, move the object -- across the root
	 */
	void	btn3pick(OI_d_tech *, XEvent *, char **, unsigned int *);
	void	btn3move(OI_d_tech *, XEvent *, char **, unsigned int *);
	void	btn3drop(OI_d_tech *, XEvent *, char **, unsigned int *);
};
Drag::~Drag() { }	/* defined out of line for compiler */

/*
 *	DirIcon
 *
 *	DirIcon is used to display directories.  It is derived from icon
 *	and adds a static directory icon that gets cloned for every
 *	directory that we have.
 */
class DirIcon : public Drag {
 public:
		DirIcon( const char *dir_name );
		~DirIcon();
	void	select();
};
DirIcon::~DirIcon() { }	/* defined out of line for compiler */

/*
 *	ExecIcon
 *
 *	ExecIcon is used to display executable files.  It is derived from
 *	icon and adds a static directory icon that gets cloned for every
 *	executable that we have.
 */
class ExecIcon : public Drag {
 public:
		ExecIcon( const char *exec );
		~ExecIcon();
	void	select();
};
ExecIcon::~ExecIcon() { }	/* defined out of line for compiler */

/*
 *	AsciiIcon
 *
 *	AsciiIcon is used to display normal files.  It is derived from icon
 *	and adds a static directory icon that gets cloned for every normal
 *	file that we have.
 */
class AsciiIcon : public Drag {
 public:
		AsciiIcon( const char *ascii );
		~AsciiIcon();
	void	select();
};
AsciiIcon::~AsciiIcon() { }	/* defined out of line for compiler */

/*
 *	FileArray::FileArray
 *
 *	Constructor for FileArray.  Preallocate the number of entries that
 *	the user wants.
 */
FileArray::FileArray( int to_alloc )
{
	num_elem = 0;
	num_alloc = to_alloc;
	files = (File **) malloc( sizeof( File * ) * num_alloc );
	for (int i = 0; i < num_alloc; i++)
		files[i] = 0;
}

/*
 *	FileArray::~FileArray
 *
 *	Cleans up memory that we have in use.
 */
FileArray::~FileArray()
{
	for (int i = 0; i < num_elem; i++)
		delete files[i];
	free( files );
}

/*
 *	file_comp
 *
 *	Helper function for FileArray::sort.  Called by qsort to compare
 *	two files.
 */
int file_comp( const void *a, const void *b )
{
	File *f1 = *(File **) a;
	File *f2 = *(File **) b;

	return strcmp( f1->filename, f2->filename );
}

/*
 *	FileArray::sort
 *
 *	Just does a qsort of the allocated elements of the array.
 */ 
void FileArray::sort()
{
	qsort( files, num_elem, sizeof( File * ), &file_comp );
}

/*
 *	FileArray::insert
 *
 *	Inserts an element into the array.  It will reallocate the array
 *	if we grow too much.
 */
void
FileArray::insert( File *f )
{
	if (num_elem == num_alloc) {
		files = (File **)
			realloc( files, sizeof( File * ) * num_alloc * 2 );
		num_alloc *= 2;
	}

	files[num_elem++] = f;
}

/*
 *	FileArray::layout_files
 *
 *	I'm not sure this belongs in the FileArray class, but it is here
 *	for now.  It lays out all of the files in icons that we create.
 */
void
FileArray::layout_files( OI_box *bp )
{
	int	row;
	int	col;
	int	i;
	Icon	*icon;
	int	nrows;
	
	row = 1;
	col = 1;
	nrows = (num_elem + 3) / 4;
	for (i = 0; i < num_elem; i++) {
		if ((files[i]->mode & S_IFMT) == S_IFDIR)
			icon = new DirIcon( files[i]->filename );
		else if (files[i]->mode & S_IEXEC)
			icon = new ExecIcon( files[i]->filename );
		else
			icon = new AsciiIcon( files[i]->filename );
		icon->layout_associated_object( bp, col, row, OI_ACTIVE );
		row++;
		if (row > nrows) {
			row = 1;
			col++;
		}
	}
}

/*
 *	Icon::Icon
 *
 *	Constructor for Icon class.  Makes a box, puts an icon and label in it.
 */
Icon::Icon( const char *lab)
		: OI_box( clsp, "iconBox", 1, 1 )
{
	OI_static_text	*stp;
	char		path[MAXPATHLEN];

	/*
	 *	Set frame width to 0 and make this layout.
	 */
	set_layout( OI_layout_row );
	set_frame_width( 0 );
	
	/*
	 *	Create a centered label.
	 */
	stp = oi_create_static_text( "label", (char *) lab );
	stp->set_gravity( OI_grav_center );
	stp->disallow_cut_paste();
	stp->layout_associated_object( this, 2, 1, OI_ACTIVE );


	/*
	 *	Set the pathname.
	 */
	pathname = strdup( realpath( lab, path ) );

	/*
	 *	Set the translation functions.
	 */
	((Drag *) this)->apply_translations();
}

/*
 *	Icon::~Icon
 *
 *	Cleanup.
 */
Icon::~Icon()
{
	free( pathname );
}

/*
 *	Icon::create_icon
 *
 *	Creates an icon of the specifed color, using the bits bm and
 *	bm_bg.  Both of these bitmaps must be the same size, or the
 *	glyph code hickups.
 */
void Icon::create_icon(
	char *bm,
	char *bm_bg,
	OI_number width,
	OI_number height,
	char *color )
{
	OI_glyph		*gp;
	PIXEL			bg_pxl;
	OI_pic_spec_mask	*masks[2];
	
	/*
	 *	If the connection is color, allocate the named color.
	 *	Otherwise just use the background color.
	 */
	if( connection()->is_color() )
		connection()->str_color( color, &bg_pxl );
	else
		bg_pxl = bkg_pixel();
		
	/*
	 *	Create the masks, then create the glyph.
	 */
	masks[0] = oi_create_pic_data_mask(bm,OI_PIC_FG,OI_PIC_NONE,0,bg_pxl );
	masks[1] = oi_create_pic_data_mask(bm_bg,OI_PIC_NONE,OI_PIC_FG,bg_pxl,0 );
	gp = oi_create_glyph("icon",2,masks,"",width,height,OI_YES,OI_NO );

	/*
	 *	finally, apply the drag/drop and click translations
	 */
	gp->layout_associated_object( this, 1, 1, OI_ACTIVE );
}


/*
 *	DirIcon::DirIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
DirIcon::DirIcon( const char *name ) : Drag( name )
{
#include "../bitmaps/dir.xbm"
#include "../bitmaps/dir-bg.xbm"
	create_icon(dir_bits,dir_bg_bits,dir_width,dir_height,"MediumPurple" );
}

/*
 *	DirIcon::select
 *
 *	Overrides the icon select function.  Here we set the directory.
 *	One thing we might want to do is to set the title of the application
 *	to the directory.
 */
void DirIcon::select()
{
	XrmQuark	q ;
	Directory	*ndir ;
	char		path[ 1024 ];
	OI_app_window	*win ;

	win = app_window();
	win->set_working();
	if (pathname[0] == '/')
		strcpy( path, pathname );
	else
		sprintf( path, "%s/%s", XrmQuarkToString( ((Directory*)win)->DirQuark() ), pathname );
		
	q = XrmStringToQuark( path );
	for (ndir = activeDirectories; ndir ; ndir = ndir->next() )
		if (ndir->DirQuark() == q) {
			XRaiseWindow( ndir->display(), ndir->outside_X_window() );
			break ;
		}
	if (!ndir) {
		if (ndir = new Directory( "branch", path )) {
			ndir->set_associated_object(
				ndir->root(),OI_DEF_LOC, OI_DEF_LOC,OI_ACTIVE );
		}
	}
	win->clear_working();
}

/*
 *	ExecIcon::ExecIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
ExecIcon::ExecIcon( const char *name ) : Drag( name )
{
#include "../bitmaps/exec.xbm"
#include "../bitmaps/exec-bg.xbm"
	create_icon( exec_bits, exec_bg_bits, exec_width, exec_height, "Plum" );
}

/*
 *	ExecIcon::select
 *
 *	Overrides the icon select function.  Just execute this command.
 */
void ExecIcon::select()
{
	char		command[MAXPATHLEN + 4];

	sprintf( command, "%s &", pathname );
	system( command );
}
 
/*
 *	AsciiIcon::AsciiIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
AsciiIcon::AsciiIcon( const char *name ) : Drag( name )
{
#include "../bitmaps/ascii.xbm"
#include "../bitmaps/ascii-bg.xbm"
	create_icon( ascii_bits, ascii_bg_bits, ascii_width, ascii_height, "PaleTurquoise" );
}

/*
 *	AsciiIcon::select
 *
 *	Overrides the icon select function.  Create a scroll text in a
 *	dialog box to display the file to the user.
 */
void AsciiIcon::select()
{
	OI_dialog_box		*dbp;
	OI_scroll_text		*stp;

	dbp = oi_create_dialog_box( "fileDB", 1, 1 );
	dbp->set_layout( OI_layout_row );
	dbp->disallow_pushpin();
	dbp->set_title( "File" );
	dbp->set_longterm( pathname );
	dbp->buttons()->numbered_cell(1)->del();
	dbp->buttons()->numbered_cell(0)->set_label( "Dismiss" );
	dbp->buttons()->numbered_cell(0)->change_action(
		dbp, (OI_action_memfn*)&OI_dialog_box::delete_all_delayed, NULL );
		
	stp = oi_create_scroll_text(
		"fileViewer", OI_SCROLL_BAR_BOTH, 24, 80, 24, 80 );
	stp->allow_auto_controller_visibility();
	stp->set_text_to_file( pathname );
	stp->set_size_track( OI_size_track_full );
	stp->disallow_kb_input();
	stp->layout_associated_object( dbp, 1, 1, OI_ACTIVE );

	dbp->set_associated_object(
		dbp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE );
}

/*
 *	make_dir_box
 *
 *	Creates a box that has all the entries in the directory in it.
 *	No sorting of the directory is done just yet, as that would
 *	obscure the OI techniques being demonstrated.
 */
void make_dir_box( OI_scroll_box *boxp, const char *dir )
{
	struct stat		sb;
	DIR			*dirp;
	struct dirent		*dirent;
	FileArray		*files;
	char			path[MAXPATHLEN];
	
	/*
	 *	Stat the directory to see if it is there.
	 */
	if (stat( dir, &sb )) {
		fprintf( stderr, "Can't stat directory %s\n", dir );
		exit( 1 );
	}

	/*
	 *	Check to see if it is a directory.  If not fail.
	 */
	if ((sb.st_mode & S_IFMT) != S_IFDIR) {
		fprintf( stderr, "%s is not a directory", dir );
		exit( 1 );
	}

	/*
	 *	chdir to the directory and set the title of this app
	 *	window to be the current directory.
	 */
	if (chdir( dir )) {
		perror( dir );
		exit( 1 );
	}	
	boxp->app_window()->set_longterm( realpath( ".", path ) );

	/*
	 *	A suspend layout is done to avoid O(n^2) layout behaviour.
	 */
	boxp->suspend_layout();

	/*
	 *	Open the directory.  While this shouldn't fail, I'll be
	 *	paranoid just in case...
	 */
	if ((dirp = opendir( "." )) == NULL ) {
		fprintf( stderr, "Logic says this can't happen, but I can't opendir %s\n", dir );
		exit( 1 );
	}

	/*
	 *	Read the directory in.  For each file, make a new icon
	 *	specific to each type of file.  Then layout the new icon.
	 *	Be really dumb about placement of icons.  A more advanced
	 *	program would sort them and arrange them into n columns.
	 */
	files = new FileArray( 100 );
	readdir( dirp );		// Skip .
	while (dirent = readdir( dirp )) {
		if (stat( dirent->d_name, &sb ))
			continue;
		files->insert(new File( dirent->d_name, sb.st_mode ));
	}
	closedir( dirp );

	/*
	 *	Now sort and layout all these glyphs.
	 */
	files->sort();
	files->layout_files(boxp->object_box());
	delete files;
	
	/*
	 *	The resume layout here is supposed to be faster since we
	 *	only layout each object once, not n times.
	 */
	boxp->resume_layout();
}

/*
 *	protocol delete callback
 *
 *	Setting protocol delete on every top level window permits us to
 *	delete views without killing the application.  We need to first
 *	remove this view from the linked list, then remove it from the
 *	screen (by deleting it).
 */
static void
delete_directory_view( OI_app_window *win, void * )
{
	Directory	*dsp ;
	Directory	*lnk ;

	dsp = (Directory *) win ;

	if (dsp == activeDirectories) {
		activeDirectories = dsp->next();
		if (!activeDirectories)
			OI_end_interaction();
	}
	else for (lnk = activeDirectories; lnk ; lnk = lnk->next()) {
		if (lnk->next() == dsp) {
			lnk->set_next( dsp->next() );
			break ;
		}
	}
	dsp->delete_all_delayed();
}

/*
 *	create a directory view
 */
Directory::Directory( char *np, const char *dir )
	: OI_app_window( np, 1, 1, "Directory" )
{
	char		*absdir ;
	OI_scroll_box	*dir_box;

	set_layout( OI_layout_row );

	/*
	 *	Create a box to hold the directory.  We use a scroll
	 *	box that size tracks for the box.
	 */
	dir_box = oi_create_scroll_box(
		"dirBox", OI_SCROLL_BAR_BOTH, 30, 30, 1, 1, 10, 10 );
	dir_box->set_layout( OI_layout_column );
	dir_box->set_size_track( OI_size_track_full );
	dir_box->allow_auto_controller_visibility();

	/*
	 *	Create a new directory object.
	 */
	absdir = NULL ;
	if (dir)
		absdir = OI_translate_filename( dir );
	qdir = absdir ? XrmStringToQuark( absdir ) : 0 ;
	dir_box->layout_associated_object( this, 1, 1, OI_ACTIVE );
	make_dir_box( dir_box, absdir );

	/*
	 *	Set up the termination routines.
	 */
	set_protocol_delete( (OI_destroy_fn *) delete_directory_view );

	/*
	 *	Add me into the linked list of directories
	 */
	set_next( activeDirectories );
	activeDirectories = this ;
}

/*
 *	Main
 *
 *	Main program.
 */
int main(int argc, char *argv[])
{
	OI_connection	*conp;
	XGCValues	gcv ;				/* for initializing outline_gc */

	/*
	 *	Open a connection.
	 */
	if (conp = OI_init( &argc, argv, "Complex Directory Listing program" ))
	{
		/*
		 *	create the cursor and dashed line inversion context
		 *	for dragging and moving objects.
		 */
		gcv.function = GXinvert ;
		gcv.subwindow_mode = IncludeInferiors ;
		gcv.line_style = LineOnOffDash ;
		dashed_box = XCreateGC(
			conp->display(), conp->root()->X_window(),
			GCFunction | GCSubwindowMode | GCLineStyle, &gcv );
		movement_cursor = XCreateFontCursor( conp->display(), XC_crosshair );

		/*
		 *	make top level directory, then begin interaction
		 */
        	Icon::clsp = OI_register_class( "Icon", "OI_box", NULL_PMF, NULL_PMF );
		activeDirectories = new Directory( "toplevel", argc > 1 ? argv[1] : "." );
		activeDirectories->set_associated_object(
			activeDirectories->root(),OI_DEF_LOC, OI_DEF_LOC,OI_ACTIVE );
		OI_begin_interaction();
	}

	/*
	 * Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
	return( 0 );
}

/*
 *	on ButtonPress
 *		- initialize ix and iy
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as ix and iy.
 *		- show the dragbox over the source object
 *
 *	You MUST remember that event x,y is relative to the object
 * 
 */
void
Drag::btn2pick(OI_d_tech *objp, XEvent *ep, char **, unsigned int *)
{
	/*
	 *	figure out which object was selected.
	 */
	drag_ref = (Icon*) (objp->is_derived_from( Icon::clsp ) ?
			objp : objp->ancestor_derived_from( Icon::clsp ));
	drag_ref->ancestor_loc( drag_ref->app_window(), &drag_ix, &drag_iy );

	/*
	 *	create and place a clone of the object.
	 */
	drag_obj = (Icon*) drag_ref->clone();
	drag_obj->set_associated_object(
		drag_ref->app_window(), drag_ix, drag_iy, OI_NOT_DISPLAYED );

	/*
	 *	note where and when the activity occurred.
	 */
	drag_ix -= ep->xbutton.x ;
	drag_iy -= ep->xbutton.y ;
	drag_time = ep->xbutton.time ;
	moved = OI_NO ;
}

/*
 *	on MotionNotify (drag event)
 *		- capture the event that caused this procedure to be called.
 *		- process all MotionNotify events in the queue (less choppy movement)
 *		- change the dragBox location from the old location to the new location
 */
void
Drag::btn2move(OI_d_tech *, XEvent *eventp, char **, unsigned int *)
{
	XEvent	event;	/* used so we don't corrupt the initial event pointer */

	if (eventp->type == MotionNotify) {
		if (!moved) {
			moved = OI_YES ;
			drag_obj->set_state( OI_ACTIVE );
		}
		event = *eventp;
		while (XCheckTypedEvent(drag_ref->display(), MotionNotify, &event));
		drag_obj->set_loc( drag_ix + event.xmotion.x, drag_iy + event.xmotion.y );
	}
}

/*
 *	on ButtonRelease
 *	refresh the object color.
 */
void
Drag::btn2drop(OI_d_tech *objp, XEvent *eventp, char **, unsigned int *)
{
	if ((!moved) && ((eventp->xbutton.time - drag_time) < OI_click_delta))
		((Icon*)objp)->select();
	drag_obj->del();
}

static void
Process_drop_location( Icon *objp, XEvent *ep )
{
	OI_number	nd ;
	OI_d_tech	*dtp ;
	Directory	*dst ;
	Directory	*ndir ;
	Directory	**vec ;

	for (nd=0, ndir = activeDirectories; ndir ; ndir = ndir->next(), nd++ );
	vec = (Directory**) malloc( nd * sizeof( Directory * ));
	for (nd=0, ndir = activeDirectories; ndir ; ndir = ndir->next(), nd++ )
		vec[ nd ] = ndir ;
	if (dtp = OI_find_obj( ep, nd, (OI_d_tech**) vec )) {
		dst = (Directory *) dtp->app_window();
		if (objp->app_window() == (OI_app_window *) dst)
			objp->push_help_str( "Dropped on source window", OI_YES );
		else {
			/*
			 *	reparent to new location
			 */
			objp->layout_associated_object( dst->descendant("dirBox"), (OI_number) 0, (OI_number) 0, OI_ACTIVE );
		}
	}
	if (vec)
		free( vec );
}

/*
 *	on ButtonPress
 *		- initialize ix and iy
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as ix and iy.
 *		- show the dragbox over the source object
 *
 *	You MUST remember that event x,y is relative to the object
 * 
 */
void
Drag::btn1pick(OI_d_tech *objp, XEvent *ep, char **, unsigned int *)
{
	long		itim ;		/* tim of original button press */
	unsigned long	delta ;		/* how long we've been looping */
	OI_connection	*conp ;		/* connection */
	Display		*dsp ;
	Window		xwin ;
	XRectangle	R[2] ;		/* rectangle specs for old/new object outlines */
	XEvent		ev ;		/* primary event buffer */
	XEvent		exep ;		/* secondary event buffer */
	int		x,y ;		/* root window coordinates of origin for object to move */
	int		ox, oy;		/* objects previous location */
	int		orx,ory ;	/* original origin of rectangle for object in absolute root coordinates */
	Window		dmy_win ;	/* trash place for XTranslateCoordinates return window */

	conp	= objp->connection();
	dsp	= conp->display();
	xwin	= conp->abs_root()->X_window();
	drag_ref = (Icon*) (objp->is_derived_from( Icon::clsp ) ?
			objp : objp->ancestor_derived_from( Icon::clsp ));
	//-----------------------------------------------
	//	WARNING: POINTER IS BEING GRABBED.
	//-----------------------------------------------
	if (XGrabPointer(
		dsp, objp->X_window(), True, (unsigned int) ButtonReleaseMask, GrabModeAsync,
		GrabModeAsync, None, movement_cursor, CurrentTime) == GrabSuccess)
	{

		delta = 0;
		itim = (long) ep->xbutton.time ;

		while (delta < OI_click_delta) {
			if (XCheckTypedEvent( dsp, ButtonRelease, &ev )) {
				XUngrabPointer( dsp, CurrentTime ) ;
				//-----------------------------------------------
				//	POINTER HAS BEEN RELEASED -- click
				//-----------------------------------------------
				((Icon*)objp)->select();
				goto done ;
			}
			else
				delta = (unsigned long) ((long)conp->time() - itim );
		}
		//-----------------------------------------------
		//	WARNING: POINTER IS GRABBED -- move
		//-----------------------------------------------
		XTranslateCoordinates(
			dsp, objp->outside_X_window(), xwin, -objp->bdr_width(),
			-objp->bdr_width(), &orx, &ory, &dmy_win) ;
		R[0].width = R[1].width = drag_ref->space_x() - 1 ;
		R[0].height = R[1].height = drag_ref->space_y() - 1 ;
		XQueryPointer(	dsp, xwin, &ev.xmotion.root, &ev.xmotion.subwindow, &x,
				&y, &ev.xmotion.x, &ev.xmotion.y, &ev.xmotion.state) ;
		R[1].x = x ;
		R[1].y = y ;

		ox = oy = -1;

		XDrawRectangle(	dsp, xwin, dashed_box, R[1].x, R[1].y, R[1].width, R[1].height) ;
		while (1) {
			if (XCheckTypedEvent(dsp,ButtonRelease, &ev))
				break ;
			if (XCheckTypedEvent( dsp, Expose, &exep)) {
				//
				// warning: notice the use of a separate `expose' event pointer
				//
				XDrawRectangle( dsp, xwin, dashed_box,
						R[1].x,R[1].y,R[1].width,R[1].height);
				OI_handle_event( &exep );
				XDrawRectangle( dsp, xwin, dashed_box,
						R[1].x,R[1].y,R[1].width,R[1].height);
			}
			else {
				XQueryPointer(	dsp, xwin, &ev.xmotion.root,
						&ev.xmotion.subwindow, &x, &y, &ev.xmotion.x, &ev.xmotion.y,
						&ev.xmotion.state) ;
				R[0].x = R[1].x ;
				R[0].y = R[1].y ;
				R[1].x = x ;
				R[1].y = y ;
				if  ((ox != x ) || (oy != y)) {
					XDrawRectangle( dsp, xwin, dashed_box,
							R[0].x, R[0].y, R[1].width, R[1].height);
					XDrawRectangle( dsp, xwin, dashed_box,
							R[1].x, R[1].y, R[1].width, R[1].height);
					ox = x;
					oy = y;
				}
			}
		}
		XDrawRectangle( dsp, xwin, dashed_box, R[1].x, R[1].y, R[1].width, R[1].height ) ;
		XUngrabPointer( dsp, CurrentTime );
		//-----------------------------------------------
		//	POINTER HAS BEEN RELEASED
		//-----------------------------------------------
		Process_drop_location( drag_ref, &ev );
 done: ;
	}
	return ;
}

/*
 *	on ButtonPress
 *		- initialize ix and iy
 *		- if this was a ButtonPress Event,
 *			capture the actual event location and store as ix and iy.
 *		- show the dragbox over the source object
 *
 *	You MUST remember that event x,y is relative to the object
 * 
 */
void
Drag::btn3pick(OI_d_tech *objp, XEvent *ep, char **, unsigned int *)
{
	XSetWindowAttributes	attrib ;

	/*
	 *	figure out which object was selected.
	 */
	drag_ref = (Icon*) (objp->is_derived_from( Icon::clsp ) ?
			objp : objp->ancestor_derived_from( Icon::clsp ));

	/*
	 *	create and place a clone of the object.
	 */
	drag_obj = (Icon*) drag_ref->clone();

	attrib.save_under = True;
	XChangeWindowAttributes( drag_obj->display(), drag_obj->outside_X_window(), CWSaveUnder, &attrib );
fprintf( stderr, "outside X window: %08lx\n", drag_obj->outside_X_window() );
fflush( stderr );

	drag_obj->allow_override_redirect();
	drag_ix = ep->xbutton.x_root - ep->xbutton.x ;
	drag_iy = ep->xbutton.y_root - ep->xbutton.y ;
	drag_obj->set_associated_object( drag_ref->root(), drag_ix, drag_iy, OI_NOT_DISPLAYED );

	drag_ix -= ep->xbutton.x ;	/* subtract off base, so we can just add in event for placement */
	drag_iy -= ep->xbutton.y ;
	drag_time = ep->xbutton.time ;
	moved = OI_NO ;
}

/*
 *	on MotionNotify (drag event)
 *		- capture the event that caused this procedure to be called.
 *		- process all MotionNotify events in the queue (less choppy movement)
 *		- change the dragBox location from the old location to the new location
 */
void
Drag::btn3move(OI_d_tech *, XEvent *eventp, char **, unsigned int *)
{
	XEvent	event;	/* used so we don't corrupt the initial event pointer */

	if (eventp->type == MotionNotify) {
		if (!moved) {
			moved = OI_YES ;
			drag_obj->set_state( OI_ACTIVE );
		}
		event = *eventp;
		while (XCheckTypedEvent(drag_ref->display(), MotionNotify, &event));
		drag_obj->set_loc( drag_ix + event.xmotion.x, drag_iy + event.xmotion.y );
	}
}

/*
 *	on ButtonRelease
 *	refresh the object color.
 */
void
Drag::btn3drop(OI_d_tech *objp, XEvent *eventp, char **, unsigned int *)
{
	if ((!moved) && ((eventp->xbutton.time - drag_time) < OI_click_delta))
		((Icon*)objp)->select();
//	drag_obj->del();
}


/*
 *	The OI_actions_rec structure allows the user to specify
 *	either C-functions, or C++ member functions, to be called
 *	when the specified mouse event occurs.
 */
static OI_actions_rec Drag_actions[] = {
	{"btn1pick", NULL, NULL, (OI_translation_memfn*)&Drag::btn1pick },
	{"btn2pick", NULL, NULL, (OI_translation_memfn*)&Drag::btn2pick },
	{"btn2move", NULL, NULL, (OI_translation_memfn*)&Drag::btn2move },
	{"btn2drop", NULL, NULL, (OI_translation_memfn*)&Drag::btn2drop },
	{"btn3pick", NULL, NULL, (OI_translation_memfn*)&Drag::btn3pick },
	{"btn3move", NULL, NULL, (OI_translation_memfn*)&Drag::btn3move },
	{"btn3drop", NULL, NULL, (OI_translation_memfn*)&Drag::btn3drop },
};

/*
 *	The Drag_translations character array specify the translations
 *	to be applied to the file glyphs.
 *	Please note that these translations could also be written out to
 *	a config file, and added to the application at runtime using
 *		app -config trans.cf
 *	Please see example transC.C for an example of how to use that
 *	method of applying translations to other objects.
 */
static	char *Drag_translations = "#override	\n\
		<Btn1Down>:	btn1pick()			\n\
		<Btn2Down>:	btn2pick()			\n\
		<Btn2Motion>:	btn2move()			\n\
		<Btn2Up>:	btn2drop()			\n\
		<Btn3Down>:	btn3pick()			\n\
		<Btn3Motion>:	btn3move()			\n\
		<Btn3Up>:	btn3drop()			\n\
	";

void
Drag::apply_translations()
{
	int		i ;
	int		nc ;
	OI_d_tech	*dtp ;

	for (nc = num_props(), i=0; i<nc; i++) {
		dtp = numbered_child( i );
		dtp->push_actions( Drag_actions, OI_count( Drag_actions ));
		dtp->override_translations( Drag_translations );
	}
	push_actions( Drag_actions, OI_count( Drag_actions ));
	override_translations( Drag_translations );
}
