/* Interactive Printer Queue Viewer -- User Interface */

#include "qview.h"
#include <strings.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

/* Forward declarations */
extern void parsecommand _ARGS((int argc, char **argv));
extern void usage _ARGS((void));
extern nonreturning void panic _ARGS((char *message));
extern void setup _ARGS((void));
extern void makemenus _ARGS((void));
extern void fillprintersmenu _ARGS((void));
extern void makewindow _ARGS((void));
extern void maketitle _ARGS((char *title, int size));
extern void mainloop _ARGS((void));
extern void do_select _ARGS((EVENT *ep));
extern void selectnew _ARGS((int which));
extern void do_selprev _ARGS((void));
extern void do_selnext _ARGS((void));
extern void do_mainmenu _ARGS((int item));
extern void do_interval _ARGS((void));
extern void do_formatmenu _ARGS((int item));
extern void do_printersmenu _ARGS((int item));
extern void showformat _ARGS((void));
extern void showpossible _ARGS((void));
extern void refreshinfo _ARGS((void));
extern void shortrefresh _ARGS((void));
extern void normalrefresh _ARGS((void));
extern void longrefresh _ARGS((void));
extern void drawproc _ARGS((WINDOW *win, int left, int top, int right, int bottom));
extern void drawshort _ARGS((int left, int top, int right, int bottom));
extern void drawnormal _ARGS((int left, int top, int right, int bottom));
extern void drawlong _ARGS((int left, int top, int right, int bottom));
extern void wdrawright _ARGS((int h, int v, char *text, int len));
extern bool interesting _ARGS((char *user));
extern void hilite _ARGS((int which));
extern void unhilite _ARGS((int which));
extern void getrect _ARGS((int which, int *ph1, int *pv1, int *ph2, int *pv2));

int
main(argc, argv)
        int argc;
        char **argv;
{
        winitargs(&argc, &argv);
        parsecommand(argc, argv);
        setup();
        mainloop();
        wdone();
        exit(0);
}

/* Format codes */
#define SHORTFORMAT     0
#define NORMALFORMAT    1
#define LONGFORMAT      2

/* Command line options and their defaults */
char *spool=            "/usr/spool";   /* -S spooldir; mainly for debugging */
char *printer=          NULL;           /* -Pprinter; default ${PRINTER-psc} */
int format=             NORMALFORMAT;   /* -s, -n, -l */
int interval=           30;             /* +<number> */
char **userlist=        NULL;           /* remaining arguments */

char *progname= "qview";

void
parsecommand(argc, argv)
        int argc;
        char **argv;
{
        if (argc > 0) {
                progname= strrchr(argv[0], '/');
                if (progname != NULL)
                        ++progname;
                else
                        progname= argv[0];
        }
        for (;;) {
                int c= getopt(argc, argv, "S:P:snl");
                if (c == EOF)
                        break;
                switch (c) {
                default:
                        usage();
                        /*NOTREACHED*/
                case 'S':
                        spool= optarg;
                        break;
                case 'P':
                        printer= optarg;
                        break;
                case 's':
                        format= SHORTFORMAT;
                        break;
                case 'n':
                        format= NORMALFORMAT;
                        break;
                case 'l':
                        format= LONGFORMAT;
                        break;
                }
        }
        if (printer == NULL) {
                printer= getenv("PRINTER");
                if (printer == NULL)
                        printer= "lpd";
        }
        if (optind < argc && argv[optind][0] == '+') {
                if (!isdigit(argv[optind][1]))
                        usage();
                interval= atoi(argv[optind] + 1);
                ++optind;
        }
        if (optind < argc)
                userlist= argv + optind;
        else {
                static char *defuserlist[]= {NULL, NULL};
                defuserlist[0]= getenv("USER");
                if (defuserlist[0] != NULL)
                        userlist= defuserlist;
        }
}

void
usage()
{
        wdone();
        fprintf(stderr,
         "usage: %s [-S spooldir] [-Pprinter] [-snl] [+interval] [user] ...\n",
         progname);
        fprintf(stderr, "  -s: short, -n: normal; -l: long format\n");
        fprintf(stderr, "  +interval must be last option if given\n");
        exit(2);
}

void
panic(message)
        char *message;
{
        wdone();
        fprintf(stderr, "%s: panic: %s\n", progname, message);
        exit(1);
}

#ifndef HAVE_GETHOSTNAME

#include <sys/utsname.h> /* defines struct utsname */

static int
gethostname(name,namelen)
        char* name;
        int namelen;
{
        struct utsname utsname;

        if (uname(&utsname)<0) return -1;
        strncpy(name,utsname.nodename,namelen);
        if (namelen>0) name[namelen-1]='\0';
        return 0;
}

#endif

void
setup()
{
        gotospooldir();
        makemenus();
        makewindow();
}

MENU *mainmenu, *formatmenu, *printersmenu;

#define MAINMENU        1
#define FORMATMENU      2
#define PRINTERSMENU    3

/* Main menu items */
#define DELETEITEM      0
#define TOPQUEUEITEM    1
#define RESTARTITEM     2
#define REFRESHITEM     3
/* --- */
#define INTERVALITEM    5
/* --- */
#define QUITITEM        7

/* Format menu items */
#define SHORTITEM       0
#define NORMALITEM      1
#define LONGITEM        2

void
makemenus()
{
        mainmenu= wmenucreate(MAINMENU, "Command");
        if (mainmenu == NULL)
                panic("can't create Command menu");
        wmenuadditem(mainmenu, "Delete job", 'D');
        wmenuadditem(mainmenu, "Move job to queue top", 'T');
        wmenuadditem(mainmenu, "Restart daemon", -1);
        wmenuadditem(mainmenu, "Refresh display", 'R');
        wmenuadditem(mainmenu, "", -1);
        wmenuadditem(mainmenu, "Set refresh interval...", -1);
        wmenuadditem(mainmenu, "", -1);
        wmenuadditem(mainmenu, "Quit", 'Q');

        formatmenu= wmenucreate(FORMATMENU, "Format");
        if (formatmenu == NULL)
                panic("can't create Format menu");
        wmenuadditem(formatmenu, "Short", 'S');
        wmenuadditem(formatmenu, "Normal", 'N');
        wmenuadditem(formatmenu, "Long", 'L');

        printersmenu= wmenucreate(PRINTERSMENU, "Printers");
        if (printersmenu == NULL)
                panic("can't create Printers menu");
        fillprintersmenu();
}

/* Should really get this from /etc/printcap, but for now... */

char *printers[]= {
        "console",
        "har",
        "hqpsc",
        "lp",
        "nvpd",
        "psc",
        "vpd",
        NULL,   /* Slot for other printer */
        NULL    /* Sentinel */
};

int whichprinter;

void
fillprintersmenu()
{
        int i;
        whichprinter= -1;
        for (i= 0; printers[i] != NULL; ++i) {
                wmenuadditem(printersmenu, printers[i], -1);
                if (strcmp(printers[i], printer) == 0)
                        whichprinter= i;
        }
        if (whichprinter < 0) {
                printers[i]= printer;
                wmenuadditem(printersmenu, printers[i], -1);
                whichprinter= i;
        }
        wmenucheck(printersmenu, whichprinter, TRUE);
}

WINDOW *win;

void
makewindow()
{
        char title[BUFSIZ];
        maketitle(title, sizeof title);
        win= wopen(title, drawproc);
        if (win == NULL)
                panic("can't open window");
}

void
maketitle(title, size)
        char *title;
        int size;
{
        char hostname[128];
        gethostname(hostname, sizeof hostname);
        sprintf(title, "qview(%s@%s)", printer, hostname);
        if (userlist != NULL) {
                char *p= strchr(title, EOS);
                char **u;
                strcpy(p, " for");
                for (u= userlist; *u != NULL; ++u) {
                        p= strchr(title, EOS);
                        if (p + strlen(*u) + 2 >= &title[size])
                                break;
                        sprintf(p, " %s", *u);
                }
        }
}

void
mainloop()
{
        showformat();
        refreshinfo();
        for (;;) {
                EVENT e;
                wgetevent(&e);
                switch (e.type) {
                case WE_COMMAND:
                        switch (e.u.command) {
                        case WC_CLOSE:
                                return;
                        case WC_RETURN:
                                refreshinfo();
                                break;
                        case WC_UP:
                                do_selprev();
                                break;
                        case WC_DOWN:
                                do_selnext();
                                break;
                        }
                        break;
                case WE_TIMER:
                case WE_SIZE:
                        refreshinfo();
                        break;
                case WE_MOUSE_DOWN:
                case WE_MOUSE_MOVE:
                case WE_MOUSE_UP:
                        do_select(&e);
                        break;
                case WE_MENU:
                        switch (e.u.m.id) {
                        case MAINMENU:
                                if (e.u.m.item == QUITITEM)
                                        return;
                                do_mainmenu(e.u.m.item);
                                break;
                        case FORMATMENU:
                                do_formatmenu(e.u.m.item);
                                break;
                        case PRINTERSMENU:
                                do_printersmenu(e.u.m.item);
                                break;
                        }
                        break;
                }
        }
}

/* Lay-out information for short format */
int nrows, ncols;
int colwidth, colheight;

int selected= -1;

void
do_select(ep)
        EVENT *ep;
{
        int which, row, col;
        int lh= wlineheight();

        switch (format) {
        case SHORTFORMAT:
                row= (ep->u.where.v / colheight) - 1;
                col= (ep->u.where.h / colwidth);
                if (row >= 0 && row < nrows && col >= 0 && col < ncols)
                        which= row + col*nrows;
                else
                        which= -1;
                break;
        case NORMALFORMAT:
        case LONGFORMAT:
                which= (ep->u.where.v / lh) - 2;
                break;
        default:
                which= -1;
                break;
        }

        selectnew(which);
}

void
selectnew(which)
        int which;
{
        if (which >= nfiles)
                which= -1;
        if (which != selected) {
                if (selected >= 0)
                        unhilite(selected);
                selected= which;
                if (selected >= 0)
                        hilite(selected);
        }
        if (selected >= 0) {
                /* Delay refresh to give user time to do something */
                wsettimer(win,
                        10 * (selected < 0 ? interval : MAX(interval, 10)));
                showpossible();
        }
        else {
                /* Don't refresh at all while the user is selecting */
                wsettimer(win, 0);
        }
}

void
do_selprev()
{
        if (selected < 0)
                selectnew(nfiles-1);
        else
                selectnew(selected-1);
}

void
do_selnext()
{
        selectnew(selected+1);
}

void
do_mainmenu(item)
        int item;
{
        switch (item) {
        case DELETEITEM:
                if (selected < 0)
                        return;
                if (do_delete(selected))
                        selected= -1;
                else
                        wfleep();
                break;
        case TOPQUEUEITEM:
                if (selected < 0)
                        return;
                if (do_topqueue(selected))
                        selected= -1;
                else
                        wfleep();
                break;
        case RESTARTITEM:
                if (!do_restart())
                        wfleep();
                break;
        case REFRESHITEM:
                break;
        case INTERVALITEM:
                do_interval();
                break;
        }
        refreshinfo();
}

void
do_interval()
{
        char buffer[256];
        char c;
        int i;
        sprintf(buffer, "%d", interval);
        if (!waskstr("Set refresh interval (seconds):", buffer, sizeof buffer))
                return;
        if (sscanf(buffer, " %d %c", &i, &c) != 1) {
                wmessage("That's not a number - please try again");
                return;
        }
        if (i < 1) {
                wmessage("Please use a minimal interval of 1 second");
                return;
        }
        interval= i;
}

void
do_formatmenu(item)
        int item;
{
        switch (item) {
        case SHORTITEM:
                format= SHORTFORMAT;
                break;
        case NORMALITEM:
                format= NORMALFORMAT;
                break;
        case LONGITEM:
                format= LONGFORMAT;
                break;
        }
        showformat();
        refreshinfo();
}

void
do_printersmenu(item)
        int item;
{
        char title[BUFSIZ];
        if (item == whichprinter)
                return;
        wmenucheck(printersmenu, whichprinter, FALSE);
        whichprinter= item;
        wmenucheck(printersmenu, whichprinter, TRUE);
        printer= printers[whichprinter];
        /* Should recover from can't chdir, but for now... */
        gotospooldir();
        maketitle(title, sizeof title);
        wsettitle(win, title);
        refreshinfo();
}

void
showformat()
{
        int item;
        int i;
        switch (format) {
        case SHORTFORMAT:       item= SHORTITEM;        break;
        case NORMALFORMAT:      item= NORMALITEM;       break;
        case LONGFORMAT:        item= LONGITEM;         break;
        default:                item= -1;               break;
        }
        for (i= 0; i < 3; ++i)
                wmenucheck(formatmenu, i, i == item);
}

void
showpossible()
{
        wmenuenable(mainmenu, DELETEITEM, selected >= 0);
        wmenuenable(mainmenu, TOPQUEUEITEM, selected >= 0);
}

void
refreshinfo()
{
        selected= -1;
        showpossible();
        if (queuerefresh()) {
                switch (format) {
                case SHORTFORMAT:
                        shortrefresh();
                        break;
                case NORMALFORMAT:
                        normalrefresh();
                        break;
                case LONGFORMAT:
                        longrefresh();
                        break;
                }
        }
        wchange(win, 0, 0, 32000, 32000);
        wsettimer(win, 10*interval);
}

void
shortrefresh()
{
        int width, height;
        colwidth= wtextwidth("000*0000K mmmmmmmm  ", -1);
        colheight= wlineheight();
        wgetwinsize(win, &width, &height);
        nrows= (height - colheight) / colheight;
        CLIPMIN(nrows, 1);
        ncols= (nfiles + nrows-1) / nrows;
        CLIPMIN(ncols, 1);
        wsetdocsize(win, ncols*colwidth, (nrows+1)*colheight);
}

void
normalrefresh()
{
        wsetdocsize(win, 0, wlineheight() * (2 + nfiles));
}

void
longrefresh()
{
        normalrefresh();
}

void
drawproc(win, left, top, right, bottom)
        WINDOW *win;
        int left, top, right, bottom;
{
        switch (format) {
        case SHORTFORMAT:
                drawshort(left, top, right, bottom);
                break;
        case NORMALFORMAT:
                drawnormal(left, top, right, bottom);
                break;
        case LONGFORMAT:
                drawlong(left, top, right, bottom);
                break;
        }
        if (selected >= 0) {
                int h1, v1, h2, v2;
                getrect(selected, &h1, &v1, &h2, &v2);
                winvert(h1, v1, h2, v2);
        }
}

void
drawshort(left, top, right, bottom)
        int left, top, right, bottom;
{
        int dh1= wtextwidth("000", -1);
        int dh2= wtextwidth("*", 1);
        int dh3= wtextwidth("0000K ", -1);
        int i;

        if (top < colheight && bottom > 0) {
                if (lock != NULL)
                        wdrawtext(0, 0, lock, -1);
                else if (status != NULL)
                        wdrawtext(0, 0, status, -1);
        }

        for (i= 0; i < nfiles; ++i) {
                char buf[256];
                int h1, v1, h2, v2;
                int h, v;
                char *user;
                getrect(i, &h1, &v1, &h2, &v2);
                if (h1 >= right)
                        break;
                if (h2 <= left || v1 >= bottom || v2 <= top)
                        continue;
                h= h1;
                v= v1;
                user= username(filelist[i].uid);
                h += dh1;
                wdrawright(h, v, filelist[i].name + 3, 3);
                if (filelist[i].active) {
                        wdrawtext(h, v, "*", 1);
                        h+= wtextwidth("*", 1);
                }
                else
                        h += dh2;
                sprintf(buf, "%ldK ", (filelist[i].size + 1023) / 1024);
                h += dh3;
                wdrawright(h, v, buf, -1);
                wdrawtext(h, v, user, -1);
                if (interesting(user))
                        wdrawline(h1, v2-1, h2, v2-1);
        }
}

void
drawnormal(left, top, right, bottom)
        int left, top, right, bottom;
{
        int lh= wlineheight();
        int h1= wtextwidth("0000", 4);
        int h2= h1 + wtextwidth("  ", 1);
        int h3= h2 + wtextwidth("*  ", 2);
        int h4= h3 + wtextwidth("cfA000hostname    ", -1);
        int h5= h4 + wtextwidth("mmmmmmmm    0000K", -1);
        int h6= h5 + wtextwidth("    00", -1);
        int v= 0;
        int i;

        if (lock != NULL && top < v+lh && v < bottom)
                wdrawtext(0, v, lock, -1);
        v += lh;
        if (status != NULL && top < v+lh && v < bottom)
                wdrawtext(0, v, status, -1);

        for (i= 0; i < nfiles; ++i) {
                char buf[256];
                char *user;
                v += lh;
                if (v >= bottom)
                        break;
                if (top >= v+lh)
                        continue;
                user= username(filelist[i].uid);
                sprintf(buf, "%d", i);
                wdrawright(h1, v, buf, -1);
                if (filelist[i].active)
                        wdrawtext(h2, v, "*", 1);
                wdrawtext(h3, v, filelist[i].name, -1);
                if (filelist[i].uid >= 0) {
                        wdrawtext(h4, v, user, -1);
                }
                sprintf(buf, "%ldK", (filelist[i].size + 1023) / 1024);
                wdrawright(h5, v, buf, -1);
                sprintf(buf, "%d", filelist[i].ndata);
                wdrawright(h6, v, buf, -1);
                if (interesting(user))
                        wdrawline(0, v+lh-1, h6, v+lh-1);
        }
}

void
drawlong(left, top, right, bottom)
        int left, top, right, bottom;
{
        drawnormal(left, top, right, bottom);
}

void
wdrawright(h, v, text, len)
        int h, v;
        char *text;
        int len;
{
        wdrawtext(h - wtextwidth(text, len), v, text, len);
}

bool
interesting(user)
        char *user;
{
        char **u;
        if (userlist == NULL)
                return FALSE;
        for (u= userlist; *u != NULL; ++u) {
                if (strcmp(user, *u) == 0)
                        return TRUE;
        }
        return FALSE;
}

void
hilite(which)
        int which;
{
        int h1, v1, h2, v2;
        getrect(which, &h1, &v1, &h2, &v2);
        wbegindrawing(win);
        winvert(h1, v1, h2, v2);
        wenddrawing(win);
        wshow(win, h1, v1, h2, v2);
}

void
unhilite(which)
        int which;
{
        int h1, v1, h2, v2;
        getrect(which, &h1, &v1, &h2, &v2);
        wbegindrawing(win);
        winvert(h1, v1, h2, v2);
        wenddrawing(win);
}

void
getrect(which, ph1, pv1, ph2, pv2)
        int which;
        int *ph1, *pv1, *ph2, *pv2;
{
        int h1, v1, h2, v2;
        int row, col;
        int lh;
        switch (format) {
        case SHORTFORMAT:
                row= which % nrows;
                col= which / nrows;
                h1= col*colwidth;
                v1= (row+1)*colheight;
                h2= h1 + colwidth;
                v2= v1 + colheight;
                break;
        case NORMALFORMAT:
        case LONGFORMAT:
                lh= wlineheight();
                h1= 0;
                v1= (which+2)*lh;
                h2= 32000;
                v2= v1 + lh;
                break;
        default:
                panic("getrect: bad format");
        }
        *ph1= h1;
        *pv1= v1;
        *ph2= h2;
        *pv2= v2;
}
