/* Input line implementation
   Copyright (C) 1994 Miguel de Icaza, Mauricio Plaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <sys/types.h>
#include "input.h"
#include "util.h"
#include "global.h"
#include "main.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/input.c,v 1.3 1994/05/14 21:34:47 miguel Exp $";

void create_input (Input *in, int x, int y, int len, char *buffer,
     WINDOW *win, int color, int field_len, char *def_text)
{
    int i;
    
    in->start_x = x;
    in->start_y = y;
    in->max_len = len;
    in->window = win;
    in->buffer = buffer ? buffer : (char *) xmalloc (len+1, "create_input");
    in->color = color;
    in->field_len = field_len;
    in->first = 1;
    leaveok (win, FALSE);
    
    wstandend (win);
    strcpy (in->buffer, def_text);
    wmove (win, in->start_y, in->start_x);
    wattron (win, color);
    wprintw (win, "%s", def_text);
    for (i = strlen (buffer); i < field_len; i++)
	waddch (win, ' ');
    in->pos = strlen (buffer);
    wmove (win, in->start_y, in->start_x+strlen (buffer));
    wrefresh (win);
}

void update_input (Input *in)
{
    int    x = in->start_x;
    int    y = in->start_y;
    WINDOW *w = in->window;
    int    i;
    
    wstandend (w);
    wattron (w, in->color);
    in->first = 0;
    wmove (w, y, x);
    for (i = 0; i < in->field_len; i++)
	waddch (in->window, ' ');
    wmove (w, y, x);
    wprintw (w, "%s", in->buffer);
    wmove (w, y, x + in->pos);
    wrefresh (w);
}

static void insert_char (Input *in, int c_code)
{
    int i;
    
    if (strlen (in->buffer)+1 < in->field_len){
	int l = strlen (&in->buffer [in->pos]);
	for (i = l+2; i > 0; i--)
	    in->buffer [in->pos+i] = in->buffer [in->pos+i-1];
	in->buffer [in->pos] = c_code;
	in->pos++;
    }
}

void handle_char (Input *in, int c_code)
{
    int    i;
    char   minbuf [80], *p;
    static char memory [100] = "";
    static int x_buf = 0, y_buf = 0; 

    if (quote){
	insert_char (in, c_code);
	update_input (in);
	return;
    }

    switch (c_code){
    case 1:			/* C-a */
	in->pos = 0;
	break;

    case 5:			/* C-e */
	in->pos = strlen (in->buffer);
	break;
	
    case 2:			/* C-b */
    case KEY_LEFT:
	if (in->pos)
	    in->pos--;
	break;

    case KEY_BACKSPACE:
    case 0177:
    case 8:			/* C-h */
	if (!in->pos)
	    break;
	for (i = in->pos; in->buffer [i-1]; i++)
	    in->buffer [i-1] = in->buffer [i];
	in->pos--;
	break;

    case 0:			/* C-2 */
	x_buf = in->pos;
	if (y_buf < x_buf)
	    y_buf= x_buf;
	break;

    case XCTRL('w'):			/* C-w */
	y_buf = in->pos;
	
    case ALT_W:
	if (y_buf < x_buf)
	    x_buf=0;
	strncpy (memory, &in->buffer [x_buf], y_buf - x_buf);
	memory [y_buf-x_buf] = 0;
	break;
	
    case 0x19:			/* C-y */
	for (p = memory; *p; p++)
	    insert_char (in, *p);
	break;

    case 0xb:			/* C-k */
	strcpy (memory, &in->buffer [in->pos]);
	in->buffer [in->pos]=0;
	break;
	
    case 6:			/* C-f */
    case KEY_RIGHT:
	if (in->buffer [in->pos])
	    in->pos++;
	break;

    default:
	if (c_code > 255 || c_code < ' ')
	    break;
	if (in->first){
	    *in->buffer = 0;
	    in->pos = 0;
	}
	insert_char (in, c_code);
    }
    update_input (in);
}
