------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ E V A L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.73 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Itypes;   use Itypes;
with Namet;    use Namet;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Output;   use Output;
with Rat;      use Rat;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Stringt;  use Stringt;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;

package body Sem_Eval is

   -----------------------------------------
   -- Handling of Compile Time Evaluation --
   -----------------------------------------

   --  The compile time evaluation of expressions is distributed over several
   --  Eval_XXX procedures. These procedures are called immediatedly after
   --  a subexpression is resolved and is therefore accomplished in a bottom
   --  up fashion.

   --  Note that there are two types of expressions that can be evaluated at
   --  compile time, those that the Ada language defines as static and the
   --  more general category of expressions that do not need run-time support.
   --  For example the expression (True and then True) can be evaluated at
   --  compile time but is not considered static by Ada 83 since "and then"
   --  is not an operator (Ada 9X considers this static). All expressions that
   --  can be evaluated without run-time support are folded.

   --  The Is_Folded flag on the result node is set to True if the operation
   --  has been performed at compile time and the result expression can be
   --  determined at compile time. The Is_Static flag on the result node is
   --  set to True only if it meets the Ada definition of staticness as
   --  described above. It is the responsibility of each of the Eval_XXX
   --  procedures to set these flags (Is_Folded, Is_Static) appropriately.

   --  Rewrite_Substitute_Tree is used to replace the original expression
   --  with the node containing the result of the evaluation. The original
   --  expression tree can still be retrieved (for purposes of conformance
   --  checking, source recreation, and ASIS compliance) by using the
   --  function Original_Node.

   --  Identifiers representing user-defined constants which are static are
   --  not replaced by other nodes but are just marked as being static/folded.

   --  For the arithmetic results no overflow checking is done here (i.e.
   --  the exact result is stored as a Uint). Overflow checking will be
   --  performed as part of the expander pass.

   --  If the expression evaluation shows that the expression would raise
   --  Constraint_Error during run-time (e.g. divide by zero), a special
   --  N_Expression_Actions node is created to raise Constraint_Error
   --  (see Create_Constraint_Error).

   ----------------
   -- Local Data --
   ----------------

   type Bits is array (Nat range <>) of Boolean;
   --  Used to convert unsigned (modular) values for folding logical ops

   Pragma_Preelaborable_Enabled : Boolean := False;
   --  This flag is used to enable checking for pragma Preelaborate, but this
   --  circuit is not enabled yet, so there is no way for this to get set True
   --  ??? to be implemented later.

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Check_Preevaluate (N : Node_Id);
   --  If the pragma Preelaborate switch is set, this subprogram checks for
   --  a violation of preelaborate rules and generates an appropriate message

   function Compare_Strings (S1, S2 : String_Id) return Int;
   --  Compares two strings, used for folding string comparisons. Returns
   --  negative, zero or positive for S1 < S2, S1 = S2, S1 > S2 respectively.

   procedure Create_Str_Value_Node (N : Node_Id; Str_Id : String_Id);
   --  Create an N_String_Literal node for the result of compile time
   --  evaluation of the "&" operation on string literals, character literals
   --  or constants.

   function From_Bits (B : Bits) return Uint;
   --  Converts a bit string of length B'Length to a Uint value

   function Get_String_Val (N : Node_Id) return Node_Id;
   --  Given a tree node for a folded string or character value, returns
   --  the corresponding string literal or character literal (one of the
   --  two must be available, or the operand would not have been marked
   --  as folded in the earlier analysis of the operands).

   function Is_False (U : Uint) return Boolean renames UI_Is_Zero;
   --  The argument is a Uint value which is the Boolean'Pos value of a
   --  Boolean operand (i.e. is either 0 for False, or 1 for True). This
   --  function simply tests if it is False (i.e. zero)

   function Is_True (U : Uint) return Boolean;
   pragma Inline (Is_True);
   --  The argument is a Uint value which is the Boolean'Pos value of a
   --  Boolean operand (i.e. is either 0 for False, or 1 for True). This
   --  function simply tests if it is True (i.e. non-zero)

   function Is_Foldable_Subtype (Typ : Entity_Id) return Boolean;
   --  Determine if the bounds of subtype are known at compile time.

   procedure Rewrite_Node (N : Node_Id; Val : Uint);
   --  Replace N with a (N_Integer_Literal, N_Identifier, N_Character_Literal)
   --  node as the result of the compile time evaluation of the node N. Val
   --  is the result in the integer case and is the position of the literal
   --  in the literals list for the enumeration case. This procedure does set
   --  the Is_Folded flag, but it does not set the Is_Static flag.

   procedure Rewrite_Node (N : Node_Id; Real_Lit_Node : Node_Id);
   --  Replace N with an N_Real_LIteral node as a result of the compile time
   --  evaluation of the node N. The Is_Folded flag is set, but not Is_Static.

   function Test (Cond : Boolean) return Uint;
   pragma Inline (Test);
   --  This function simply returns the appropriate Boolean'Pos value
   --  corresponding to the value of Cond as a universal integer. It is
   --  used for producing the result of the static evaluation of the
   --  logical operators

   procedure To_Bits (U : Uint; B : out Bits);
   --  Converts a Uint value to a bit string of length B'Length

   -----------------------
   -- Check_Preevaluate --
   -----------------------

   procedure Check_Preevaluate (N : Node_Id) is
   begin
      if Pragma_Preelaborable_Enabled then
         Error_Msg_N ("Evaluation of this expression is not preevaluable", N);
      end if;
   end Check_Preevaluate;

   ---------------------
   -- Compare_Strings --
   ---------------------

   function Compare_Strings (S1, S2 : String_Id) return Int is
      L1 : constant Nat := String_Length (S1);
      L2 : constant Nat := String_Length (S2);
      LM : Nat;

      C1 : Char_Code;
      C2 : Char_Code;

   begin
      if L1 <= L2 then
         LM := L1;
      else
         LM := L2;
      end if;

      for J in 1 .. LM loop
         C1 := Get_String_Char (S1, J);
         C2 := Get_String_Char (S2, J);

         if C1 /= C2 then
            return Int (C1) - Int (C2);
         end if;
      end loop;

      return Int (L1) - Int (L2);
   end Compare_Strings;

   ---------------------------
   -- Create_Str_Value_Node --
   ---------------------------

   --  Note that this procedure sets the Is_Folded flag, but it is up to the
   --  caller to set Is_Static if the resulting string constant is to be
   --  regarded as static (possible only in Ada 9X mode of course).

   procedure Create_Str_Value_Node (N : Node_Id; Str_Id : String_Id) is
      Str_Literal_Node : Node_Id;
      Subtype_Id       : Entity_Id;

   begin
      Str_Literal_Node := New_Node (N_String_Literal, Sloc (N));
      Set_Strval (Str_Literal_Node, Str_Id);
      Set_Is_Folded (Str_Literal_Node, True);

      --  Create a special subtype for the N_String_Literal node which
      --  becomes its Etype.

      Subtype_Id :=
        New_Itype (Str_Literal_Node, Scope_Id => Scope (Etype (N)));
      Set_Ekind (Subtype_Id, E_String_Literal_Subtype);
      Set_Etype (Subtype_Id, Etype (N));
      Set_Component_Type (Subtype_Id, Component_Type (Etype (N)));
      Set_String_Literal_Length
        (Subtype_Id, UI_From_Int (String_Length (Str_Id)));
      Set_Etype (Str_Literal_Node, Subtype_Id);
      Rewrite_Substitute_Tree (N, Str_Literal_Node);
   end Create_Str_Value_Node;

   --------------------
   -- Eval_Attribute --
   --------------------

   --  Note: a number of attributes corresponding to constant values were
   --  rewritten already during semantic processing. These include:

   --    Address_Size
   --    Denorm
   --    Emax
   --    Epsilon
   --    Large
   --    Machine_Emax
   --    Machine_Emin
   --    Machine_Mantissa
   --    Machine_Overflows
   --    Machine_Radix
   --    Machine_Rounds
   --    Mantissa
   --    Max_Intrerrupt_Priority
   --    Max_Priority
   --    Model_Emin
   --    Model_Epsilon
   --    Model_Mantissa
   --    Model_Small
   --    Safe_Emax
   --    Safe_First
   --    Safe_Large
   --    Safe_Small
   --    Signed_Zeros
   --    Storage_Unit
   --    Universal_Literal_String
   --    Word_Size

   --  Note also that the Range attribute has at this stage been rewritten as
   --  an explicit range using references to the First and Last attributes.

   procedure Eval_Attribute (N : Node_Id) is
      Aname     : constant Name_Id      := Attribute_Name (N);
      Id        : constant Attribute_Id := Get_Attribute_Id (Aname);
      P         : constant Node_Id      := Prefix (N);
      E1        : Node_Id;
      E2        : Node_Id;
      Typ       : Entity_Id;

   begin
      if Present (Expressions (N)) then
         E1 := First (Expressions (N));
         E2 := Next (E1);
      else
         E1 := Empty;
         E2 := Empty;
      end if;

      --  We can only fold a small subset of attributes. Note that many
      --  attributes, such as the floating-point attributes, were already
      --  folded to constants during the semantic processing.

      case Id is
         when Attribute_First  |
              Attribute_Digits |
              Attribute_Last   |
              Attribute_Max    |
              Attribute_Min    |
              Attribute_Pos    |
              Attribute_Pred   |
              Attribute_Size   |
              Attribute_Succ   |
              Attribute_Val
                             => null;

         when others => return;
      end case;

      --  Can't fold if previous error

      if Etype (P) = Any_Type or else
         (Present (E1) and then Etype (E1) = Any_Type)
      then
         return;
      end if;

      --  Establish the prefix type, treating 'Base specially. If the prefix
      --  is other than a name or 'Base reference, then we cannot fold.

      if Nkind (P) in N_Entity_Name then
         Typ := Entity (P);

      elsif Nkind (P) = N_Attribute_Reference
        and then Attribute_Name (P) = Name_Base
      then
         Typ := Etype (P);

      else
         return;
      end if;

      --  Now look at the type, we can only fold a type or foldable subtype
      --  where there are no expressions, or the expressions themselves are
      --  foldable if they are present.

      if not Is_Type (Typ) or else not Is_Foldable_Subtype (Typ) then
         return;

      elsif Present (E1) then
         if not Is_Folded (E1) then
            return;
         else
            if Present (E2) then
               if not Is_Folded (E2) then
                  return;
               end if;
            end if;
         end if;
      end if;

      --  Remaining processing depends on particular attribute

      case Id is

         --  Digits attribute. Rewrite with appropriate value

         when Attribute_Digits =>
            Rewrite_Node (N, Digits_Value (Typ));

         --  Processing for 'First attribute. For any scalar type T or for
         --  any subtype T of a scalar type, T'First yields the upper bound of
         --  T. The value of this attribute has the same type as T. [LRM 3.5]

         when Attribute_First =>
            if Is_Float_Type (Typ) then
               Rewrite_Node (N, Type_Low_Bound (Typ));
            else
               Rewrite_Node (N, Expr_Value (Type_Low_Bound (Typ)));
            end if;

         --  Processing for 'Last attribute. For any scalar type T or for
         --  any subtype T of a scalar type, T'First yields the upper bound of
         --  T. The value of this attribute has the same type as T. [LRM 3.5]

         when Attribute_Last =>
            if Is_Float_Type (Typ) then
               Rewrite_Node (N, Type_High_Bound (Typ));
            else
               Rewrite_Node (N, Expr_Value (Type_High_Bound (Typ)));
            end if;

         --  Processing for 'Max attribute, if both operands are static,
         --  result is static and we compute the max at compile time

         when Attribute_Max =>
            if Is_Real_Type (Typ) then
               Rewrite_Node (N, Rat_Max (E1, E2));
            else
               Rewrite_Node (N, UI_Max (Expr_Value (E1), Expr_Value (E2)));
            end if;

         --  Processing for 'Min attribute, if both operands are static,
         --  result is static and we compute the min at compile time

         when Attribute_Min =>
            if Is_Real_Type (Typ) then
               Rewrite_Node (N, Rat_Min (E1, E2));
            else
               Rewrite_Node (N, UI_Min (Expr_Value (E1), Expr_Value (E2)));
            end if;

         --  Pos attribute. The result is the position number of the value
         --  of the parameter. [LRM 3.5.5]

         when Attribute_Pos =>
            Rewrite_Node (N, Expr_Value (E1));

         --  For T'PRED (X) The parameter X must be a value of the base type
         --  of T. The result type is the base type of T. The result is the
         --  value whose position number is one less than that of X. The
         --  exception CONSTRAINT_ERROR is raised if X equals T'BASE'FIRST.

         when Attribute_Pred =>
            if UI_Eq (Expr_Value (E1),
              Expr_Value (Type_Low_Bound (Base_Type (Typ))))
            then
               Raise_Warning (N, Standard_Constraint_Error,
                 "Pred of type'First?!");
               Create_Raise_Expression (N, Standard_Constraint_Error);
               return;
            else
               Rewrite_Node (N, UI_Difference (Expr_Value (E1), Uint_1));
            end if;

         --  Size attribute just returns the size (note that we could only
         --  be called for size applied to an appropriate type to yield
         --  a static type)

         when Attribute_Size =>
            Rewrite_Node (N, Esize (Typ));

         --  For T'SUCC (X) The parameter X must be a value of the base type
         --  of T. The result type is the base type of T. The result is the
         --  value whose position number is one greater than that of X. The
         --  exception CONSTRAINT_ERROR is raised if X equals T'BASE'LAST.

         when Attribute_Succ =>
            if UI_Eq (Expr_Value (E1),
                      Expr_Value (Type_High_Bound (Base_Type (Typ))))
            then
               Raise_Warning (N, Standard_Constraint_Error,
                 "Succ of type'Last?!");
               Create_Raise_Expression (N, Standard_Constraint_Error);
               return;
            else
               Rewrite_Node (N,  UI_Sum (Expr_Value (E1), Uint_1));
            end if;

         --  For T'VAL (X) This attribute is a special function with a single
         --  parameter which  can be of any integer type. The result type is
         --  the base type of T. The result is the value whose position number
         --  is the universal_integer value corresponding to X. The exception
         --  CONSTRAINT_ERROR is raised if the universal_integer value
         --  corresponding to X is not in the range T'POS(T'BASE'FIRST)
         --  ..  T'POS(T'BASE'LAST).

         when Attribute_Val =>
            if UI_Lt (Expr_Value (E1),
                      Expr_Value (Type_Low_Bound (Base_Type (Typ)))) or else
               UI_Gt (Expr_Value (E1),
                      Expr_Value (Type_High_Bound (Base_Type (Typ))))
            then
               Raise_Warning (N, Standard_Constraint_Error,
                 "pos out of range?!");
               Create_Raise_Expression (N, Standard_Constraint_Error);
               return;
            else
               Rewrite_Node (N, Expr_Value (E1));
            end if;

         --  Remaining attributes cannot be folded

         when others => return;
      end case;

      if Is_Static_Subtype (Typ) then
         if Present (E1) then
            if Present (E2) then
               Set_Is_Static (N, Is_Static (E1) and Is_Static (E2));
            else
               Set_Is_Static (N, Is_Static (E1));
            end if;
         else
            Set_Is_Static (N);
         end if;
      end if;
   end Eval_Attribute;

   ------------------------
   -- Eval_Arithmetic_Op --
   ------------------------

   procedure Eval_Arithmetic_Op (N : Node_Id) is
      Left      : constant Node_Id := Left_Opnd (N);
      Right     : constant Node_Id := Right_Opnd (N);
      Left_Int  : Uint;
      Right_Int : Uint;
      Result    : Uint;

   begin
      if not (Is_Folded (Left) and then Is_Folded (Right)) then
         return;

      elsif Is_Integer_Type (Etype (Left))
        and then Is_Integer_Type (Etype (Right))
      then
         Left_Int := Expr_Value (Left);
         Right_Int := Expr_Value (Right);

         case Nkind (N) is

            when N_Op_Add =>
               Result := UI_Sum (Left_Int, Right_Int);

            when N_Op_Subtract =>
               Result := UI_Difference (Left_Int, Right_Int);

            when N_Op_Multiply =>
               Result := UI_Product (Left_Int, Right_Int);

            when N_Op_Divide =>

               --  The exception Constraint_Error is raised by integer
               --  division, rem and mod if the right operand is zero.

               if UI_Is_Zero (Right_Int) then
                  Raise_Warning (N, Standard_Constraint_Error,
                    "division by zero?!");
                  Create_Raise_Expression (N, Standard_Constraint_Error);
                  return;
               else
                  Result := UI_Quotient (Left_Int, Right_Int);
               end if;

            when N_Op_Mod =>

               --  The exception Constraint_Error is raised by integer
               --  division, rem and mod if the right operand is zero.

               if UI_Is_Zero (Right_Int) then
                  Raise_Warning (N, Standard_Constraint_Error,
                    "mod with zero right operand?!");
                  Create_Raise_Expression (N, Standard_Constraint_Error);
                  return;
               else
                  Result := UI_Mod (Left_Int, Right_Int);
               end if;

            when N_Op_Rem =>

               --  The exception Constraint_Error is raised by integer
               --  division, rem and mod if the right operand is zero.

               if UI_Is_Zero (Right_Int) then
                  Raise_Warning (N, Standard_Constraint_Error,
                    "rem with zero right operand?!");
                  Create_Raise_Expression (N, Standard_Constraint_Error);
                  return;
               else
                  Result := UI_Rem (Left_Int, Right_Int);
               end if;

            when others =>
               pragma Assert (False); null;
         end case;

         --  Adjust the result by the modulus if the type is a modular type

         if Is_Modular_Integer_Type (Etype (N)) then
            Rewrite_Node (N, UI_Mod (Result, Modulus (Etype (N))));
         else
            Rewrite_Node (N, Result);
         end if;

      elsif Is_Float_Type (Etype (Left))
        and then Is_Float_Type (Etype (Right))
      then
         if Nkind (N) = N_Op_Add then
            Rewrite_Node (N, Rat_Sum (Left, Right));

         elsif Nkind (N) = N_Op_Subtract then
            Rewrite_Node (N, Rat_Difference (Left, Right));

         elsif Nkind (N) = N_Op_Multiply then
            Rewrite_Node (N, Rat_Product (Left, Right));

         else
            pragma Assert (Nkind (N) = N_Op_Divide);
            Rewrite_Node (N, Rat_Quotient (Left, Right));
         end if;
      end if;

      Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));
   end Eval_Arithmetic_Op;

   ----------------------------
   -- Eval_Character_Literal --
   ----------------------------

   procedure Eval_Character_Literal (N : Node_Id) is
   begin
      Set_Is_Folded (N);
      Set_Is_Static (N);
   end Eval_Character_Literal;

   ------------------------
   -- Eval_Concatenation --
   ------------------------

   --  Attempt to constant fold concatenation between combinations of
   --  string_literal, character_literals and static user defined
   --  constants.

   procedure Eval_Concatenation (N : Node_Id) is
      Left  : constant Node_Id := Left_Opnd (N);
      Right : constant Node_Id := Right_Opnd (N);

   begin
      if Is_Folded (Left) and then Is_Folded (Right) then
         declare
            Left_Str  : constant Node_Id := Get_String_Val (Left);
            Right_Str : constant Node_Id := Get_String_Val (Right);

         begin
            --  Establish new string literal, and store left operand. We make
            --  sure to use the special Start_String that takes an operand if
            --  the left operand is a string literal. Since this is optimized
            --  in the case where that is the most recently created string
            --  literal, we ensure efficient time/space behavior for the
            --  case of a concatenation of a series of string literals.

            if Nkind (Left_Str) = N_String_Literal then
               Start_String (Strval (Left_Str));
            else
               Start_String;
               Store_String_Char (Char_Literal_Value (Left_Str));
            end if;

            --  Now append the characters of the right operand

            if Nkind (Right_Str) = N_String_Literal then
               declare
                  S : constant String_Id := Strval (Right_Str);

               begin
                  for J in 1 .. String_Length (S) loop
                     Store_String_Char (Get_String_Char (S, J));
                  end loop;
               end;
            else
               Store_String_Char (Char_Literal_Value (Right_Str));
            end if;

            Create_Str_Value_Node (N, End_String);

            if Ada_9X then
               Set_Is_Static (N, Is_Static (Left) and Is_Static (Right));
            end if;
         end;
      end if;
   end Eval_Concatenation;

   ----------------------
   -- Eval_Entity_Name --
   ----------------------

   --  This procedure is used for identifiers and expanded names

   procedure Eval_Entity_Name (N : Node_Id) is
      Def_Id : constant Entity_Id := Entity (N);

   begin
      --  Enumeration literals are always considered to be constants

      if Ekind (Def_Id) = E_Enumeration_Literal then
         Set_Is_Folded (N);
         Set_Is_Static (N);

      --  A constant explicitly declared by a constant declaration with
      --  a static subtype, and initialized with a static expression.
      --  The expression value given for a constant is evaluated when the
      --  object declaration is processed so here we can just test its
      --  Is_Folded flag.

      elsif Ekind (Def_Id) = E_Constant then
         declare
            Val : constant Node_Id   := Constant_Value (Def_Id);
            Typ : constant Entity_Id := Etype (Def_Id);

         begin
            if (Is_Scalar_Type (Typ) or else Nkind (Val) = N_String_Literal)
              and then Is_Folded (Constant_Value (Def_Id))
            then
               Set_Is_Folded (N);
               Set_Is_Static (N, Is_Static (Constant_Value (Def_Id)));
            end if;
         end;
      end if;
   end Eval_Entity_Name;

   ---------------------------
   -- Eval_Integer_Literal --
   ---------------------------

   --  Note: we don't check that the literal is in range, since such a check
   --  at this stage would be premature (the value might be part of a static
   --  expression which gets folded to something that is in range). We leave
   --  the check for literals in range up to Gigi.

   procedure Eval_Integer_Literal (N : Node_Id) is
   begin
      Set_Is_Folded (N);
      Set_Is_Static (N);
   end Eval_Integer_Literal;

   ---------------------
   -- Eval_Logical_Op --
   ---------------------

   procedure Eval_Logical_Op (N : Node_Id) is
      Left      : constant Node_Id := Left_Opnd (N);
      Right     : constant Node_Id := Right_Opnd (N);
      Left_Int  : Uint;
      Right_Int : Uint;

   begin
      if Is_Folded (Left) and then Is_Folded (Right) then
         Left_Int := Expr_Value (Left);
         Right_Int := Expr_Value (Right);

         if Is_Modular_Integer_Type (Etype (N)) then
            declare
               Left_Bits  : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);
               Right_Bits : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);

            begin
               To_Bits (Left_Int, Left_Bits);
               To_Bits (Right_Int, Right_Bits);

               --  Note: should really be able to use array ops instead of
               --  these loops, but they weren't working at the time ???

               if Nkind (N) = N_Op_And then
                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) and Right_Bits (J);
                  end loop;

               elsif Nkind (N) = N_Op_Or then
                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) or Right_Bits (J);
                  end loop;

               else
                  pragma Assert (Nkind (N) = N_Op_Xor);

                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) xor Right_Bits (J);
                  end loop;
               end if;

               Rewrite_Node (N, From_Bits (Left_Bits));
            end;

         else
            pragma Assert (Is_Boolean_Type (Etype (N)));
            if Nkind (N) = N_Op_And then
               Rewrite_Node (N,
                 Test (Is_True (Left_Int) and then Is_True (Right_Int)));

            elsif Nkind (N) = N_Op_Or then
               Rewrite_Node (N,
                 Test (Is_True (Left_Int) or else Is_True (Right_Int)));

            else
               pragma Assert (Nkind (N) = N_Op_Xor);
               Rewrite_Node (N,
                 Test (Is_True (Left_Int) xor Is_True (Right_Int)));
            end if;
         end if;

         Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));
      end if;
   end Eval_Logical_Op;

   ------------------------
   -- Eval_Membership_Op --
   ------------------------

   procedure Eval_Membership_Op (N : Node_Id) is
      Def_Id   : Entity_Id;
      Left     : constant Node_Id := Left_Opnd (N);
      Left_Int : Uint;

   begin
      if Nkind (Right_Opnd (N)) in N_Entity_Name then
         Def_Id := Entity (Right_Opnd (N));
      else
         Def_Id := Etype (Right_Opnd (N));
      end if;

      if Is_Folded (Left) and then Is_Foldable_Subtype (Def_Id) then
         Left_Int := Expr_Value (Left);

         if Nkind (N) = N_Op_In then
            Rewrite_Node (N, Test (
              UI_Le (Expr_Value (Type_Low_Bound (Def_Id)), Left_Int)
                and then
                  UI_Ge (Expr_Value (Type_High_Bound (Def_Id)), Left_Int)));

         else
            pragma Assert (Nkind (N) = N_Op_Not_In);
            Rewrite_Node (N, Test (
              UI_Gt (Expr_Value (Type_Low_Bound (Def_Id)), Left_Int)
                or else
                  UI_Lt (Expr_Value (Type_High_Bound (Def_Id)), Left_Int)));
         end if;

         Set_Is_Static
           (N, Is_Static (Left) and then Is_Static_Subtype (Def_Id));
      end if;
   end Eval_Membership_Op;

   -----------------
   -- Eval_Op_Not --
   -----------------

   procedure Eval_Op_Not (N : Node_Id) is
      Right : constant Node_Id := Right_Opnd (N);
      Rint  : Uint;

   begin
      if Is_Folded (Right) then
         Rint := Expr_Value (Right);

         if Is_Modular_Integer_Type (Etype (N)) then
            declare
               Right_Bits : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);
            begin
               To_Bits (Rint, Right_Bits);

               for I in Right_Bits'range loop
                  Right_Bits (I) := not Right_Bits (I);
               end loop;

               Rewrite_Node (N, From_Bits (Right_Bits));
            end;

         else
            pragma Assert (Is_Boolean_Type (Etype (N)));
            Rewrite_Node (N, Test (not Is_True (Rint)));
         end if;

         Set_Is_Static (N, Is_Static (Right));
      end if;
   end Eval_Op_Not;

   -------------------
   -- Eval_Op_Expon --
   -------------------

   procedure Eval_Op_Expon (N : Node_Id) is
      Left      : constant Node_Id := Left_Opnd (N);
      Right     : constant Node_Id := Right_Opnd (N);
      Right_Int : Uint;
      Result    : Uint;

   begin
      if Is_Folded (Left) and then Is_Folded (Right) then
         Right_Int := Expr_Value (Right);

         if Is_Integer_Type (Etype (Left)) then

            --  Exponentiation of an integer raises the exception
            --  Constraint_Error for a negative exponent. [LRM 4.5.6]

            if UI_Is_Negative (Right_Int) then
               Raise_Warning (N, Standard_Constraint_Error,
                 "negative exponent for integer base?!");
               Create_Raise_Expression (N, Standard_Constraint_Error);
               return;
            else
               Result := UI_Exponentiate (Expr_Value (Left), Right_Int);

               if Is_Modular_Integer_Type (Etype (N)) then
                  Result := UI_Mod (Result, Modulus (Etype (N)));
               end if;

               Rewrite_Node (N, Result);
            end if;

         else
            --  Cannot have a zero base with a negative exponent

            if UI_Is_Negative (Right_Int)
              and then Rat_Is_Zero (Left)
            then
               Raise_Warning (N, Standard_Constraint_Error,
                 "negative exponent for zero base?!");
               Create_Raise_Expression (N, Standard_Constraint_Error);
               return;
            else
               Rewrite_Node (N, Rat_Exponentiate (Left, Right_Int));
            end if;
         end if;

         Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));
      end if;
   end Eval_Op_Expon;

   -------------------------------
   -- Eval_Qualified_Expression --
   -------------------------------

   procedure Eval_Qualified_Expression (N : Node_Id) is
   begin
      if Is_Folded (Expression (N)) then
         Rewrite_Substitute_Tree (N, Expression (N));
         Set_Paren_Count (N, 0);
      end if;
   end Eval_Qualified_Expression;

   -----------------------
   -- Eval_Real_Literal --
   -----------------------

   procedure Eval_Real_Literal (N : Node_Id) is
   begin
      Set_Is_Folded (N);
      Set_Is_Static (N);
   end Eval_Real_Literal;

   ------------------------
   -- Eval_Relational_Op --
   ------------------------

   procedure Eval_Relational_Op (N : Node_Id) is
      Left      : constant Node_Id   := Left_Opnd (N);
      Right     : constant Node_Id   := Right_Opnd (N);
      Typ       : constant Entity_Id := Etype (Left);
      Left_Int  : Uint;
      Right_Int : Uint;
      Result    : Boolean;

   begin
      if Is_Folded (Left) and then Is_Folded (Right) then

         --  Integer and Enumeration type cases

         if Is_Integer_Type (Typ) or else Is_Enumeration_Type (Typ) then
            Left_Int := Expr_Value (Left);
            Right_Int := Expr_Value (Right);

            case Nkind (N) is
               when N_Op_Eq => Result := UI_Eq (Left_Int, Right_Int);
               when N_Op_Ne => Result := UI_Ne (Left_Int, Right_Int);
               when N_Op_Lt => Result := UI_Lt (Left_Int, Right_Int);
               when N_Op_Le => Result := UI_Le (Left_Int, Right_Int);
               when N_Op_Gt => Result := UI_Gt (Left_Int, Right_Int);
               when N_Op_Ge => Result := UI_Ge (Left_Int, Right_Int);

               when others => pragma Assert (False); null;
            end case;

            Rewrite_Node (N, Test (Result));
            Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));

         --  Real type case

         elsif Is_Real_Type (Typ) then
            case Nkind (N) is
               when N_Op_Eq => Result := Rat_Eq (Left, Right);
               when N_Op_Ne => Result := Rat_Ne (Left, Right);
               when N_Op_Lt => Result := Rat_Lt (Left, Right);
               when N_Op_Le => Result := Rat_Le (Left, Right);
               when N_Op_Gt => Result := Rat_Gt (Left, Right);
               when N_Op_Ge => Result := Rat_Ge (Left, Right);

               when others => pragma Assert (False); null;
            end case;

            Rewrite_Node (N, Test (Result));
            Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));

         --  String case

         elsif Is_String_Type (Typ) then
            declare
               LS  : constant String_Id := Strval (Get_String_Val (Left));
               RS  : constant String_Id := Strval (Get_String_Val (Right));
               Cmp : constant Int       := Compare_Strings (LS, RS);

            begin
               case Nkind (N) is
                  when N_Op_Eq => Result := Cmp =  0;
                  when N_Op_Ne => Result := Cmp /= 0;
                  when N_Op_Lt => Result := Cmp <  0;
                  when N_Op_Le => Result := Cmp <= 0;
                  when N_Op_Gt => Result := Cmp >  0;
                  when N_Op_Ge => Result := Cmp >= 0;

                  when others => pragma Assert (False); null;
               end case;
            end;

            --  Note that the result is never static in this case, since
            --  string comparison operators are not static functions.

            Rewrite_Node (N, Test (Result));
         end if;

      end if;
   end Eval_Relational_Op;

   ----------------
   -- Eval_Shift --
   ----------------

   --  Shifts are not static, but they should be folded when possible. To
   --  be added later ???

   procedure Eval_Shift (N : Node_Id) is
   begin
      null;
   end Eval_Shift;

   ------------------------
   -- Eval_Short_Circuit --
   ------------------------

   procedure Eval_Short_Circuit (N : Node_Id) is
      Left      : constant Node_Id := Left_Opnd (N);
      Right     : constant Node_Id := Right_Opnd (N);
      Left_Int  : Uint;
      Right_Int : Uint;

   begin
      if Is_Folded (Left) and then Is_Folded (Right) then
         Left_Int := Expr_Value (Left);
         Right_Int := Expr_Value (Right);

         if Nkind (N) = N_Op_And_Then then
            Check_Preevaluate (N);
            Rewrite_Node (N,
              Test (Is_True (Left_Int) and then Is_True (Right_Int)));

         else
            pragma Assert (Nkind (N) = N_Op_Or_Else);
            Check_Preevaluate (N);
            Rewrite_Node (N,
              Test (Is_True (Left_Int) or else Is_True (Right_Int)));
         end if;

         --  Note that short circuit operations are never static in Ada 83

         if Ada_9X then
            Set_Is_Static (N, Is_Static (Left) and then Is_Static (Right));
         end if;
      end if;
   end Eval_Short_Circuit;

   -------------------------
   -- Eval_String_Literal --
   -------------------------

   procedure Eval_String_Literal (N : Node_Id) is
   begin
      Set_Is_Folded (N);

      if Ada_9X and then Is_Static_Subtype (Component_Type (Etype (N))) then
         Set_Is_Static (N);
      end if;
   end Eval_String_Literal;

   --------------------------
   -- Eval_Type_Conversion --
   --------------------------

   procedure Eval_Type_Conversion (N : Node_Id) is
      Operand     : Node_Id   := Expression (N);
      Target_Type : Entity_Id := Etype (N);

   begin
      --  For now we are handling only the case of type converions that are
      --  of a discrete type.

      if Is_Static_Subtype (Target_Type)
        and then Is_Static (Operand)
        and then Is_Discrete_Type (Target_Type)
        and then Is_Discrete_Type (Etype (Operand))
      then
         --  Convert user defined static constants to the corresponding literal
         --  node (either N_Integer_Literal or N_Identifier) so that they can
         --  be marked with the new Etype if it is within the proper bounds.

         if Nkind (Operand) in N_Entity_Name then
            Rewrite_Node (Operand, Expr_Value (Operand));
         end if;

         --  Check if the operand is in the bounds of the target type. If it
         --  is, then it is retyped with the target type and marked as static;
         --  otherwise it is converted into a raise node and an appropriate
         --  warning is issued.

         if  UI_Lt (Expr_Value (Operand),
                    Expr_Value (Type_Low_Bound (Target_Type)))
           or else
             UI_Gt (Expr_Value (Operand),
                    Expr_Value (Type_High_Bound (Target_Type)))
         then
            Raise_Warning (N, Standard_Constraint_Error,
              "expression of subtype is out of range ?!");
         else
            Set_Etype (Operand, Target_Type);
            Rewrite_Substitute_Tree (N, Relocate_Node (Operand));
            Set_Is_Folded (N);

            --  Type conversions are only considered static in Ada 9X

            if Ada_9X then
               Set_Is_Static (N);
            end if;
         end if;
      end if;
   end Eval_Type_Conversion;

   -------------------
   -- Eval_Unary_Op --
   -------------------

   procedure Eval_Unary_Op (N : Node_Id) is
      Right : constant Node_Id := Right_Opnd (N);
      Rint  : Uint;

   begin
      if not Is_Folded (Right) then
         return;

      elsif Is_Integer_Type (Etype (N)) then
         Rint := Expr_Value (Right);

         --  In the case of modular unary plus and abs there is no need to
         --  adjust the result of the operation since if the original operand
         --  was in bounds the result will be in the bounds of the modular
         --  type. However, in the case of modular unary minus the result
         --  may go out of the bounds of the modular type and needs adjustment.

         if Nkind (N) = N_Op_Plus then
            Rewrite_Node (N, Rint);

         elsif Nkind (N) = N_Op_Minus then
            if Is_Modular_Integer_Type (Etype (N)) then
               Rewrite_Node (N,
                 UI_Mod (UI_Negate (Rint), Modulus (Etype (N))));
            else
               Rewrite_Node (N, UI_Negate (Rint));
            end if;

         else
            pragma Assert (Nkind (N) = N_Op_Abs);
            Rewrite_Node (N, UI_Abs (Rint));
         end if;

      elsif Is_Float_Type (Etype (N)) then

         if Nkind (N) = N_Op_Plus then
            Rewrite_Node (N, New_Copy (Right));

         elsif Nkind (N) = N_Op_Minus then
            Rewrite_Node (N, Rat_Negate (Right));

         else
            pragma Assert (Nkind (N) = N_Op_Abs);
            Rewrite_Node (N, Rat_Abs (Right));
         end if;
      end if;

      Set_Is_Static (N, Is_Static (Right));
   end Eval_Unary_Op;

   ----------------
   -- Expr_Value --
   ----------------

   function Expr_Value (N : Node_Id) return Uint is
      Def_Id : Entity_Id;
      Kind   : Node_Kind;

   begin
      Kind := Nkind (N);

      if Kind = N_Identifier or else Kind = N_Expanded_Name then
         Def_Id := Entity (N);

         --  An enumeration literal that was either in the source or
         --  created as a result of static evaluation.

         if Ekind (Def_Id) = E_Enumeration_Literal then
            return Enumeration_Pos (Def_Id);

         --  A user defined static constant

         else
            pragma Assert (Ekind (Def_Id) = E_Constant);
            return Expr_Value (Constant_Value (Def_Id));
         end if;

      --  An integer literal that was either in the source or created as
      --  a result of static evaluation.

      elsif Kind = N_Integer_Literal then
         return Intval (N);

      else
         pragma Assert (Kind = N_Character_Literal);

         --  Since Character literals of type Standard.Character don't
         --  have any defining character literals built for them, they
         --  do not have their Entity set, so just use their Char
         --  code. Otherwise for user-defined character literals use
         --  their Pos value as usual.

         if No (Entity (N)) then
            return UI_From_Int (Int (Char_Literal_Value (N)));
         else
            return Enumeration_Pos (Entity (N));
         end if;
      end if;

   end Expr_Value;

   ---------------
   -- From_Bits --
   ---------------

   function From_Bits (B : Bits) return Uint is
      T : Uint := Uint_0;

   begin
      for J in 0 .. B'Last loop
         if B (J) then
            T := UI_Sum (T, UI_Exponentiate (Uint_2, UI_From_Int (J)));
         end if;
      end loop;

      return T;
   end From_Bits;

   --------------------
   -- Get_String_Val --
   --------------------

   function Get_String_Val (N : Node_Id) return Node_Id is
   begin
      if Nkind (N) = N_String_Literal then
         return N;

      elsif Nkind (N) = N_Character_Literal then
         return N;

      else
         pragma Assert (Nkind (N) in N_Entity_Name);
         return Get_String_Val (Constant_Value (Entity (N)));
      end if;
   end Get_String_Val;

   -------------------------
   -- Is_Foldable_Subtype --
   -------------------------

   function Is_Foldable_Subtype (Typ : Entity_Id) return Boolean is
      Base_T : Entity_Id := Base_Type (Typ);

   begin
      if Is_Generic_Type (Base_T) or
         not Is_Scalar_Type (Base_T)
      then
         return False;

      elsif Base_T = Typ then
         return True;

      --  ??? For now always consider floating point types as foldable since
      --  we currently reject ranges that are put on floating points.

      elsif Is_Float_Type (Typ) then
         return True;

      else
         return Is_Foldable_Subtype (Base_T)
           and then Is_Folded (Type_Low_Bound (Typ))
           and then Is_Folded (Type_High_Bound (Typ));
      end if;
   end Is_Foldable_Subtype;

   -----------------------
   -- Is_Static_Subtype --
   -----------------------

   --  A static subtype is either a scalar base type, other than a generic
   --  formal type; or a scalar subtype formed by imposing on a static
   --  subtype either a static range constraint, or a floating or fixed
   --  point constraint whose range constraint, if any, is static. [LRM 4.9]

   function Is_Static_Subtype (Typ : Entity_Id) return Boolean is
      Base_T : constant Entity_Id := Base_Type (Typ);

   begin
      if Is_Generic_Type (Base_T) or else not Is_Scalar_Type (Base_T) then
         return False;

      elsif Base_T = Typ then
         return True;

      --  ??? For now always consider floating point types as static since
      --  we currently reject ranges that are put on floating points.

      elsif Is_Float_Type (Typ) then
         return True;
      else
         return Is_Static_Subtype (Base_T)
           and then Is_Static (Type_Low_Bound (Typ))
           and then Is_Static (Type_High_Bound (Typ));
      end if;
   end Is_Static_Subtype;

   -------------
   -- Is_True --
   -------------

   function Is_True (U : Uint) return Boolean is
   begin
      return not Is_False (U);
   end Is_True;

   ------------------
   -- Rewrite_Node --
   ------------------

   procedure Rewrite_Node (N : Node_Id; Val : Uint) is
      Int_Literal : Node_Id;
      Lit         : Entity_Id;
      Pos         : Int;

   begin
      if Is_Integer_Type (Etype (N)) then

         --  Substitute an N_Integer_Literal node for the result of the
         --  compile time evaluation of an operation on an integer literal or
         --  constant (i.e. results of arithmetic operators or attributes).

         Int_Literal := Make_Integer_Literal (Sloc (N), Intval => Val);
         Set_Etype (Int_Literal, Etype (N));
         Rewrite_Substitute_Tree (N, Int_Literal);

      elsif Is_Enumeration_Type (Etype (N)) then

         --  Substitute an N_Identifier or N_Character_Literal node for
         --  the result of the compile time evaluation of an operation on
         --  an enumeration literal or constant (i.e. results of logical
         --  operators or attributes).

         --  In the case where the literal is either of type Wide_Character
         --  or Character, there needs to be some special handling since
         --  there is no explicit chain of literals to search. Instead, an
         --  N_Character_Literal node is created with the appropriate
         --  Char_Code and Chars fields.

         if Base_Type (Etype (N)) = Standard_Character
           or else Base_Type (Etype (N)) = Standard_Wide_Character
         then
            Pos := UI_To_Int (Val);
            Name_Buffer (1) := ''';
            Name_Buffer (2) := Character'Val (Pos);
            Name_Buffer (3) := ''';
            Name_Len := 3;
            Lit := New_Node (N_Character_Literal, Sloc (N));
            Set_Chars (Lit, Name_Find);
            Set_Char_Literal_Value (Lit, Char_Code (Pos));
            Set_Etype (Lit, Etype (N));
            Rewrite_Substitute_Tree (N, Lit);

         else
            --  Iterate through the literals list until the one in the
            --  desired position in the chain is found. Note that since
            --  this offset is relative to the original enumeration type
            --  start at the first literal of the base type.

            Pos := UI_To_Int (Val) - 1;
            Lit := First_Literal (Base_Type (Etype (N)));

            for J in 0 .. Pos loop
               Lit := Next_Literal (Lit);
            end loop;

            Rewrite_Substitute_Tree (N, New_Occurrence_Of (Lit, Sloc (N)));
         end if;
      end if;

      Set_Is_Folded (N);
   end Rewrite_Node;

   procedure Rewrite_Node (N : Node_Id; Real_Lit_Node : Node_Id) is
   begin
      Set_Etype (Real_Lit_Node, Etype (N));
      Rewrite_Substitute_Tree (N, Real_Lit_Node);
      Set_Is_Folded (N);
   end Rewrite_Node;

   ----------
   -- Test --
   ----------

   function Test (Cond : Boolean) return Uint is
   begin
      if Cond then
         return Uint_1;
      else
         return Uint_0;
      end if;
   end Test;

   --------------
   -- To_Bits --
   --------------

   procedure To_Bits (U : Uint; B : out Bits) is
   begin
      for J in 0 .. B'Last loop
         B (J) := not UI_Is_Zero (UI_Mod (UI_Quotient (U,
           UI_Exponentiate (Uint_2, UI_From_Int (J))), Uint_2));
      end loop;
   end To_Bits;

end Sem_Eval;
