------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                               I T Y P E S                                --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.7 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains declarations for handling of implicit types

with Sem_Util;
with Types;    use Types;

package Itypes is

   --  Implicit types are types created by the semantic or the expander either
   --  by convenience or to stick with the Ada formal semantics. They are
   --  entities which don't appear as a Defining_Identifier of some declaration
   --  Node in order to save tree expension or when there is no Ada syntax for
   --  defining them (the class wide type associated to a tagged type for
   --  instance). They are attached to the Tree through the Node related
   --  to their creation.

   --  Here is the list of nodes that may contain implicit types:

   --    * N_Full_Type_Declaration:
   --        - Array types  : implicit base type, implicit index types
   --        - Tagged types : class wide type

   --    * N_Subtype_Declaration
   --        - Array types  : implicit index types

   --    * N_Formal_Object_Declaration:
   --    * N_Component_Declaration:
   --    * N_Object_Declaration:
   --        - discriminated or indexed objects: implicit constrained subtype
   --        - anonymous array objects: implicit type and indexes
   --        - dispatch table object: anonymous access to primitive types

   --    * N_Iteration_Scheme
   --        -  implicit range type

   --    * N_Function_Specification:
   --    * N_Procedure_Specification:
   --    * N_Entry_Body:
   --    * N_Accept_Statement:
   --        -  implicit formal access type

   --    * N_Entry_Declaration:
   --        - implicit formal access type
   --        - implicit index (for families)

   --    * N_Freeze_Entity:
   --        - implicit types attached to a N_Object_Declaration may be moved
   --          to their own N_Freeze node when this one is inserted before the
   --          object declaration

   --  Sunexpression nodes

   --    * N_Slice:
   --        - implicit index type
   --        - implicit slice Subtype

   --    * N_Aggregate:
   --        - implicit constrained subtypes

   --    * N_String_Literal:
   --        - implicit string subtype

   --    * N_Allocator:
   --        - implicit access type
   --        - explicit constrained designated subtype (this one is not an
   --          implicit type, i.e. the tree is generated for its declaration)
   --          this is a peculiarity.

   --    * N_Reference
   --        - implicit access type

   --    * N_Expression_Actions:
   --        - implicit types attached with the Expression Node are moved up
   --          to the expression action in order to ease Gigi processing.

   --  Note: the set of nodes listed above is covered by the subtype
   --  N_Has_Itypes. Such nodes have a First_Itype field that is either
   --  Empty if there are no associated Itypes, or points to the first
   --  entity on the list of Itypes if there are Itypes. In addition such
   --  nodes have a Has_Dynamic_Itype flag that is set if any of the
   --  entities in the list is for a dynamic implicit type (one which has
   --  variable bounds or other data that must be elaborated).

   --  Note: N_Has_Itypes includes N_Subtype_Indication, even though it is
   --  not listed above. This is for purely technical reasons, to accomodate
   --  the restriction that the node types for which Itypes are present must
   --  have contiguous codes so that N_Has_Itypes can be declared. The two
   --  fields are always unused, i.e. First_Itype is always Empty for a
   --  subtype indication node, and Has_Dynamic_Itype is always False.

   --------------------------------------------
   -- Procedures for Implicit Types Handling --
   --------------------------------------------

   --  In the new approach to Itype handling, a list of Itype entities may
   --  be associated with any node that is not an entity, the list uses
   --  Next_Itype as the link pointer (we had originally thought to use
   --  normal lists, since the Paren (Declaration_Of) field is not needed
   --  for implicit types, but that doesn't work because of exchanging of
   --  private type entities.

   procedure Transfer_Itypes (From : Node_Id; To : Node_Id);
   --  Move the implicit types from node From to node To.

   function Is_Itype (E : Entity_Id) return Boolean;
   --  Return True is the Entity is an Implicit type.

   procedure Attach_Itype_To (N : Node_Id; E : Entity_Id);
   --  Attach the Itype E at the end of the list of Itypes associated with
   --  node N, which must be one of the node types listed above.

   function New_Itype (
     In_Node      : Node_Id;
     Related_Id   : Entity_Id := Empty;
     Suffix       : Character := ' ';
     Suffix_Index : Nat       := 0;
     Scope_Id     : Entity_Id := Sem_Util.Current_Scope)
     return         Entity_Id;
   --  This is used in cases where there is a new underlying type or subtype
   --  introduced implicitly. Examples include:
   --
   --   a. the implicit unconstrained array type type introduced by a
   --      constrained array type declaration.
   --
   --   b. the implicit subtype introduced by a subtype indication given as
   --      "Integer range 1 .. 2".
   --
   --   For each implicit type or subtype that is created we also create
   --   an N_Implicit_Type node which is inserted in the declaration list
   --   to reflect the present of the created type. The only information
   --   it contains is a pointer to the created entity for the type.
   --   (which is in the Defining_Identifier field). This list is collected
   --   for all such implicit entities generated for a given declaration
   --   and is inserted before the declaration in Analyze_Declarations.
   --
   --   Related_Id is present only if the implicit type name may be referenced
   --   as a public symbol, and thus needs a unique external name. The name
   --   is created by a call to:
   --
   --     New_External_Name (Chars (Related_Id), Suffix, Suffix_Index, 'T')
   --
   --   If the implicit type does not need an external name, then the
   --   Related_Id parameter is omitted (and hence empty). In this case
   --   Suffix and Suffix_Index are ignored and the implicit type name is
   --   created by a call to New_Internal_Name ('T').
   --
   --   Note that in all cases, the name starts with "T". This is used
   --   to identify implicit types in the error message handling circuits.
   --
   --  The Scope_Id parameter specifies the scope of the created type, and
   --  is normally the Current_Scope as shown, but can be set otherwise.

end Itypes;
