-----------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 7                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.31 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains virtually all expansion mechanisms related to
--    - controlled types
--    - transient scopes

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Expand;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Util; use Exp_Util;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sinfo;    use Sinfo;
with Sem;      use Sem;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;

package body Exp_Ch7 is

   -------------------------
   --  Local Subprograms  --
   -------------------------

   function Find_Closest_Final_List (E : Entity_Id) return Node_Id;
   --  E is an entity representing a controlled object or type. This function
   --  returns a reference to the final list attached to the first
   --  "controllable" scope Entity (E_Block, E_Function, E_Procedure,
   --  E_Task_Type, E_Entry), creating this final list if necessary.

   function Find_Node_To_Be_Wrapped (N : Node_Id) return Node_Id;
   --  N is a node wich may generate a transient scope.
   --  Loop over the parent pointers of N until it find the appropriate node
   --  to wrap. It it returns Empty, it means that no transient scope is needed
   --  in this context.

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id;
   --  Find the first primitive operation of type T whose name is 'Name'.
   --  this function allows the use of a primitive operation which is not
   --  directly visible

   function Make_Clean
     (Clean : Entity_Id;
      Mark  : Entity_Id;
      Flist : Entity_Id)
      return  Node_Id;
   --  Expand a the clean-up procedure for controlled and/or transient block.
   --  Clean is the entity for such a procedure.  Mark is the entity for the
   --  secondary stack mark, if empty only controlled block clean-up will be
   --  performed. Flist is the entity for the local final list, if empty
   --  only transient scope clean-up will be performed.

   procedure Set_Scope_Is_Transient (V : Boolean := True);

   procedure Set_Node_To_Be_Wrapped (N : Node_Id);

   function      Scope_Uses_Sec_Stack return Boolean;
   procedure Set_Scope_Uses_Sec_Stack (V : Boolean := True);

   function Make_Transient_Block
     (Loc         : Source_Ptr;
      Flist       : Entity_Id;
      Instruction : Node_Id)
      return        Node_Id;
   --  Create a transient block whose name is Scope, which is also a controlled
   --  block if Flist is not empty and whose only instruction is Instruction.

   ---------------------------
   -- Expand_N_Package_Body --
   ---------------------------

   --  Add call to Activate_Tasks if body is an activator (actual
   --  processing is in chapter 9).

   procedure Expand_N_Package_Body (N : Node_Id) is
   begin
      Build_Task_Activation_Call (N);
   end Expand_N_Package_Body;

   ------------------
   -- Find_Prim_Op --
   ------------------

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id is
      Prim : Elmt_Id;
      Typ  : Entity_Id := T;

   begin
      if Is_Class_Wide_Type (Typ) then
         Typ := Etype (Typ);
      end if;

      Prim := First_Elmt (Primitive_Operations (Typ));
      while Chars (Node (Prim)) /= Name loop
         Prim := Next_Elmt (Prim);
      end loop;

      return Node (Prim);
   end Find_Prim_Op;

   -----------------------------------
   --  Controlled types Management  --
   -----------------------------------

   --  When a controlled object is declared in a local scope, this scope is
   --  expanded into a "controlled scope". That is to say a scope generating
   --  the finalization of all controlled objects declared within. Here is an
   --  example of the expansion of a controlled block :

   --    declare
   --       X : Controlled ;
   --       Y : Controlled := Init;
   --    begin
   --       X := Y;
   --    end;
   --
   --  is expanded into
   --
   --    declare
   --       Local_Final_List : System.FI.Finalizable_Ptr;

   --       procedure _Clean is
   --       begin
   --          Abort_Defer;
   --          System.FI.Finalize_List (Local_Final_List);
   --          Abort_Undefer
   --       end _Clean;

   --       X : Controlled;
   --       Initialize (X);
   --       System.FI.Attach_To_Final_List (Local_Final_List, Finalizable (X));
   --       Y : Controlled := Init;
   --       System.FI.Attach_To_Final_List (Local_Final_List, Finalizable (Y));

   --    begin
   --       Finalize (X);
   --       X := Y;
   --       Adjust (X);
   --    at end
   --       _Clean;
   --    end;

   ----------------------------
   -- Expand_Cleanup_Actions --
   ----------------------------

   procedure Expand_Cleanup_Actions (N : Node_Id) is
      S          : constant Entity_Id := Current_Scope;
      Decls      :  List_Id := Declarations (N);
      Loc        : constant Source_Ptr := Sloc (N);
      Clean      : Entity_Id;
      Flist      : constant Entity_Id := Finalization_Chain_Entity (S);
      Mark       : Entity_Id := Empty;
      Clean_Decl : Node_Id;
      Added_Decl : Node_Id := Empty;

   begin
      --  This procedure is directly called from the Semantics, so we don't
      --  want to execute it if not in expansion mode. Furthermore, if no data
      --  allocated in the secondary stack nor local finalizable objects are
      --  present, nothing needs to be done.

      if not (Expander_Active
               and then (Scope_Uses_Sec_Stack or else Present (Flist)))
      then
         return;
      end if;

      if Decls = No_List then
         Decls := New_List;
         Set_Declarations (N, Decls);
      end if;

      if Scope_Uses_Sec_Stack then

         --  Expand :
         --    _Mxx : constant Mark_Id := SS_Mark;

         Mark := Make_Defining_Identifier (Loc, New_Internal_Name ('M'));
         Added_Decl :=
           Make_Object_Declaration (Loc,
             Defining_Identifier => Mark,
             Object_Definition => New_Reference_To (RTE (RE_Mark_Id), Loc),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (RTE (RE_SS_Mark), Loc)));
         Prepend_To (Decls, Added_Decl);
         Analyze (First (Decls));
      end if;

      if Present (Flist) then

         --  Expand :
         --   Local_Final_List : System.FI.Finalizable_Ptr;

         Prepend_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Flist,
             Object_Definition =>
               New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));
         Analyze (First (Decls));

         if No (Added_Decl) then
            Added_Decl := First (Decls);
         end if;
      end if;

      --  Clean-up procedure definition

      Clean := Make_Defining_Identifier (Loc, Name_uClean);
      Insert_After (Added_Decl, Make_Clean (Clean, Mark, Flist));
      Analyze (Next (Added_Decl));
      Protect_Statements (N, Clean);
   end Expand_Cleanup_Actions;

   ------------------------------
   --  Find_Closest_Final_List --
   ------------------------------

   function Find_Closest_Final_List (E : Entity_Id) return Node_Id is
      S  : Entity_Id := E;
      Id : Entity_Id;

   begin
      while S /= Standard_Standard
        and then Ekind (S) /= E_Block
        and then Ekind (S) /= E_Function
        and then Ekind (S) /= E_Procedure
        and then Ekind (S) /= E_Task_Type
        and then Ekind (S) /= E_Entry
      loop
         S := Scope (S);
      end loop;

      if S = Standard_Standard then
         return New_Reference_To (RTE (RE_Global_Final_List), Sloc (E));
      else
         if No (Finalization_Chain_Entity (S)) then

            Id := Make_Defining_Identifier (Sloc (S), New_Internal_Name ('F'));
            Set_Finalization_Chain_Entity (S, Id);

            --  Set momentarily some semantics attributes to allow normal
            --  analysis of expansions containing references to this chain.
            --  Will be fully decorated during the expansion of the scope
            --  itself

            Set_Ekind (Id, E_Variable);
            Set_Etype (Id, RTE (RE_Finalizable_Ptr));
         end if;

         return New_Reference_To (Finalization_Chain_Entity (S), Sloc (E));
      end if;
   end Find_Closest_Final_List;

   ----------------------------
   -- Make_Init_Attach_Calls --
   ----------------------------

   --  An initialize call generated by the compiler is always followed by
   --  an attach to the final list, thus these two calls are groups in the same
   --  subprogram.

   function  Make_Init_Attach_Calls
     (Ref  : Node_Id;
      Scop : Entity_Id;
      Typ  : Entity_Id)
      return List_Id
   is
      Loc : constant Source_Ptr := Sloc (Ref);

   begin
      return New_List (
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (Find_Prim_Op (Typ, Name_Initialize), Loc),
          Parameter_Associations => New_List (Ref)),
        Make_Attach_Ctrl_Object (New_Copy_Tree (Ref), Scop));
   end  Make_Init_Attach_Calls;

   --------------------------------------
   -- Make_Init_Attach_Components_Call --
   --------------------------------------

   function  Make_Init_Attach_Components_Call
     (Ref  : Node_Id;
      Scop : Entity_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
   begin
      return Empty;
   end  Make_Init_Attach_Components_Call;

   -----------------------------
   -- Make_Attach_Ctrl_Object --
   -----------------------------

   function Make_Attach_Ctrl_Object
     (Ref  : Node_Id;
      Scop : Entity_Id)
      return Node_Id
   is
      Flist : constant Node_Id := Find_Closest_Final_List (Scop);
      Loc   : constant Source_Ptr := Sloc (Ref);

   begin

      --  System.FI.Attach_To_Final_List (Flist, Ref)

      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Attach_To_Final_List), Loc),
          Parameter_Associations => New_List (
            Flist,
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Finalizable), Loc),
              Expression => Ref)));
   end Make_Attach_Ctrl_Object;

   ----------------------
   -- Make_Adjust_Call --
   ----------------------

   --  Generates a call to the Adjust primitive

   function Make_Adjust_Call
     (Ref  : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (Ref);
      N   : Node_Id;

   begin
      --  Even a constant target has to be adjusted

      Set_Assignment_OK (Ref);

      N :=
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (Find_Prim_Op (Typ, Name_Adjust), Loc),
          Parameter_Associations => New_List (Ref));

      return N;
   end Make_Adjust_Call;

   ---------------------------------
   -- Make_Adjust_Components_Call --
   ---------------------------------

   function Make_Adjust_Components_Call
     (Ref  : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
   begin
      return Empty;
   end Make_Adjust_Components_Call;

   ------------------------
   -- Make_Finalize_Call --
   ------------------------

   function Make_Finalize_Call
     (Ref  : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
      Loc : constant Source_Ptr := Sloc (Ref);
      N   : Node_Id;

   begin
      N :=
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (Find_Prim_Op (Typ, Name_Finalize), Loc),
          Parameter_Associations => New_List (Ref));
      return N;
   end Make_Finalize_Call;

   -----------------------------------
   -- Make_Finalize_Components_Call --
   -----------------------------------

   function Make_Finalize_Components_Call
     (Ref  : Node_Id;
      Typ  : Entity_Id)
      return Node_Id
   is
   begin
      return Empty;
   end Make_Finalize_Components_Call;

   -----------------------------------
   --  Transient Scopes Management  --
   -----------------------------------

   --  A transient scope is created when temporary objects are created by the
   --  compiler. These temporary objects are allocated on the secondary stack
   --  and the transient scope is responsible for finalizing the object when
   --  appropriate and reclaiming the memory at the right time. The temporary
   --  objects are generally the objects allocated to store the result of a
   --  function returning an unconstrained or a tagged value.  Expressions
   --  needing to be wrapped in a transient scope (functions calls returning
   --  unconstrained or tagged values) may appear in 3 different contexts which
   --  lead to 3 different kinds of transient scope expansion:

   --   1. In a simple statement (procedure call, assignment, ...). In
   --      this case the instruction is wrapped into a transient block.
   --      (See Wrap_Transient_Statement for details)

   --   2. In an expression of a control structure (test in a IF statement,
   --      expression in a CASE statement, ...). In this case this expression
   --      is wrapped into an Expression_Action containing a transient block.
   --      (See Wrap_Transient_Expression for details)

   --   3. In a expression of an object_declaration. No wrapping is possible
   --      here, so the finalization actions, if any are done right after the
   --      declaration and the secondary stack deallocation is done in the
   --      proper enclosing scope (see Wrap_Transient_Declaration for details)

   --  Note about function returning tagged types: It has been decided to
   --  always allocate their result in the secondary stack while it is not
   --  absolutely mandatory when the tagged type is constrained because the
   --  caller knows the size of the returned object and thus could allocate the
   --  result in the primary stack. But, allocating them always in the
   --  secondary stack simplifies many implementation hassles:

   --    - If it is dispatching function call, the computation of the size of
   --      the result is possible but complex from the outside.

   --    - If the returned type is controlled, the assignment of the returned
   --      value to the anonymous object involves an Adjust, and we have no
   --      easy way to access the anonymous object created by the back-end

   --    - If the returned type is class-wide, this is an unconstrained type
   --      anyway

   --  Furthermore, the little loss in efficiency which is the result of this
   --  decision is not such a big deal because function returning tagged types
   --  are not very much used in real life as opposed to functions returning
   --  access to a tagged type

   -------------------------------
   -- Establish_Transient_Scope --
   -------------------------------

   --  This procedure is called each time a transient block has to be inserted
   --  that is to say for each call to a function with unconstrained ot tagged
   --  result. It creates a new scope on the stack scope in order to enclose
   --  all transient variables generated

   procedure Establish_Transient_Scope (N : Node_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Wrap_Node : Node_Id;

   begin
      --  Only create a new transient scope if the current one is not

      if not Scope_Is_Transient then
         Wrap_Node := Find_Node_To_Be_Wrapped (N);

         --  Case of no wrap node, false alert, no transient scope needed

         if No (Wrap_Node) then
            null;

         --  Transient scope is required

         else
            New_Scope (New_Internal_Entity (E_Block, Current_Scope, Loc, 'B'));
            Set_Scope_Is_Transient;
            Set_Scope_Uses_Sec_Stack;
            Set_Node_To_Be_Wrapped (Wrap_Node);

            if Debug_Flag_W then
               Write_Str ("    <Transient>");
               Write_Eol;
            end if;
         end if;
      end if;
   end Establish_Transient_Scope;

   ------------------------
   -- Scope_Is_Transient --
   ------------------------

   function Scope_Is_Transient  return Boolean is
   begin
      return Scope_Stack.Table (Scope_Stack.Last).Is_Transient;
   end Scope_Is_Transient;

   ----------------------------
   -- Set_Scope_Is_Transient --
   ----------------------------

   procedure Set_Scope_Is_Transient (V : Boolean := True) is
   begin
      Scope_Stack.Table (Scope_Stack.Last).Is_Transient := V;
   end Set_Scope_Is_Transient;

   ------------------------
   -- Node_To_Be_Wrapped --
   ------------------------

   function Node_To_Be_Wrapped return Node_Id is
   begin
      return Scope_Stack.Table (Scope_Stack.Last).Node_To_Be_Wrapped;
   end Node_To_Be_Wrapped;

   ----------------------------
   -- Set_Node_To_Be_Wrapped --
   ----------------------------

   procedure Set_Node_To_Be_Wrapped (N : Node_Id) is
   begin
      Scope_Stack.Table (Scope_Stack.Last).Node_To_Be_Wrapped := N;
   end Set_Node_To_Be_Wrapped;

   --------------------------
   -- Scope_Uses_Sec_Stack --
   --------------------------

   function Scope_Uses_Sec_Stack return Boolean is
   begin
      return Scope_Stack.Table (Scope_Stack.Last).Uses_Sec_Stack;
   end Scope_Uses_Sec_Stack;

   ------------------------------
   -- Set_Scope_Uses_Sec_Stack --
   ------------------------------

   procedure Set_Scope_Uses_Sec_Stack (V : Boolean := True) is
   begin
      Scope_Stack.Table (Scope_Stack.Last).Uses_Sec_Stack := V;
   end Set_Scope_Uses_Sec_Stack;

   --------------------------------
   --  Expand_Ctrl_Function_Call --
   --------------------------------

   --  Transform F (x) into

   --    [_V : Finalizable_Ptr;
   --     _V := Finalizable_Ptr (F (x)'Ref);
   --     Attach_To_Final_List ("Final_List_Of_Current_Scope", _V.all);

   --   Type_Of_F!(_V.all)]

   procedure Expand_Ctrl_Function_Call (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      V     : constant Entity_Id :=
                Make_Defining_Identifier (Loc, New_Internal_Name ('V'));
      Flist : constant Node_Id := Find_Closest_Final_List (Current_Scope);
      Act   : List_Id;
      Expr  : Node_Id;
      Rtype : Entity_Id;

   begin
      Act := New_List (
        Make_Object_Declaration (Loc,
          Defining_Identifier => V,
          Object_Definition =>
            New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)),

        Make_Assignment_Statement (Loc,
          Name => New_Reference_To (V, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (RTE (RE_Finalizable_Ptr), Loc),
              Expression => Make_Reference (Loc, New_Copy (N)))),

        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE_Attach_To_Final_List), Loc),
          Parameter_Associations => New_List (
            Flist,
            Make_Explicit_Dereference (Loc, New_Reference_To (V, Loc)))));

      Expr :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Rtype, Loc),
          Expression   =>
            Make_Explicit_Dereference (Loc, New_Reference_To (V, Loc)));

      Replace_Substitute_Tree (N,
        Make_Expression_Actions (Loc,
          Actions    => Act,
          Expression => Expr));

      Analyze (N);
      Resolve (N, Rtype);
   end Expand_Ctrl_Function_Call;

   -----------------------------
   -- Find_Node_To_Be_Wrapped --
   -----------------------------

   function Find_Node_To_Be_Wrapped (N : Node_Id) return Node_Id is
      P          : Node_Id;
      The_Parent : Node_Id;

   begin
      The_Parent := N;
      loop
         P := The_Parent;
         pragma Assert (P /= Empty);
         The_Parent := Parent (P);

         case Nkind (The_Parent) is

            --  Simple statements are ideal nodes to be wrapped

            when N_Assignment_Statement     |
                 N_Procedure_Call_Statement |
                 N_Entry_Call_Statement     =>
               return The_Parent;

            --  Object declarations are also a boundary for the transient scope
            --  even if they are not really wrapped
            --  (see Wrap_Transient_Declaration)

            when N_Object_Declaration =>
               return The_Parent;

            --  The expression itself is to be wrapped if its parent is a
            --  compound statement or any other statement where the expression
            --  is known to be scalar

            when N_Accept_Alternative               |
                 N_Attribute_Definition_Clause      |
                 N_Case_Statement                   |
                 N_Code_Statement                   |
                 N_Delay_Alternative                |
                 N_Delay_Until_Statement            |
                 N_Delay_Relative_Statement         |
                 N_Discriminant_Association         |
                 N_Elsif_Part                       |
                 N_Entry_Body_Formal_Part           |
                 N_Exit_Statement                   |
                 N_If_Statement                     |
                 N_Index_Or_Discriminant_Constraint |
                 N_Iteration_Scheme                 |
                 N_Subtype_Indication               |
                 N_Terminate_Alternative            =>
               return P;

            --  The N_range case is peculiar because this is the only case
            --  where the wrapping don't take place at the Statement boundary.
            --  Anyway it has to take place at this level for "for loops" and
            --  will generate wrapping in some type and subtype declarations,
            --  and slice expressions which is correct.

            when N_Range                       =>
               return P;

            --  The following nodes contains "dummy calls" which don't
            --  need to be wrapped.

            when N_Parameter_Specification     |
                 N_Discriminant_Specification  |
                 N_Component_Declaration       =>
               return Empty;

            --  The expression of a return statement is not to be wrapped if it
            --  is in a function returning an unconstrained or tagged type
            --  because in this case the wrapping take place around the call

            when N_Return_Statement            =>
               if Is_Unconstrained (P)
                 or else Is_Tagged_Type (Etype (P))
               then
                  return Empty;
               else
                  return P;
               end if;

            --  If we leave a scope without having been able to find a node to
            --  wrap, something is going wrong

            when N_Subprogram_Body     |
                 N_Package_Declaration |
                 N_Package_Body        |
                 N_Block_Statement     =>
               pragma Assert (False); null;

            --  otherwise continue the search

            when others =>
               null;
         end case;
      end loop;
   end Find_Node_To_Be_Wrapped;

   --------------------------------
   -- Wrap_Transient_Declaration --
   --------------------------------

   --  If a transient scope has been established during the processing of the
   --  Expression of an Object_Declaration, it is not possible to wrap the
   --  declaration into a transient block as usual case, otherwise the object
   --  would be itself declared in the wrong scope. Therefore, all entities (if
   --  any) defined in the transient block are moved to the proper enclosing
   --  scope, furthermore, if they are controlled they are finalized right
   --  after the declaration. The finalization list of the transient scope is
   --  defined as a renaming of the enclosing one so during their
   --  initialization they will be attached to the proper finalization
   --  list. For instance, the following declaration :

   --        X : Typ := F (G (A), G (B));

   --  (where G(A) and G(B) return controlled values, expanded as _v1 and _v2)
   --  is expanded into :

   --    _local_final_list_1 : Finalizable_Ptr renames _Local_Final_List;
   --    X : Typ := [ complex Expression-Action ];
   --    Finalize_One (_v1);
   --    Finalize_One (_v2);

   procedure Wrap_Transient_Declaration (N : Node_Id) is
      S           : Entity_Id;
      Ent         : Entity_Id;
      Node        : Node_Id;
      Loc         : constant Source_Ptr := Sloc (N);
      Enclosing_S : Entity_Id;
      I           : Int;

   begin
      S := Current_Scope;

      --  Expand the node before leaving the transient scope

      Set_Scope_Is_Transient (False);
      Expand (N);

      Pop_Scope;

      --  Find the proper enclosing scope

      I := Scope_Stack.Last;
      loop
         Enclosing_S := Scope_Stack.Table (I).Entity;

         if Enclosing_S = Standard_Standard then

            --  Transient data declared in an outer scope will stay for
            --  ever on the secondary stack

            exit;

         elsif Ekind (Enclosing_S) = E_Block
           or else Ekind (Enclosing_S) = E_Function
           or else Ekind (Enclosing_S) = E_Procedure
           or else Ekind (Enclosing_S) = E_Task_Type
           or else Ekind (Enclosing_S) = E_Entry
         then

            --  the enclosing scope marked as containing transient data

            Scope_Stack.Table (I).Uses_Sec_Stack := True;
            exit;

         else
            I := I - 1;
         end if;
      end loop;

      --  Renaming declaration to point to the righ finalization chain

      if Present (Finalization_Chain_Entity (S)) then
         Node :=
           Make_Object_Renaming_Declaration (Loc,
             Defining_Identifier => Finalization_Chain_Entity (S),
             Subtype_Mark => New_Reference_To (RTE (RE_Finalizable_Ptr), Loc),
             Name => New_Reference_To (Find_Closest_Final_List (S), Loc));

         Insert_Before (N, Node);
         Analyze (Node);
      end if;

      --  Put the local entities back in the enclosing scope

      Ent := First_Entity (S);
      while Present (Ent) loop

         Set_Scope (Ent, Current_Scope);

         if Has_Controlled (Etype (Ent)) then

            --  not done yet ???
            null;
         end if;

         if Is_Controlled (Etype (Ent)) then

            --  Generate the Finalization calls

            Node :=
              Make_Procedure_Call_Statement (Loc,
                Name => New_Reference_To (RTE (RE_Finalize_One), Loc),
                Parameter_Associations => New_List (
                  New_Reference_To (
                    Finalization_Chain_Entity (Current_Scope), Loc),

                  Make_Unchecked_Type_Conversion (Loc,
                    Subtype_Mark =>
                      New_Reference_To (RTE (RE_Finalizable), Loc),
                    Expression => New_Reference_To (Ent, Loc))));

            Insert_After (N, Node);
            Analyze (Node);
         end if;

         Ent := Next_Entity (Ent);
      end loop;
   end Wrap_Transient_Declaration;

   -------------------------------
   -- Wrap_Transient_Expression --
   -------------------------------

   --  Transform <Expression> into

   --  (lines marked with <CTRL> are expanded only in presence of Controlled
   --   objects needing finalization)

   --    [_E : Etyp;
   --     declare
   --        _M : constant Mark_Id := SS_Mark;
   --        Local_Final_List : System.FI.Finalizable_Ptr;    <CTRL>

   --        procedure _Clean is
   --        begin
   --           Abort_Defer;
   --           System.FI.Finalize_List (Local_Final_List);   <CTRL>
   --           SS_Release (M);
   --           Abort_Undefer;
   --        end _Clean;

   --     begin
   --        _E := <Expression>;
   --     at end
   --        _Clean;
   --     end;

   --    _E]

   procedure Wrap_Transient_Expression (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      E       : constant Entity_Id :=
                  Make_Defining_Identifier (Loc, New_Internal_Name ('E'));
      Etyp    : constant Entity_Id := Etype (N);
      New_Exp : constant Node_Id := Relocate_Node (N);

   begin
      Replace_Substitute_Tree (N,
        Make_Expression_Actions (Loc,
          Actions => New_List (

            Make_Object_Declaration (Loc,
              Defining_Identifier => E,
              Object_Definition   => New_Reference_To (Etyp, Loc)),

            Make_Transient_Block (Loc,
              Flist       => Finalization_Chain_Entity (Current_Scope),
              Instruction =>
                Make_Assignment_Statement (Loc,
                  Name       => New_Reference_To (E, Loc),
                  Expression => New_Exp))),

          Expression =>  New_Reference_To (E, Loc)));

      --  Expand the node before leaving the transient scope

      Set_Scope_Is_Transient (False);
      Expand (New_Exp);

      Pop_Scope;
      Analyze (N);
      Resolve (N, Etyp);
   end Wrap_Transient_Expression;

   ------------------------------
   -- Wrap_Transient_Statement --
   ------------------------------

   --  Transform <Instruction> into

   --  (lines marked with <CTRL> are expanded only in presence of Controlled
   --   objects needing finalization)

   --    declare
   --       _M : Mark_Id := SS_Mark;
   --       Local_Final_List : System.FI.Finalizable_Ptr ;    <CTRL>

   --       procedure _Clean is
   --       begin
   --          Abort_Defer;
   --          System.FI.Finalize_List (Local_Final_List);    <CTRL>
   --          SS_Release (_M);
   --          Abort_Undefer;
   --       end _Clean;

   --    begin
   --       <Instruction>;
   --    at end
   --       _Clean;
   --    end;

   procedure Wrap_Transient_Statement (N : Node_Id) is
      Loc           : constant Source_Ptr := Sloc (N);
      Block         : Node_Id;
      New_Statement : constant Node_Id := Relocate_Node (N);

   begin
      Block :=
        Make_Transient_Block (Loc,
          Flist       => Finalization_Chain_Entity (Current_Scope),
          Instruction => New_Statement);

      Replace_Substitute_Tree (N, Block);

      --  Expand the node before leaving the transient scope

      Set_Scope_Is_Transient (False);
      Expand (New_Statement);

      --  When the transient scope was established, we pushed the entry for
      --  the transient scope onto the scope stack, so that the scope was
      --  active for the installation of finalizable entities etc. Now we
      --  must remove this entry, since we have constructed a proper block.

      Pop_Scope;

      --  With the scope stack back to normal, we can call analyze on the
      --  resulting block. At this point, the transient scope is being
      --  treated like a perfectly normal scope, so there is nothing
      --  special about it.

      --  Note: Wrap_Transient_Statement is called with the node already
      --  analyzed (i.e. Analyzed (N) is True). This is important, since
      --  otherwise we would get a recursive processing of the node when
      --  we do this Analyze call.

      Analyze (N);
   end Wrap_Transient_Statement;

   --------------------------
   -- Make_Transient_Block --
   --------------------------

   --  Lines marked <CTRL> are expanded only in presence of Controlled
   --  objects needing finalization

   --  Lines marked <SEC_STACK> are expanded only when something is allocated
   --  in the secondary stack

   --   declare
   --      _M : Mark_Id := SS_Mark;                          <SEC_STACK>
   --      Local_Final_List : System.FI.Finalizable_Ptr ;    <CTRL>

   --      procedure _Clean is
   --      begin
   --         Abort_Defer;
   --         System.FI.Finalize_List (Local_Final_List);    <CTRL>
   --         SS_Release (_M);                               <SEC_STACK>
   --         Abort_Undefer;
   --      end _Clean;

   --   begin
   --      <Instruction>;
   --   at end
   --      _Clean;
   --   end;

   function Make_Transient_Block
     (Loc : Source_Ptr;
      Flist       : Entity_Id;
      Instruction : Node_Id)
      return        Node_Id
   is
      Decls : constant List_Id := New_List;
      Clean : constant Entity_Id
                := Make_Defining_Identifier (Loc, Name_uClean);
      Mark  : Entity_Id := Empty;

   begin
      if Scope_Uses_Sec_Stack then
         Mark := Make_Defining_Identifier (Loc, Name_uM);
         Append_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Mark,
             Object_Definition => New_Reference_To (RTE (RE_Mark_Id), Loc),
             Expression =>
                Make_Function_Call (Loc,
                  Name => New_Reference_To (RTE (RE_SS_Mark), Loc))));
      end if;

      if Present (Flist) then
         Append_To (Decls,
           Make_Object_Declaration (Loc,
             Defining_Identifier => Flist,
             Object_Definition =>
               New_Reference_To (RTE (RE_Finalizable_Ptr), Loc)));
      end if;

      Append_To (Decls, Make_Clean (Clean, Mark, Flist));

      return
        Make_Block_Statement (Loc,
          Identifier => New_Reference_To (Current_Scope, Loc),
          Declarations => Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (Instruction),
              Identifier => New_Occurrence_Of (Clean, Loc)),
          Has_Created_Identifier => True);
   end Make_Transient_Block;

   -----------------
   --  Make_Clean --
   -----------------

   function Make_Clean
     (Clean : Entity_Id;
      Mark  : Entity_Id;
      Flist : Entity_Id)
      return  Node_Id
   is
      Loc   : constant Source_Ptr := Sloc (Clean);
      Stmt  : List_Id := New_List;
      Sbody  : Node_Id;

   begin
      if Present (Flist) then
         Append_To (Stmt,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_Finalize_List), Loc),
             Parameter_Associations => New_List (
                    New_Reference_To (Flist, Loc))));
      end if;

      if Present (Mark) then
         Append_To (Stmt,
           Make_Procedure_Call_Statement (Loc,
             Name => New_Reference_To (RTE (RE_SS_Release), Loc),
             Parameter_Associations => New_List (
                    New_Reference_To (Mark, Loc))));
      end if;

      Sbody :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Procedure_Specification (Loc,
              Defining_Unit_Name => Clean),

          Declarations  => New_List,

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => Stmt));

      Wrap_Cleanup_Procedure (Sbody);
      return Sbody;
   end Make_Clean;
end Exp_Ch7;
