/*
 * Copyright 2010 Jerome Glisse <glisse@freedesktop.org>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * on the rights to use, copy, modify, merge, publish, distribute, sub
 * license, and/or sell copies of the Software, and to permit persons to whom
 * the Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR(S) AND/OR THEIR SUPPLIERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 * USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Authors:
 *      Jerome Glisse
 */
#define _FILE_OFFSET_BITS 64
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>
#include <errno.h>
#include "radeon_priv.h"
#include "xf86drm.h"
#include "radeon_drm.h"

struct radeon_bo *radeon_bo(struct radeon *radeon, unsigned handle,
			unsigned size, unsigned alignment, void *ptr)
{
	struct radeon_bo *bo;
	int r;

	bo = calloc(1, sizeof(*bo));
	if (bo == NULL) {
		return NULL;
	}
	bo->size = size;
	bo->handle = handle;
	bo->refcount = 1;
	bo->alignment = alignment;

	if (handle) {
		struct drm_gem_open open_arg;

		memset(&open_arg, 0, sizeof(open_arg));
		open_arg.name = handle;
		r = drmIoctl(radeon->fd, DRM_IOCTL_GEM_OPEN, &open_arg);
		if (r != 0) {
			free(bo);
			return NULL;
		}
		bo->handle = open_arg.handle;
		bo->size = open_arg.size;
	} else {
		struct drm_radeon_gem_create args;

		args.size = size;
		args.alignment = alignment;
		args.initial_domain = RADEON_GEM_DOMAIN_CPU;
		args.flags = 0;
		args.handle = 0;
		r = drmCommandWriteRead(radeon->fd, DRM_RADEON_GEM_CREATE,
					&args, sizeof(args));
		bo->handle = args.handle;
		if (r) {
			fprintf(stderr, "Failed to allocate :\n");
			fprintf(stderr, "   size      : %d bytes\n", size);
			fprintf(stderr, "   alignment : %d bytes\n", alignment);
			free(bo);
			return NULL;
		}
	}
	if (ptr) {
		if (radeon_bo_map(radeon, bo)) {
			fprintf(stderr, "%s failed to copy data into bo\n", __func__);
			return radeon_bo_decref(radeon, bo);
		}
		memcpy(bo->data, ptr, size);
		radeon_bo_unmap(radeon, bo);
	}
	return bo;
}

int radeon_bo_map(struct radeon *radeon, struct radeon_bo *bo)
{
	struct drm_radeon_gem_mmap args;
	void *ptr;
	int r;

	if (bo->map_count != 0) {
		goto success;
	}
	/* Zero out args to make valgrind happy */
	memset(&args, 0, sizeof(args));
	args.handle = bo->handle;
	args.offset = 0;
	args.size = (uint64_t)bo->size;
	r = drmCommandWriteRead(radeon->fd, DRM_RADEON_GEM_MMAP,
				&args, sizeof(args));
	if (r) {
		fprintf(stderr, "error mapping %p 0x%08X (error = %d)\n",
			bo, bo->handle, r);
		return r;
	}
	ptr = mmap(0, args.size, PROT_READ|PROT_WRITE, MAP_SHARED, radeon->fd, args.addr_ptr);
	if (ptr == MAP_FAILED) {
		fprintf(stderr, "%s failed to map bo\n", __func__);
		return -errno;
	}
	bo->data = ptr;

success:
	bo->map_count++;

	return 0;
}

void radeon_bo_unmap(struct radeon *radeon, struct radeon_bo *bo)
{
	if (--bo->map_count > 0) {
		return;
	}
	munmap(bo->data, bo->size);
	bo->data = NULL;
}

struct radeon_bo *radeon_bo_incref(struct radeon *radeon, struct radeon_bo *bo)
{
	bo->refcount++;
	return bo;
}

struct radeon_bo *radeon_bo_decref(struct radeon *radeon, struct radeon_bo *bo)
{
	struct drm_gem_close args;

	if (bo == NULL)
		return NULL;
	if (--bo->refcount > 0) {
		return NULL;
	}

	if (bo->map_count) {
		munmap(bo->data, bo->size);
	}
	memset(&args, 0, sizeof(args));
	args.handle = bo->handle;
	drmIoctl(radeon->fd, DRM_IOCTL_GEM_CLOSE, &args);
	memset(bo, 0, sizeof(struct radeon_bo));
	free(bo);
	return NULL;
}

int radeon_bo_wait(struct radeon *radeon, struct radeon_bo *bo)
{
	struct drm_radeon_gem_wait_idle args;
	int ret;

	/* Zero out args to make valgrind happy */
	memset(&args, 0, sizeof(args));
	args.handle = bo->handle;
	do {
		ret = drmCommandWriteRead(radeon->fd, DRM_RADEON_GEM_WAIT_IDLE,
					&args, sizeof(args));
	} while (ret == -EBUSY);
	return ret;
}
