/*
 * Copyright (C) 2001-2003 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: wizards.c,v 1.21 2003/04/08 22:30:28 guenter Exp $
 *
 * stuff to run on first startup / setup wizards
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <X11/Xlib.h>

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <stdio.h>
#include <string.h>

#include "globals.h"
#include "gtkxine.h"
#include "utils.h"
#include "desktop_integration.h"

#include "ok.xpm"
#include "fail.xpm"

typedef struct {
  char *msg;
  char *explanation;
} hc_msg_t;

static void details_cb (GtkWidget* widget, gpointer data) {
  
  hc_msg_t *msg = data;

  display_info ("Health Check Results", msg->msg);

  display_info ("Health Check Explanation", msg->explanation);
}


static GtkWidget *do_health_check (GtkWidget *dlg, GtkWidget *headline) {

  GtkWidget           *vbox, *l, *table, *b;
  xine_health_check_t  hc;
  xine_health_check_t *results;
  int                  check;
  GdkPixmap           *ok_pix, *fail_pix;
  hc_msg_t            *msg;

  ok_pix   = gdk_pixmap_create_from_xpm_d (dlg->window, 
					   NULL, NULL, ok_xpm);
  fail_pix = gdk_pixmap_create_from_xpm_d (dlg->window, 
					   NULL, NULL, fail_xpm);
  

  vbox = gtk_vbox_new (0, 0);

  table = gtk_table_new (1, 1, FALSE);

  l = gtk_label_new ("The xine engine runs certain checks on your\n"
		     "system configuration, results follow:");

  gtk_label_set_justify (GTK_LABEL (l), GTK_JUSTIFY_LEFT);

  gtk_misc_set_alignment (GTK_MISC (l), 0, 0.5);

  gtk_box_pack_start (GTK_BOX(vbox), l, FALSE, FALSE, 5);

  /* If input.vcd_device is empty, we will set it to the default "/dev/cdrom" */
  hc.cdrom_dev = gtk_xine_config_register_string(GTK_XINE(gtx),
						 "input.vcd_device",
						 "/dev/cdrom",
						 "device used for cdrom drive",
						 NULL, 0, NULL, NULL);
  
  /* If input.dvd_device is empty, we will set it to the default "/dev/dvd" */
  hc.dvd_dev = gtk_xine_config_register_string(GTK_XINE(gtx),
					       "input.dvd_device",
					       "/dev/dvd",
					       "device used for dvd drive",
					       NULL, 0, NULL, NULL);
  
  /* Run tests */
  check = 0;
  while (1) {
    results = xine_health_check (&hc, check);

    if (results->status == XINE_HEALTH_CHECK_NO_SUCH_CHECK)
      break;

    gtk_table_resize ( GTK_TABLE (table), 3, check+1);

    l = gtk_label_new (results->title);
    gtk_table_attach (GTK_TABLE (table), l, 0, 1, check, check+1, GTK_FILL,
		      GTK_SHRINK, 2, 5);
    gtk_misc_set_alignment (GTK_MISC (l), 0, 0.5);

    switch (results->status) {
    case XINE_HEALTH_CHECK_UNSUPPORTED:
    case XINE_HEALTH_CHECK_OK:
      l = gtk_pixmap_new (ok_pix, NULL);
      break;
    case XINE_HEALTH_CHECK_FAIL:
      msg = malloc (sizeof (hc_msg_t));

      msg->msg = results->msg;
      msg->explanation = results->explanation;

      l = gtk_pixmap_new (fail_pix, NULL);

      b = gtk_button_new_with_label ("Details");
      gtk_signal_connect (GTK_OBJECT (b), "clicked",
			  GTK_SIGNAL_FUNC (details_cb), msg);
      gtk_table_attach (GTK_TABLE (table), b, 2, 3, check, check+1, GTK_FILL,
			GTK_SHRINK, 2, 5);
      break;
    }

    gtk_table_attach (GTK_TABLE (table), l, 1, 2, check, check+1, GTK_FILL,
		      GTK_SHRINK, 2, 5);

    check++;
  } ;

  gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, FALSE, 5);

  gtk_widget_show_all (vbox);

  return vbox;
}

static GtkWidget *cb_moz;
static GtkWidget *cb_gnome;
static GtkWidget *cb_kde;
static GtkWidget *cb_mailcap;

static GtkWidget *ask_integration_wizard (GtkWidget *w, GtkWidget *headline) {

  GtkWidget           *vbox, *l;

  vbox = gtk_vbox_new (0, 0);

  l = gtk_label_new ("Register gxine with the following applications\n"
		     "as a media handler/helper:");

  gtk_label_set_justify (GTK_LABEL (l), GTK_JUSTIFY_LEFT);
  gtk_misc_set_alignment (GTK_MISC (l), 0, 0.5);

  gtk_box_pack_start (GTK_BOX(vbox), l, FALSE, FALSE, 5);


  cb_mailcap = gtk_check_button_new_with_label ("~/.mailcap");
  gtk_box_pack_start (GTK_BOX(vbox), cb_mailcap, FALSE, FALSE, 5);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb_mailcap), TRUE);

  cb_moz = gtk_check_button_new_with_label ("Mozilla (Plugin)");
  gtk_box_pack_start (GTK_BOX(vbox), cb_moz, FALSE, FALSE, 5);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb_moz), TRUE);

  cb_gnome = gtk_check_button_new_with_label ("Gnome / Nautilus");
  gtk_box_pack_start (GTK_BOX(vbox), cb_gnome, FALSE, FALSE, 5);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb_gnome), TRUE);

  cb_kde = gtk_check_button_new_with_label ("KDE / Konqueror");
  gtk_box_pack_start (GTK_BOX(vbox), cb_kde, FALSE, FALSE, 5);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb_kde), TRUE);

  gtk_widget_show_all (vbox);

  return vbox;
}

static void di_register_mozilla (void) {

  gchar       *fname, *sname;
  char         buf[4096];
  int          fds, fdd, n;
  
  printf ("wizards: installing browser plugin...\n");
  
  /* just make sure the directory exists */
  
  fname = g_strconcat (g_get_home_dir(), "/.mozilla", NULL);
  mkdir (fname, 0755);
  g_free (fname);
  
  fname = g_strconcat(g_get_home_dir(), "/.mozilla/plugins", NULL);
  mkdir (fname, 0755);
  g_free (fname);
  
  fname = g_strconcat (g_get_home_dir(), "/.mozilla/plugins/gxineplugin.so", NULL);
  sname = g_strconcat (GXINE_PLUGINDIR, "/gxineplugin.so", NULL);
  
  /* copy */
  fds = open (sname, O_RDONLY);
  if (fds<0) {
    printf ("wizards: error, cannot open %s\n", fname);
    g_free (fname); g_free (sname);
    return;
  }
  
  fdd = open (fname, O_WRONLY | O_CREAT | O_TRUNC, 0644);
  if (fdd<0) {
    printf ("wizards: error, cannot open %s\n", fname);
    g_free (fname); g_free (sname);
    close (fds);
    return;
  }

  do {
    
    n = read  (fds, buf, 4096);
    
    /* printf ("wizards: read %d bytes\n", n); */
    
    if (n>0)
      write (fdd, buf, n);
    
  } while (n>0);
  
  close (fdd);
  close (fds);
  
  g_free (fname); g_free (sname);
}

static void desktop_integration (void) {

  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb_mailcap))) {
    di_register_mailcap();
  }
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb_gnome))) {
    di_register_gnome();
  }
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb_kde))) {
    di_register_kde();
  }
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb_moz))) {
    di_register_mozilla();
  }
}

#define RESPONSE_CLOSE 0
#define RESPONSE_NEXT  1

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  int *b = data;
  *b = RESPONSE_CLOSE;
  gtk_main_quit();

  return TRUE;
}

static void next_cb (GtkWidget* widget, gpointer data) {
  int *b = data;
  *b = RESPONSE_NEXT;
  gtk_main_quit();
}

#define WIZARDS_LEVEL 6

void run_wizards (gboolean requested) {

  int        b;
  GtkWidget *dlg, *button, *headline, *sep, *content, *w, *hbox;
  int        state;

  if (requested 
      || (gtk_xine_config_register_num (GTK_XINE(gtx), "misc.wizards_shown", 0,
					"Keep track of wheter user has seen wizards yet",
					NULL, 20, NULL, NULL) < WIZARDS_LEVEL)) {

    xine_cfg_entry_t  entry;

    if (gtk_xine_config_lookup_entry (GTK_XINE(gtx), "misc.wizards_shown", &entry)) {
      entry.num_value = WIZARDS_LEVEL;
      gtk_xine_config_update_entry (GTK_XINE(gtx), &entry);
    }

    /* set up dialog which all wizards will use */

    dlg = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (dlg), "gxine setup wizards");
    gtk_window_set_default_size (GTK_WINDOW (dlg), 450, 400);
    gtk_signal_connect( GTK_OBJECT (dlg), "delete_event",
			GTK_SIGNAL_FUNC (close_cb), &b );

    /*
     * set up buttons
     */
    
    button = gtk_button_new_from_stock (GTK_STOCK_CLOSE);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC (close_cb),
			&b);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area),
			button, TRUE, TRUE, 0);  

    button = gtk_button_new_from_stock (GTK_STOCK_GO_FORWARD);
    gtk_signal_connect (GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC (next_cb),
			&b);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->action_area),
			button, TRUE, TRUE, 0);  
    
    
    /*
     * contents: headline, separator, wizard-specific part
     */
    
    hbox = gtk_hbox_new (0,0);

    /* headline */
    {
      gchar *pathname;
      pathname = g_strconcat (GXINE_PIXMAPDIR, "/wizards.png", NULL);
      headline = gtk_image_new_from_file (pathname);
      g_free (pathname);
    }
    gtk_box_pack_start (GTK_BOX (hbox), headline, FALSE, FALSE, 2);  

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox),
			hbox, FALSE, FALSE, 2);  
    
    sep = gtk_hseparator_new (); 
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox),
			sep, FALSE, FALSE, 5);  
    
    content = gtk_vbox_new (0, 0);
    
    /*
     * ask user if he/she wants to proceed
     */

    
    w = gtk_label_new ("Welcome to gxine!\n\n"
		       "Would you like to run some setup wizards now\n"
		       "that will check your installation and maybe\n"
		       "do some adjustments for you, if neccessary?\n\n"
		       "If you do not want to run the wizards right now,\n"
		       "just click on the 'Close' button and you will not be\n"
		       "bothered again.\n\n"
		       "You can always run the wizards (again) from the help menu.");
    gtk_label_set_line_wrap (GTK_LABEL (w), TRUE);
    gtk_box_pack_start (GTK_BOX(content), w,  TRUE, TRUE, 0);  
  
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox),
		      content, TRUE, TRUE, 0);  

    gtk_widget_set_usize (w, 500, 300);

    gtk_widget_show_all (dlg);

    gtk_main ();

    if (b==RESPONSE_CLOSE) {
      gtk_widget_hide (dlg);
      return;
    }


    state = 0;
    while (TRUE) {

      gtk_container_remove (GTK_CONTAINER (content), w);

      switch (state) {
	
      case 0:
	w = do_health_check (dlg, headline);

	gtk_box_pack_start (GTK_BOX(content), w, TRUE, TRUE, 0);  
	gtk_widget_set_usize (w, 500, 300);

	gtk_main ();

	if (b==RESPONSE_CLOSE) {
	  gtk_widget_hide (dlg);
	  return;
	}

	state ++;
	break;

      case 1:
	w = ask_integration_wizard (dlg, headline);

	gtk_box_pack_start (GTK_BOX(content), w, TRUE, TRUE, 0);  
	gtk_widget_set_usize (w, 500, 300);

	gtk_main ();

	if (b==RESPONSE_CLOSE) {
	  gtk_widget_hide (dlg);
	  return;
	}
	
	desktop_integration ();
	  
      default:
	gtk_widget_hide (dlg);
	return;
      }
    }

    gtk_widget_hide (dlg);
  }
}
