/*
 * Copyright (C) 2001-2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: stream_info.c,v 1.6 2003/03/01 23:37:56 guenter Exp $
 *
 * stream_info/metadata display dialog implementation
 */

#include <config.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <stdio.h>

#include "globals.h"
#include "gtkxine.h"
#include "stream_info.h"

static GtkWidget    *dlg;
static int           is_visible;
static GtkListStore *meta_store;

static void add_meta_entry (const gchar *l, int info) {

  GtkTreeIter  iter;
  const char *str;

  str = gtk_xine_get_meta_info (GTK_XINE (gtx), info);
  if (!str)
    return;

  gtk_list_store_append (meta_store, &iter);

  gtk_list_store_set (meta_store, &iter, 0, l, 1, 
		      g_utf8_normalize (str, -1, G_NORMALIZE_DEFAULT), -1);
}

static void add_string_entry (const gchar *l, char *info) {

  GtkTreeIter  iter;

  gtk_list_store_append (meta_store, &iter);

  gtk_list_store_set (meta_store, &iter, 0, l, 1, 
		      g_utf8_normalize (info, -1, G_NORMALIZE_DEFAULT), -1);
}


void update_metadata (void) {

  char   str[80];
  int    channels, bits, samplerate, w, h, bps;
  double fps, ratio;

  gtk_list_store_clear (meta_store);

  add_meta_entry ("Title:", XINE_META_INFO_TITLE);
  add_meta_entry ("Artist:", XINE_META_INFO_ARTIST);
  add_meta_entry ("Comment:", XINE_META_INFO_COMMENT);
  add_meta_entry ("Date:",    XINE_META_INFO_YEAR);
  add_meta_entry ("Audio Codec:", XINE_META_INFO_AUDIOCODEC);

  channels   = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_AUDIO_CHANNELS);
  bits       = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_AUDIO_BITS);
  samplerate = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_AUDIO_SAMPLERATE);
  bps        = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_AUDIO_BITRATE) ;

  if (channels) {
    snprintf (str, 80, "%d channels, %d bit, %3.1fkHz, %d bps",
	      channels, bits, samplerate/1000.0, bps);

    add_string_entry ("Audio Format:", str);
  }


  add_meta_entry ("Video Codec:", XINE_META_INFO_VIDEOCODEC);

  w = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_VIDEO_WIDTH);
  h = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_VIDEO_HEIGHT);

  if (gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_FRAME_DURATION) != 0.0)
    fps = 90000.0 / gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_FRAME_DURATION);
  else
    fps = 0.0;

  ratio = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_VIDEO_RATIO) / 10000.0;
  bps   = gtk_xine_get_stream_info (GTK_XINE (gtx), XINE_STREAM_INFO_VIDEO_BITRATE) ;
  if (w) {
    snprintf (str, 80, "%dx%d, %2.1f fps, %4.2f:1, %d bps", w, h, fps, ratio, bps);

    add_string_entry ("Video Format:", str);
  }

  add_meta_entry ("System Layer:", XINE_META_INFO_SYSTEMLAYER);

}

void stream_info_show (void) {

  if (is_visible) {

    is_visible = FALSE;
    gtk_widget_hide (dlg);

  } else {

    update_metadata ();

    is_visible = TRUE;
    gtk_widget_show_all (dlg);
    gtk_widget_map (dlg);
  }
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  is_visible = FALSE;
  gtk_widget_unmap (dlg);

  return TRUE;
}

void stream_info_init (void) {

  GtkWidget            *tree_view, *button, *scrolled_window;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;

  /* create dialog */

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), "Stream Meta Info");
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 400);
  g_signal_connect (GTK_OBJECT (dlg), "delete_event",
		    G_CALLBACK (close_cb), NULL );

  button = gtk_dialog_add_button (GTK_DIALOG (dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(button), "clicked", 
		    G_CALLBACK(close_cb), 
		    NULL);
  
  /* add list to hold meta info */

  meta_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  
  tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL(meta_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("field",
						     cell,
						     "text", 0,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes ("data from stream",
						     cell,
						     "text", 1,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view),
			       GTK_TREE_VIEW_COLUMN (column));
  
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree_view);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  is_visible = FALSE;
}

