/*
 *  Copyright (C) 2002 Jorn Baayen <jorn@nl.linux.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtkwidget.h>
#include <gtk/gtktoolbar.h>
#include <string.h>

#include "galeon-favicon.h"
#include "galeon-shell.h"
#include "galeon-shell.h"
#include "pixbuf-cache.h"

static void galeon_favicon_class_init (GaleonFaviconClass *klass);
static void galeon_favicon_init (GaleonFavicon *ma);
static void galeon_favicon_finalize (GObject *object);
static void galeon_favicon_set_property (GObject *object,
					 guint prop_id,
					 const GValue *value,
					 GParamSpec *pspec);
static void galeon_favicon_get_property (GObject *object,
					 guint prop_id,
					 GValue *value,
					 GParamSpec *pspec);
static void galeon_favicon_update_image (GaleonFavicon *favicon);
static void cache_changed_cb (GaleonFaviconCache *cache,
		              const char *url,
		              GaleonFavicon *favicon);

struct GaleonFaviconPrivate
{
	GaleonFaviconCache *cache;
	gboolean isdefault;

	char *url;
	GdkPixbuf *default_pixbuf;
};

enum
{
	PROP_0,
	PROP_CACHE,
	PROP_URL
};

enum
{
	CHANGED,
	LAST_SIGNAL
};

static guint galeon_favicon_signals[LAST_SIGNAL] = { 0 };

static GObjectClass *parent_class = NULL;

GType
galeon_favicon_get_type (void)
{
	static GType galeon_favicon_type = 0;

	if (galeon_favicon_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (GaleonFaviconClass),
			NULL,
			NULL,
			(GClassInitFunc) galeon_favicon_class_init,
			NULL,
			NULL,
			sizeof (GaleonFavicon),
			0,
			(GInstanceInitFunc) galeon_favicon_init
		};

		galeon_favicon_type = g_type_register_static (GTK_TYPE_IMAGE,
							      "GaleonFavicon",
							      &our_info, 0);
	}

	return galeon_favicon_type;
}

static void
galeon_favicon_class_init (GaleonFaviconClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = galeon_favicon_finalize;

	object_class->set_property = galeon_favicon_set_property;
	object_class->get_property = galeon_favicon_get_property;

	g_object_class_install_property (object_class,
					 PROP_CACHE,
					 g_param_spec_object ("cache",
							      "Favicon cache",
							      "Favicon cache",
							      GALEON_TYPE_FAVICON_CACHE,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (object_class,
					 PROP_URL,
					 g_param_spec_string ("url",
							      "Associated URL",
							      "Associated URL",
							      NULL,
							      G_PARAM_READWRITE));

	galeon_favicon_signals[CHANGED] =
		g_signal_new ("changed",
			      G_OBJECT_CLASS_TYPE (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GaleonFaviconClass, changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE,
			      0);
}

static void
galeon_favicon_init (GaleonFavicon *ma)
{
	GtkWidget *dummy;

	ma->priv = g_new0 (GaleonFaviconPrivate, 1);
	ma->priv->isdefault = TRUE;

	dummy = gtk_toolbar_new ();
	g_object_ref (dummy);
	gtk_object_sink (GTK_OBJECT (dummy));

	ma->priv->default_pixbuf = gtk_widget_render_icon (dummy,
							   GALEON_STOCK_BOOKMARK,
							   GTK_ICON_SIZE_MENU, NULL);
	g_object_unref (dummy);

	gtk_widget_set_size_request (GTK_WIDGET (ma), 16, 16);
}

static void
galeon_favicon_finalize (GObject *object)
{
	GaleonFavicon *ma;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GALEON_IS_FAVICON (object));

	ma = GALEON_FAVICON (object);

	g_return_if_fail (ma->priv != NULL);

	g_free (ma->priv->url);

	if (ma->priv->default_pixbuf)
	{
		g_object_unref (G_OBJECT (ma->priv->default_pixbuf));
	}

	g_free (ma->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
galeon_favicon_set_property (GObject *object,
			     guint prop_id,
			     const GValue *value,
			     GParamSpec *pspec)
{
	GaleonFavicon *favicon = GALEON_FAVICON (object);

	switch (prop_id)
	{
	case PROP_CACHE:
		favicon->priv->cache = g_value_get_object (value);

		g_signal_connect_object (G_OBJECT (favicon->priv->cache),
				         "changed",
				         G_CALLBACK (cache_changed_cb),
				         favicon,
				         0);

		galeon_favicon_update_image (favicon);
		break;
	case PROP_URL:
		galeon_favicon_set_url (favicon, g_value_get_string (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void 
galeon_favicon_get_property (GObject *object,
			     guint prop_id,
			     GValue *value,
			     GParamSpec *pspec)
{
	GaleonFavicon *favicon = GALEON_FAVICON (object);

	switch (prop_id)
	{
	case PROP_CACHE:
		g_value_set_object (value, favicon->priv->cache);
		break;
	case PROP_URL:
		g_value_set_string (value, favicon->priv->url);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

GtkWidget *
galeon_favicon_new (const char *url)
{
	GaleonFavicon *favicon;
	GaleonFaviconCache *cache = galeon_shell_get_favicon_cache (galeon_shell);

	g_return_val_if_fail (GALEON_IS_FAVICON_CACHE (cache), NULL);

	favicon = GALEON_FAVICON (g_object_new (GALEON_TYPE_FAVICON,
						"cache", cache,
						"url", url,
						NULL));

	g_return_val_if_fail (favicon->priv != NULL, NULL);

	return GTK_WIDGET (favicon);
}

void
galeon_favicon_set_url (GaleonFavicon *favicon,
		        const char *url)
{
	g_return_if_fail (GALEON_IS_FAVICON (favicon));

	g_free (favicon->priv->url);
	favicon->priv->url = g_strdup (url);

	g_object_notify (G_OBJECT (favicon), "url");
	galeon_favicon_update_image (favicon);
}

char *
galeon_favicon_get_url (GaleonFavicon *favicon)
{
	g_return_val_if_fail (GALEON_IS_FAVICON (favicon), NULL);
	
	return g_strdup (favicon->priv->url);
}

gboolean
galeon_favicon_get_is_default (GaleonFavicon *favicon)
{
	return favicon->priv->isdefault;
}

static void
cache_changed_cb (GaleonFaviconCache *cache,
		  const char *url,
		  GaleonFavicon *favicon)
{
	if (galeon_favicon_cache_url_equal (url, favicon->priv->url))
		galeon_favicon_update_image (favicon);
}

static void
galeon_favicon_update_image (GaleonFavicon *favicon)
{
	GdkPixbuf *pixbuf;

	g_return_if_fail (GALEON_IS_FAVICON_CACHE (favicon->priv->cache));

	pixbuf = galeon_favicon_cache_lookup (favicon->priv->cache,
					      favicon->priv->url);

	favicon->priv->isdefault = (pixbuf == NULL);
	if (!pixbuf)
	{
		pixbuf = favicon->priv->default_pixbuf;
		favicon->priv->isdefault = TRUE;
	}
	else
	{
		favicon->priv->isdefault = FALSE;
	}

	gtk_image_set_from_pixbuf (GTK_IMAGE (favicon), pixbuf);

	g_signal_emit (G_OBJECT (favicon), galeon_favicon_signals[CHANGED], 0);
}	
