/*
 * sound/configure.c	- Configuration program for the Linux Sound Driver
 *
 *	(c)	Hannu Savolainen, 1993
 */

#include <stdio.h>

#define B(x)	(1 << (x))

/*
 *	Option numbers
 */

#define OPT_SB		0
#define OPT_PAS		1
#define OPT_ADLIB	2
#define OPT_MAXDEV	2

#define OPT_SBPRO	3
#define OPT_AUDIO	4
#define OPT_MIDI	5
#define OPT_YM3812_AUTO	6	/* Select this automaticly if user selects MIDI or AdLib driver */
#define OPT_YM3812	7	/* Select this if the previous one was not selected */
#define OPT_SEQUENCER	8
#define OPT_LAST	8

#define ANY_DEVS (B(OPT_AUDIO)|B(OPT_MIDI)|B(OPT_SEQUENCER))

typedef struct {
	    unsigned long conditions;
	    unsigned long exclusive_options;
	    char macro[20];
	    int verify;
	    int alias;
	} hw_entry;


/*
 *	The rule table for the driver options. The first field defines a set of options which
 *	must be selected before this entry can be selected. The second field is a set of options
 *	which are not allowed with this one. If the fourth field is zero, the option is selected
 *	without asking confirmation from the user.
 *
 *	With this version of the rule table it is possible to select just one type of hardware.
 *
 *	NOTE!	Keep the following table and the questions array in sync with the option numbering!
 */

hw_entry hw_table[] = {/* Select if one of these selected.      Don't select with these. Name	    Verify   Alias */
	/* 0 */		{0,					0,			"SB",		1,	0},
			{0,					B(OPT_SB),		"PAS",		1,	0},
			{0,					B(OPT_SB)|B(OPT_PAS),	"ADLIB",	1,	0},

			{B(OPT_SB),				0,			"SBPRO",	1,	0},
			{B(OPT_SB)|B(OPT_PAS),			0,			"AUDIO",	1,	0},
			{B(OPT_SB)|B(OPT_PAS),			0,			"MIDI",		1,	0},
			{B(OPT_ADLIB)|B(OPT_MIDI),		0,			"YM3812_AUTO",	0,	OPT_YM3812},
			{B(OPT_SB)|B(OPT_PAS)|B(OPT_ADLIB),	B(OPT_YM3812_AUTO),	"YM3812",	1,	0},
	/* 8 */		{B(OPT_MIDI)|B(OPT_YM3812)|B(OPT_YM3812_AUTO),	0,		"SEQUENCER",	0,	0}
		};

char *questions[] = {
		"SoundBlaster support",
		"ProAudioSpectrum 16 support",
		"AdLib support",

		"SoundBlaster Pro support (mixer)",
		"digitized voice support",
		"MIDI interface support",
		"This should not be asked",
		"2 operator FM synthesizer support",
		"/dev/sequencer support"
	};

unsigned long selected_options = 0;

int can_select_option(int nr)
{
	switch (nr)
	{
	   case 0:
   		fprintf(stderr,"\nSelect one of the following soundcards:\n");
   		fprintf(stderr,"	- SoundBlaster / SB Pro\n");
   		fprintf(stderr,"	- ProAudioSpectrum 16\n");
   		fprintf(stderr,"	- AdLib\n");
   		break;
	
	   case OPT_MAXDEV+1:
   		fprintf(stderr,"\nSelect one or more of the following options\n");
   		break;
	}

	if (hw_table[nr].conditions)
	   if (!(hw_table[nr].conditions & selected_options)) return 0;

	if (hw_table[nr].exclusive_options)
	   if (hw_table[nr].exclusive_options & selected_options) return 0;

	return 1;
}

int main()
{
	int i, num, def_size;
	char answ[10];

	printf("/*\tGenerated by configure. Don't edit by hand!!!!\t*/\n\n");

	fprintf(stderr, "\nConfiguring the sound support\n\n"
			"Note!	Selecting the partial driver may lead to combination\n"
			"	of features which don't compile. Select the partial driver\n"
			"	only if you have lack of memory!\n\n");

	fprintf(stderr, "Do you want to include full version of the sound driver (y/n) ? ");
	scanf("%s", answ);

	if (!strcmp(answ, "y") || !strcmp(answ, "Y"))
	   selected_options = 0xffffffff; else
	{
		/* Partial driver */
	
		for (i=0;i<=OPT_LAST;i++)
		if (can_select_option(i))
		{
			if (!(selected_options & B(i)))	/* Not selected yet */
			if (!hw_table[i].verify)
			{
				if (hw_table[i].alias)
				   selected_options |= B(hw_table[i].alias);
				else
				   selected_options |= B(i);
			} else {
				fprintf(stderr, "  %s (y/n) ? ", questions[i]);
				scanf("%s", answ);
		
				if (!strcmp(answ, "y") || !strcmp(answ, "Y"))
				   if (hw_table[i].alias)
				      selected_options |= B(hw_table[i].alias);
				   else
				      selected_options |= B(i);
			}
		}
	}	
	
	if (!(selected_options & ANY_DEVS))
	{
		printf("#undef CONFIGURE_SOUNDCARD\n");
		printf("#undef KERNEL_SOUNDCARD\n");
		fprintf(stderr, "\n*** This combination is useless. Sound driver disabled!!! ***\n\n");
		exit(0);
	} else printf("#define KERNEL_SOUNDCARD\n");

	for (i=0;i<=OPT_LAST;i++)
	if (!hw_table[i].alias)
	if (selected_options & B(i))
	   printf("#undef  EXCLUDE_%s\n", hw_table[i].macro);
	else
	   printf("#define EXCLUDE_%s\n", hw_table[i].macro);

/*
 * IRQ and DMA settings
 */
 	printf("\n");

 	if (selected_options & B(OPT_SB) && selected_options & (B(OPT_AUDIO)|B(OPT_MIDI)) )
 	{
 		fprintf(stderr,	"\nIRQ number for SoundBlaster?\n"
 				"The IRQ adress is defined by the jumpers on your card and\n"
 				"7 is the factory default. Valid values are 2, 5, 7 and 10.\n"
 				"Enter the value: ");

 		scanf("%d", &num);
 		if (num != 2 && num != 5 && num != 7 && num != 10)
 		{

 			fprintf(stderr, "*** Illegal input! ***\n");
 			num = 7;
 		}
 		fprintf(stderr,"SoundBlaster IRQ set to %d\n", num);
 		printf("#define SBC_IRQ %d\n", num);
 	}

 	if (selected_options & B(OPT_PAS))
 	{
 	   if (selected_options & (B(OPT_AUDIO) | B(OPT_MIDI)) )
 	   {
 		fprintf(stderr,	"\nIRQ number for ProAudioSpectrum?\n"
				"The recommended value is the IRQ used under DOS.\n"
				"Please refer to the ProAudioSpectrum User's Guide.\n"
				"The default value is 7.\n"
 				"Enter the value: ");

 		scanf("%d", &num);
 		if (num == 6 || num < 3 || num > 15)	/* Used for floppy */
 		{

 			fprintf(stderr, "*** Illegal input! ***\n");
 			num = 7;
 		}
 		fprintf(stderr,"ProAudioSpectrum IRQ set to %d\n", num);
 		printf("#define PAS_IRQ %d\n", num);
 	   }

	   if (selected_options & B(OPT_AUDIO))
	   {
 		fprintf(stderr,	"\nDMA number for ProAudioSpectrum?\n"
				"The recommended value is the DMA channel under DOS.\n"
				"Please refer to the ProAudioSpectrum User's Guide.\n"
				"The default value is 3\n"
 				"Enter the value: ");

 		scanf("%d", &num);
 		if (num == 4 || num < 0 || num > 7)
 		{

 			fprintf(stderr, "*** Illegal input! ***\n");
 			num = 3;
 		}
 		fprintf(stderr,"\nProAudioSpectrum DMA set to %d\n", num);
 		printf("#define PAS_DMA %d\n", num);
 	   }
 	}

 	if (selected_options & B(OPT_AUDIO))
 	{
 		def_size = 16384;

 		if (selected_options & B(OPT_SBPRO)) def_size = 32768;
 		if ((selected_options & B(OPT_PAS)) && !(selected_options & B(OPT_SB))) def_size = 65536;

		fprintf(stderr,	"\nSelect the DMA buffer size (4096, 16384, 32768 or 65536 bytes)\n"
				"%d is recommended value for this configuration.\n"
				"Enter the value: ", def_size);

 		scanf("%d", &num);
 		if (num != 4096 && num != 16384 && num != 32768 && num != 65536)
 		{

 			fprintf(stderr, "*** Illegal input! ***\n");
 			num = def_size;
 		}
 		fprintf(stderr, "The DMA buffer size set to %d\n", num);
 		printf("#define DSP_BUFFSIZE %d\n", num);
 	}

 	fprintf(stderr,	"The sound driver is now configured.\n");

	exit(0);
}
